/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is a test script.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgtestmacro.h"
#include "skgimportexportmanager.h"
#include "skgbankincludes.h"

/**
 * The main function of the unit test
 * @param argc the number of arguments
 * @param argv the list of arguments
 */
int main(int argc, char** argv)
{
    Q_UNUSED(argc);
    Q_UNUSED(argv);

    // Init test
    SKGINITTEST(true);

    QDate now = QDate::currentDate();

    {
        // Test import SKGImportExportManager::CSV skrooge
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);

        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT_CSV", err);
            SKGImportExportManager impmissing(&document1, QUrl("missingfile.csv"));
            impmissing.setAutomaticApplyRules(true);
            SKGTESTERROR("imp1.importFile", impmissing.importFile(), false);

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/skrooge_partial.csv"));
            imp1.getImportMimeTypeFilter();
            imp1.getExportMimeTypeFilter();
            imp1.setCodec("");
            SKGTESTERROR("imp1.importFile", imp1.importFile(), true);
        }
        SKGAccountObject account;
        SKGTESTERROR("ACCOUNT.getObjectByName", SKGNamedObject::getObjectByName(&document1, "v_account", "Courant steph", account), true);
        SKGTESTERROR("ACCOUNT.load", account.load(), true);
        SKGTEST("ACCOUNT:getCurrentAmount", SKGServices::doubleToString(account.getCurrentAmount()), "-935");
    }

    {
        // Test import QIF 1
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGBankObject bank(&document1);
        SKGAccountObject account;
        SKGUnitObject unit_euro(&document1);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT_INIT", err);

            // Creation bank
            SKGTESTERROR("BANK:setName", bank.setName("CREDIT COOP"), true);
            SKGTESTERROR("BANK:save", bank.save(), true);

            // Creation account
            SKGTESTERROR("BANK:addAccount", bank.addAccount(account), true);
            SKGTESTERROR("ACCOUNT:setName", account.setName("Courant steph"), true);
            SKGTESTERROR("ACCOUNT:setNumber", account.setNumber("12345P"), true);
            SKGTESTERROR("ACCOUNT:save", account.save(), true);

            // Creation unit
            SKGTESTERROR("UNIT:setName", unit_euro.setName("euro"), true);
            SKGTESTERROR("UNIT:save", unit_euro.save(), true);

            // Creation unitvalue
            SKGUnitValueObject unit_euro_val1;
            SKGTESTERROR("UNIT:addUnitValue", unit_euro.addUnitValue(unit_euro_val1), true);
            SKGTESTERROR("UNITVALUE:setQuantity", unit_euro_val1.setQuantity(1), true);
            SKGTESTERROR("UNITVALUE:setDate", unit_euro_val1.setDate(now), true);
            SKGTESTERROR("UNITVALUE:save", unit_euro_val1.save(), true);
        }

        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT_CSV", err);
            // Date;Libelle;Libelle complementaire;Montant;Sens;Numero de cheque
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/coopanet.csv"));
            QMap<QString, QString> parameters = imp1.getImportParameters();
            parameters["automatic_search_header"] = 'N';
            parameters["columns_positions"] = "date|comment||amount|sign|number";
            imp1.setImportParameters(parameters);
            SKGTESTERROR("QIF.setDefaultAccount", imp1.setDefaultAccount(&account), true);
            SKGTESTERROR("imp1.importFile", imp1.importFile(), true);
        }
        SKGTESTERROR("ACCOUNT.load", account.load(), true);
        SKGTEST("ACCOUNT:getCurrentAmount", SKGServices::doubleToString(account.getCurrentAmount()), "-680.28");
    }

    {
        // Test import bankperfect
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGBankObject bank(&document1);
        SKGAccountObject account1;
        SKGAccountObject account2;
        SKGUnitObject unit_euro(&document1);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT_INIT", err);

            // Creation bank
            SKGTESTERROR("BANK:setName", bank.setName("CREDIT COOP"), true);
            SKGTESTERROR("BANK:save", bank.save(), true);

            // Creation account
            SKGTESTERROR("BANK:addAccount", bank.addAccount(account1), true);
            SKGTESTERROR("ACCOUNT:setName", account1.setName("Courant steph"), true);
            SKGTESTERROR("ACCOUNT:setNumber", account1.setNumber("12345P"), true);
            SKGTESTERROR("ACCOUNT:save", account1.save(), true);

            SKGTESTERROR("BANK:addAccount", bank.addAccount(account2), true);
            SKGTESTERROR("ACCOUNT:setName", account2.setName("PEL"), true);
            SKGTESTERROR("ACCOUNT:save", account2.save(), true);

            // Creation unit
            SKGTESTERROR("UNIT:setName", unit_euro.setName("euro"), true);
            SKGTESTERROR("UNIT:save", unit_euro.save(), true);

            // Creation unitvalue
            SKGUnitValueObject unit_euro_val1;
            SKGTESTERROR("UNIT:addUnitValue", unit_euro.addUnitValue(unit_euro_val1), true);
            SKGTESTERROR("UNITVALUE:setQuantity", unit_euro_val1.setQuantity(1), true);
            SKGTESTERROR("UNITVALUE:setDate", unit_euro_val1.setDate(now), true);
            SKGTESTERROR("UNITVALUE:save", unit_euro_val1.save(), true);
        }

        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT_BP_CSV", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/bankperfect.csv"));
            SKGTESTERROR("QIF.setDefaultAccount", imp1.setDefaultAccount(&account1), true);
            SKGTESTERROR("imp1.importFile", imp1.importFile(), true);
        }

        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT_BP_CSV", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/bankperfect2.csv"));
            SKGTESTERROR("QIF.setDefaultAccount", imp1.setDefaultAccount(&account2), true);
            SKGTESTERROR("imp1.importFile", imp1.importFile(), true);

            int NbOperationsMerged = 0;
            SKGTESTERROR("imp1.findAndGroupTransfers", imp1.findAndGroupTransfers(NbOperationsMerged), true);
            SKGTEST("imp1:NbOperationsMerged", NbOperationsMerged, 6);
        }
        SKGTESTERROR("ACCOUNT.load", account1.load(), true);
        SKGTEST("ACCOUNT:getCurrentAmount", SKGServices::doubleToString(account1.getCurrentAmount()), "2624.071111");

        SKGTESTERROR("ACCOUNT.load", account2.load(), true);
        SKGTEST("ACCOUNT:getCurrentAmount", SKGServices::doubleToString(account2.getCurrentAmount()), "1500");

        SKGImportExportManager exp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("OUT") % "/skgtestimportcsv/bankperfect.csv"));
        SKGTESTERROR("SKGImportExportManager::CSV.exportFile", exp1.exportFile(), true);

        {
            SKGStringListList oTable;
            SKGTESTERROR("SKGImportExportManager::CSV.exportFile", document1.getConsolidatedView("v_operation_display", "d_DATEMONTH", "t_CATEGORY", "f_CURRENTAMOUNT", "SUM", "", oTable), true);
            QStringList dump = SKGServices::tableToDump(oTable, SKGServices::DUMP_TEXT);
            int nbl = dump.count();
            for (int i = 0; i < nbl; ++i) {
                SKGTRACE << dump[i] << endl;
            }
        }
        {
            SKGStringListList oTable;
            SKGTESTERROR("SKGImportExportManager::CSV.exportFile", document1.getConsolidatedView("v_operation_display", "", "t_CATEGORY", "f_CURRENTAMOUNT", "SUM", "", oTable), true);
            QStringList dump = SKGServices::tableToDump(oTable, SKGServices::DUMP_TEXT);
            int nbl = dump.count();
            for (int i = 0; i < nbl; ++i) {
                SKGTRACE << dump[i] << endl;
            }
        }
        {
            SKGStringListList oTable;
            SKGTESTERROR("SKGImportExportManager::CSV.exportFile", document1.getConsolidatedView("v_operation_display", "d_DATEWEEK", "", "f_CURRENTAMOUNT", "SUM", "", oTable), true);
            SKGTESTERROR("SKGImportExportManager::CSV.exportFile", document1.getConsolidatedView("v_operation_display", "d_DATEQUARTER", "", "f_CURRENTAMOUNT", "SUM", "", oTable), true);
            SKGTESTERROR("SKGImportExportManager::CSV.exportFile", document1.getConsolidatedView("v_operation_display", "d_DATESEMESTER", "", "f_CURRENTAMOUNT", "SUM", "", oTable), true);
            SKGTESTERROR("SKGImportExportManager::CSV.exportFile", document1.getConsolidatedView("v_operation_display", "d_DATEYEAR", "", "f_CURRENTAMOUNT", "SUM", "", oTable), true);
            SKGTESTERROR("SKGImportExportManager::CSV.exportFile", document1.getConsolidatedView("v_operation_display", "d_date", "", "f_CURRENTAMOUNT", "SUM", "", oTable), true);
            SKGTESTERROR("SKGImportExportManager::CSV.exportFile", document1.getConsolidatedView("v_operation_display", "d_DATEMONTH", "", "f_CURRENTAMOUNT", "SUM", "", oTable), true);
            QStringList dump = SKGServices::tableToDump(oTable, SKGServices::DUMP_TEXT);
            int nbl = dump.count();
            for (int i = 0; i < nbl; ++i) {
                SKGTRACE << dump[i] << endl;
            }
        }
    }

    {
        // Test import skrooge
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT_SKROOGE_CSV", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("OUT") % "/skgtestimportcsv/bankperfect.csv"));
            SKGTESTERROR("imp1.importFile", imp1.importFile(), true);
        }

        SKGImportExportManager exp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("OUT") % "/skgtestimportcsv/bankperfect2.csv"));
        SKGTESTERROR("SKGImportExportManager::CSV.exportFile", exp1.exportFile(), true);
    }

    {
        // Test import skrooge+optimization
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT_OPTIM", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/bankperfect.csv"));
            SKGTESTERROR("OPTIM.importFile", imp1.importFile(), true);
        }

        SKGTESTERROR("OPTIM.undoRedoTransaction", document1.undoRedoTransaction(SKGDocument::UNDOLASTSAVE), true);
    }

    {
        // Test import skrooge in double to check merge
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/remi_1.csv"));
            SKGTESTERROR("OPTIM.importFile", imp1.importFile(), true);
        }
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/remi_1.csv"));
            SKGTESTERROR("OPTIM.importFile", imp1.importFile(), true);
        }

        SKGAccountObject account(&document1);
        SKGTESTERROR("ACCOUNT.load", account.setName("remi 1"), true);
        SKGTESTERROR("ACCOUNT.load", account.load(), true);
        SKGTEST("ACCOUNT:getCurrentAmount", SKGServices::doubleToString(account.getCurrentAmount()), "-767.26");
    }
    {
        // Test import 2638120
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/XXXXXXMxxxXXXXXXX.csv"));
            SKGTESTERROR("CSV.importFile", imp1.importFile(), true);
        }

        SKGAccountObject account(&document1);
        SKGTESTERROR("ACCOUNT.load", account.setName("XXXXXXMxxxXXXXXXX"), true);
        SKGTESTERROR("ACCOUNT.load", account.load(), true);
        SKGTEST("ACCOUNT:getCurrentAmount", SKGServices::doubleToString(account.getCurrentAmount()), "8114.26");
    }
    {
        // Test import 206894
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/206894.csv"));
            SKGTESTERROR("CSV.importFile", imp1.importFile(), true);
        }

        SKGAccountObject account(&document1);
        SKGTESTERROR("ACCOUNT.load", account.setName("206894"), true);
        SKGTESTERROR("ACCOUNT.load", account.load(), true);
        SKGTEST("ACCOUNT:getCurrentAmount", SKGServices::doubleToString(account.getCurrentAmount()), "-2986.39");
    }

    {
        // Test import with tabulation
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/mutual fund.csv"));
            SKGTESTERROR("CSV.importFile", imp1.importFile(), true);
        }


        SKGAccountObject account(&document1);
        SKGTESTERROR("ACCOUNT.load", account.setName("Janus Twenty Fund"), true);
        SKGTESTERROR("ACCOUNT.load", account.load(), true);
        SKGTEST("ACCOUNT:getCurrentAmount", SKGServices::doubleToString(account.getCurrentAmount()), "24.51428572");
    }

    {
        // Test import shares with original amount
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/209705.csv"));
            SKGTESTERROR("CSV.importFile", imp1.importFile(), true);
        }
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/209705_2.csv"));
            SKGTESTERROR("CSV.importFile", imp1.importFile(), true);
        }
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT", err);
            SKGImportExportManager imp1(&document1);

            int out = 0;
            SKGTESTERROR("CSV.findAndGroupTransfers", imp1.findAndGroupTransfers(out), true);
            SKGTEST("CSV:nb", out, 2);
        }

        SKGObjectBase::SKGListSKGObjectBase grouped;
        SKGTESTERROR("CSV.getObjects", document1.getObjects("operation", "i_group_id!=0", grouped), true);
        SKGTEST("CSV:grouped.count", grouped.count(), 2);
    }

    {
        // Test import transactions splited and grouped
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/compte.csv"));
            SKGTESTERROR("CSV.importFile", imp1.importFile(), true);
        }

        SKGObjectBase::SKGListSKGObjectBase grouped;
        SKGTESTERROR("CSV.getObjects", document1.getObjects("operation", "", grouped), true);
        SKGTEST("CSV:grouped.count", grouped.count(), 4);
        SKGTESTERROR("CSV.getObjects", document1.getObjects("operation", "i_group_id!=0", grouped), true);
        SKGTEST("CSV:grouped.count", grouped.count(), 2);
    }

    {
        // Test import mmex
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/mmex.csv"));
            SKGTESTERROR("CSV.importFile", imp1.importFile(), true);
        }

        int nb = 0;
        SKGTESTERROR("CATEGORY.getNbObjects", document1.getNbObjects("category", "t_fullname='Alimentation > Restaurant'", nb), true);
        SKGTEST("CATEGORY:nb", nb, 1);

        SKGAccountObject account(&document1);
        SKGTESTERROR("ACCOUNT.load", account.setName("mmex"), true);
        SKGTESTERROR("ACCOUNT.load", account.load(), true);
        SKGTEST("ACCOUNT:getCurrentAmount", SKGServices::doubleToString(account.getCurrentAmount()), "1418.44");
    }
    {
        // Test import mmex
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/mmex_no_header.csv"));
            QMap<QString, QString> parameters = imp1.getImportParameters();
            parameters["automatic_search_columns"] = 'N';
            parameters["columns_positions"] = "date|comment|sign|amount|category";
            parameters["automatic_search_header"] = 'N';
            parameters["header_position"] = '0';
            imp1.setImportParameters(parameters);
            SKGTESTERROR("CSV.importFile", imp1.importFile(), true);
        }


        SKGAccountObject account(&document1);
        SKGTESTERROR("ACCOUNT.load", account.setName("mmex no header"), true);
        SKGTESTERROR("ACCOUNT.load", account.load(), true);
        SKGTEST("ACCOUNT:getCurrentAmount", SKGServices::doubleToString(account.getCurrentAmount()), "1418.44");
    }
    {
        // 263263
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/263263.csv"));
            QMap<QString, QString> parameters = imp1.getImportParameters();
            parameters["mapping_debit"] = "kreditrente";
            parameters["automatic_search_columns"] = 'N';
            parameters["columns_positions"] = "date||number|sign|comment|amount|amount";
            parameters["automatic_search_header"] = 'N';
            parameters["header_position"] = '1';
            imp1.setImportParameters(parameters);
            SKGTESTERROR("CSV.importFile", imp1.importFile(), true);
        }

        SKGAccountObject account(&document1);
        SKGTESTERROR("ACCOUNT.load", account.setName("263263"), true);
        SKGTESTERROR("ACCOUNT.load", account.load(), true);
        SKGTEST("ACCOUNT:getCurrentAmount", SKGServices::doubleToString(account.getCurrentAmount()), "800.09");
    }
    {
        // CREDIT-DEBIT
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/credit-debit.csv"));
            SKGTESTERROR("CSV.importFile", imp1.importFile(), true);
        }

        SKGAccountObject account(&document1);
        SKGTESTERROR("ACCOUNT.load", account.setName("credit debit"), true);
        SKGTESTERROR("ACCOUNT.load", account.load(), true);
        SKGTEST("ACCOUNT:getCurrentAmount", SKGServices::doubleToString(account.getCurrentAmount()), "1500");
    }
    {
        // BACKSLASHES
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/backslashes.csv"));
            SKGTESTERROR("CSV.importFile", imp1.importFile(), true);
        }

        SKGAccountObject account(&document1);
        SKGTESTERROR("ACCOUNT.load", account.setName("backslashes"), true);
        SKGTESTERROR("ACCOUNT.load", account.load(), true);
        SKGTEST("ACCOUNT:getCurrentAmount", SKGServices::doubleToString(account.getCurrentAmount()), "1000");
    }
    {
        // MULTILINE
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/multiline.csv"));
            SKGTESTERROR("CSV.importFile", imp1.importFile(), true);
        }

        SKGAccountObject account(&document1);
        SKGTESTERROR("ACCOUNT.load", account.setName("multiline"), true);
        SKGTESTERROR("ACCOUNT.load", account.load(), true);
        SKGTEST("ACCOUNT:getCurrentAmount", SKGServices::doubleToString(account.getCurrentAmount()), "3000");
    }

    {
        // MULTILINE
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/non_numerical_amount.csv"));
            SKGTESTERROR("CSV.importFile", imp1.importFile(), true);
        }

        SKGAccountObject account(&document1);
        SKGTESTERROR("ACCOUNT.load", account.setName("non numerical amount"), true);
        SKGTESTERROR("ACCOUNT.load", account.load(), true);
        SKGTEST("ACCOUNT:getCurrentAmount", SKGServices::doubleToString(account.getCurrentAmount()), "-119.56");
    }

    {
        // Test import 320112
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/320112.csv"));
            QMap<QString, QString> parameters = imp1.getImportParameters();
            parameters["automatic_search_columns"] = 'N';
            parameters["columns_positions"] = "date|mode|payee|comment|amount";
            imp1.setImportParameters(parameters);
            SKGTESTERROR("CSV.importFile", imp1.importFile(), true);
        }

        SKGAccountObject account(&document1);
        SKGTESTERROR("ACCOUNT.load", account.setName("320112"), true);
        SKGTESTERROR("ACCOUNT.load", account.load(), true);
        SKGTEST("ACCOUNT:getCurrentAmount", SKGServices::doubleToString(account.getCurrentAmount()), "6.13");
    }

    {
        // Test import date DDMMMYYYY
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/date_DDMMMYYYY.csv"));
            SKGTESTERROR("CSV.importFile", imp1.importFile(), true);
        }

        document1.dump(DUMPOPERATION | DUMPACCOUNT);
        SKGAccountObject account(&document1);
        SKGTESTERROR("ACCOUNT.load", account.setName("date DDMMMYYYY"), true);
        SKGTESTERROR("ACCOUNT.load", account.load(), true);
        SKGTEST("ACCOUNT:getAmount", SKGServices::doubleToString(account.getAmount(QDate(2004, 12, 31))), "35");
    }

    {
        // Test import separator tab
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/tabs.csv"));
            SKGTESTERROR("CSV.importFile", imp1.importFile(), true);
        }
    }

    {
        // Test import separator comma
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/commas.csv"));
            SKGTESTERROR("CSV.importFile", imp1.importFile(), true);
        }
    }

    {
        // Test import separator comma
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT", err);
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportcsv/349961.csv"));
            SKGTESTERROR("CSV.importFile", imp1.importFile(), true);
        }
        int nb = 0;
        SKGTESTERROR("CSV.getObjects", document1.getNbObjects("operation", "", nb), true);
        SKGTEST("CSV:nb operations", nb, 1);
        SKGTESTERROR("CSV.getObjects", document1.getNbObjects("suboperation", "d_date='2015-07-07'", nb), true);
        SKGTEST("CSV:nb suboperations 2015-07-07", nb, 1);
        SKGTESTERROR("CSV.getObjects", document1.getNbObjects("suboperation", "d_date='2015-07-08'", nb), true);
        SKGTEST("CSV:nb suboperations 2015-07-08", nb, 1);
        SKGTESTERROR("CSV.getObjects", document1.getNbObjects("suboperation", "d_date='2015-07-09'", nb), true);
        SKGTEST("CSV:nb suboperations 2015-07-09", nb, 1);
    }
    // End test
    SKGENDTEST();
}
