/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for operation management.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgoperationpluginwidget.h"

#include <KStandardDirs>
#include <KStandardAction>
#include <KMessageBox>
#include <KInputDialog>
#include <KColorScheme>

#include <QtGui/QHeaderView>
#include <QDomDocument>
#include <QDir>
#include <QMap>
#include <QtCore/QFile>
#include <QTableWidget>
#include <QScriptEngine>

#include "skgmainpanel.h"
#include "skgobjectmodel.h"
#include "skgobjectbase.h"
#include "skgbankincludes.h"
#include "skgtraces.h"
#include "skgservices.h"
#include "skgsplittabledelegate.h"
#include "skgpayeeobject.h"
#include "skgoperation_settings.h"
#include "skgshow.h"
#include "skgcalculatoredit.h"
#include "skgtreeview.h"

SKGOperationPluginWidget::SKGOperationPluginWidget(SKGDocumentBank* iDocument)
    : SKGTabPage(iDocument), m_lastFastEditionOperationFound(0), m_showClosedAccounts(false),
      m_numberFieldIsNotUptodate(true), m_modeInfoZone(0)
{
    SKGTRACEINFUNC(1);
    if (!iDocument) {
        return;
    }

    m_timer.setSingleShot(true);
    connect(&m_timer, SIGNAL(timeout()), this, SLOT(onRefreshInformationZone()), Qt::QueuedConnection);

    ui.setupUi(this);

    ui.kAccountLabel2->setText(i18n("%1:", iDocument->getDisplay("t_ACCOUNT")));
    ui.kDateLabel->setText(i18n("%1:", iDocument->getDisplay("d_date")));
    ui.kAmountLabel->setText(i18n("%1:", iDocument->getDisplay("f_CURRENTAMOUNT")));
    ui.kPayeeLabel->setText(i18n("%1:", iDocument->getDisplay("t_payee")));
    ui.kTypeLabel->setText(i18n("%1:", iDocument->getDisplay("t_mode")));
    ui.kNumberEdit->setClickMessage(iDocument->getDisplay("i_number"));
    ui.kCategoryLabel->setText(i18n("%1:", iDocument->getDisplay("t_CATEGORY")));
    ui.kCommentLabel->setText(i18n("%1:", iDocument->getDisplay("t_comment")));
    ui.kTrackerLabel->setText(i18n("%1:", iDocument->getDisplay("t_REFUND")));

    ui.kUnitEdit->setDocument(iDocument);

    ui.kTitle->hide();
    ui.kResetInternalFilter->hide();
    ui.kReconciliatorFrame2->hide();
    ui.kReconciliateAccount->hide();

    m_attributesForSplit  << "d_date" << "t_category" << "f_value" << "t_comment"  << "t_refund";
    int nb = m_attributesForSplit.count();
    ui.kSubOperationsTable->setColumnCount(nb);
    for (int i = 0; iDocument && i < nb; ++i) {
        QString att = m_attributesForSplit.at(i);
        QTableWidgetItem* item = new QTableWidgetItem(iDocument->getIcon(att), iDocument->getDisplay(att));
        ui.kSubOperationsTable->setHorizontalHeaderItem(i, item);
    }

    {
        // Bind operation view
        m_objectModel = new SKGObjectModel(static_cast<SKGDocumentBank*>(getDocument()), "v_operation_display_all", "1=0", this, "", false);
        ui.kOperationView->setModel(m_objectModel);

        // Add registered global action in contextual menu
        if (SKGMainPanel::getMainPanel()) {
            m_fastEditionAction = SKGMainPanel::getMainPanel()->getGlobalAction("fast_edition");
            ui.kOperationView->getView()->insertGlobalAction();
            ui.kOperationView->getView()->insertGlobalAction("open");
            ui.kOperationView->getView()->insertGlobalAction("open_report");
            ui.kOperationView->getView()->insertGlobalAction("edit_find");
            ui.kOperationView->getView()->insertGlobalAction();
            ui.kOperationView->getView()->insertGlobalAction("edit_delete");
            ui.kOperationView->getView()->insertGlobalAction();
            ui.kOperationView->getView()->insertGlobalAction("edit_switch_highlight");
            ui.kOperationView->getView()->insertGlobalAction("edit_point_selected_operation");
            ui.kOperationView->getView()->insertGlobalAction("edit_group_operation");
            ui.kOperationView->getView()->insertGlobalAction("edit_ungroup_operation");
            ui.kOperationView->getView()->insertGlobalAction("validate_imported_operation");
            ui.kOperationView->getView()->insertGlobalAction("merge_imported_operation");
            ui.kOperationView->getView()->insertGlobalAction("merge_sub_operations");
            ui.kOperationView->getView()->insertGlobalAction();
            ui.kOperationView->getView()->insertGlobalAction("edit_duplicate_operation");
            ui.kOperationView->getView()->insertGlobalAction("edit_template_operation");
            ui.kOperationView->getView()->insertGlobalAction("edit_apply_template");
            ui.kOperationView->getView()->insertGlobalAction("schedule_operation");
            ui.kOperationView->getView()->insertGlobalAction("add_property");
        }

        connect(ui.kOperationView->getView(), SIGNAL(clickEmptyArea()), this, SLOT(cleanEditor()));
        connect(ui.kOperationView->getView(), SIGNAL(doubleClicked(QModelIndex)), this, SLOT(onDoubleClick()));
        connect(ui.kOperationView->getView(), SIGNAL(selectionChangedDelayed()), this, SLOT(onSelectionChanged()));
    }

    // Add Standard KDE Icons to buttons to Operations
    ui.kModifyOperationBtn->setIcon(KIcon("dialog-ok-apply"));
    ui.kAddOperationBtn->setIcon(KIcon("list-add"));
    ui.kCleanBtn->setIcon(KIcon("edit-clear"));
    ui.kResetInternalFilter->setIcon(KIcon("dialog-cancel"));
    ui.kReconciliatorButton->setIcon(KIcon("object-rotate-left"));
    ui.kValidate->setIcon(KIcon("dialog-ok-apply"));
    ui.kValidate->setIconSize(QSize(48, 48));
    ui.kAutoPoint->setIcon(KIcon("games-solve"));
    ui.kCreateFakeOperation->setIcon(KIcon("list-add"));
    ui.kFastEditBtn->setIcon(KIcon("games-solve"));

    {
        SKGWidgetSelector::SKGListQWidget list;
        list.push_back(ui.SKGBasicSection);
        list.push_back(ui.SKGPayeeModeSection);
        list.push_back(ui.SKGSmallButtons);
        list.push_back(ui.SKGEditionButtonsWidget);
        list.push_back(ui.SKGSingleOpSection);
        ui.kWidgetSelector->addButton(KIcon("dialog-ok-apply"), i18n("Standard"), i18n("Display the edit panel for standard operations"), list);
    }

    {
        SKGWidgetSelector::SKGListQWidget list;
        list.push_back(ui.SKGBasicSection);
        list.push_back(ui.SKGPayeeModeSection);
        list.push_back(ui.SKGSmallButtons);
        list.push_back(ui.SKGEditionButtonsWidget);
        list.push_back(ui.SKGSplitOpSection);
        ui.kWidgetSelector->addButton(KIcon("split"), i18n("Split"), i18n("Display the edit panel for split operations"), list);
    }
    {
        SKGWidgetSelector::SKGListQWidget list;
        list.push_back(ui.SKGBasicSection);
        list.push_back(ui.SKGPayeeModeSection);
        list.push_back(ui.SKGSmallButtons);
        list.push_back(ui.SKGEditionButtonsWidget);
        list.push_back(ui.SKGSingleOpSection);
        list.push_back(ui.kTargetAccountEdit);
        list.push_back(ui.kTargetAccountLabel);
        ui.kWidgetSelector->addButton(KIcon("view-financial-transfer"), i18n("Transfer"), i18n("Display the edit panel for transfers between accounts"), list);
    }
    {
        SKGWidgetSelector::SKGListQWidget list;
        list.push_back(ui.SKGBasicSection);
        list.push_back(ui.SKGPayeeModeSection);
        list.push_back(ui.SKGSmallButtons);
        list.push_back(ui.SKGEditionButtonsWidget);
        list.push_back(ui.SKGSharesSection);
        ui.kWidgetSelector->addButton(KIcon("view-stock-account"), i18n("Shares"), i18n("Display the edit panel for purchasing or selling shares"), list);
    }
    connect(ui.kWidgetSelector, SIGNAL(selectedModeChanged(int)), this, SLOT(onBtnModeClicked(int)));

    ui.kFreezeBtn->setIcon(KIcon("skrooge_freeze"));

    // Fast edition
    connect(qApp, SIGNAL(focusChanged(QWidget*,QWidget*)), this, SLOT(onFocusChanged()));
    connect(m_fastEditionAction, SIGNAL(triggered(bool)), this, SLOT(onFastEdition()));

    // SubOperations
    connect(ui.kAmountEdit, SIGNAL(textChanged(QString)), this, SLOT(onQuantityChanged()));
    connect(ui.kDateEdit, SIGNAL(dateChanged(QDate)), this, SLOT(onDateChanged(QDate)));
    connect(ui.kSubOperationsTable, SIGNAL(cellChanged(int,int)), this, SLOT(onSubopCellChanged(int,int)));
    connect(ui.kSubOperationsTable->verticalHeader(), SIGNAL(sectionClicked(int)), this, SLOT(onRemoveSubOperation(int)));

    ui.kSubOperationsTable->horizontalHeader()->setResizeMode(QHeaderView::Interactive);
    ui.kSubOperationsTable->verticalHeader()->setResizeMode(QHeaderView::Fixed);
    ui.kSubOperationsTable->setWordWrap(false);
    m_tableDelegate = new SKGSplitTableDelegate(ui.kSubOperationsTable, getDocument(), m_attributesForSplit);
    m_tableDelegate->addParameterValue("total", '0');
    ui.kSubOperationsTable->setItemDelegate(m_tableDelegate);
    ui.kSubOperationsTable->setTextElideMode(Qt::ElideMiddle);
    connect(ui.kSubOperationsTable, SIGNAL(removeLine(int)), this, SLOT(onRemoveSubOperation(int)));

    ui.kTargetAccountEdit->hide();
    ui.kTargetAccountLabel->hide();
    ui.SKGSplitOpSection->hide();
    ui.SKGSharesSection->hide();

    ui.kWidgetSelector->setSelectedMode(0);

    // Set Event filters to catch CTRL+ENTER or SHIFT+ENTER
    this->installEventFilter(this);

    // Set Event filters for locking widgets
    ui.kTypeEdit->lineEdit()->installEventFilter(this);
    ui.kTypeEdit->installEventFilter(this);
    ui.kUnitEdit->lineEdit()->installEventFilter(this);
    ui.kUnitEdit->installEventFilter(this);
    ui.kCategoryEdit->lineEdit()->installEventFilter(this);
    ui.kCategoryEdit->installEventFilter(this);
    ui.kCommentEdit->lineEdit()->installEventFilter(this);
    ui.kCommentEdit->installEventFilter(this);
    ui.kPayeeEdit->lineEdit()->installEventFilter(this);
    ui.kPayeeEdit->installEventFilter(this);
    ui.kTrackerEdit->lineEdit()->installEventFilter(this);
    ui.kTrackerEdit->installEventFilter(this);
    ui.kAccountEdit->installEventFilter(this);
    ui.kAmountEdit->installEventFilter(this);
    ui.kNumberEdit->installEventFilter(this);

    m_frozenLogo = KStandardDirs::locate("icon", QString::fromLatin1("hicolor/16x16/actions/skrooge_freeze.png"));

    connect(getDocument(), SIGNAL(tableModified(QString,int,bool)), this, SLOT(dataModified(QString,int,bool)), Qt::QueuedConnection);

    connect(ui.kUnitEdit, SIGNAL(editTextChanged(QString)), this, SLOT(onOperationCreatorModified()), Qt::QueuedConnection);
    connect(ui.kAmountEdit, SIGNAL(textChanged(QString)), this, SLOT(onOperationCreatorModified()), Qt::QueuedConnection);
    connect(ui.kAmountSharesEdit, SIGNAL(textChanged(QString)), this, SLOT(onOperationCreatorModified()), Qt::QueuedConnection);
    connect(ui.kCommissionEdit, SIGNAL(textChanged(QString)), this, SLOT(onOperationCreatorModified()), Qt::QueuedConnection);
    connect(ui.kTaxEdit, SIGNAL(textChanged(QString)), this, SLOT(onOperationCreatorModified()), Qt::QueuedConnection);
    connect(ui.kAccountEdit, SIGNAL(currentIndexChanged(int)), this, SLOT(onOperationCreatorModified()), Qt::QueuedConnection);
    connect(ui.kOperationView->getShowWidget(), SIGNAL(stateChanged()), this, SLOT(onFilterChanged()), Qt::QueuedConnection);



    dataModified("", 0);
    onOperationCreatorModified();

    setAllWidgetsEnabled();
}

SKGOperationPluginWidget::~SKGOperationPluginWidget()
{
    SKGTRACEINFUNC(1);
    m_objectModel = NULL;
    m_fastEditionAction = NULL;
}

QString SKGOperationPluginWidget::currentAccount()
{
    QStringList accounts = SKGServices::splitCSVLine(ui.kOperationView->getShowWidget()->getState());
    foreach(const QString & item, accounts) {
        if (item.startsWith(QLatin1String("##_"))) {
            return item.right(item.length() - 3);
        }
    }
    return "";
}

bool SKGOperationPluginWidget::isWidgetEditionEnabled(QWidget* iWidget)
{
    return (iWidget && (!iWidget->property("frozen").isValid() || !iWidget->property("frozen").toBool()));
}

void SKGOperationPluginWidget::setWidgetEditionEnabled(QWidget* iWidget, bool iEnabled)
{
    if (iWidget && isWidgetEditionEnabled(iWidget) != iEnabled) {
        KLineEdit* line = qobject_cast<KLineEdit*>(iWidget);
        if (iEnabled) {
            iWidget->setStyleSheet("background-image:none;");  // TODO(Stephane MANKOWSKI): Correction to avoid missing bleue line around field
            iWidget->setProperty("frozen", false);
            if (line && iWidget->property("clearButtonShown").toBool() == true) {
                line->setClearButtonShown(true);

                line->setReadOnly(true);
                line->setReadOnly(false);
            }
        } else {
            QString align = "right";
            if (line) {
                if (line->alignment()&Qt::AlignRight) {
                    align = "left";
                }

                if (line->isClearButtonShown()) {
                    iWidget->setProperty("clearButtonShown", true);
                    line->setClearButtonShown(false);
                }
                iWidget->setStyleSheet("background-image:url(" % m_frozenLogo % ");background-repeat:no-repeat;background-clip: padding; background-position: top " % align % "; background-origin: content;");
                iWidget->setProperty("frozen", true);
                line->setReadOnly(true);
                line->setReadOnly(false);
            }
        }

        QString addOn = i18nc("A tool tip", "This field is frozen (it will not be affected by Fast Edition). Double click to unfreeze it");
        QString t = iWidget->toolTip().remove('\n' % addOn).remove(addOn);
        if (!iEnabled) {
            t = iWidget->toolTip();
            if (!t.isEmpty()) {
                t += '\n';
            }
            t += addOn;
        }
        iWidget->setToolTip(t);

        // 348619: Freeze the unit when amount is frozen
        if (iWidget == ui.kAmountEdit) {
            setWidgetEditionEnabled(ui.kUnitEdit->lineEdit(), iEnabled);
        }
    }
}

bool SKGOperationPluginWidget::eventFilter(QObject* iObject, QEvent* iEvent)
{
    if (iEvent && iEvent->type() == QEvent::MouseButtonDblClick) {
        KLineEdit* line = qobject_cast<KLineEdit*>(iObject);
        if (line) {
            setWidgetEditionEnabled(line, !isWidgetEditionEnabled(line));
        }
    } else if (iEvent && iEvent->type() == QEvent::FocusIn) {
        KLineEdit* line = qobject_cast<KLineEdit*>(iObject);
        if (line) {
            m_previousValue = line->text();
        } else {
            SKGComboBox* cmb = qobject_cast<SKGComboBox*>(iObject);
            if (cmb) {
                m_previousValue = cmb->text();
            }
        }
    } else if (iEvent && iEvent->type() == QEvent::FocusOut) {
        KLineEdit* line = qobject_cast<KLineEdit*>(iObject);
        if (line) {
            if (m_previousValue != line->text()) {
                setWidgetEditionEnabled(line, false);
            }
        } else {
            SKGComboBox* cmb = qobject_cast<SKGComboBox*>(iObject);
            if (cmb) {
                if (m_previousValue != cmb->text()) {
                    setWidgetEditionEnabled(cmb->lineEdit(), false);
                }
            }
        }
    } else if (iEvent && iEvent->type() == QEvent::KeyPress) {
        QKeyEvent* keyEvent = static_cast<QKeyEvent*>(iEvent);
        if ((keyEvent->key() == Qt::Key_Return || keyEvent->key() == Qt::Key_Enter) && iObject == this) {
            if (QApplication::keyboardModifiers() &Qt::ControlModifier && ui.kAddOperationBtn->isEnabled()) {
                ui.kAddOperationBtn->click();
            } else if (QApplication::keyboardModifiers() &Qt::ShiftModifier && ui.kModifyOperationBtn->isEnabled()) {
                ui.kModifyOperationBtn->click();
            }
        }
    }

    return false;
}

void SKGOperationPluginWidget::onFreeze()
{
    if (ui.kFreezeBtn->isChecked() == false) {
        ui.kFreezeBtn->setIcon(KIcon("skrooge_freeze"));
        // At least one fiels is already frozen ==> unfreeze
        setAllWidgetsEnabled();
    } else {
        QStringList overlay;
        overlay.push_back("edit-delete");
        ui.kFreezeBtn->setIcon(KIcon("skrooge_freeze", NULL, overlay));
        // No wildget frozen ==> freeze widget containing test
        if (!ui.kTypeEdit->text().isEmpty()) {
            setWidgetEditionEnabled(ui.kTypeEdit->lineEdit(), false);
        }
        if (!ui.kUnitEdit->text().isEmpty()) {
            setWidgetEditionEnabled(ui.kUnitEdit->lineEdit(), false);
        }
        if (!ui.kCategoryEdit->text().isEmpty()) {
            setWidgetEditionEnabled(ui.kCategoryEdit->lineEdit(), false);
        }
        if (!ui.kCommentEdit->text().isEmpty()) {
            setWidgetEditionEnabled(ui.kCommentEdit->lineEdit(), false);
        }
        if (!ui.kPayeeEdit->text().isEmpty()) {
            setWidgetEditionEnabled(ui.kPayeeEdit->lineEdit(), false);
        }
        if (!ui.kTrackerEdit->text().isEmpty()) {
            setWidgetEditionEnabled(ui.kTrackerEdit->lineEdit(), false);
        }
        // if(!ui.kAccountEdit->text().isEmpty()) setWidgetEditionEnabled(ui.kAccountEdit, false);
        if (!ui.kAmountEdit->text().isEmpty()) {
            setWidgetEditionEnabled(ui.kAmountEdit, false);
        }
        if (!ui.kNumberEdit->text().isEmpty()) {
            setWidgetEditionEnabled(ui.kNumberEdit, false);
        }
    }
}

void SKGOperationPluginWidget::setAllWidgetsEnabled()
{
    SKGTRACEINFUNC(10);
    // Enable widgets
    setWidgetEditionEnabled(ui.kTypeEdit->lineEdit(), true);
    setWidgetEditionEnabled(ui.kUnitEdit->lineEdit(), true);
    setWidgetEditionEnabled(ui.kCategoryEdit->lineEdit(), true);
    setWidgetEditionEnabled(ui.kCommentEdit->lineEdit(), true);
    setWidgetEditionEnabled(ui.kPayeeEdit->lineEdit(), true);
    setWidgetEditionEnabled(ui.kTrackerEdit->lineEdit(), true);
    setWidgetEditionEnabled(ui.kAccountEdit, true);
    setWidgetEditionEnabled(ui.kAmountEdit, true);
    setWidgetEditionEnabled(ui.kNumberEdit, true);
}

QString SKGOperationPluginWidget::getAttributeOfSelection(const QString& iAttribute)
{
    QString output;
    SKGObjectBase::SKGListSKGObjectBase selectedObjects = ui.kOperationView->getView()->getSelectedObjects();
    int nb = selectedObjects.count();
    for (int i = 0; i < nb ; ++i) {
        SKGObjectBase obj = selectedObjects.at(i);
        QString val = obj.getAttribute(iAttribute);
        if (i > 0 && val != output) {
            output = NOUPDATE;
            break;
        }
        output = val;
    }

    return output;
}

void SKGOperationPluginWidget::onSelectionChanged()
{
    SKGTRACEINFUNC(10);

    int mode = ui.kWidgetSelector->getSelectedMode();

    // Enable widgets
    setAllWidgetsEnabled();
    ui.kFreezeBtn->setChecked(false);
    ui.kFreezeBtn->setIcon(KIcon("skrooge_freeze"));

    // Mapping
    int nbSelect = ui.kOperationView->getView()->getNbSelectedObjects();
    bool onConsolidatedTable = false;
    if (nbSelect && m_objectModel) {
        SKGObjectBase objbase = ui.kOperationView->getView()->getFirstSelectedObject();
        SKGOperationObject obj;
        onConsolidatedTable = (objbase.getTable() == "v_suboperation_consolidated");
        if (onConsolidatedTable) {
            obj = SKGOperationObject(obj.getDocument(), SKGServices::stringToInt(obj.getAttribute("i_OPID")));
        } else {
            obj = objbase;
        }

        ui.kDateEdit->setDate(SKGServices::stringToTime(objbase.getAttribute("d_date")).date());
        m_previousDate = ui.kDateEdit->date();
        ui.kCommentEdit->setText(objbase.getAttribute(onConsolidatedTable ? "t_REALCOMMENT" : "t_comment"));
        QString number = objbase.getAttribute("i_number");
        if (number == "0") {
            number = "";
        }
        ui.kNumberEdit->setText(number);
        QString accountName = objbase.getAttribute("t_ACCOUNT");
        if (!m_showClosedAccounts && !accountName.isEmpty() && !ui.kAccountEdit->contains(accountName)) {
            // Refresh list of accounts if a closed account is selected
            m_showClosedAccounts = true;
            dataModified("", 0);
        }
        ui.kAccountEdit->setText(accountName);
        ui.kPayeeEdit->setText(objbase.getAttribute("t_PAYEE"));
        ui.kTypeEdit->setText(objbase.getAttribute("t_mode"));
        QString unit = objbase.getAttribute("t_UNIT");
        ui.kUnitEdit->setText(unit);
        QString cat = objbase.getAttribute("t_REALCATEGORY");
        if (cat.isEmpty()) {
            cat = objbase.getAttribute("t_CATEGORY");
        }
        ui.kCategoryEdit->setText(cat);
        ui.kTrackerEdit->setText(objbase.getAttribute(onConsolidatedTable ? "t_REALREFUND" : "t_REFUND"));
        QString quantity = objbase.getAttribute("f_REALQUANTITY");
        if (quantity.isEmpty()) {
            quantity = objbase.getAttribute("f_QUANTITY");
        }
        double quantityVal = SKGServices::stringToDouble(quantity);
        SKGUnitObject unitObject = ui.kUnitEdit->getUnit();
        int nbDec = unitObject.getNumberDecimal();
        if (nbDec == 0) {
            nbDec = 2;
        }
        quantity = KGlobal::locale()->formatMoney(qAbs(quantityVal), "", nbDec);
        if (quantity.startsWith(KGlobal::locale()->positiveSign())) {
            quantity = quantity.right(quantity.length() - KGlobal::locale()->positiveSign().length());
        }
        if (quantityVal > 0) {
            quantity = '+' % quantity;
        } else {
            quantity = '-' % quantity;
        }
        ui.kAmountEdit->setText(quantity);

        if (nbSelect > 1) {
            // In case of multi selection
            if (mode >= 0) {
                ui.kWidgetSelector->setSelectedMode(0);
            }
            ui.kAccountEdit->setText(getAttributeOfSelection("t_ACCOUNT"));
            ui.kTypeEdit->setText(getAttributeOfSelection("t_mode"));
            ui.kUnitEdit->setText(getAttributeOfSelection("t_UNIT"));
            ui.kCategoryEdit->setText(getAttributeOfSelection(onConsolidatedTable ? "t_REALCATEGORY" : "t_CATEGORY"));
            ui.kTrackerEdit->setText(getAttributeOfSelection(onConsolidatedTable ? "t_REALREFUND" : "t_REFUND"));
            ui.kCommentEdit->setText(getAttributeOfSelection(onConsolidatedTable ? "t_REALCOMMENT" : "t_comment"));
            ui.kPayeeEdit->setText(getAttributeOfSelection("t_PAYEE"));

            ui.kDateEdit->setDate(QDate());
            ui.kAmountEdit->setText("");
            ui.kNumberEdit->setText("");

            if (m_modeInfoZone != 1) {
                displaySelectionAmount();
            }
        } else {
            if (obj.getStatus() == SKGOperationObject::POINTED) {
                displayReconciliationInfo();
            } else if (m_modeInfoZone != 1) {
                displayBalance();
            }

            // It is a single selection
            // Is it a split ?
            int nbSubOperations = obj.getNbSubOperations();
            if (nbSubOperations > 1 && !onConsolidatedTable) {
                // yes, it is a split
                if (mode >= 0) {
                    ui.kWidgetSelector->setSelectedMode(1);
                }

                displaySubOperations();
            } else {
                // Is it a transfer ?
                SKGOperationObject op2;
                obj.isTransfer(op2);
                if (op2.exist()) {
                    // yes it is a transfer
                    SKGAccountObject account2;
                    op2.getParentAccount(account2);
                    QString accountName2 = account2.getName();
                    if (!m_showClosedAccounts && !ui.kTargetAccountEdit->contains(accountName2)) {
                        // Refresh list of accounts if a closed account is selected
                        m_showClosedAccounts = true;
                        dataModified("", 0);
                    }
                    ui.kTargetAccountEdit->setText(accountName2);
                    if (mode >= 0) {
                        ui.kWidgetSelector->setSelectedMode(2);
                    }
                } else {
                    if (mode >= 0) {
                        ui.kWidgetSelector->setSelectedMode(0);
                    }
                }
            }
        }
    }

    ui.kDateEdit->setEnabled(nbSelect <= 1);
    ui.kAmountEdit->setEnabled(nbSelect <= 1);
    ui.kNumberEdit->setEnabled(nbSelect <= 1);

    bool splitTest = nbSelect <= 1 && !onConsolidatedTable;
    ui.kWidgetSelector->setEnabledMode(1, splitTest);
    if (!splitTest && mode == 1) {
        ui.kWidgetSelector->setSelectedMode(0);
    }

    onOperationCreatorModified();

    Q_EMIT selectionChanged();

    if (m_modeInfoZone == 2) {
        onRefreshInformationZoneDelayed();
    }
}

void SKGOperationPluginWidget::onOperationCreatorModified()
{
    SKGTRACEINFUNC(10);

    int mode = ui.kWidgetSelector->getSelectedMode();

    // Set icons
    if (!isTemplateMode()) {
        ui.kModifyOperationBtn->setIcon(KIcon("dialog-ok-apply"));
        ui.kAddOperationBtn->setIcon(KIcon("list-add"));
    } else {
        QStringList overlay;
        overlay.push_back("skrooge_template");
        ui.kModifyOperationBtn->setIcon(KIcon("dialog-ok-apply", NULL, overlay));
        ui.kAddOperationBtn->setIcon(KIcon("list-add", NULL, overlay));
    }

    // Is it an existing unit ?
    QString unitName = ui.kUnitEdit->currentText();
    SKGUnitObject unit(getDocument());
    unit.setName(unitName);
    unit.setSymbol(unitName);
    if (unit.load().isSucceeded()) {
        ui.kWidgetSelector->setEnabledMode(3, true);
        if (mode == 3 && unit.getType() == SKGUnitObject::SHARE) {
            // Update units
            SKGServices::SKGUnitInfo unitOfUnitName = static_cast<SKGDocumentBank*>(getDocument())->getPrimaryUnit();
            SKGUnitObject unitOfUnit;
            unit.getUnit(unitOfUnit);
            if (unitOfUnit.exist()) {
                unitOfUnitName.Symbol = unitOfUnit.getSymbol();
                unitOfUnitName.NbDecimal = unitOfUnit.getNumberDecimal();
            }
            ui.kUnitShare->setText(unitOfUnitName.Symbol);
            ui.kUnitCommission->setText(unitOfUnitName.Symbol);
            ui.kUnitTax->setText(unitOfUnitName.Symbol);

            // Update total in "purchase / sale share" page
            double total = ui.kAmountSharesEdit->value() + (ui.kCommissionEdit->value() + ui.kTaxEdit->value()) * (ui.kAmountEdit->value() > 0 ? 1 : -1);
            ui.KTotal->setText(KGlobal::locale()->formatMoney(total, unitOfUnitName.Symbol, unitOfUnitName.NbDecimal));
        } else {
            // BUG 2692665
            ui.kUnitShare->setText(unitName);
            ui.kUnitCommission->setText(unitName);
            ui.kUnitTax->setText(unitName);
        }

    } else {
        ui.kWidgetSelector->setEnabledMode(3, false);
        if (mode == 3) {
            ui.kWidgetSelector->setSelectedMode(0);
        }
    }

    bool activated = mode != -1 &&
                     !ui.kAccountEdit->currentText().isEmpty() &&
                     ((!ui.kAmountEdit->text().isEmpty() && ui.kAmountEdit->valid()) || !ui.kAmountEdit->isEnabled()) &&
                     !unitName.isEmpty() &&
                     (mode != 3 || !ui.kAmountSharesEdit->text().isEmpty());

    int nbSelect = getNbSelectedObjects();

    ui.kAddOperationBtn->setEnabled(activated);
    ui.kModifyOperationBtn->setEnabled(activated && nbSelect > 0 && (ui.kWidgetSelector->getSelectedMode() == 0 ||  ui.kWidgetSelector->getSelectedMode() == 1 ||  ui.kWidgetSelector->getSelectedMode() == 2));

    m_numberFieldIsNotUptodate = true;
    if (ui.kNumberEdit->hasFocus()) {
        fillNumber();
    }
}

void SKGOperationPluginWidget::onUpdateOperationClicked()
{
    SKGError err;
    SKGTRACEINFUNCRC(10, err);
    // Get Selection
    SKGObjectBase::SKGListSKGObjectBase selection = getSelectedObjects();

    int nb = selection.count();
    {
        SKGBEGINPROGRESSTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Operation update"), err, nb);
        err = updateSelection(selection);
    }

    // status bar
    IFOKDO(err, SKGError(0, i18nc("Successful message after an user action", "Operation canceled")))
    else {
        err.addError(ERR_FAIL, i18nc("Error message",  "Operation update failed"));
    }

    // Display error
    SKGMainPanel::displayErrorMessage(err);

    // Set focus on date
    ui.kDateEdit->setFocus();
}

SKGError SKGOperationPluginWidget::updateSelection(const SKGObjectBase::SKGListSKGObjectBase& iSelection, bool forceCreation)
{
    SKGError err;
    SKGTRACEINFUNCRC(10, err);

    // Initialisation
    double ratio = -1;
    bool refreshSubOperation = true;

    // Get Selection
    SKGObjectBase::SKGListSKGObjectBase selection = getSelectedObjects();
    int nb = iSelection.count();

    for (int i = 0; !err && i < nb; ++i) {
        SKGObjectBase obj = iSelection.at(i);
        SKGOperationObject operationObj;
        if (obj.getTable() == "v_suboperation_consolidated") {
            operationObj = SKGOperationObject(obj.getDocument(), SKGServices::stringToInt(obj.getAttribute("i_OPID")));
        } else {
            operationObj = SKGOperationObject(obj.getDocument(), obj.getID());
        }

        SKGObjectBase::SKGListSKGObjectBase gops;
        IFOKDO(err, operationObj.getGroupedOperations(gops))
        if (gops.count() == 2 &&  ui.kWidgetSelector->getSelectedMode() < 2) {
            getDocument()->sendMessage(i18nc("An information message", "You modified one part of a transfer"), SKGDocument::Warning);
        }

        // Update operation if single selection
        if (ui.kWidgetSelector->getSelectedMode() == 0) {
            // Get subop
            SKGSubOperationObject subOp;
            int nbSubop = 0;
            if (obj.getTable() == "v_suboperation_consolidated") {
                // It's a sub operation
                subOp = SKGSubOperationObject(obj.getDocument(), SKGServices::stringToInt(obj.getAttribute("i_SUBOPID")));
                nbSubop = 1;
            } else {
                // It's a real operation, we take the first one
                SKGObjectBase::SKGListSKGObjectBase subOps;
                IFOKDO(err, operationObj.getSubOperations(subOps))
                nbSubop = subOps.count();
                if (nbSubop) {
                    subOp = subOps[0];
                }
            }

            QString trackerName = ui.kTrackerEdit->text().trimmed();
            if (!err && trackerName != NOUPDATE) {
                SKGTrackerObject tracker;
                err = SKGTrackerObject::createTracker(static_cast<SKGDocumentBank*>(getDocument()), trackerName, tracker, true);
                IFOKDO(err, subOp.setTracker(tracker))
            }

            SKGCategoryObject cat;
            QString catName = ui.kCategoryEdit->text().trimmed();
            if (!err &&  catName != NOUPDATE) {
                err = SKGCategoryObject::createPathCategory(static_cast<SKGDocumentBank*>(getDocument()), catName, cat, true);
                IFOKDO(err, subOp.setCategory(cat))
            } else {
                // Get current category to be able to find the appropriate sign
                subOp.getCategory(cat);
            }

            if (!err && nb == 1) {
                if (nbSubop > 1) {
                    err = SKGError(25, i18nc("Error message", "Cannot update a split operation"));
                } else {
                    double val = ui.kAmountEdit->value();

                    // Is the sign forced ?
                    if (ui.kAmountEdit->sign() == 0) {
                        // No
                        SKGObjectBase cat2(cat.getDocument(), "v_category_display", cat.getID());

                        // Are we able to find to sign with the category ?
                        if (cat2.getAttribute("t_TYPEEXPENSE") == "-") {
                            val = -val;
                        }
                    }
                    err = subOp.setQuantity(val);
                }
            }

            if (!err && ui.kCommentEdit->text() != NOUPDATE) {
                err = subOp.setComment(ui.kCommentEdit->text());
            }
            IFOKDO(err, subOp.save())
        } else if (ui.kWidgetSelector->getSelectedMode() == 1) {
            // Case split
            refreshSubOperation = false;
            int nbsubop = ui.kSubOperationsTable->rowCount();
            QList<int> listIdSubOp;
            for (int j = 0; !err && j < nbsubop; ++j) {
                // Get values
                QTableWidgetItem* item = ui.kSubOperationsTable->item(j, m_attributesForSplit.indexOf("t_category"));
                int id = (forceCreation ? 0 : item->data(Qt::UserRole).toInt());
                QString catName = item->text();

                item = ui.kSubOperationsTable->item(j, m_attributesForSplit.indexOf("t_comment"));
                QString comment = item->text();

                item = ui.kSubOperationsTable->item(j, m_attributesForSplit.indexOf("f_value"));
                double val = SKGServices::stringToDouble(item->text());
                QString formula = item->toolTip();

                item = ui.kSubOperationsTable->item(j, m_attributesForSplit.indexOf("t_refund"));
                QString trackerName = item->text();

                item = ui.kSubOperationsTable->item(j, m_attributesForSplit.indexOf("d_date"));
                QDate date = SKGServices::stringToTime(item->text()).date();
                if (!date.isValid()) {
                    date = ui.kDateEdit->date();
                }

                SKGSubOperationObject subOperation;
                if (id) {
                    // Update existing sub op
                    subOperation = SKGSubOperationObject(static_cast<SKGDocumentBank*>(getDocument()), id);
                } else {
                    // Create new sub op
                    err = operationObj.addSubOperation(subOperation);
                }

                // Create sub operation object
                IFOK(err) {
                    SKGCategoryObject cat;
                    err = SKGCategoryObject::createPathCategory(static_cast<SKGDocumentBank*>(getDocument()), catName, cat, true);
                    IFOKDO(err, subOperation.setCategory(cat))
                }
                IFOK(err) {
                    SKGTrackerObject tracker;
                    err = SKGTrackerObject::createTracker(static_cast<SKGDocumentBank*>(getDocument()), trackerName, tracker, true);
                    IFOKDO(err, subOperation.setTracker(tracker))
                }
                IFOKDO(err, subOperation.setOrder(ui.kSubOperationsTable->visualRow(j)))
                IFOKDO(err, subOperation.setDate(date))
                IFOKDO(err, subOperation.setComment(comment))
                IFOKDO(err, subOperation.setQuantity(val))
                if (formula.startsWith(QLatin1String("=")) && !err) {
                    err = subOperation.setFormula(formula);
                }
                IFOKDO(err, subOperation.save())

                // The sub operation created or updated mustn't be removed
                listIdSubOp.push_back(subOperation.getID());
            }

            // Remove useless subop
            IFOK(err) {
                SKGObjectBase::SKGListSKGObjectBase subOps;
                err = operationObj.getSubOperations(subOps);
                int nbsubop2 = subOps.count();
                for (int j = 0; !err && j < nbsubop2; ++j) {
                    SKGObjectBase sop = subOps.at(j);
                    if (!listIdSubOp.contains(sop.getID())) {
                        err = sop.remove(false);
                    }
                }
            }
        } else if (ui.kWidgetSelector->getSelectedMode() == 2) {
            // Case transfer
            // Create sub operation object
            double operationQuantity = ui.kAmountEdit->value();


            SKGSubOperationObject subOperation;
            SKGObjectBase::SKGListSKGObjectBase subOps;
            IFOKDO(err, operationObj.getSubOperations(subOps))
            IFOK(err) {
                subOperation = subOps.at(0);

                double oldQuantity = subOperation.getQuantity();

                if (ui.kAmountEdit->sign() == 0) {
                    operationQuantity = qAbs(operationQuantity);
                } else if (ui.kAmountEdit->sign() > 0) {
                    operationQuantity = -qAbs(operationQuantity);
                } else {
                    operationQuantity = qAbs(operationQuantity);
                    if (oldQuantity == 0) {
                        err = getDocument()->sendMessage(i18nc("An information message",  "Absolute value has been used for transfer creation."));
                    }
                }

                if (ui.kAmountEdit->isEnabled()) {
                    err = subOperation.setQuantity(-operationQuantity);
                } else {
                    operationQuantity = -subOperation.getQuantity();
                }
            }

            QString trackerName = ui.kTrackerEdit->text().trimmed();
            if (!err && trackerName != NOUPDATE) {
                SKGTrackerObject tracker;
                err = SKGTrackerObject::createTracker(static_cast<SKGDocumentBank*>(getDocument()), trackerName, tracker, true);
                IFOKDO(err, subOperation.setTracker(tracker))
            }

            SKGCategoryObject cat;
            QString catName = ui.kCategoryEdit->text().trimmed();
            if (!err &&  catName != NOUPDATE) {
                err = SKGCategoryObject::createPathCategory(static_cast<SKGDocumentBank*>(getDocument()), ui.kCategoryEdit->text(), cat, true);
                IFOKDO(err, subOperation.setCategory(cat))
            }
            IFOKDO(err, subOperation.setComment(operationObj.getComment()))
            IFOKDO(err, subOperation.save())

            // Get account
            SKGAccountObject accountObj2(getDocument());
            IFOKDO(err, accountObj2.setName(ui.kTargetAccountEdit->currentText()))
            IFOKDO(err, accountObj2.load())

            // Check unit of target account
            SKGUnitObject unit;
            IFOKDO(err, operationObj.getUnit(unit))

            // Correction bug 2299303 vvv
            SKGUnitObject unitTargetAccount;
            IFOKDO(err, accountObj2.getUnit(unitTargetAccount))
            if (!err && unitTargetAccount.exist() && unit != unitTargetAccount) {
                // The unit of the operation is not compliant with the unit of the target account
                // We ask to the user if he wants to continue or convert into the target account
                bool ok = false;
                QApplication::setOverrideCursor(QCursor(Qt::ArrowCursor));
                int decimal = qMax(unit.getNumberDecimal(), unitTargetAccount.getNumberDecimal());
                double newval = KInputDialog::getDouble(i18nc("Question", "Confirmation"),
                                                        i18nc("Question", "The operation's unit is not compatible with the target account.\n"
                                                                "Click Cancel if you want to continue anyway; "
                                                                "otherwise, enter the value in the target account's unit (%1):", unitTargetAccount.getSymbol()),
                                                        SKGUnitObject::convert(operationQuantity, unit, unitTargetAccount, ui.kDateEdit->date()),
                                                        -DBL_MAX, DBL_MAX, 0.1, decimal, &ok, this);
                QApplication::restoreOverrideCursor();
                if (ok) {
                    operationQuantity = newval;
                    unit = unitTargetAccount;
                }
            }
            // Correction bug 2299303 ^^^

            // create transferred operation
            SKGOperationObject operation2;

            IFOK(err) {
                if (gops.count() == 2) {
                    operation2 = (obj == SKGOperationObject(gops.at(0)) ? gops.at(1) : gops.at(0));
                    if (ui.kTargetAccountEdit->text() != NOUPDATE) {
                        err = operation2.setParentAccount(accountObj2);
                    }
                } else {
                    err = accountObj2.addOperation(operation2);
                }
            }
            if (nb == 1) {
                IFOKDO(err, operation2.setMode(ui.kTypeEdit->currentText()))
                QString payeeName = ui.kPayeeEdit->currentText();
                if (!err &&  payeeName != NOUPDATE) {
                    SKGPayeeObject payeeObject;
                    err = SKGPayeeObject::createPayee(static_cast<SKGDocumentBank*>(getDocument()), payeeName, payeeObject, true);
                    IFOKDO(err, operation2.setPayee(payeeObject))
                }
                IFOKDO(err, operation2.setNumber(SKGServices::stringToInt(ui.kNumberEdit->text())))
                IFOKDO(err, operation2.setComment(ui.kCommentEdit->text()))
                IFOKDO(err, operation2.setDate(ui.kDateEdit->date(), refreshSubOperation))
                IFOKDO(err, operation2.setUnit(unit))

            } else {
                IFOKDO(err, operation2.setMode(operationObj.getMode()))
                IFOKDO(err, operation2.setAttribute("r_payee_id", operationObj.getAttribute("r_payee_id")))
                IFOKDO(err, operation2.setNumber(operationObj.getNumber()))
                IFOKDO(err, operation2.setComment(operationObj.getComment()))
                IFOKDO(err, operation2.setDate(operationObj.getDate(), refreshSubOperation))
                IFOKDO(err, operation2.setUnit(unit))
            }
            IFOKDO(err, operation2.setGroupOperation(operationObj))
            IFOKDO(err, operationObj.load())
            IFOKDO(err, operation2.setTemplate(isTemplateMode()))
            IFOKDO(err, operation2.save())

            // Create sub operation object
            SKGSubOperationObject subOperation2;

            SKGObjectBase::SKGListSKGObjectBase subops;
            IFOKDO(err, operation2.getSubOperations(subops))
            IFOK(err) {
                if (subops.count()) {
                    subOperation2 = subops.at(0);
                } else {
                    err = operation2.addSubOperation(subOperation2);
                }
            }
            IFOKDO(err, subOperation2.setQuantity(operationQuantity))
            IFOKDO(err, subOperation2.setCategory(cat))
            IFOKDO(err, subOperation2.setComment(operationObj.getComment()))
            IFOKDO(err, subOperation2.save())
        }

        IFOKDO(err, operationObj.setTemplate(isTemplateMode()))

        if (nb == 1) {
            IFOKDO(err, operationObj.setNumber(SKGServices::stringToInt(ui.kNumberEdit->text())))
            IFOKDO(err, operationObj.setDate(ui.kDateEdit->date(), refreshSubOperation))
        }
        if (!err && ui.kCommentEdit->text() != NOUPDATE) {
            if (obj.getTable() != "v_suboperation_consolidated") {
                err = operationObj.setComment(ui.kCommentEdit->text());
            }
        }

        if (!err && ui.kAccountEdit->text() != NOUPDATE) {
            SKGAccountObject account(getDocument());
            err = account.setName(ui.kAccountEdit->text());
            IFOKDO(err, account.load())
            IFOKDO(err, operationObj.setParentAccount(account))
        }
        if (!err && ui.kTypeEdit->text() != NOUPDATE) {
            err = operationObj.setMode(ui.kTypeEdit->text());
        }
        QString payeeName = ui.kPayeeEdit->currentText();
        if (!err &&  payeeName != NOUPDATE) {
            SKGPayeeObject payeeObject;
            err = SKGPayeeObject::createPayee(static_cast<SKGDocumentBank*>(getDocument()), payeeName, payeeObject, true);
            IFOKDO(err, operationObj.setPayee(payeeObject))
        }
        if (!err && ui.kUnitEdit->text() != NOUPDATE) {
            // Correction bug 282983 vvv
            // Check unit of target account
            SKGAccountObject account;
            IFOKDO(err, operationObj.getParentAccount(account))
            SKGUnitObject unitTargetAccount;
            IFOKDO(err, account.getUnit(unitTargetAccount))

            SKGUnitObject unit = ui.kUnitEdit->getUnit();

            if (!err && unitTargetAccount.exist() && unit != unitTargetAccount) {
                // Correction bug 283842 vvvv
                bool ok = false;
                if (ratio == -1) {
                    // The unit of the operation is not compliant with the unit of the target account
                    // We ask to the user if he wants to continue or convert into the target account
                    QApplication::setOverrideCursor(QCursor(Qt::ArrowCursor));
                    double currentAmount = ui.kAmountEdit->value();
                    int decimal = qMax(unit.getNumberDecimal(), unitTargetAccount.getNumberDecimal());
                    double newval = KInputDialog::getDouble(i18nc("Question", "Confirmation"),
                                                            i18nc("Question", "The operation's unit is not compatible with the target account.\n"
                                                                    "Click Cancel if you want to continue anyway; "
                                                                    "otherwise, enter the value in the target account's unit (%1):", unitTargetAccount.getSymbol()),
                                                            SKGUnitObject::convert(currentAmount, unit, unitTargetAccount, ui.kDateEdit->date()),
                                                            -DBL_MAX, DBL_MAX, 0.1, decimal, &ok, this);
                    ratio = newval / currentAmount;
                    QApplication::restoreOverrideCursor();
                }
                if (ok) {
                    // Apply ratio to all operation
                    SKGObjectBase::SKGListSKGObjectBase subops;
                    err = operationObj.getSubOperations(subops);
                    int nbsubops = subops.count();
                    for (int j = 0; !err && j < nbsubops; ++j) {
                        SKGSubOperationObject subop(subops.at(j));
                        err = subop.setQuantity(subop.getQuantity() * ratio);
                        IFOKDO(err, subop.save(true, false))
                    }

                    // Change unit
                    unit = unitTargetAccount;
                } else {
                    err = getDocument()->sendMessage(i18nc("Warning message", "You created an operation in %1 in an account in %2.", unit.getSymbol(), unitTargetAccount.getSymbol()), SKGDocument::Warning);
                }
                // Correction bug 283842 ^^^^
            }
            // Correction bug 282983 ^^^
            IFOKDO(err, operationObj.setUnit(unit))
        }

        // Save
        IFOKDO(err, operationObj.save())

        // Send message
        if (!forceCreation) {
            IFOKDO(err, operationObj.getDocument()->sendMessage(i18nc("An information message", "The operation '%1' has been updated", operationObj.getDisplayName()), SKGDocument::Hidden));
        }

        IFOKDO(err, getDocument()->stepForward(i + 1))
    }

    return err;
}

void SKGOperationPluginWidget::onAddOperationClicked()
{
    SKGError err;
    SKGTRACEINFUNC(10);

    // Get parameters
    QString accountName = ui.kAccountEdit->currentText();
    QString catName = ui.kCategoryEdit->currentText();
    QString trackerName = ui.kTrackerEdit->currentText();

    SKGOperationObject operation;
    {
        SKGBEGINTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Operation creation"), err);

        // Check enties
        if (ui.kAccountEdit->text() == NOUPDATE ||
            ui.kTypeEdit->text() == NOUPDATE ||
            ui.kUnitEdit->text() == NOUPDATE ||
            ui.kCategoryEdit->text() == NOUPDATE ||
            ui.kTrackerEdit->text() == NOUPDATE ||
            ui.kCommentEdit->text() == NOUPDATE ||
            ui.kPayeeEdit->text() == NOUPDATE) {
            err = SKGError(ERR_FAIL, i18nc("Error message", "Impossible to create an operation with one attribute valuated with %1", NOUPDATE));
        }

        // Get account
        SKGAccountObject accountObj(getDocument());
        IFOKDO(err, accountObj.setName(accountName))
        IFOKDO(err, accountObj.load())

        // Create operation object
        IFOKDO(err, accountObj.addOperation(operation))
        IFOKDO(err, operation.setMode(ui.kTypeEdit->currentText()))
        SKGPayeeObject payeeObject;
        IFOK(err) {
            QString payeeName = ui.kPayeeEdit->currentText();
            err = SKGPayeeObject::createPayee(static_cast<SKGDocumentBank*>(getDocument()), payeeName, payeeObject, true);
            IFOKDO(err, operation.setPayee(payeeObject))
        }
        IFOKDO(err, operation.setNumber(SKGServices::stringToInt(ui.kNumberEdit->text())))
        IFOKDO(err, operation.setComment(ui.kCommentEdit->text()))
        IFOKDO(err, operation.setDate(ui.kDateEdit->date()))
        IFOKDO(err, operation.setTemplate(isTemplateMode()))
        SKGUnitObject unit = ui.kUnitEdit->getUnit();
        IFOKDO(err, operation.setUnit(unit))
        if (skgoperation_settings::automaticPointInReconciliation() && m_modeInfoZone == 1) {
            IFOKDO(err, operation.setStatus(SKGOperationObject::POINTED))
        }
        IFOKDO(err, operation.save())

        if (ui.kWidgetSelector->getSelectedMode() <= 2) {
            // STD OPERATION (SPLIT , TRANSFER OR NOT)
            // We must create a suboperation, just be sure to be able to update it
            SKGSubOperationObject subOperation;
            IFOKDO(err, operation.addSubOperation(subOperation))
            IFOKDO(err, subOperation.setQuantity(0))
            IFOKDO(err, subOperation.save())

            SKGObjectBase::SKGListSKGObjectBase list;
            list << operation;
            IFOKDO(err, updateSelection(list, true))

        } else if (ui.kWidgetSelector->getSelectedMode() == 3) {
            // PURCHASE OR SALE SHARE
            // Create sub operation object
            SKGSubOperationObject subOperation;
            double val = ui.kAmountEdit->value();
            IFOKDO(err, operation.addSubOperation(subOperation))
            IFOKDO(err, subOperation.setQuantity(val))
            IFOKDO(err, subOperation.save())

            if (!err && val > 0) {
                err = operation.setProperty("SKG_OP_ORIGINAL_AMOUNT", SKGServices::doubleToString(ui.kAmountSharesEdit->value()));
            }
            IFOKDO(err, operation.save())

            // Get account
            SKGAccountObject accountObj2(getDocument());
            IFOKDO(err, accountObj2.setName(ui.kPaymentAccountEdit->currentText()))
            IFOKDO(err, accountObj2.load())

            // create paiement operation for shares
            SKGOperationObject operation2;
            IFOKDO(err, accountObj2.addOperation(operation2))
            IFOKDO(err, operation2.setMode(ui.kTypeEdit->currentText()))
            IFOKDO(err, operation2.setPayee(payeeObject))
            IFOKDO(err, operation2.setNumber(SKGServices::stringToInt(ui.kNumberEdit->text())))
            IFOKDO(err, operation2.setComment(ui.kCommentEdit->text()))
            IFOKDO(err, operation2.setDate(ui.kDateEdit->date()))
            IFOK(err) {
                SKGUnitObject unitOfUnit;
                err = unit.getUnit(unitOfUnit);
                IFKO(err) {
                    unitOfUnit = SKGUnitObject(getDocument());
                    err = unitOfUnit.setSymbol(static_cast<SKGDocumentBank*>(getDocument())->getPrimaryUnit().Symbol);
                    IFOKDO(err, unitOfUnit.load())
                }

                IFOKDO(err, operation2.setUnit(unitOfUnit))
            }
            IFOKDO(err, operation2.setGroupOperation(operation))
            IFOKDO(err, operation2.setTemplate(isTemplateMode()))
            IFOKDO(err, operation2.save())

            // Create main sub operation
            SKGSubOperationObject subOperation2;
            IFOKDO(err, operation2.addSubOperation(subOperation2))
            IFOKDO(err, subOperation2.setComment(i18nc("Noun", "Shares")))
            IFOKDO(err, subOperation2.setQuantity(ui.kAmountSharesEdit->value() * (val > 0 ? -1 : 1)))
            IFOKDO(err, subOperation2.save())

            // Create commission sub operation
            if (ui.kCommissionEdit->value()) {
                SKGSubOperationObject subOperation3;
                IFOKDO(err, operation2.addSubOperation(subOperation3))
                IFOKDO(err, subOperation3.setComment(skgoperation_settings::commentCommissionOperation()))

                QString category = skgoperation_settings::categoryCommissionOperation();
                if (!category.isEmpty()) {
                    SKGCategoryObject c;
                    IFOKDO(err, SKGCategoryObject::createPathCategory(static_cast<SKGDocumentBank*>(getDocument()), category, c, true))
                    IFOKDO(err, subOperation3.setCategory(c))
                }

                IFOKDO(err, subOperation3.setQuantity(-ui.kCommissionEdit->value()))
                IFOKDO(err, subOperation3.save())
            }

            // Create tax sub operation
            if (ui.kTaxEdit->value()) {
                SKGSubOperationObject subOperation4;
                IFOKDO(err, operation2.addSubOperation(subOperation4))
                IFOKDO(err, subOperation4.setComment(skgoperation_settings::commentTaxOperation()))

                QString category = skgoperation_settings::categoryTaxOperation();
                if (!category.isEmpty()) {
                    SKGCategoryObject c;
                    IFOKDO(err, SKGCategoryObject::createPathCategory(static_cast<SKGDocumentBank*>(getDocument()), category, c, true))
                    IFOKDO(err, subOperation4.setCategory(c))
                }

                IFOKDO(err, subOperation4.setQuantity(-ui.kTaxEdit->value()))
                IFOKDO(err, subOperation4.save())
            }
        }

        // Send message
        IFOKDO(err, operation.getDocument()->sendMessage(i18nc("An information to the user that something was added", "The operation '%1' has been added", operation.getDisplayName()), SKGDocument::Hidden));
    }

    // status bar
    IFOK(err) {
        err = SKGError(0, i18nc("Successful message after an user action", "Operation created"));
        ui.kOperationView->getView()->selectObject(operation.getUniqueID());
    } else {
        err.addError(ERR_FAIL, i18nc("Error message",  "Operation creation failed"));
    }

    // Notification for object creation
    SKGMainPanel::getMainPanel()->displayMessage("", SKGDocument::Positive);

    // Display error
    SKGMainPanel::displayErrorMessage(err);

    // Set focus on date
    ui.kDateEdit->setFocus();
}

SKGTreeView* SKGOperationPluginWidget::getTableView()
{
    return ui.kOperationView->getView();
}

QString SKGOperationPluginWidget::getState()
{
    SKGTRACEINFUNC(10);
    QDomDocument doc("SKGML");
    QDomElement root;
    if (m_lastState.hasChildNodes()) {
        doc = m_lastState;
        root = doc.documentElement();
    } else {
        root = doc.createElement("parameters");
        doc.appendChild(root);
    }

    root.setAttribute("currentPage", SKGServices::intToString(ui.kWidgetSelector->getSelectedMode()));
    root.setAttribute("modeInfoZone", SKGServices::intToString(m_modeInfoZone));
    root.setAttribute("reconcilitorAmount", ui.kReconcilitorAmountEdit->text());
    root.removeAttribute("account");

    // Memorize table settings
    root.setAttribute("view", ui.kOperationView->getState());

    return doc.toString();
}

void SKGOperationPluginWidget::setState(const QString& iState)
{
    SKGTRACEINFUNC(10);
    QDomDocument doc("SKGML");
    doc.setContent(iState);
    QDomElement root = doc.documentElement();

    QString account = root.attribute("account");
    QString currentPage = root.attribute("currentPage");
    QString title = root.attribute("title");
    QString title_icon = root.attribute("title_icon");
    QString modeInfoZoneS = root.attribute("modeInfoZone");
    QString reconcilitorAmountS = root.attribute("reconcilitorAmount");
    QString operationTable = root.attribute("operationTable");
    QString selection = root.attribute("selection");
    QString templates = root.attribute("template");
    m_operationWhereClause = root.attribute("operationWhereClause");

    // Default values in case of reset
    if (currentPage.isEmpty()) {
        currentPage = '0';
    }
    if (operationTable.isEmpty()) {
        if (m_operationWhereClause.isEmpty()) {
            operationTable = "v_operation_display_all";
        } else {
            operationTable = "v_operation_display";
        }
    }

    // Set
    SKGAccountObject acc;
    SKGNamedObject::getObjectByName(getDocument(), "v_account", account, acc);
    if (acc.isClosed() && m_showClosedAccounts == false) {
        m_showClosedAccounts = true;
        dataModified("", 0);
    }
    bool previous = ui.kReconcilitorAmountEdit->blockSignals(true);
    ui.kReconcilitorAmountEdit->setText(reconcilitorAmountS);
    ui.kReconcilitorAmountEdit->blockSignals(previous);
    ui.kWidgetSelector->setSelectedMode(SKGServices::stringToInt(currentPage));
    if (!title.isEmpty()) {
        QFontMetrics fm(fontMetrics());
        ui.kTitle->setComment("<html><body><b>" % SKGServices::stringToHtml(fm.elidedText(title, Qt::ElideMiddle, 2000)) % "</b></body></html>");
        ui.kTitle->setToolTip(title);
        ui.kTitle->show();
    }
    if (!title_icon.isEmpty()) {
        ui.kTitle->setPixmap(KIcon(title_icon).pixmap(22, 22), KTitleWidget::ImageLeft);
    }

    if (m_objectModel) {
        m_objectModel->setTable(operationTable);
    }
    if (m_objectModel && !m_operationWhereClause.isEmpty()) {
        ui.kOperationView->getShowWidget()->setEnabled(false);
        m_objectModel->setFilter(m_operationWhereClause);
    }
    if (!operationTable.isEmpty() || !m_operationWhereClause.isEmpty()) {
        // We keep a copy of given state in case of bookmark
        m_lastState = doc;
    } else {
        m_lastState = QDomDocument();
    }

    // Update model
    if (m_objectModel) {
        previous = m_objectModel->blockRefresh(true);
        onAccountChanged();
        m_objectModel->blockRefresh(previous);
    }

    // !!! Must be done here after onFilterChanged
    QString v = root.attribute("view");
    if (!v.isEmpty()) {
        ui.kOperationView->setState(v);
    }
    if (!account.isEmpty()) {
        QStringList parameters = SKGServices::splitCSVLine(ui.kOperationView->getShowWidget()->getState());
        if (parameters.count() == 0) {
            parameters.push_back("");
            parameters.push_back("operations");
            parameters.push_back("hide");
        }
        parameters[0] = "##_" % account;
        ui.kOperationView->getShowWidget()->setState(SKGServices::stringsToCsv(parameters));
    }

    if (templates == "Y") {
        QStringList parameters = SKGServices::splitCSVLine(ui.kOperationView->getShowWidget()->getState());
        parameters.removeAll("operations");
        parameters.push_back("templates");
        ui.kOperationView->getShowWidget()->setState(SKGServices::stringsToCsv(parameters));
    }

    if (!selection.isEmpty()) {
        QStringList uuids = SKGServices::splitCSVLine(selection);
        ui.kOperationView->getView()->selectObjects(uuids, true);  // FIXME // TODO(Stephane MANKOWSKI)
        onSelectionChanged();
    }

    // Refresh of the information zone
    if (!modeInfoZoneS.isEmpty()) {
        m_modeInfoZone = SKGServices::stringToInt(modeInfoZoneS) - 1;
    } else {
        m_modeInfoZone = -1;
    }
    onRotateAccountTools();
}

QString SKGOperationPluginWidget::getDefaultStateAttribute()
{
    if (m_objectModel && m_objectModel->getTable() == "v_suboperation_consolidated") {
        return "SKGOPERATION_CONSOLIDATED_DEFAULT_PARAMETERS";
    }

    if (!m_operationWhereClause.isEmpty()) {
        return "";
    }

    return "SKGOPERATION_DEFAULT_PARAMETERS";
}

void SKGOperationPluginWidget::fillTargetAccount()
{
    int nbAccounts = ui.kAccountEdit->count();
    QString current = ui.kAccountEdit->text();
    QString currentTarget = ui.kTargetAccountEdit->text();
    QString currentRecon = ui.kReconciliateAccount->text();
    ui.kTargetAccountEdit->clear();
    ui.kReconciliateAccount->clear();
    ui.kReconciliateAccount->addItem("");
    for (int i = 0; i < nbAccounts; ++i) {
        if (ui.kAccountEdit->itemText(i) != current) {
            ui.kTargetAccountEdit->addItem(ui.kAccountEdit->itemIcon(i), ui.kAccountEdit->itemText(i));
            ui.kReconciliateAccount->addItem(ui.kAccountEdit->itemIcon(i), ui.kAccountEdit->itemText(i));
        }
    }
    if (ui.kTargetAccountEdit->contains(currentTarget)) {
        ui.kTargetAccountEdit->setText(currentTarget);
    }

    SKGError err;
    SKGAccountObject currentActObj(getDocument());
    IFOKDO(err, currentActObj.setName(current))
    IFOKDO(err, currentActObj.load())

    SKGAccountObject linkedActObj;
    IFOKDO(err, currentActObj.getLinkedAccount(linkedActObj))
    if (linkedActObj.getID()) {
        currentRecon = linkedActObj.getName();
    }

    if (ui.kReconciliateAccount->contains(currentRecon)) {
        ui.kReconciliateAccount->setText(currentRecon);
    }
}

void SKGOperationPluginWidget::dataModified(const QString& iTableName, int iIdTransaction, bool iLightTransaction)
{
    SKGTRACEINFUNC(10);
    Q_UNUSED(iIdTransaction);

    // Refresh widgets
    QSqlDatabase* db = getDocument()->getDatabase();
    setEnabled(db != NULL);
    if (db != NULL) {
        if (iTableName == "account" || iTableName.isEmpty()) {
            SKGShow* showWidget = ui.kOperationView->getShowWidget();
            QString current = currentAccount();
            QString currentState = showWidget->getState();

            // Disconnect combo filter account
            disconnect(showWidget, SIGNAL(stateChanged()), this, SLOT(onAccountChanged()));
            disconnect(showWidget, SIGNAL(stateChanged()), this, SLOT(onRefreshInformationZoneDelayed()));
            disconnect(showWidget, SIGNAL(stateChanged()), this, SLOT(onOperationCreatorModified()));
            disconnect(ui.kAccountEdit, SIGNAL(currentIndexChanged(int)), this, SLOT(fillTargetAccount()));

            // Clear
            ui.kAccountEdit->clear();
            ui.kPaymentAccountEdit->clear();

            SKGStringListList listAccount;
            getDocument()->executeSelectSqliteOrder(QString("SELECT t_ICON, t_name, t_bookmarked from v_account_display ") % (m_showClosedAccounts ? "" : "where t_close='N' ")
                                                    % "order by t_bookmarked DESC, t_name ASC", listAccount);

            int nbAccounts = listAccount.count() - 1;
            if (!m_lastState.hasChildNodes()) {
                ui.kTitle->hide();
            }

            // Set show widget
            showWidget->clear();

            if (nbAccounts > 1) {
                showWidget->addGroupedItem("all", i18nc("Option to for display of operations", "All"), "", "1=1", "account", Qt::META + Qt::Key_A);
                showWidget->addSeparator();
            }
            QDir dirLogo(KStandardDirs::locate("data", QString::fromLatin1("skrooge/images/logo/")));
            QString uncheck;
            bool accountSeparatorAdded = false;
            for (int i = 1; i <= nbAccounts; ++i) {  // Ignore header
                QString iconName = dirLogo.absoluteFilePath(listAccount.at(i).at(0));
                QIcon icon(iconName);
                QString text = listAccount.at(i).at(1);
                QString id = "##_" % text;
                if (nbAccounts == 1) {
                    QStringList items = SKGServices::splitCSVLine(currentState);
                    if (items.count() == 0) {
                        items.push_back("all");
                    }
                    if (items[0] == "all" || !items[0].startsWith(QLatin1String("##_"))) {
                        items[0] = id;
                    }
                    if (items.count() == 1) {
                        items.push_back("operations");
                        items.push_back("hide");
                    }
                    currentState = SKGServices::stringsToCsv(items);
                }
                ui.kAccountEdit->addItem(icon, text);
                ui.kPaymentAccountEdit->addItem(icon, text);

                if (!accountSeparatorAdded && listAccount.at(i).at(2) == "N" && i > 1) {
                    showWidget->addSeparator();
                    accountSeparatorAdded = true;
                }

                showWidget->addGroupedItem(id, text, iconName, "t_ACCOUNT='" + SKGServices::stringToSqlString(text) + '\'', "account",
                                           (i < 10 ? QKeySequence::fromString("Meta+" % SKGServices::intToString(i)) : QKeySequence()));

                if (!uncheck.isEmpty()) {
                    uncheck = uncheck % ";";
                }
                uncheck = uncheck % id;
            }

            int nb = showWidget->count();
            for (int i = 1; i < nb; ++i) {
                showWidget->setListIdToUncheckWhenChecked(i, uncheck % ";all");
            }
            if (nbAccounts > 1) {
                showWidget->setListIdToUncheckWhenChecked(0, uncheck);
            }

            showWidget->addSeparator();
            showWidget->addGroupedItem("operations", i18nc("Option to for display of operations", "Operations"), "view-financial-list", "d_date!='0000-00-00' AND t_template='N'",
                                       "type", Qt::META + Qt::Key_O);
            showWidget->addGroupedItem("templates", i18nc("Option to for display of operations", "Templates"), "skrooge_template", "d_date!='0000-00-00' AND t_template='Y'",
                                       "type", Qt::META + Qt::Key_T);
            showWidget->addSeparator();
            showWidget->addItem("hidepointed", i18nc("Option to for display of operations", "Hide pointed operations"), "dialog-ok", "t_status<>'P'", "", "", "", "", Qt::META + Qt::Key_P);
            showWidget->addItem("hide", i18nc("Option to for display of operations", "Hide checked operations"), "dialog-ok-apply", "t_status<>'Y'", "", "", "", "", Qt::META + Qt::Key_C);
            showWidget->addSeparator();
            showWidget->addPeriodItem("period");
            showWidget->setMode(SKGShow::AND);
            if (currentState.isEmpty()) {
                showWidget->setDefaultState("all;operations;hide");
            } else {
                showWidget->setState(currentState);
            }

            if (!current.isEmpty()) {
                ui.kAccountEdit->setText(current);
            }

            // Connect combo filter account
            connect(ui.kAccountEdit, SIGNAL(currentIndexChanged(int)), this, SLOT(fillTargetAccount()), Qt::QueuedConnection);
            connect(showWidget, SIGNAL(stateChanged()), this, SLOT(onAccountChanged()), Qt::QueuedConnection);
            connect(showWidget, SIGNAL(stateChanged()), this, SLOT(onRefreshInformationZoneDelayed()), Qt::QueuedConnection);
            connect(showWidget, SIGNAL(stateChanged()), this, SLOT(onOperationCreatorModified()), Qt::QueuedConnection);

            fillTargetAccount();

            if (nbAccounts == 0) {
                ui.kTitle->setText(i18nc("Message", "First you have to create an account."));
                ui.kTitle->setPixmap(KIcon("dialog-information").pixmap(22, 22) , KTitleWidget::ImageLeft);
                ui.kTitle->show();
                showWidget->hide();
            } else {
                showWidget->show();
            }
        }

        if (iTableName == "refund" || iTableName.isEmpty()) {
            SKGMainPanel::fillWithDistinctValue(ui.kTrackerEdit, getDocument(), "refund", "t_name", "t_close='N'");
        }
        if (!iLightTransaction) {
            if (iTableName == "category" || iTableName.isEmpty()) {
                SKGMainPanel::fillWithDistinctValue(ui.kCategoryEdit, getDocument(), "category", "t_fullname", "");
            }
            if (iTableName == "payee" || iTableName.isEmpty()) {
                SKGMainPanel::fillWithDistinctValue(ui.kPayeeEdit, getDocument(), "payee", "t_name", "");
            }
            if (iTableName == "operation" || iTableName.isEmpty()) {
                SKGMainPanel::fillWithDistinctValue(ui.kTypeEdit, getDocument(), "operation", "t_mode", "", true);
                SKGMainPanel::fillWithDistinctValue(ui.kCommentEdit, getDocument(), "v_operation_all_comment", "t_comment", "", true);

                // Set type number
                m_numberFieldIsNotUptodate = true;

                // Correction 215658: vvv because the table is modified, the selection is modified
                onSelectionChanged();
                // Correction 215658: ^^^

                if (m_modeInfoZone != 2) {
                    onRefreshInformationZoneDelayed();    // Because mode 2 is linked to selection (see onSelectionChanged)
                }
            }
        } else if (iTableName == "operation" || iTableName.isEmpty()) {
            onRefreshInformationZoneDelayed();
        }
    }
}

void SKGOperationPluginWidget::onDoubleClick()
{
    _SKGTRACEINFUNC(10);

    // Get selection
    SKGObjectBase::SKGListSKGObjectBase selection = getSelectedObjects();
    if (selection.count() == 1) {
        SKGOperationObject op(selection.at(0));

        if (op.isTemplate() && selection.at(0).getRealTable() == "operation") {
            // this is a template, we have to create an operation
            SKGError err;
            SKGBEGINTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Operation creation"), err);
            SKGOperationObject operation;
            err = op.duplicate(operation);
            if (skgoperation_settings::automaticPointInReconciliation() && m_modeInfoZone == 1) {
                IFOKDO(err, operation.setStatus(SKGOperationObject::POINTED))
                IFOKDO(err, operation.save())
            }

            // Send message
            IFOKDO(err, operation.getDocument()->sendMessage(i18nc("An information to the user that something was added", "The operation '%1' has been added", operation.getDisplayName()), SKGDocument::Hidden));

            // status bar
            IFOK(err) {
                setTemplateMode(false);
                err = SKGError(0, i18nc("Successful message after an user action", "Operation created"));
                ui.kOperationView->getView()->selectObject(operation.getUniqueID());
            } else {
                err.addError(ERR_FAIL, i18nc("Error message",  "Operation creation failed"));
            }

            // Display error
            SKGMainPanel::displayErrorMessage(err);

        } else {
            // This is not a template, we have to open it
            SKGMainPanel::getMainPanel()->getGlobalAction("open")->trigger();
        }
    }
}

void SKGOperationPluginWidget::onResetInternalFilter()
{
    SKGTRACEINFUNC(10);

    // Initialisation
    m_lastState.clear();

    if (m_objectModel) {
        m_objectModel->setTable("v_operation_display_all");
    }
    ui.kOperationView->getShowWidget()->setEnabled(true);
    ui.kTitle->hide();

    setState(getDocument()->getParameter(getDefaultStateAttribute()));

    onFilterChanged();
}

void SKGOperationPluginWidget::onRefreshInformationZoneDelayed()
{
    m_timer.start(300);
}

void SKGOperationPluginWidget::onRefreshInformationZone()
{
    SKGTRACEINFUNC(1);
    ui.kReconciliateAccount->hide();

    SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(getDocument());
    QString current = currentAccount();
    if (doc) {
        SKGServices::SKGUnitInfo unit1 = doc->getPrimaryUnit();
        SKGServices::SKGUnitInfo unit2 = doc->getSecondaryUnit();
        if (m_modeInfoZone == 0) {
            // Refresh info area
            // Compute where clause
            QString filter = "1=1";
            if (!current.isEmpty()) {
                filter = "t_name='" % SKGServices::stringToSqlString(current) % '\'';
            }
            SKGStringListList listTmp;
            getDocument()->executeSelectSqliteOrder(
                "SELECT TOTAL(f_CURRENTAMOUNT), TOTAL(f_CHECKED), TOTAL(f_COMING_SOON), TOTAL(f_COMING_SOON_FROM_LINKED_ACCOUNT) from v_account_display WHERE " % filter,
                listTmp);
            if (listTmp.count() == 2) {
                if (!current.isEmpty()) {
                    SKGAccountObject account(getDocument());
                    if (account.setName(current).isSucceeded()) {
                        if (account.load().isSucceeded()) {
                            SKGUnitObject unitAccount;
                            if (account.getUnit(unitAccount).isSucceeded()) {
                                if (!unitAccount.getSymbol().isEmpty()) {
                                    unit1.Symbol = unitAccount.getSymbol();
                                    unit1.Value = SKGServices::stringToDouble(unitAccount.getAttribute("f_CURRENTAMOUNT"));

                                    if (unit1.Symbol != static_cast<SKGDocumentBank*>(getDocument())->getPrimaryUnit().Symbol) {
                                        unit2 = static_cast<SKGDocumentBank*>(getDocument())->getPrimaryUnit();
                                    }
                                }
                            }
                        }
                    }
                }

                double v1 = SKGServices::stringToDouble(listTmp.at(1).at(0));
                double v2 = SKGServices::stringToDouble(listTmp.at(1).at(1));
                double v3 = SKGServices::stringToDouble(listTmp.at(1).at(2));
                double v4 = SKGServices::stringToDouble(listTmp.at(1).at(3));
                QString s1 = doc->formatMoney(v1, unit1);
                QString s2 = doc->formatMoney(v2, unit1);
                QString s3 = doc->formatMoney(v3, unit1);
                QString s4 = doc->formatMoney(v4, unit1);
                QString zero = doc->formatMoney(0, unit1);
                ui.kInfo->setText(i18nc("Message", "Balance: %1     Checked: %2     To be Checked: %3", s1, s2, !current.isEmpty() && s4 != zero ? s3 % " + " % s4 : s3));
                if (!unit2.Symbol.isEmpty() && unit2.Value) {
                    s1 = doc->formatMoney(v1, unit2);
                    s2 = doc->formatMoney(v2, unit2);
                    s3 = doc->formatMoney(v3, unit2);
                    s4 = doc->formatMoney(v4, unit2);
                }
                ui.kInfo->setToolTip(i18nc("Message", "<p>Balance: %1</p><p>Checked: %2</p><p>To be Checked: %3</p>", s1, s2, !current.isEmpty() && s4 != zero ? s3 % " + " % s4 : s3));
            }
        } else if (m_modeInfoZone == 1) {
            // Refresh reconciliation area
            // Compute where clause
            QString filter = '\'' % SKGServices::stringToSqlString(currentAccount()) % '\'';
            SKGStringListList listTmp;
            getDocument()->executeSelectSqliteOrder(
                "SELECT ABS(TOTAL(f_CURRENTAMOUNT_EXPENSE)),TOTAL(f_CURRENTAMOUNT_INCOME) FROM v_operation_display WHERE t_status='P' AND t_ACCOUNT=" % filter,
                listTmp);
            if (listTmp.count() == 2) {
                SKGAccountObject::AccountType accountType = SKGAccountObject::OTHER;
                if (!current.isEmpty()) {
                    SKGAccountObject account(getDocument());
                    if (account.setName(current).isSucceeded()) {
                        if (account.load().isSucceeded()) {
                            accountType = account.getType();

                            SKGUnitObject unitAccount;
                            if (account.getUnit(unitAccount).isSucceeded()) {
                                if (!unitAccount.getSymbol().isEmpty()) {
                                    unit1.Symbol = unitAccount.getSymbol();
                                    unit1.Value = SKGServices::stringToDouble(unitAccount.getAttribute("f_CURRENTAMOUNT"));

                                    if (unit1.Symbol != static_cast<SKGDocumentBank*>(getDocument())->getPrimaryUnit().Symbol) {
                                        unit2 = static_cast<SKGDocumentBank*>(getDocument())->getPrimaryUnit();
                                    }
                                }
                            }
                        }
                    }
                }
                if (accountType == SKGAccountObject::SKGAccountObject::CREDITCARD) {
                    ui.kReconciliationTitle->setText(i18nc("A title", "Total amount:"));
                    ui.kReconciliateAccount->show();
                } else {
                    ui.kReconciliationTitle->setText(i18nc("A title", "Final balance:"));
                }

                ui.kAutoPoint->setVisible(accountType != SKGAccountObject::WALLET);

                SKGStringListList listTmp2;
                double diff = 0;
                getDocument()->executeSelectSqliteOrder(
                    "SELECT TOTAL(f_CHECKED) from v_account_display WHERE t_name=" % filter,
                    listTmp2);
                if (listTmp2.count() == 2) {
                    diff = SKGServices::stringToDouble(listTmp2.at(1).at(0)) - ui.kReconcilitorAmountEdit->value() * unit1.Value;
                }

                // Set tooltip
                if (current.isEmpty()) {
                    ui.kReconciliatorInfo->setText(i18nc("Description", "You must select only one account to use reconciliation."));
                    ui.kReconciliatorInfo->setToolTip(ui.kReconciliatorInfo->text());
                } else {
                    double v1 = SKGServices::stringToDouble(listTmp.at(1).at(0));
                    double v2 = SKGServices::stringToDouble(listTmp.at(1).at(1));
                    QString sdiff = doc->formatMoney(diff, unit1);
                    QString s1 = doc->formatMoney(v1, unit1);
                    QString s2 = doc->formatMoney(v2, unit1);
                    ui.kReconciliatorInfo->setText(i18nc("Message", "%1 - Delta: %2     Expenditure: %3     Income: %4", unit1.Symbol, sdiff, s1, s2));

                    // Comparison
                    QString zero = doc->formatMoney(0, unit1);
                    QString negativezero = doc->formatMoney(-EPSILON, unit1);
                    ui.kValidate->setVisible(sdiff == zero || sdiff == negativezero);
                    ui.kCreateFakeOperation->setVisible(!ui.kValidate->isVisible());
                    ui.kAutoPoint->setVisible(!ui.kValidate->isVisible());

                    if (!unit2.Symbol.isEmpty() && unit2.Value) {
                        sdiff = doc->formatMoney(diff, unit2);
                        s1 = doc->formatMoney(v1, unit2);
                        s2 = doc->formatMoney(v2, unit2);
                    }
                    ui.kReconciliatorInfo->setToolTip(i18nc("Message", "<p>Delta: %1</p><p>Expenditure: %2</p><p>Income: %3</p>", sdiff, s1, s2));
                }
            }
        } else if (m_modeInfoZone == 2) {
            // Refresh info area with selection
            SKGObjectBase::SKGListSKGObjectBase selection = getSelectedObjects();
            double amount = 0;
            int nb = selection.count();
            QMap<QString, double> amountPerUnit;
            QMap<QString, int> decimalPerUnit;
            for (int i = 0; i < nb; ++i) {
                SKGObjectBase obj(selection.at(i));
                SKGOperationObject op;
                double val;
                double quantity;
                if (obj.getTable() == "v_suboperation_consolidated") {
                    // It's a sub operation
                    SKGSubOperationObject subOp = SKGSubOperationObject(obj.getDocument(), SKGServices::stringToInt(obj.getAttribute("i_SUBOPID")));
                    subOp.getParentOperation(op);

                    val = SKGServices::stringToDouble(obj.getAttribute("f_REALCURRENTAMOUNT"));
                    quantity = SKGServices::stringToDouble(obj.getAttribute("f_REALQUANTITY"));
                } else {
                    op = obj;
                    val = op.getCurrentAmount();
                    quantity = SKGServices::stringToDouble(op.getAttribute("f_QUANTITY"));
                }

                amount += val;

                SKGUnitObject unit;
                op.getUnit(unit);
                amountPerUnit[unit.getSymbol()] += quantity;
                decimalPerUnit[unit.getSymbol()] = unit.getNumberDecimal();
            }

            QString addition;
            QMapIterator<QString, double> i(amountPerUnit);
            while (i.hasNext()) {
                i.next();

                double amo = i.value();
                if (!addition.isEmpty() && amo > 0) {
                    addition += '+';
                }
                SKGServices::SKGUnitInfo u2;
                u2.Symbol = i.key();
                u2.Value = 1.0;
                u2.NbDecimal = decimalPerUnit[i.key()];
                addition += doc->formatMoney(amo, u2);
            }

            QString v2 = doc->formatMoney(amount / unit1.Value, unit1);
            if (nb == 0 || v2 == addition) {
                addition = "";
            } else {
                v2 += '=';
                v2 += addition;
            }
            if (nb) {
                ui.kInfo->setText(i18np("Selection: %1 operation for %2", "Selection: %1 operations for %2", nb, v2));
                if (!unit2.Symbol.isEmpty() && unit2.Value) {
                    v2 = addition % doc->formatMoney(amount / unit2.Value, unit2);
                }
                ui.kInfo->setToolTip(i18np("Selection: %1 operation for %2", "Selection: %1 operations for %2", nb, v2));
            } else {
                ui.kInfo->setText(i18nc("Noun", "Selection: none"));
                ui.kInfo->setToolTip(i18nc("Noun", "Selection: none"));
            }
        }
    }
}

void SKGOperationPluginWidget::onAccountChanged()
{
    SKGTRACEINFUNC(1);
    if (!currentAccount().isEmpty() && ui.kOperationView->getView()->getNbSelectedObjects() == 0) {
        // Get account object
        SKGAccountObject act(getDocument());
        SKGError err = act.setName(currentAccount());
        IFOKDO(err, act.load())

        // Get unit
        SKGUnitObject unit;
        IFOKDO(err, act.getUnit(unit))
        if (!err && !unit.getSymbol().isEmpty()) {
            ui.kUnitEdit->setText(unit.getSymbol());
        }
    }
    onFilterChanged();
}

void SKGOperationPluginWidget::onFilterChanged()
{
    SKGTRACEINFUNC(1);
    if (!isEnabled()) {
        return;
    }
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

    // Enable/ disable widgets
    bool onOneAccount = (!currentAccount().isEmpty());
    ui.kReconciliatorFrame2->setEnabled(onOneAccount);
    if (!onOneAccount && m_modeInfoZone == 1) {
        ui.kReconciliatorFrame2->hide();
        ui.kInfo->show();
        m_modeInfoZone = 0;
    }

    QString current = currentAccount();
    if (!current.isEmpty()) {
        ui.kAccountEdit->setText(current);
    }

    if (m_operationWhereClause.isEmpty()) {
        ui.kResetInternalFilter->hide();
    } else {
        ui.kResetInternalFilter->show();
    }
    QApplication::restoreOverrideCursor();
}

void SKGOperationPluginWidget::fillNumber()
{
    SKGTRACEINFUNC(10);
    QStringList list;
    QString account = ui.kAccountEdit->text();
    QString wc;
    if (!account.isEmpty()) {
        wc = "rd_account_id IN (SELECT id FROM account WHERE t_name='" + SKGServices::stringToSqlString(account) + "')";
    }
    getDocument()->getDistinctValues("v_operation_next_numbers", "i_number", wc, list);

    // Fill completion
    KCompletion* comp = ui.kNumberEdit->completionObject();
    comp->clear();
    comp->insertItems(list);

    m_numberFieldIsNotUptodate = false;
}

void SKGOperationPluginWidget::onFocusChanged()
{
    _SKGTRACEINFUNC(10);
    if (!qApp->closingDown()) {
        if (SKGMainPanel::getMainPanel() && SKGMainPanel::getMainPanel()->currentPage() == this) {
            if (m_numberFieldIsNotUptodate && ui.kNumberEdit->hasFocus()) {
                fillNumber();
            }

            bool test = ui.kTypeEdit->hasFocus() ||
                        //                  ui.kAmountEdit->hasFocus() ||
                        //              ui.kNumberEdit->hasFocus() ||
                        ui.kUnitEdit->hasFocus() ||
                        ui.kCategoryEdit->hasFocus() ||
                        ui.kTrackerEdit->hasFocus() ||
                        ui.kCommentEdit->hasFocus() ||
                        ui.kPayeeEdit->hasFocus();
            if (m_fastEditionAction) {
                m_fastEditionAction->setEnabled(test);
            }
        }
    }
}

void SKGOperationPluginWidget::onFastEdition()
{
    if (SKGMainPanel::getMainPanel()->currentPage() != this) {
        return;
    }
    SKGTRACEINFUNC(10);
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
    SKGError err;

    // Get widget item
    QWidget* w = QApplication::focusWidget();

    // Build the where clause
    QString wc;
    if (ui.kTypeEdit->hasFocus()) {
        wc = "t_mode LIKE '" % SKGServices::stringToSqlString(ui.kTypeEdit->text()) % "%'";
    } else if (ui.kUnitEdit->hasFocus()) {
        wc = "t_UNIT LIKE '" % SKGServices::stringToSqlString(ui.kUnitEdit->text()) % "%'";
    } else if (ui.kCategoryEdit->hasFocus()) {
        wc = "t_CATEGORY LIKE '" % SKGServices::stringToSqlString(ui.kCategoryEdit->text()) % "%'";
    } else if (ui.kTrackerEdit->hasFocus()) {
        wc = "t_REFUND LIKE '" % SKGServices::stringToSqlString(ui.kTrackerEdit->text()) % "%'";
    } else if (ui.kCommentEdit->hasFocus()) {
        wc = "t_comment LIKE '" % SKGServices::stringToSqlString(ui.kCommentEdit->text()) % "%'";
    } else if (ui.kPayeeEdit->hasFocus()) {
        wc = "t_PAYEE LIKE '" % SKGServices::stringToSqlString(ui.kPayeeEdit->text()) % "%'";
    }

    if (!wc.isEmpty()) {
        // Read Setting
        QString fasteditmode = skgoperation_settings::fasteditmode();

        /*
        0-Search in templates only
        1-Search first in templates and after in operations
        2-Search in operations only
        3-Search first in operations and after in templates
        */
        if (fasteditmode == "0") {
            wc += " AND t_template='Y'";
        } else if (fasteditmode == "2") {
            wc += " AND t_template='N'";
        }

        if (wc != m_lastFastEditionWhereClause) {
            m_lastFastEditionWhereClause = wc;
            m_lastFastEditionOperationFound = 0;
        }

        // Look for last operation
        if (m_lastFastEditionOperationFound != 0) {
            wc += " AND id<" % SKGServices::intToString(m_lastFastEditionOperationFound);
        }

        // Add order by
        wc += " ORDER BY ";
        if (fasteditmode == "1") {
            wc += " t_template DESC, ";
        } else if (fasteditmode == "3") {
            wc += " t_template ASC, ";
        }
        wc += "d_date DESC, id DESC LIMIT 1";

        SKGObjectBase::SKGListSKGObjectBase operations;
        err = getDocument()->getObjects("v_operation_display_all", wc, operations);
        if (!err && operations.count()) {
            SKGOperationObject op(operations.at(0));

            m_lastFastEditionOperationFound = op.getID();
            if (isWidgetEditionEnabled(ui.kTypeEdit->lineEdit())) {
                ui.kTypeEdit->setText(op.getMode());
            }
            if (isWidgetEditionEnabled(ui.kUnitEdit->lineEdit())) {
                ui.kUnitEdit->setText(op.getAttribute("t_UNIT"));
            }
            if (isWidgetEditionEnabled(ui.kCategoryEdit->lineEdit())) {
                ui.kCategoryEdit->setText(op.getAttribute("t_CATEGORY"));
            }
            if (isWidgetEditionEnabled(ui.kCommentEdit->lineEdit())) {
                ui.kCommentEdit->setText(op.getComment());
            }
            if (isWidgetEditionEnabled(ui.kPayeeEdit->lineEdit())) {
                ui.kPayeeEdit->setText(op.getAttribute("t_PAYEE"));
            }
            if (isWidgetEditionEnabled(ui.kTrackerEdit->lineEdit())) {
                ui.kTrackerEdit->setText(op.getAttribute("t_REFUND"));
            }
            if (currentAccount().isEmpty()) {
                ui.kAccountEdit->setText(op.getAttribute("t_ACCOUNT"));
            }
            if (isWidgetEditionEnabled(ui.kAmountEdit)) {
                QString quantity = op.getAttribute("f_QUANTITY");

                double quantityVal = SKGServices::stringToDouble(quantity);
                SKGUnitObject unitObject = ui.kUnitEdit->getUnit();
                int nbDec = unitObject.getNumberDecimal();
                if (nbDec == 0) {
                    nbDec = 2;
                }
                quantity = KGlobal::locale()->formatMoney(qAbs(quantityVal), "", nbDec);
                if (quantity.startsWith(KGlobal::locale()->positiveSign())) {
                    quantity = quantity.right(quantity.length() - KGlobal::locale()->positiveSign().length());
                }
                if (quantityVal > 0) {
                    quantity = '+' % quantity;
                } else {
                    quantity = '-' % quantity;
                }
                ui.kAmountEdit->setText(quantity);
            }

            // set next number
            if (isWidgetEditionEnabled(ui.kNumberEdit)) {
                int number = op.getNumber();
                if (number == 0) {
                    ui.kNumberEdit->setText("");
                } else {
                    if (m_numberFieldIsNotUptodate) {
                        fillNumber();
                    }

                    KCompletion* comp = ui.kNumberEdit->completionObject();
                    if (comp) {
                        QStringList list = comp->items();
                        int nb = list.count();
                        int cpt = 0;
                        while (nb >= 0 && cpt >= 0 && cpt < 1000) {
                            ++number;

                            if (list.contains(SKGServices::intToString(number))) {
                                cpt = -2;
                            }
                            ++cpt;
                        }

                        if (cpt < 0) {
                            ui.kNumberEdit->setText(SKGServices::intToString(number));
                        }
                    }
                }
            }

            // Get nb operation linked
            SKGObjectBase::SKGListSKGObjectBase groupedOperations;
            op.getGroupedOperations(groupedOperations);
            int nbGroupedOp = groupedOperations.count();

            // Get nb sub op
            SKGObjectBase::SKGListSKGObjectBase subOperations;
            op.getSubOperations(subOperations);
            int nbSupOp = subOperations.count();

            if (nbSupOp > 1) {
                // It is a SPLIT operation
                ui.kWidgetSelector->setSelectedMode(1);
                displaySubOperations(op , false);
            } else {
                if (nbGroupedOp > 1) {
                    // It is a TRANSFER
                    SKGOperationObject op2(groupedOperations.at(0));
                    if (op2 == op) {
                        op2 = groupedOperations.at(1);
                    }

                    SKGAccountObject targetAccount;
                    op2.getParentAccount(targetAccount);

                    ui.kTargetAccountEdit->setText(targetAccount.getName());
                } else {
                    ui.kWidgetSelector->setSelectedMode(0);
                }
            }

        } else {
            m_lastFastEditionWhereClause = "";
            m_lastFastEditionOperationFound = 0;
        }
    }

    if (w) {
        w->setFocus(Qt::OtherFocusReason);
    }
    QApplication::restoreOverrideCursor();

    // Display error
    SKGMainPanel::displayErrorMessage(err);
}

bool SKGOperationPluginWidget::isTemplateMode()
{
    QAction* act = ui.kOperationView->getShowWidget()->getAction("templates");
    return (act && act->isChecked());
}

void SKGOperationPluginWidget::setTemplateMode(bool iTemplate)
{
    SKGTRACEINFUNC(10);

    if (iTemplate != isTemplateMode()) {
        QAction* act = ui.kOperationView->getShowWidget()->getAction("templates");
        if (act) {
            act->setChecked(iTemplate);
        }

        act = ui.kOperationView->getShowWidget()->getAction("operations");
        if (act) {
            act->setChecked(!iTemplate);
        }
    }
}

void SKGOperationPluginWidget::onBtnModeClicked(int mode)
{
    SKGTRACEINFUNC(10);
    if (mode != 1 && mode != -1) {
        ui.kSubOperationsTable->setRowCount(0);
        ui.kSubOperationsTable->clearContents();
    }

    if (mode == 1) {
        if (ui.kSubOperationsTable->rowCount() == 0) {
            addSubOperationLine(0, ui.kDateEdit->date(), ui.kCategoryEdit->text(), ui.kTrackerEdit->text(), ui.kCommentEdit->text(), ui.kAmountEdit->value(), 0);
        }
    }
    onOperationCreatorModified();
}

void SKGOperationPluginWidget::displaySubOperations(const SKGOperationObject& iOperation , bool iKeepId)
{
    SKGTRACEINFUNC(10);
    ui.kSubOperationsTable->setRowCount(0);
    ui.kSubOperationsTable->clearContents();

    int nbSubOperations = 0;

    QList<SKGObjectBase> subOperations;
    SKGError err =  iOperation.getSubOperations(subOperations);
    nbSubOperations = subOperations.count();
    for (int i = 0; i < nbSubOperations; ++i) {
        SKGSubOperationObject subOperation(subOperations.at(i));

        SKGCategoryObject category;
        subOperation.getCategory(category);

        SKGTrackerObject tracker;
        subOperation.getTracker(tracker);

        addSubOperationLine(i, subOperation.getDate(), category.getFullName(), tracker.getName(),
                            subOperation.getComment(), subOperation.getQuantity(), subOperation.getFormula(),
                            (iKeepId ? subOperation.getID() : 0));
    }

    onQuantityChanged();
}

void SKGOperationPluginWidget::displaySubOperations()
{
    SKGTRACEINFUNC(10);
    SKGOperationObject operation;
    if (getSelectedOperation(operation).isSucceeded()) {
        displaySubOperations(operation);
    }
}

double SKGOperationPluginWidget::getRemainingQuantity()
{
    SKGTRACEINFUNC(10);
    double sumQuantities = 0;
    int nbSubOperations = ui.kSubOperationsTable->rowCount();

    for (int i = 0; i < nbSubOperations ; ++i) {
        QTableWidgetItem* quantityItem = ui.kSubOperationsTable->item(i, m_attributesForSplit.indexOf("f_value"));
        if (quantityItem) {
            sumQuantities = sumQuantities + SKGServices::stringToDouble(quantityItem->text());
        }
    }

    return ui.kAmountEdit->value() - sumQuantities;
}

void SKGOperationPluginWidget::onDateChanged(const QDate& iDate)
{
    SKGTRACEINFUNC(10);
    bool previous = ui.kSubOperationsTable->blockSignals(true);  // Disable signals so that filling cell doesn't create new lines
    if (sender() == ui.kDateEdit && iDate.isValid() && m_previousDate.isValid()) {
        // Refresh dates
        int nbSubOperations = ui.kSubOperationsTable->rowCount();
        for (int i = 0; i < nbSubOperations ; ++i) {
            QTableWidgetItem* dateItem = ui.kSubOperationsTable->item(i, m_attributesForSplit.indexOf("d_date"));
            if (dateItem) {
                QDate previousSubDate = SKGServices::stringToTime(dateItem->text()).date();
                if (previousSubDate.isValid()) {
                    int delta = m_previousDate.daysTo(iDate);
                    dateItem->setText(SKGServices::dateToSqlString(previousSubDate.addDays(delta)));
                }
            }
        }
    }
    m_previousDate = iDate;
    ui.kSubOperationsTable->blockSignals(previous);    // Reenable signals
}

void SKGOperationPluginWidget::onQuantityChanged()
{
    SKGTRACEINFUNC(10);
    int nbSubOperations = ui.kSubOperationsTable->rowCount();

    bool previous = ui.kSubOperationsTable->blockSignals(true);  // Disable signals so that filling cell doesn't create new lines
    if (sender() == ui.kAmountEdit) {
        // Update the total amount
        m_tableDelegate->addParameterValue("total", ui.kAmountEdit->value());

        // Refresh computed amounts
        for (int i = 0; i < nbSubOperations ; ++i) {
            QTableWidgetItem* quantityItem = ui.kSubOperationsTable->item(i, m_attributesForSplit.indexOf("f_value"));
            if (quantityItem) {
                QString formula = quantityItem->toolTip();
                if (formula.startsWith(QLatin1String("="))) {
                    formula = formula.right(formula.length() - 1);
                    formula.replace(',', '.');  // Replace comma by a point in case of typo
                    formula.remove(' ');
                    formula.replace("total", SKGServices::doubleToString(ui.kAmountEdit->value()));

                    QScriptEngine myEngine;
                    QScriptValue result = myEngine.evaluate(formula);
                    if (result.isNumber()) {
                        QString t = SKGServices::doubleToString(result.toNumber());
                        quantityItem->setText(t);
                    }
                }
            }
        }
    }

    // This code put the remaining quantity on the all sub operations with the same ratios ^^^
    // Specific code for the last one to avoid "round" error
    QTableWidgetItem* remainingQuantityItem = ui.kSubOperationsTable->item(nbSubOperations - 1, m_attributesForSplit.indexOf("f_value"));
    if (remainingQuantityItem) {
        QString t = SKGServices::doubleToString(SKGServices::stringToDouble(remainingQuantityItem->text()) + getRemainingQuantity());
        remainingQuantityItem->setText(t);
        remainingQuantityItem->setToolTip(t);
    }
    ui.kSubOperationsTable->blockSignals(previous);    // Reenable signals
}

void SKGOperationPluginWidget::onSubopCellChanged(int row, int column)
{
    SKGTRACEINFUNC(10);
    QTableWidgetItem* subop_cell = ui.kSubOperationsTable->item(row, column);
    QBrush base_brush = ui.kSubOperationsTable->item(row, 0)->foreground();

    int nbSubOperations = ui.kSubOperationsTable->rowCount();
    if (row == nbSubOperations - 1 && column == m_attributesForSplit.indexOf("f_value")) {
        // If the quantity in the last line is edited, we add a new
        // line with the new remaining quantity
        addSubOperationLine(nbSubOperations, ui.kDateEdit->date(), "", "", "", 0, "");
    }
    if (column == m_attributesForSplit.indexOf("f_value")) {
        if (subop_cell->text().toDouble() != 0) {
            onQuantityChanged();
        } else {
            base_brush = KColorScheme(QPalette::Normal).foreground(KColorScheme::NegativeText);
        }
        subop_cell->setForeground(base_brush);
    }
}

void SKGOperationPluginWidget::onRemoveSubOperation(int iRow)
{
    SKGTRACEINFUNC(10);
    bool previous = ui.kSubOperationsTable->blockSignals(true);
    ui.kSubOperationsTable->removeRow(iRow);

    // If all rows removed, add an empty line
    if (ui.kSubOperationsTable->rowCount() == 0) {
        addSubOperationLine(0, ui.kDateEdit->date(), "", "", "", 0, "");
    }

    onQuantityChanged();
    ui.kSubOperationsTable->blockSignals(previous);
}

void SKGOperationPluginWidget::onRotateAccountTools()
{
    SKGTRACEINFUNC(10);
    if (m_modeInfoZone == 0) {
        displayReconciliationInfo();
    } else if (m_modeInfoZone == 1) {
        displaySelectionAmount();
    } else {
        displayBalance();
    }
}


void SKGOperationPluginWidget::displayBalance()
{
    if (m_modeInfoZone != 0) {
        ui.kReconciliatorFrame2->hide();
        ui.kInfo->show();
        m_modeInfoZone = 0;
        onRefreshInformationZoneDelayed();
    }
}

void SKGOperationPluginWidget::displayReconciliationInfo()
{
    if (!currentAccount().isEmpty()) {
        // Only show reconciliation info if only one account is displayed
        ui.kReconciliatorFrame2->show();
        ui.kInfo->hide();
        m_modeInfoZone = 1;
        onRefreshInformationZoneDelayed();
    } else {
        // If more than one account is displayed, skip reconciliation mode
        // (it doesn't make sense to reconciliate several accounts at once)
        // and move to the next modeInfoZone
        m_modeInfoZone = 1;
        onRotateAccountTools();
    }
}

void SKGOperationPluginWidget::displaySelectionAmount()
{
    ui.kReconciliatorFrame2->hide();
    ui.kInfo->show();
    m_modeInfoZone = 2;
    onRefreshInformationZoneDelayed();
}


void SKGOperationPluginWidget::onAutoPoint()
{
    SKGError err;
    SKGTRACEINFUNCRC(10, err);

    {
        SKGBEGINTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Auto point account"), err);
        SKGAccountObject act(getDocument());
        err = act.setName(currentAccount());
        IFOKDO(err, act.load())
        IFOKDO(err, act.autoReconcile(ui.kReconcilitorAmountEdit->value()))

        // Send message
        IFOKDO(err, act.getDocument()->sendMessage(i18nc("An information message", "The account '%1' has been auto pointed", act.getDisplayName()), SKGDocument::Hidden));
    }
    // status bar
    IFOKDO(err, SKGError(0, i18nc("Successful message after an user action", "Account auto pointed.")))

    // Display error
    SKGMainPanel::displayErrorMessage(err);
}

void SKGOperationPluginWidget::onAddFakeOperation()
{
    SKGError err;
    SKGTRACEINFUNCRC(10, err);
    {
        SKGBEGINTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Create fake operation"), err);

        SKGAccountObject accountObj(getDocument());
        IFOKDO(err, accountObj.setName(currentAccount()))
        IFOKDO(err, accountObj.load())

        SKGOperationObject op;
        IFOKDO(err, accountObj.addOperation(op))
        IFOKDO(err, op.setDate(QDate::currentDate()))
        IFOKDO(err, op.setComment(skgoperation_settings::commentFakeOperation()))
        QString payee = skgoperation_settings::payeeFakeOperation();
        if (!payee.isEmpty()) {
            SKGPayeeObject p;
            IFOKDO(err, SKGPayeeObject::createPayee(static_cast<SKGDocumentBank*>(getDocument()), payee, p, true))
            IFOKDO(err, op.setPayee(p))
        }

        SKGUnitObject unit;
        IFOKDO(err, accountObj.getUnit(unit))
        IFOKDO(err, op.setUnit(unit))
        if (skgoperation_settings::automaticPointInReconciliation() && m_modeInfoZone == 1) {
            IFOKDO(err, op.setStatus(SKGOperationObject::POINTED))
        }
        IFOKDO(err, op.save())

        SKGSubOperationObject sop;
        IFOKDO(err, op.addSubOperation(sop))

        SKGStringListList listTmp2;
        double diff = 0;
        getDocument()->executeSelectSqliteOrder(
            "SELECT TOTAL(f_CHECKED) from v_account_display WHERE t_name='" % SKGServices::stringToSqlString(currentAccount()) % '\'',
            listTmp2);
        if (listTmp2.count() == 2) {
            diff = SKGServices::stringToDouble(listTmp2.at(1).at(0)) * unit.getAmount() - ui.kReconcilitorAmountEdit->value();
        }

        IFOKDO(err, sop.setQuantity(-diff))
        IFOKDO(err, sop.setComment(skgoperation_settings::commentFakeOperation()))
        QString category = skgoperation_settings::categoryFakeOperation();
        if (!category.isEmpty()) {
            SKGCategoryObject c;
            IFOKDO(err, SKGCategoryObject::createPathCategory(static_cast<SKGDocumentBank*>(getDocument()), category, c, true))
            IFOKDO(err, sop.setCategory(c))
        }
        IFOKDO(err, sop.save())

        // Send message
        IFOKDO(err, op.getDocument()->sendMessage(i18nc("An information to the user that something was added", "The operation '%1' has been added", op.getDisplayName()), SKGDocument::Hidden));
    }

    // status bar
    IFOKDO(err, SKGError(0, i18nc("Successful message after an user action", "Fake operation created.")))
    else {
        err.addError(ERR_FAIL, i18nc("Error message",  "Creation failed"));
    }

    // Display error
    SKGMainPanel::displayErrorMessage(err);
}


void SKGOperationPluginWidget::onValidatePointedOperations()
{
    SKGError err;
    SKGTRACEINFUNCRC(10, err);

    QString account = currentAccount();
    if (!account.isEmpty()) {
        // Get reconciliated account
        SKGAccountObject act(getDocument());
        IFOKDO(err, act.setName(account))
        IFOKDO(err, act.load())

        QString bindAccount = ui.kReconciliateAccount->currentText();

        if (act.getType() == SKGAccountObject::CREDITCARD && !bindAccount.isEmpty()) {
            //
            IFOK(err) {
                SKGBEGINPROGRESSTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Switch to checked"), err, 3);
                SKGAccountObject accountObj2(getDocument());
                IFOKDO(err, accountObj2.setName(bindAccount))
                IFOKDO(err, accountObj2.load())
                IFOKDO(err, getDocument()->stepForward(1))

                IFOKDO(err, act.transferDeferredOperations(accountObj2))
                IFOKDO(err, getDocument()->stepForward(2))

                // Change reconciliation date of the account
                IFOKDO(err, act.setReconciliationDate(QDate::currentDate()))
                IFOKDO(err, act.setLinkedAccount(accountObj2))
                IFOKDO(err, act.save());

                // Send message
                IFOKDO(err, act.getDocument()->sendMessage(i18nc("An information message", "The account '%1' has been reconciliated", act.getDisplayName()), SKGDocument::Hidden));

                IFOKDO(err, getDocument()->stepForward(3))
            }
        } else {
            // Change state of all operations
            SKGObjectBase::SKGListSKGObjectBase list;
            IFOKDO(err, getDocument()->getObjects("v_operation_display", "t_status='P' AND t_ACCOUNT='" % SKGServices::stringToSqlString(account) % '\'', list));
            int nb = list.count();
            IFOK(err) {
                SKGBEGINPROGRESSTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Switch to checked"), err, nb + 1);
                for (int i = 0; !err && i < nb; ++i) {
                    // Set operation checked
                    SKGOperationObject op(list[i]);
                    err = op.setStatus(SKGOperationObject::CHECKED);
                    IFOKDO(err, op.save())

                    // Send message
                    IFOKDO(err, op.getDocument()->sendMessage(i18nc("An information message", "The operation '%1' has been checked", op.getDisplayName()), SKGDocument::Hidden));

                    IFOKDO(err, getDocument()->stepForward(i + 1))
                }

                // Change reconciliation date of the account
                IFOKDO(err, act.setReconciliationDate(QDate::currentDate()))
                IFOKDO(err, act.save())

                // Send message
                IFOKDO(err, act.getDocument()->sendMessage(i18nc("An information message", "The account '%1' has been reconciliated", act.getDisplayName()), SKGDocument::Hidden));

                IFOKDO(err, getDocument()->stepForward(nb + 1))
            }
        }
    }

    // status bar
    IFOKDO(err, SKGError(0, i18nc("Successful message after an user action", "Operation checked.")))
    else {
        err.addError(ERR_FAIL, i18nc("Error message",  "Switch failed"));
    }

    // Display error
    SKGMainPanel::displayErrorMessage(err);
}

void SKGOperationPluginWidget::addSubOperationLine(int row, const QDate& date, const QString& category, const QString& tracker, const QString& comment, double quantity, const QString& formula, int id)
{
    SKGTRACEINFUNC(10);
    bool previous = ui.kSubOperationsTable->blockSignals(true);

    ui.kSubOperationsTable->insertRow(row);

    // Add a delete icon on the line:
    QTableWidgetItem* hitem = new QTableWidgetItem(KIcon("edit-delete"), "");
    ui.kSubOperationsTable->setVerticalHeaderItem(row, hitem);
    QHeaderView* headerView = ui.kSubOperationsTable->verticalHeader();
    headerView->setMovable(true);

    // Category
    QTableWidgetItem* categoryItem = new QTableWidgetItem(category);
    categoryItem->setToolTip(category);
    categoryItem->setData(Qt::UserRole, id);
    ui.kSubOperationsTable->setItem(row, m_attributesForSplit.indexOf("t_category"), categoryItem);

    // Comment
    QTableWidgetItem* commentItem = new QTableWidgetItem(comment);
    commentItem->setToolTip(comment);
    ui.kSubOperationsTable->setItem(row, m_attributesForSplit.indexOf("t_comment"), commentItem);

    // Quantity
    QString t = SKGServices::doubleToString(quantity);
    QTableWidgetItem* quantityItem = new QTableWidgetItem(t);
    quantityItem->setTextAlignment(Qt::AlignVCenter | Qt::AlignRight);
    quantityItem->setToolTip(formula.isEmpty() ? t : formula);
    ui.kSubOperationsTable->setItem(row, m_attributesForSplit.indexOf("f_value"), quantityItem);

    // Refund
    QTableWidgetItem* trackerItem = new QTableWidgetItem(tracker);
    trackerItem->setToolTip(tracker);
    categoryItem->setData(Qt::UserRole, id);
    ui.kSubOperationsTable->setItem(row, m_attributesForSplit.indexOf("t_refund"), trackerItem);

    // Date
    QTableWidgetItem* dateItem = new QTableWidgetItem(SKGServices::dateToSqlString(date));
    dateItem->setToolTip(SKGServices::dateToSqlString(date));
    ui.kSubOperationsTable->setItem(row, m_attributesForSplit.indexOf("d_date"), dateItem);

    ui.kSubOperationsTable->blockSignals(previous);

    ui.kSubOperationsTable->resizeColumnsToContents();
    if (row == 0 && category.isEmpty()) {
        ui.kSubOperationsTable->horizontalHeader()->resizeSection(0, 300);
    }
}

QWidget* SKGOperationPluginWidget::mainWidget()
{
    return ui.kOperationView->getView();
}

SKGError SKGOperationPluginWidget::getSelectedOperation(SKGOperationObject& operation)
{
    SKGError err;
    SKGObjectBase::SKGListSKGObjectBase selectedOperations = getSelectedObjects();
    if (selectedOperations.count() > 0) {
        operation = selectedOperations.at(0);
        err.setReturnCode(0);
    } else {
        err.setReturnCode(1).setMessage(i18nc("Error message",  "No Operation Selected"));
    }
    return err;
}

void SKGOperationPluginWidget::cleanEditor()
{
    if (getNbSelectedObjects() == 0 || sender() == ui.kCleanBtn) {
        ui.kOperationView->getView()->clearSelection();
        ui.kDateEdit->setDate(QDate::currentDate());
        ui.kPayeeEdit->setText("");
        ui.kCategoryEdit->setText("");
        ui.kTrackerEdit->setText("");
        ui.kAmountEdit->setText("");
        ui.kTypeEdit->setText("");
        ui.kCommentEdit->setText("");
        ui.kNumberEdit->setText("");

        if (!currentAccount().isEmpty()) {
            ui.kAccountEdit->setText(currentAccount());
        }

        setAllWidgetsEnabled();
        m_previousDate = QDate::currentDate();
    }
    if (sender() == ui.kCleanBtn) {
        ui.kWidgetSelector->setSelectedMode(0);
    }
}

bool SKGOperationPluginWidget::isEditor()
{
    return true;
}

void SKGOperationPluginWidget::activateEditor()
{
    if (ui.kWidgetSelector->getSelectedMode() == -1) {
        ui.kWidgetSelector->setSelectedMode(0);
    }
    ui.kPayeeEdit->setFocus();
}

#include "skgoperationpluginwidget.moc"
