/*******************************************************************
 KNotes -- Notes for the KDE project

 Copyright (c) 1997-2002, The KNotes Developers

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*******************************************************************/

#include <qlabel.h>
#include <qsizegrip.h>
#include <qbitmap.h>
#include <qcursor.h>
#include <qpaintdevicemetrics.h>
#include <qsimplerichtext.h>

#include <dcopclient.h>
#include <kapplication.h>
#include <kaction.h>
#include <kxmlgui.h>
#include <kprinter.h>
#include <klocale.h>
#include <kstandarddirs.h>
#include <ksimpleconfig.h>
#include <kdebug.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kprocess.h>
#include <klineeditdlg.h>
#include <kpopupmenu.h>
#include <kmdcodec.h>
#include <kio/netaccess.h>
#include <kglobalsettings.h>
#include <kcolordialog.h>

#include "knote.h"
#include "knotebutton.h"
#include "knoteedit.h"
//#include "knoteconfigdlg.h"
//#include "version.h"
#define KNOTES_VERSION "2.3"

#include <kwin.h>
#include <netwm.h>

// fscking X headers
#ifdef FocusIn
#undef FocusIn
#endif
#ifdef FocusOut
#undef FocusOut
#endif

extern Atom qt_sm_client_id;


// -------------------- Initialisation -------------------- //
KNote::KNote( const Note& note, KONotesView* view, QWidget* parent, const char* name )
  : QFrame( parent, name, WStyle_Customize | WStyle_NoBorder | WDestructiveClose ), mNotesView( view ), mNote(note)
{
    XChangeProperty( x11Display(), winId(), qt_sm_client_id, XA_STRING, 8, 
        PropModeReplace, 0, 0 );

    m_menu = new KPopupMenu(this);
    m_edit_menu = new KPopupMenu(this);

    // create the menu items for the note - not the editor...
    // rename, mail, print, insert date, close, delete, new note
    (new KAction( i18n("New"), "filenew", 0, this, SLOT(slotNewNote()), actionCollection(), "new_note" ))->plug(m_menu);
    (new KAction( i18n("Insert Date"), "knotesdate", 0 , this, SLOT(slotInsDate()), actionCollection(), "insert_date" ))->plug(m_menu);
    (new KAction( i18n("Mail..."), "mail_send", 0, this, SLOT(slotMail()), actionCollection(), "mail_note" ))->plug(m_menu);
    /*new KAction( i18n("Print..."), "fileprint", 0, this, SLOT(slotPrint()), actionCollection(), "print_note" );*/
    /*new KAction( i18n("Preferences..."), "configure", 0, this, SLOT(slotPreferences()), actionCollection(), "configure_note" );*/
    (new KAction( i18n("Color..."), 0, 0, this, SLOT(slotColor()), actionCollection(), "color_note" ))->plug(m_menu);
    (new KAction( i18n("Hide"), "fileclose" , 0, this, SLOT(slotClose()), actionCollection(), "hide_note" ))->plug(m_menu);
    (new KAction( i18n("Delete"), "knotesdelete", 0, this, SLOT(slotKill()), actionCollection(), "delete_note" ))->plug(m_menu);

    m_alwaysOnTop = new KToggleAction( i18n("Always on Top"), "attach", 0, this, SLOT(slotToggleAlwaysOnTop()), actionCollection(), "always_on_top" );
    m_alwaysOnTop->plug(m_menu);

    connect( m_alwaysOnTop, SIGNAL(toggled(bool)), m_alwaysOnTop, SLOT(setChecked(bool)) );
    m_toDesktop = new KListAction( i18n("To Desktop"), 0, this, SLOT(slotToDesktop(int)), actionCollection(), "to_desktop" );
    connect( m_toDesktop->popupMenu(), SIGNAL(aboutToShow()), this, SLOT(slotUpdateDesktopActions()) );

    // create the note header, button and label...
    m_button = new KNoteButton( this );
    m_button->setPixmap( BarIcon( "knotesclose" ) );
    connect( m_button, SIGNAL( clicked() ), this, SLOT( slotClose() ) );

    m_label = new QLabel( this );
    m_label->setAlignment( AlignHCenter );
    m_label->installEventFilter( this );  // recieve events (for dragging & action menu)

    // create the note editor
    m_editor = new KNoteEdit( this );
    m_editor->installEventFilter( this ); // recieve events (for modified)
    m_editor->viewport()->installEventFilter( this );

    setFocusProxy( m_editor );
    
    // create the resize handle
    m_editor->setCornerWidget( new QSizeGrip( this ) );
    {
      int _width = m_editor->cornerWidget()->width();
      int _height = m_editor->cornerWidget()->height();
      QBitmap mask;
      mask.resize( _width, _height );
      mask.fill( color0 );
      QPointArray array;
      array.setPoints( 3, 0, _height, _width, _height, _width, 0 );
      QPainter p;
      p.begin( &mask );
      p.setBrush( color1 );
      p.drawPolygon( array );
      p.end();
      m_editor->cornerWidget()->setMask( mask );
    }
    // set up the look&feel of the note
    setMinimumSize( 20, 20 );
    setFrameStyle( WinPanel | Raised );
    setLineWidth( 1 );

    m_editor->setMargin( 5 );
    m_editor->setFrameStyle( NoFrame );
    m_editor->setBackgroundMode( PaletteBase );

    resize( 200,150 );
    if( !mNote.geometry.isEmpty() ) {
      int x, y;
      unsigned int w, h;
      int rc = XParseGeometry( mNote.geometry.local8Bit(), &x, &y, &w, &h );
      if( rc & WidthValue ) resize( w, height() );
      if( rc & HeightValue ) resize( width(), h );
      if( rc & XValue ) {
	if( rc & XNegative ) move( QApplication::desktop()->width()-width()-x, KNote::y() );
	else move( x, KNote::y() );
      }
      if( rc & YValue ) {
	if( rc & YNegative ) move( KNote::x(), QApplication::desktop()->height()-height()-y );
	else move( KNote::x(), y );	
      }
    }
    kdDebug() << "COLOR is " << mNote.color.name() << endl;
    if( mNote.color.isValid() ) setColor( m_editor->colorGroup().text(), mNote.color );
    setText( mNote.text );
    /* save a bit of mem */
    mNote.text = QString::null;
}

KNote::~KNote()
{
  kdDebug(5500) << k_funcinfo << endl;

 if( m_editor->isModified() ) { 
   saveData();
 }
 emit sigKilled( m_label->text() );
}


// -------------------- public member functions -------------------- //

void KNote::saveData()
{
  if( mNotesView ) {
    mNote.text = m_editor->text();
    mNote.geometry = QString("%1x%2+%3+%4").arg(width()).arg(height()).arg(x()).arg(y());
    mNotesView->updateNote( mNote );
    m_editor->setModified( false );
  }
}

void KNote::setReadOnly( bool ro )
{
  if( m_editor->isReadOnly() == ro ) return;
  if( ro ) {
    if( m_editor->isModified() ) { 
      saveData();
    }
  }
  m_editor->setReadOnly(ro);
}

void KNote::saveDisplayConfig() const
{
kdDebug(5500) << k_funcinfo << endl;
/*
    KSimpleConfig config( m_noteDir.absFilePath( m_configFile ) );
    NETWinInfo wm_client( qt_xdisplay(), winId(), qt_xrootwin(), NET::WMDesktop | NET::WMState );

    config.setGroup( "WindowDisplay" );
    config.writeEntry( "desktop", wm_client.desktop() );
    config.writeEntry( "state", wm_client.state() );
    config.writeEntry( "position", pos() );
*/
}

QString KNote::noteId() const
{
  return mNote.id;
}

QString KNote::name() const
{
  return m_label->text();
}

QString KNote::text() const
{
    return m_editor->text();
}

void KNote::setName( const QString& name )
{
    m_label->setText( name );

    //saveConfig();
}

void KNote::setText( const QString& text )
{
    m_editor->setText( text );
}

void KNote::setNote( const Note& note )
{
  mNote = note;
  setText( mNote.text );
}


// -------------------- public slots -------------------- //

void KNote::slotNewNote()
{
  if( mNotesView ) mNotesView->newNote();
}

void KNote::slotRename()
{
    //pop up dialog to get the new name
    bool ok;
    QString newname = KLineEditDlg::getText( i18n("Please enter the new name:"),
                                             m_label->text(), &ok, this );
    if ( !ok ) // handle cancel
        return;

    if ( newname.isEmpty() ) {
        KMessageBox::sorry( this, i18n("A name must have at least one character") );
        return;
    }

    emit sigRenamed( m_label->text(), newname );
}

void KNote::slotClose()
{
kdDebug(5500) << k_funcinfo << endl;
    m_editor->clearFocus();
    hide(); //just hide the note so it's still available from the dock window
    saveData();    
}

void KNote::slotKill()
{
    if ( KMessageBox::warningYesNo( this,
         i18n("Do you really want to delete this note?"),
         i18n("Delete \"%1\"").arg( m_label->text() ) ) == KMessageBox::Yes )
    {
      kdDebug() << "KNote::slotKill()" << endl;
      if( mNotesView ) mNotesView->deleteNote( mNote );
    }
}

void KNote::slotInsDate()
{
    m_editor->insert( KGlobal::locale()->formatDateTime(QDateTime::currentDateTime()) );
}

void KNote::slotColor()
{
  QColor c;
  if( KColorDialog::getColor(c, backgroundColor(), this) == QDialog::Accepted ) {
    setColor( m_editor->colorGroup().text(), c );
    mNote.color = c;
    saveData();
  }
}

void KNote::slotToggleAlwaysOnTop()
{
    if ( KWin::info(winId()).state & NET::StaysOnTop )
        KWin::clearState( winId(), NET::StaysOnTop );
    else
        KWin::setState( winId(), KWin::info(winId()).state | NET::StaysOnTop );
}

void KNote::slotToDesktop( int id )
{
    if ( id == 0 || id == NETWinInfo::OnAllDesktops )
        KWin::setOnAllDesktops( winId(), true );
    else
        KWin::setOnDesktop( winId(), id );
}

void KNote::slotUpdateDesktopActions()
{
    NETRootInfo wm_root( qt_xdisplay(), NET::NumberOfDesktops | NET::DesktopNames );
    NETWinInfo wm_client( qt_xdisplay(), winId(), qt_xrootwin(), NET::WMDesktop );

    QStringList desktops;
    desktops.append( i18n("&All Desktops") );
    desktops.append( QString::null );           // Separator

    int count = wm_root.numberOfDesktops();
    for ( int n = 1; n <= count; n++ )
        desktops.append( QString("&%1 %2").arg( n ).arg( QString::fromUtf8(wm_root.desktopName( n )) ) );

    m_toDesktop->setItems( desktops );

    if ( wm_client.desktop() == NETWinInfo::OnAllDesktops )
        m_toDesktop->setCurrentItem( 0 );
    else
        m_toDesktop->setCurrentItem( wm_client.desktop() );
}

void KNote::slotMail() //const
{
  DCOPClient* client = kapp->dcopClient();
  QByteArray data;
  QDataStream arg(data, IO_WriteOnly);
  int pos = m_editor->text().find("\n");
  QString subj;
  if( pos > 0 ) subj = m_editor->text().left(pos);
  else subj = m_editor->text();
  arg << "" << "" << "" << subj << m_editor->text() << false;
  if( !client->send( "kmail", "KMailIface", "openComposer(QString,QString,QString,QString,QString,bool)", data ) ) {
    kdWarning() << "DCOP error" << endl;
  }    
}

void KNote::slotPrint() const
{
#if 0
    saveData();

    KPrinter printer;
    printer.setFullPage( true );

    if ( printer.setup() )
    {
        KSimpleConfig config( m_noteDir.absFilePath( m_configFile ), true );
        config.setGroup( "Editor" );

        QFont font( KGlobalSettings::generalFont() );
        font = config.readFontEntry( "font", &font );

        QPainter painter;
        painter.begin( &printer );
        
        const int margin = 40;  // pt

        QPaintDeviceMetrics metrics( painter.device() );
        int marginX = margin * metrics.logicalDpiX() / 72;
        int marginY = margin * metrics.logicalDpiY() / 72;

        QRect body( marginX, marginY,
                    metrics.width() - marginX * 2,
                    metrics.height() - marginY * 2 );

        QString content;
        if ( m_editor->textFormat() == PlainText )
            content = QStyleSheet::convertFromPlainText( m_editor->text() );
        else
            content = m_editor->text();

        QSimpleRichText text( content, font, m_editor->context(),
                              m_editor->styleSheet(), m_editor->mimeSourceFactory(),
                              body.height() /*, linkColor, linkUnderline? */ );

        text.setWidth( &painter, body.width() );
        QRect view( body );

        int page = 1;

        for (;;) {
            text.draw( &painter, body.left(), body.top(), view, colorGroup() );
            view.moveBy( 0, body.height() );
            painter.translate( 0, -body.height() );

            // page numbers
            painter.setFont( font );
            painter.drawText(
                view.right() - painter.fontMetrics().width( QString::number( page ) ),
                view.bottom() + painter.fontMetrics().ascent() + 5, QString::number( page )
            );

            if ( view.top() >= text.height() )
                break;

            printer.newPage();
            page++;
        }

        painter.end();
    }
#endif
}


// -------------------- private slots -------------------- //

void KNote::slotApplyConfig()
{
#if 0
    KSimpleConfig config( m_noteDir.absFilePath( m_configFile ) );

    //do the Editor group: tabsize, autoindent, textformat, font, fontsize, fontstyle
    config.setGroup( "Editor" );

    bool richtext = config.readBoolEntry( "richtext", false );
    if ( richtext )
        m_editor->setTextFormat( RichText );
    else
    {
        m_editor->setTextFormat( PlainText );
        m_editor->setText( m_editor->text() );
    }

    QFont def( KGlobalSettings::generalFont() );
    def = config.readFontEntry( "font", &def );
    m_editor->setTextFont( def );

    def = config.readFontEntry( "titlefont", &def );
    m_label->setFont( def );

    uint tab_size = config.readUnsignedNumEntry( "tabsize", 4 );
    m_editor->setTabStop( tab_size );

    bool indent = config.readBoolEntry( "autoindent", true );
    m_editor->setAutoIndentMode( indent );

    //do the Data Group- name, data
    config.setGroup( "Data" );

    // TODO
    if ( m_label->text().isEmpty() )
    {
        QString notename = config.readEntry( "name", m_configFile );
        m_label->setText( notename );
    }

    // do Display group - bgcolor, fgcolor, transparent
    config.setGroup( "Display" );

    // create a pallete...
    QColor bg = config.readColorEntry( "bgcolor", &(Qt::yellow) );
    QColor fg = config.readColorEntry( "fgcolor", &(Qt::black) );

    setColor( fg, bg );
    
    emit sigConfigChanged();
#endif
}

void KNote::setColor( const QColor &fg, const QColor &bg )
{
    QPalette newpalette = palette();
    newpalette.setColor( QColorGroup::Background, bg );
    newpalette.setColor( QColorGroup::Foreground, fg );
    newpalette.setColor( QColorGroup::Base,       bg ); // text background
    newpalette.setColor( QColorGroup::Text,       fg ); // text color

    // the shadow
    newpalette.setColor( QColorGroup::Midlight, bg.light(110) );
    newpalette.setColor( QColorGroup::Shadow, bg.dark(116) );
    newpalette.setColor( QColorGroup::Light, bg.light(180) );
    newpalette.setColor( QColorGroup::Dark, bg.dark(108) );
    setPalette( newpalette );

    // set the text color
    m_editor->setTextColor( fg );

    // set darker values for the label and button...
    m_button->setBackgroundColor( palette().active().shadow() );
    
    // to set the color of the title
    updateFocus();
}

void KNote::updateFocus()
{
    if ( hasFocus() )
    {
        m_label->setBackgroundColor( palette().active().shadow() );
        m_button->show();
        m_editor->cornerWidget()->show();
    }
    else
    {
        m_label->setBackgroundColor( palette().active().background() );
        m_button->hide();
        m_editor->cornerWidget()->hide();
    }
}

void KNote::updateLayout()
{
    // DAMN, Qt 3.1 still has no support for widgets with a fixed aspect ratio :-(
    // So we have to write our own layout manager...

    int headerHeight = m_label->sizeHint().height();
    int margin = m_editor->margin();

    m_button->setGeometry( 
                frameRect().width() - headerHeight - 2,
                frameRect().y() + 2, 
                headerHeight, 
                headerHeight 
             );

    m_label->setGeometry( 
                frameRect().x() + 2, 
                frameRect().y() + 2,
                frameRect().width() - (m_button->isHidden()?0:headerHeight) - 4,
                headerHeight 
             );
              
    m_editor->setGeometry( 
                contentsRect().x(), 
                contentsRect().y() + headerHeight + 2,
                contentsRect().width(),
                contentsRect().height() - headerHeight - 4
             );

    setMinimumSize( m_editor->cornerWidget()->width() + margin*2 + 4,
                    headerHeight + m_editor->cornerWidget()->height() + margin*2 + 4 );
}

// -------------------- protected methods -------------------- //

void KNote::resizeEvent( QResizeEvent* qre )
{
    QFrame::resizeEvent( qre );
    updateLayout();
}

void KNote::closeEvent( QCloseEvent* /*e*/ )
{
kdDebug(5500) << k_funcinfo << endl;
    slotClose();
}

void KNote::keyPressEvent( QKeyEvent* e )
{
    if ( e->key() == Key_Escape )
        slotClose();
    else
        e->ignore();
}

bool KNote::event( QEvent* ev )
{
    if ( ev->type() == QEvent::LayoutHint )
    {
        updateLayout();
        return true;
    }
    else
        return QFrame::event( ev );
}

bool KNote::eventFilter( QObject* o, QEvent* ev )
{
    if ( o == m_label )
    {
        QMouseEvent* e = (QMouseEvent*)ev;

        if ( ev->type() == QEvent::MouseButtonDblClick )
            slotRename();

        if ( ev->type() == QEvent::MouseButtonRelease && 
             (e->button() == LeftButton || e->button() == MidButton) )
        {
            m_dragging = false;
            m_label->releaseMouse();
            return true;
        }

        if ( ev->type() == QEvent::MouseButtonPress &&
             (e->button() == LeftButton || e->button() == MidButton)) 
        {
            m_pointerOffset = e->pos();
            m_label->grabMouse( sizeAllCursor );
            
            e->button() == LeftButton ? raise() : lower();
                
            return true;
        }

        if ( ev->type() == QEvent::MouseMove && m_label == mouseGrabber() )
        {
            if ( m_dragging )
                move( QCursor::pos() - m_pointerOffset );
            else
            {
                m_dragging = (
                    (e->pos().x() - m_pointerOffset.x()) *
                    (e->pos().x() - m_pointerOffset.x())
                    +
                    (e->pos().y() - m_pointerOffset.y()) *
                    (e->pos().y() - m_pointerOffset.y())   >= 9 
                );
            }
            return true;
        }

        if ( m_menu && ( ev->type() == QEvent::MouseButtonPress )
            && ( e->button() == RightButton ) )
        {
            m_menu->popup( QCursor::pos() );
            return true;
        }

        return false;
    }
    
    if ( o == m_editor )
    {
        if ( ev->type() == QEvent::FocusOut )
        {
            updateFocus();
            if ( m_editor->isModified() )
                saveData();
        }
        else if ( ev->type() == QEvent::FocusIn )
            updateFocus();

        return false;
    }
    /*
    if ( o == m_editor->viewport() )
    {
        if ( ev->type() == QEvent::MouseButtonPress )
            if ( m_edit_menu && ((QMouseEvent*)ev)->button() == RightButton )
            {
                m_edit_menu->popup( QCursor::pos() );
                return true;
            }
    }
    */
    return false;
}

#include "knote.moc"
#include "knotebutton.moc"
