/*
   SPDX-FileCopyrightText: 2021 Laurent Montel <montel@kde.org>

   SPDX-License-Identifier: LGPL-2.0-or-later
*/

#include "admininvitemodel.h"
#include <KLocalizedString>
#include <QDateTime>

AdminInviteModel::AdminInviteModel(QObject *parent)
    : QAbstractListModel(parent)
{
}

AdminInviteModel::~AdminInviteModel() = default;

int AdminInviteModel::rowCount(const QModelIndex &parent) const
{
    if (parent.isValid()) {
        return 0; // flat model
    }
    return mAdminInvites.count();
}

QVariant AdminInviteModel::headerData(int section, Qt::Orientation orientation, int role) const
{
    if (role == Qt::DisplayRole && orientation == Qt::Horizontal) {
        switch (static_cast<AdminInviteRoles>(section)) {
        case AdminInviteModel::UserIdentifier:
        case AdminInviteModel::RoomId:
            break;
        case AdminInviteModel::Identifier:
            return i18n("Token");
        case AdminInviteModel::CreateStr:
        case AdminInviteModel::Create:
            return i18n("Created at");
        case AdminInviteModel::Uses:
            return i18n("Uses");
        case AdminInviteModel::MaxUses:
            return i18n("Uses left");
        case AdminInviteModel::Expire:
            return i18n("Expiration");
        }
    }
    return {};
}

int AdminInviteModel::columnCount(const QModelIndex &parent) const
{
    Q_UNUSED(parent)
    return static_cast<int>(AdminInviteModel::LastColumn) + 1;
}

const QVector<InviteInfo> &AdminInviteModel::adminInvites() const
{
    return mAdminInvites;
}

void AdminInviteModel::setAdminInvites(const QVector<InviteInfo> &newAdminInvites)
{
    if (rowCount() != 0) {
        beginRemoveRows(QModelIndex(), 0, mAdminInvites.count() - 1);
        mAdminInvites.clear();
        endRemoveRows();
    }
    if (!newAdminInvites.isEmpty()) {
        beginInsertRows(QModelIndex(), 0, newAdminInvites.count() - 1);
        mAdminInvites = newAdminInvites;
        endInsertRows();
    }
}

QVariant AdminInviteModel::data(const QModelIndex &index, int role) const
{
    if (index.row() < 0 || index.row() >= mAdminInvites.count()) {
        return {};
    }
    if (role != Qt::DisplayRole) {
        return {};
    }

    const InviteInfo &inviteInfo = mAdminInvites.at(index.row());
    const int col = index.column();
    switch (col) {
    case AdminInviteModel::UserIdentifier:
        return inviteInfo.userIdentifier();
    case AdminInviteModel::Identifier:
        return inviteInfo.identifier();
    case AdminInviteModel::RoomId:
        return inviteInfo.roomId();
    case AdminInviteModel::Create:
        return inviteInfo.createDateTime();
    case AdminInviteModel::CreateStr:
        return inviteInfo.createDateTime().toString();
    case AdminInviteModel::Uses:
        return inviteInfo.uses();
    case AdminInviteModel::MaxUses:
        return inviteInfo.maxUses();
    case AdminInviteModel::Expire:
        return expireInvitation(inviteInfo);
    }
    return {};
}

QString AdminInviteModel::expireInvitation(const InviteInfo &inviteInfo) const
{
    if (inviteInfo.expireDateTime() > QDateTime::currentDateTime()) {
        return i18n("Expire in %1 days", QDateTime::currentDateTime().daysTo(inviteInfo.expireDateTime()));
    } else {
        return i18n("Expired");
    }
}

void AdminInviteModel::removeInvite(const QString &identifier)
{
    const int roomCount = mAdminInvites.count();
    for (int i = 0; i < roomCount; ++i) {
        if (mAdminInvites.at(i).identifier() == identifier) {
            beginRemoveRows(QModelIndex(), i, i);
            mAdminInvites.removeAt(i);
            endRemoveRows();
            break;
        }
    }
}
