/* This file is part of the KDE project
 * SPDX-FileCopyrightText: 2011 C. Boemann <cbo@boemann.dk>
 *
 * SPDX-License-Identifier: LGPL-2.0-or-later
 */

#include "SimpleRootAreaProvider.h"

#include "TextShape.h"

#include <KoBorder.h>
#include <KoInsets.h>
#include <KoTextLayoutObstruction.h>
#include <KoTextLayoutRootArea.h>

SimpleRootAreaProvider::SimpleRootAreaProvider(KoTextShapeData *data, TextShape *textshape)
    : m_textShape(textshape)
    , m_area(nullptr)
    , m_textShapeData(data)
    , m_fixAutogrow(false)

{
}

KoTextLayoutRootArea *
SimpleRootAreaProvider::provide(KoTextDocumentLayout *documentLayout, const RootAreaConstraint &, int requestedPosition, bool *isNewRootArea)
{
    if (m_area == nullptr) {
        *isNewRootArea = true;
        m_area = new KoTextLayoutRootArea(documentLayout);
        m_area->setAssociatedShape(m_textShape);
        m_textShapeData->setRootArea(m_area);

        return m_area;
    }
    if (requestedPosition == 0) {
        *isNewRootArea = false;
        return m_area;
    }
    return nullptr;
}

void SimpleRootAreaProvider::releaseAllAfter(KoTextLayoutRootArea *afterThis)
{
    Q_UNUSED(afterThis);
}

void SimpleRootAreaProvider::doPostLayout(KoTextLayoutRootArea *rootArea, bool isNewRootArea)
{
    Q_UNUSED(isNewRootArea);

    m_textShape->update(m_textShape->outlineRect());

    QSizeF newSize = m_textShape->size()
        - QSizeF(m_textShapeData->leftPadding() + m_textShapeData->rightPadding(), m_textShapeData->topPadding() + m_textShapeData->bottomPadding());

    KoBorder *border = m_textShape->border();
    if (border) {
        newSize -= QSizeF(border->borderWidth(KoBorder::LeftBorder) + border->borderWidth(KoBorder::RightBorder),
                          border->borderWidth(KoBorder::TopBorder) + border->borderWidth(KoBorder::BottomBorder));
    } else {
        KoInsets inset = m_textShape->strokeInsets();
        newSize -= QSizeF((inset.right + inset.left), (inset.bottom + inset.top));
    }

    if (m_textShapeData->verticalAlignment() & Qt::AlignBottom) { }
    if (m_textShapeData->verticalAlignment() & Qt::AlignVCenter) { }

    if (m_textShapeData->resizeMethod() == KoTextShapeData::AutoResize) {
        newSize = QSizeF(rootArea->right() - rootArea->left(), rootArea->bottom() - rootArea->top());
        // HACK: The size we get from the shape will cut descents,
        // so we add some reasonable (approximate) value that works ok in normal cases
        QSizeF s(0.0, 0.0);
        int lines = m_textShapeData->document()->lineCount();
        if (lines > 0) {
            s.setHeight(newSize.height() / lines * 0.2);
        }
        newSize += s;
    } else {
        if (m_textShapeData->resizeMethod() == KoTextShapeData::AutoGrowWidthAndHeight || m_textShapeData->resizeMethod() == KoTextShapeData::AutoGrowHeight) {
            qreal height = rootArea->bottom() - rootArea->top();
            if (height > newSize.height()) {
                newSize.setHeight(height);
            }
            if (m_textShape->shapeId() == "AnnotationTextShapeID") {
                if (height < newSize.height()) {
                    newSize.setHeight(rootArea->bottom() - rootArea->top());
                }
            }
        }
        if (m_textShapeData->resizeMethod() == KoTextShapeData::AutoGrowWidthAndHeight || m_textShapeData->resizeMethod() == KoTextShapeData::AutoGrowWidth) {
            qreal width = rootArea->right() - rootArea->left();
            if (width > newSize.width()) {
                newSize.setWidth(rootArea->right() - rootArea->left());
            }
        }
    }

    qreal newBottom = rootArea->top() + newSize.height();
    KoFlake::Position sizeAnchor = KoFlake::TopLeftCorner;

    if (m_textShapeData->verticalAlignment() & Qt::AlignBottom) {
        if (true /*FIXME test no page based shapes interfering*/) {
            rootArea->setVerticalAlignOffset(newBottom - rootArea->bottom());
            sizeAnchor = KoFlake::BottomLeftCorner;
        }
    }
    if (m_textShapeData->verticalAlignment() & Qt::AlignVCenter) {
        if (true /*FIXME test no page based shapes interfering*/) {
            rootArea->setVerticalAlignOffset((newBottom - rootArea->bottom()) / 2);
            sizeAnchor = KoFlake::CenteredPosition;
        }
    }
    newSize += QSizeF(m_textShapeData->leftPadding() + m_textShapeData->rightPadding(), m_textShapeData->topPadding() + m_textShapeData->bottomPadding());
    if (border) {
        newSize += QSizeF(border->borderWidth(KoBorder::LeftBorder) + border->borderWidth(KoBorder::RightBorder),
                          border->borderWidth(KoBorder::TopBorder) + border->borderWidth(KoBorder::BottomBorder));
    } else {
        KoInsets inset = m_textShape->strokeInsets();
        newSize += QSizeF((inset.right + inset.left), (inset.bottom + inset.top));
    }

    if (newSize != m_textShape->size()) {
        // OO grows to both sides so when to small the initial layouting needs
        // to keep that into account.
        if (m_fixAutogrow) {
            m_fixAutogrow = false;
            QSizeF tmpSize = m_textShape->size();
            tmpSize.setWidth(newSize.width());
            QPointF centerpos = rootArea->associatedShape()->absolutePosition(KoFlake::CenteredPosition);
            m_textShape->setSize(tmpSize);
            m_textShape->setAbsolutePosition(centerpos, KoFlake::CenteredPosition);
            centerpos = rootArea->associatedShape()->absolutePosition(sizeAnchor);
            m_textShape->setSize(newSize);
            m_textShape->setAbsolutePosition(centerpos, sizeAnchor);
        }
        m_textShape->setSize(newSize);
    }

    m_textShape->update(m_textShape->outlineRect());
}

void SimpleRootAreaProvider::updateAll()
{
    if (m_area && m_area->associatedShape()) {
        m_area->associatedShape()->update();
    }
}

QRectF SimpleRootAreaProvider::suggestRect(KoTextLayoutRootArea *rootArea)
{
    // Come up with a rect, but actually we don't need the height, as we set it to infinite below
    //  Still better keep it for completeness sake
    QRectF rect(QPointF(), m_textShape->size());
    rect.adjust(m_textShapeData->leftPadding(), m_textShapeData->topPadding(), -m_textShapeData->rightPadding(), -m_textShapeData->bottomPadding());

    KoBorder *border = m_textShape->border();
    if (border) {
        rect.adjust(border->borderWidth(KoBorder::LeftBorder),
                    border->borderWidth(KoBorder::TopBorder),
                    -border->borderWidth(KoBorder::RightBorder),
                    -border->borderWidth(KoBorder::BottomBorder));
    } else {
        KoInsets inset = m_textShape->strokeInsets();
        rect.adjust(inset.left, inset.top, -inset.right, -inset.bottom);
    }

    // In simple cases we always set height way too high so that we have no breaking
    // If the shape grows afterwards or not is handled in doPostLayout()
    rect.setHeight(1E6);

    if (m_textShapeData->resizeMethod() == KoTextShapeData::AutoGrowWidthAndHeight || m_textShapeData->resizeMethod() == KoTextShapeData::AutoGrowWidth
        || m_textShapeData->resizeMethod() == KoTextShapeData::AutoResize) {
        rootArea->setNoWrap(1E6);
    }

    // Make sure the size is not negative due to padding and border width
    // This can happen on vertical lines containing text on shape.
    if (rect.width() < 0) {
        rect.setWidth(0);
    }
    return rect;
}

QList<KoTextLayoutObstruction *> SimpleRootAreaProvider::relevantObstructions(KoTextLayoutRootArea *rootArea)
{
    Q_UNUSED(rootArea);

    QList<KoTextLayoutObstruction *> obstructions;
    /*
        m_textShape->boundingRect();
        QList<KoShape *> shapes;
        shapes = manager->shapesAt(canvasRect):
    */
    return obstructions;
}
