#include <klocale.h>
#include <klistview.h>
#include <kmessagebox.h>

#include <qbuttongroup.h>
#include <qcombobox.h>
#include <qgroupbox.h>
#include <qheader.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qlayout.h>


#include "gphoto_interface.h"
#include "camerasettings.h"

#include "setupcamera.h"

SetupCamera::SetupCamera(QWidget *parent, const char* name )
    : QWidget(parent, name)
{


    mUMSCameraNameActual = QString("Directory Browse");
    mUMSCameraNameShown  = QString("USB Mass Storage");

    mLayout = new QGridLayout(this, 1, 1, 5, -1);

    mCameraListView = new KListView(this);
    mCameraListView->addColumn(i18n("Camera Selection"));
    mCameraListView->setFullWidth(true);
    //KListView_cameras->setAllColumnsShowFocus( true );
    //KListView_cameras->setRootIsDecorated( true );

    mLayout->addMultiCellWidget(mCameraListView, 0, 3, 0, 0);


    mAutoDetectButton = new QPushButton(this);
    mAutoDetectButton->setText(i18n( "Auto-Detect" ) );
    mLayout->addWidget(mAutoDetectButton,0,1);


    mPortButtonGroup = new QButtonGroup(this);
    mPortButtonGroup->setTitle( i18n( "Port Type" ) );

    mSerialRadioButton = new QRadioButton(mPortButtonGroup);
    mSerialRadioButton->setGeometry( QRect( 20, 40, 121, 21 ) );
    mSerialRadioButton->setText( i18n( "Serial" ) );

    mUSBRadioButton = new QRadioButton(mPortButtonGroup);
    mUSBRadioButton->setGeometry( QRect( 20, 80, 121, 21 ) );
    mUSBRadioButton->setText( i18n( "USB" ) );

    mLayout->addWidget( mPortButtonGroup, 1, 1 );

    mPortSettingBox = new QGroupBox(this);
    mPortSettingBox->setTitle( i18n( "Port Setting" ) );

    mPortPathLabel = new QLabel(mPortSettingBox);
    mPortPathLabel->setGeometry( QRect( 20, 30, 160, 61 ) );
    mPortPathLabel->setText(i18n("Set port path for serial ports"));

    mPortPathComboBox = new QComboBox(false, mPortSettingBox);
    mPortPathComboBox->setGeometry( QRect( 20, 100, 161, 31 ) );
    mPortPathComboBox->setEditable(true);

    mLayout->addWidget( mPortSettingBox, 2, 1 );

    mPathSettingBox = new QGroupBox(this);
    mPathSettingBox->setTitle( i18n( "Path Setting" ) );

    mPathLabel = new QLabel(mPathSettingBox);
    mPathLabel->setGeometry( QRect( 20, 30, 160, 61 ) );
    mPathLabel->setText(i18n("For USB Mass Storage Cameras, \n"
                             "set path for mount directory \n"
                             "of the camera"));

    mPathComboBox = new QComboBox(false, mPathSettingBox);
    mPathComboBox->setGeometry( QRect( 20, 100, 161, 31 ) );
    mPathComboBox->setEditable(true);

    mLayout->addWidget( mPathSettingBox, 3, 1);


    // signals and slots connections

    connect( mAutoDetectButton, SIGNAL( clicked() ),
             this, SLOT( slot_autoDetect() ) );
    connect( mCameraListView, SIGNAL( selectionChanged(QListViewItem *) ),
             this, SLOT( slot_setModel(QListViewItem *) ) );
    connect( mPortButtonGroup, SIGNAL( clicked(int) ),
             this, SLOT( slot_setPort(int) ) );


    gpInterface = new Gphoto_Interface();
    getCameraList();
    getPortList();

    loadSettings();

}

SetupCamera::~SetupCamera()
{
    delete gpInterface;
}

void SetupCamera::slot_autoDetect()
{
    QString cameraModel;
    QString cameraPort;

    if (gpInterface->autoDetectCamera(cameraModel, cameraPort) != 0) {
        KMessageBox::error(this,i18n("Failed to auto-detect camera!\n"
                           "Please retry or try setting manually."));
        return;
    }


    QListViewItem *item = mCameraListView->firstChild();
    while(item) {
        if (item->text(0) == cameraModel) {
            mCameraListView->setSelected(item,true);
            mCameraListView->ensureItemVisible(item);
            break;
        }
        item = item->nextSibling();
    }

    if (cameraPort.startsWith("serial:")) {
        slot_setPort(mPortButtonGroup->id(mSerialRadioButton));
        mPortPathComboBox->setEnabled(true);
        mPortPathComboBox->clear();
        mPortPathComboBox->insertItem(cameraPort, 0);
    }
    else if (mCameraPort.startsWith("usb:")) {
        slot_setPort(mPortButtonGroup->id(mUSBRadioButton));
    }

    QString infoStr(i18n("Detected a "));
    infoStr +=  cameraModel + i18n(" with port ") + cameraPort;
    KMessageBox::information(this,infoStr);

}

void SetupCamera::slot_setModel(QListViewItem *item)
{
    if (!item) return;

    QString cameraModel = item->text(0);
    QStringList supportedPorts;

    gpInterface->getCameraPorts(cameraModel, supportedPorts);

    if (supportedPorts.contains("serial")) {
        mSerialRadioButton->setEnabled(true);
        mSerialRadioButton->setChecked(true);
        slot_setPort(mPortButtonGroup->id(mSerialRadioButton));
    }
    else {
        mSerialRadioButton->setChecked(false);
        mSerialRadioButton->setEnabled(false);
    }

    if (supportedPorts.contains("usb")) {
        mUSBRadioButton->setEnabled(true);
        mUSBRadioButton->setChecked(true);
        slot_setPort(mPortButtonGroup->id(mUSBRadioButton));
    }
    else {
        mUSBRadioButton->setChecked(false);
        mUSBRadioButton->setEnabled(false);
    }

    if (cameraModel == mUMSCameraNameShown) {
        mSerialRadioButton->setChecked(false);
        mSerialRadioButton->setEnabled(false);
        mUSBRadioButton->setChecked(false);
        mUSBRadioButton->setEnabled(false);
        mPathComboBox->setEnabled(true);
        mPathComboBox->clear();
        mCameraPath = QString("/mnt/camera");
        mPathComboBox->insertItem(mCameraPath,0);
        slot_setPort(-1);
    }
    else {
        mPathComboBox->setEnabled(true);
        mPathComboBox->clear();
        mCameraPath = QString("/");
        mPathComboBox->insertItem(mCameraPath,0);
        mPathComboBox->setEnabled(false);
    }
}

void SetupCamera::slot_setPort(int portType)
{
    mPortButtonGroup->setButton(portType);

    if (portType == mPortButtonGroup->id(mSerialRadioButton)) {
        mPortPathLabel->setText(i18n("Set Port Path for \n"
                                "serial port"));
        mPortPathComboBox->setEnabled(true);

        QString current = mPortPathComboBox->currentText();
        mPortPathComboBox->clear();
        if (mSupportedSerialPorts.count() > 0)
            mPortPathComboBox->insertStringList(mSupportedSerialPorts);
        for (int i=0; i<mPortPathComboBox->count(); i++) {
            if (mPortPathComboBox->text(i) == current) {
                mPortPathComboBox->setCurrentItem(i);
                break;
            }
        }
        return;
    }

    if (portType == mPortButtonGroup->id(mUSBRadioButton)) {
        mPortPathLabel->setText(i18n("No further configuration\n"
                                "needed for USB"));
        mPortPathComboBox->setEnabled(true);
        mPortPathComboBox->clear();
        mPortPathComboBox->setEnabled(false);
        return;
    }


    mPortPathLabel->setText(i18n("No further configuration\n"
                            "needed for USB\n"
                            "MassStorage Camera"));
    mPortPathComboBox->setEnabled(false);
}



void SetupCamera::getCameraList()
{
    QStringList cameraList;
    int numSupported;

    gpInterface->getSupportedCameras(cameraList, numSupported);

    for (int i=0; i<numSupported; i++)
        if (cameraList[i] == mUMSCameraNameActual)
            new KListViewItem(mCameraListView,
                              mUMSCameraNameShown);
        else
            new KListViewItem(mCameraListView, cameraList[i]);

}

void SetupCamera::getPortList()
{
    QStringList portList;
    gpInterface->getSupportedPorts(portList);

    mSupportedSerialPorts.clear();

    for (unsigned int i=0; i<portList.count(); i++) {
        if ((portList[i]).startsWith("serial:"))
            mSupportedSerialPorts.append(portList[i]);
    }
}

void SetupCamera::loadSettings()
{
    CameraSettings camSettings;
    camSettings.readSettings();

    mCameraModel = camSettings.getCameraModel();

    QListViewItem *item = mCameraListView->firstChild();
    while(item) {
        if (mCameraModel == mUMSCameraNameActual &&
            item->text(0) == mUMSCameraNameShown) {
            mCameraListView->setSelected(item, true);
            mCameraListView->ensureItemVisible(item);
            break;
        }
        if (item->text(0) == mCameraModel) {
            mCameraListView->setSelected(item, true);
            mCameraListView->ensureItemVisible(item);
            break;
        }
        item = item->nextSibling();
    }


    mCameraPort = camSettings.getCameraPort();

    if (mCameraPort.startsWith("serial:")) {
        slot_setPort(mPortButtonGroup->id(mSerialRadioButton));
        mPortPathComboBox->insertItem(mCameraPort, 0);
    }
    else if (mCameraPort.startsWith("usb:")) {
        slot_setPort(mPortButtonGroup->id(mUSBRadioButton));
    }


    mCameraPath = camSettings.getCameraPath();
    mPathComboBox->setEnabled(true);
    mPathComboBox->insertItem(mCameraPath,0);

    if (mCameraModel != mUMSCameraNameActual)
        mPathComboBox->setEnabled(false);

}

void SetupCamera::saveSettings()
{
    CameraSettings camSettings;

    if (mCameraListView->selectedItem()) {
        mCameraModel = mCameraListView->selectedItem()->text(0);
        if (mCameraModel == mUMSCameraNameShown)
            mCameraModel = mUMSCameraNameActual;
    }
    else
        mCameraModel = "";

    if (mPortButtonGroup->selected()) {
        QString type = mPortButtonGroup->selected()->text();
        if (type == i18n("Serial"))
            mCameraPort = mPortPathComboBox->currentText();
        else if (type == i18n("USB"))
            mCameraPort="usb:";
    }
    else
        mCameraPort = "None";

    if (mCameraModel == mUMSCameraNameActual)
        mCameraPath = mPathComboBox->currentText();
    else
        mCameraPath = "/";

    camSettings.setCameraModel(mCameraModel);
    camSettings.setCameraPort(mCameraPort);
    camSettings.setCameraPath(mCameraPath);
    camSettings.saveSettings();
}
