#include <qpixmap.h>
#include <qpainter.h>
#include <qcolor.h>
#include <qpen.h>
#include <qdir.h>
#include <qfile.h>
#include <qpopmenu.h>
#include <qvaluevector.h>
#include <qdragobject.h>

#include <kurl.h>
#include <kurldrag.h>
#include <kstandarddirs.h>
#include <kmdcodec.h>
#include <ktrader.h>
#include <krun.h>
#include <kservice.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kpropertiesdialog.h>
#include <kio/job.h>

#include "thumbnailcreator.h"
#include "thumbnailsize.h"
#include "customevents.h"
#include "imageviewer.h"
#include "slideshow.h"
#include "albumiconitem.h"

#include "albumiconview.h"


AlbumIconView::AlbumIconView(QWidget *_parent):
    QIconView(_parent)
{
    setGridX(mThumbSize.getSizeAsNumber() + 5);
    setAutoArrange(true);
    setSorting(true);
    setResizeMode(QIconView::Adjust);
    setSelectionMode(QIconView::Extended);
    setSpacing(5);

    mThumbCreator = new ThumbnailCreator(this);
    mThumbCreator->start();

    mThumbBaseCacheDir    = "";

    KGlobal::dirs()->addResourceType("thumbnails","share/thumbnails/");
    mThumbBaseCacheDir = QDir::cleanDirPath( KGlobal::dirs()->resourceDirs("thumbnails")[0] );


    mBusy = false;

    mSlideShowDelay = 3;
    mSlideShowTransitionEffect = true;
    mSlideShowFileNamePosition = SlideShow::Off;
    mUseInternalImageViewer = true;
    mImageViewerName = "";

    connect(this, SIGNAL(doubleClicked (QIconViewItem *)),
            this, SLOT(slot_doubleClicked(QIconViewItem *)));
    connect(this, SIGNAL(rightButtonClicked(QIconViewItem *,
                                            const QPoint&)),
            this, SLOT(slot_rightClicked(QIconViewItem *,
                                          const QPoint&)));
    connect(this, SIGNAL(itemRenamed(QIconViewItem *,
                                      const QString &)),
	   this, SLOT(slot_itemRenamed(QIconViewItem *,
                                       const QString &)) );

}


AlbumIconView::~AlbumIconView()
{
    delete mThumbCreator;
}

void AlbumIconView::setPath(const KURL& path)
{

    if (path == mCurrentPath) return;

    mThumbCreator->stop();
    mThumbCreator->forceRegeneration(false);
    clear();

    mNumThumbsRequested  = 0;
    mNumThumbsDownloaded = 0;

    mCurrentPath.setPath(QDir::cleanDirPath(path.path()));

    // Generate the thumbnail dir name
    KMD5 md5( QFile::encodeName(mCurrentPath.url()) );
    QCString hash = md5.hexDigest();
    QString thumbPath = QString::fromLatin1( hash.data(), 4 ) + "/" +
                        QString::fromLatin1( hash.data()+4, 4 ) + "/" +
                        QString::fromLatin1( hash.data()+8 ) + "/";

    // Create the thumbnail cache dirs
    QString thumbSmallCacheDir = locateLocal( "thumbnails", thumbPath + "60" + "/" );
    QString thumbMediumCacheDir = locateLocal( "thumbnails", thumbPath + "large" + "/" );
    QString thumbLargeCacheDir = locateLocal( "thumbnails", thumbPath + "huge" + "/" );

    mThumbCreator->setThumbnailCacheDirs(thumbSmallCacheDir,
                                         thumbMediumCacheDir,
                                         thumbLargeCacheDir);

}


void AlbumIconView::addItem(KFileItem* fileItem)
{

    int pixelSize = mThumbSize.getSizeAsNumber();

    QPixmap thumbnail(pixelSize,pixelSize);
    QPainter painter(&thumbnail);
    painter.eraseRect(0,0,pixelSize,pixelSize);
    painter.drawRect(0,0,pixelSize,pixelSize);

    AlbumIconItem* iconItem = new AlbumIconItem(this, fileItem->url().fileName(),
                                                thumbnail, fileItem);
    fileItem->setExtraData(this, iconItem);
    iconItem->setRenameEnabled(true);

    mThumbCreator->requestThumbnail(fileItem->url().path(), mThumbSize);
    mNumThumbsRequested++;
}

void AlbumIconView::deleteItem(KFileItem* fileItem)
{

    AlbumIconItem* iconItem =
        static_cast<AlbumIconItem*>(fileItem->extraData(this));
    if (!iconItem) return;

    delete iconItem;
    fileItem->removeExtraData(this);

    arrangeItemsInGrid();
}

void AlbumIconView::refreshItem(KFileItem* fileItem)
{
    if (!fileItem) return;
    AlbumIconItem* iconItem =
        static_cast<AlbumIconItem*>(fileItem->extraData(this));
    if (!iconItem) return;

    iconItem->setText(fileItem->text());
    sort();
}

void AlbumIconView::cancelThumbnailGeneration()
{
    mThumbCreator->stop();
}

void AlbumIconView::pauseThumbnailGeneration()
{
    if (mThumbCreator->isPaused())
        mThumbCreator->pause(false);
    else
        mThumbCreator->pause(true);
}

void AlbumIconView::customEvent(QCustomEvent *event)
{
    if (event->type() >= QCustomEvent::User) {

        switch (event->type()) {

        case EventThumbnailT: {

            EventThumbnail* ev(static_cast<EventThumbnail *>(event));

            QString directory;
            QString fileName;
            QPixmap thumbnail;

            ev->getParentFolder(directory);
            ev->getFileName(fileName);
            ev->getThumbNail(thumbnail);


            for (QIconViewItem *item = firstItem();
                 item; item=item->nextItem()) {

                if (item->text() == fileName) {
                    item->setPixmap(thumbnail);
                    mNumThumbsDownloaded++;
                    int progress = int(double(mNumThumbsDownloaded)/
                                       double(mNumThumbsRequested)*100);
                    emit signal_albumProgress(progress);
                    return;
                }

            }

            break;

        }

        case EventBusyT: {

	  EventBusy* ev(static_cast<EventBusy *>(event));
            if (ev->busy()) {
                if (!mBusy) {
                    mBusy = true;
                    emit signal_albumBusy(true);
                }
            }
            else {
                if (mBusy) {
                    mThumbCreator->forceRegeneration(false);
                    mBusy = false;
                    emit signal_albumBusy(false);
                }
            }

	    break;

        }

        default:
            break;

        }

    }

}

void AlbumIconView::setThumbnailSize(const ThumbnailSize& thumbSize)
{
    if (mThumbSize == thumbSize)
        return;
    mThumbSize = thumbSize;
    setGridX(mThumbSize.getSizeAsNumber() + 10);
}

ThumbnailSize AlbumIconView::getThumbnailSize() const
{
    return mThumbSize;
}

void AlbumIconView::resizeThumbnails(const ThumbnailSize& thumbSize)
{
    if (mThumbSize == thumbSize)
        return;

    mThumbCreator->stop();
    mThumbCreator->forceRegeneration(false);
    mNumThumbsRequested  = 0;
    mNumThumbsDownloaded = 0;

    mThumbSize = thumbSize;

    setGridX(mThumbSize.getSizeAsNumber() + 10);

    int pixelSize = mThumbSize.getSizeAsNumber();
    QPixmap thumbnail(pixelSize,pixelSize);
    QPainter painter(&thumbnail);
    painter.eraseRect(0,0,pixelSize,pixelSize);
    painter.drawRect(0,0,pixelSize,pixelSize);

    for (QIconViewItem *item = firstItem(); item; item=item->nextItem()) {
        item->setPixmap(thumbnail);
        AlbumIconItem* albumItem = static_cast<AlbumIconItem *>(item);
        mThumbCreator->requestThumbnail(albumItem->fileItem()->url().path(), mThumbSize);
        mNumThumbsRequested++;
    }

    arrangeItemsInGrid();

}



void AlbumIconView::slot_doubleClicked(QIconViewItem* item)
{
    if (!item) return;

    AlbumIconItem* albumItem = static_cast<AlbumIconItem *>(item);
    if (!albumItem) return;

    if (mUseInternalImageViewer) {
        ImageViewer* imgViewer = new ImageViewer;
        imgViewer->loadImage(albumItem->fileItem()->url().path());
        imgViewer->show();
    }
    else {
        KService::Ptr imageServicePtr;
        imageServicePtr = KService::serviceByDesktopName(mImageViewerName.lower());
        if  (imageServicePtr)
            KRun::run(*imageServicePtr, albumItem->fileItem()->url());
    }
}

void AlbumIconView::regenerateThumbnails()
{
    mThumbCreator->stop();
    mThumbCreator->forceRegeneration(true);
    mNumThumbsRequested  = 0;
    mNumThumbsDownloaded = 0;

    int pixelSize = mThumbSize.getSizeAsNumber();
    QPixmap thumbnail(pixelSize,pixelSize);
    QPainter painter(&thumbnail);
    painter.eraseRect(0,0,pixelSize,pixelSize);
    painter.drawRect(0,0,pixelSize,pixelSize);

    for (QIconViewItem *item = firstItem(); item; item=item->nextItem()) {
        item->setPixmap(thumbnail);
        AlbumIconItem* albumItem = static_cast<AlbumIconItem *>(item);
        mThumbCreator->requestThumbnail(albumItem->fileItem()->url().path(), mThumbSize);
        mNumThumbsRequested++;
    }

}

void AlbumIconView::startSlideShow()
{

    if (!count()) return;

    bool useSelected=false;

    for (QIconViewItem *item = firstItem(); item;
         item=item->nextItem()) {
        if (item->isSelected()) {
            useSelected = true;
            break;
        }
    }

    KFileItemList itemList;

    for (QIconViewItem *item = firstItem(); item;
         item=item->nextItem()) {
        if (useSelected && !(item->isSelected())) continue;
        AlbumIconItem* albumItem = static_cast<AlbumIconItem *>(item);
        itemList.append(albumItem->fileItem());
    }

    if (itemList.isEmpty()) return;

    SlideShow *slideShow = new SlideShow(itemList,mSlideShowDelay,
                                         mSlideShowTransitionEffect,
                                         mSlideShowFileNamePosition);
    slideShow->show();

}

void AlbumIconView::slot_rightClicked(QIconViewItem* item,
                                      const QPoint& pos)
{
    if (!item) return;

    AlbumIconItem *albumItem =
        static_cast<AlbumIconItem *>(item);
    if (!albumItem) return;

    QPopupMenu *popMenu = new QPopupMenu();
    QPopupMenu *openWithMenu = new QPopupMenu();

    //---------------------------------------------------------------

    QValueVector<KService::Ptr> serviceVector;

    KTrader::OfferList offers =
        KTrader::self()->query(albumItem->fileItem()->mimetype(),
                               "Type == 'Application'");

    KTrader::OfferList::Iterator iter;
    KService::Ptr ptr;
    int index = 100;
    for( iter = offers.begin(); iter != offers.end(); ++iter ) {
        ptr = *iter;
        openWithMenu->insertItem( ptr->pixmap(KIcon::Small),
                                  ptr->name(), index++);
        serviceVector.push_back(ptr);
    }

    //---------------------------------------------------------------

    popMenu->insertItem( BarIcon("image", KIcon::SizeSmall),
                        i18n("View"), 10);
    popMenu->insertItem( i18n("Open With ..."), openWithMenu, 11);
    popMenu->insertSeparator();
    popMenu->insertItem( BarIcon("pencil", KIcon::SizeSmall),
                         i18n("Rename"), 12);
    popMenu->insertItem( BarIcon("trashcan_empty", KIcon::SizeSmall),
                         i18n("Delete"), 13);
    popMenu->insertItem( i18n("Properties ..."), 14);

    //---------------------------------------------------------------

    int id = popMenu->exec(pos);
    switch (id) {

    case 10: {
        slot_doubleClicked(item);
        break;
    }

    case 12: {
        item->rename();
        break;
    }

    case 13: {
        deleteSelectedItems();
        break;
    }

    case 14: {
        (void) new KPropertiesDialog(albumItem->fileItem());
        break;
    }

    default:
        break;

    }

    //---------------------------------------------------------------

    if (id >= 100) {
        KService::Ptr imageServicePtr = serviceVector[id-100];
        KRun::run(*imageServicePtr, albumItem->fileItem()->url());
    }

    serviceVector.clear();
    delete openWithMenu;
    delete popMenu;
}

void AlbumIconView::slot_itemRenamed(QIconViewItem* item,
                                     const QString& newName)
{
    if (!item) return;
    AlbumIconItem *albumItem =
        static_cast<AlbumIconItem *>(item);
    if (!albumItem) return;

    // First Check if there is an item with same name already
    bool sameNameExists = false;
    for (QIconViewItem *it = firstItem(); it; it=it->nextItem()) {
        if (it != item && (it->text() == newName)) {
            sameNameExists = true;
            break;
        }
    }
    // If another item with same name exists, rename back to old
    if (sameNameExists)
        item->setText(albumItem->fileItem()->url().fileName());

    KURL oldurl(albumItem->fileItem()->url());
    QString newPath = oldurl.directory(false,true) + newName;
    KURL newurl(newPath);

    if ( oldurl != newurl ) {
        KURL::List lst;
        lst.append(oldurl);
        KIO::Job * job = KIO::moveAs( oldurl, newurl );
    }

}

void AlbumIconView::deleteSelectedItems()
{
    KURL::List urlList;
    QStringList nameList;

    for (QIconViewItem *it = firstItem(); it; it=it->nextItem()) {
        if (it->isSelected()) {
            AlbumIconItem *albumItem =
                static_cast<AlbumIconItem *>(it);
            urlList.append(albumItem->fileItem()->url());
            nameList.append(albumItem->fileItem()->url().fileName());
        }
    }

    if (urlList.count() <= 0) return;


    QString warnMsg(i18n("About to delete these Image(s)\nAre you sure?"));
    if (KMessageBox::warningContinueCancelList(this,
                                               warnMsg,
                                               nameList,
                                               i18n("Warning"),
                                               i18n("Delete"))
        ==  KMessageBox::Continue) {
	KIO::DeleteJob *job = KIO::del(urlList, false, true);
    }

}

void AlbumIconView::setSlideShowParams(int delay, bool effect,
                                       const QString& fileNamePos)
{
    mSlideShowDelay = delay;
    mSlideShowTransitionEffect = effect;
    if (fileNamePos == "At Top")
        mSlideShowFileNamePosition = SlideShow::AtTop;
    else if (fileNamePos == "At Bottom")
        mSlideShowFileNamePosition = SlideShow::AtBottom;
    else
        mSlideShowFileNamePosition = SlideShow::Off;

}

void AlbumIconView::setImageViewerParams(bool useInternal,
                                         const QString& viewerName)
{
    mUseInternalImageViewer = useInternal;
    mImageViewerName = viewerName;
}

// -- Drag ---------------------------------------------

void AlbumIconView::startDrag()
{
    KURL::List urlList;

    for (QIconViewItem *it = firstItem(); it; it=it->nextItem()) {
        if (it->isSelected()) {
            AlbumIconItem *albumItem =
                static_cast<AlbumIconItem *>(it);
            urlList.append(albumItem->fileItem()->url());
        }
    }

    if (urlList.isEmpty()) return;

    QUriDrag* drag = KURLDrag::newDrag( urlList, this );
    drag->setPixmap(SmallIcon("image"));
    drag->dragCopy();

}
