/***************************************************************************
    kstfit_kneefrequency_unweighted.cpp  -  knee frequency fit plugin for kst
                             -------------------
    begin                : Jun 6, 2006
    copyright            : (C) 2006 by Duncan Hanson
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <string.h>
#include <stdlib.h>
#include <math.h>

#define KNEEFREQ_XVALUES      0
#define KNEEFREQ_YVALUES      1
#define KNEEFREQ_WEIGHTS      2

#define	KNEEFREQ_YFIT         0
#define KNEEFREQ_YRESIDUALS   1
#define KNEEFREQ_PARAMETERS   2

#define KNEEFREQ_NUMPARAMETERS 5

extern "C" int parameterName(int iIndex, char** pName);
extern "C" int kstfit_kneefrequency_unweighted(const double *const inArrays[], const int inArrayLens[],
		const double inScalars[],
		double *outArrays[], int outArrayLens[],
		double outScalars[]);

int kstfit_kneefrequency_unweighted(const double *const inArrays[], const int inArrayLens[],
		const double inScalars[],
		double *outArrays[], int outArrayLens[],
		double outScalars[])
{
  if (inArrayLens[KNEEFREQ_XVALUES] != inArrayLens[KNEEFREQ_YVALUES] || inArrayLens[KNEEFREQ_XVALUES] < 1)  {
    return -2;
  }

  int inArraysLength = inArrayLens[KNEEFREQ_XVALUES];

  outArrays[KNEEFREQ_YFIT] = (double*)realloc(outArrays[KNEEFREQ_YFIT], inArraysLength*sizeof(double));
  outArrays[KNEEFREQ_YRESIDUALS] = (double*)realloc(outArrays[KNEEFREQ_YRESIDUALS], inArraysLength*sizeof(double));
  outArrays[KNEEFREQ_PARAMETERS] = (double*)realloc(outArrays[KNEEFREQ_PARAMETERS], KNEEFREQ_NUMPARAMETERS*sizeof(double));

  if (outArrays[KNEEFREQ_YFIT] == NULL || outArrays[KNEEFREQ_YRESIDUALS] == NULL || outArrays[KNEEFREQ_PARAMETERS] == NULL) {
    return -3; // memory error.
  }

  outArrayLens[KNEEFREQ_YFIT] = inArraysLength;
  outArrayLens[KNEEFREQ_YRESIDUALS] = inArraysLength;
  outArrayLens[KNEEFREQ_PARAMETERS] = KNEEFREQ_NUMPARAMETERS;

  double maxOneOverFFreq;
  double minWhiteNoiseFreq;
  double whiteNoiseC;
  double xi;
  double yi;
  int i;

  memset( outArrays[KNEEFREQ_YFIT], 0, inArraysLength * sizeof(double));
  memset( outArrays[KNEEFREQ_YRESIDUALS], 0, inArraysLength * sizeof(double));

  for (i = 0; i < KNEEFREQ_NUMPARAMETERS; i++) {
    outArrays[KNEEFREQ_PARAMETERS][i] = 0.0;
  }

  maxOneOverFFreq = inScalars[0];
  minWhiteNoiseFreq = inScalars[1];
  whiteNoiseC = inScalars[2];

  int maxOneOverFIndex;
  int minWhiteNoiseIndex;
  int i_bot = 0;
  int i_top = inArraysLength - 1;

  //
  // fast calculation of index for maxOneOverFFreq
  //

  while (i_bot + 1 < i_top) {
    int i0 = (i_top + i_bot)/2;
    if (maxOneOverFFreq < inArrays[KNEEFREQ_XVALUES][i0]) {
      i_top = i0;
    } else {
      i_bot = i0;
    }
  }
  maxOneOverFIndex = i_top; //top because we use i_bot+1.

  //
  // fast calculation of index for minWhiteNoiseFreq
  //

  i_bot = 0;
  i_top = inArraysLength - 1;

  while (i_bot + 1 < i_top) {
    int i0 = (i_top + i_bot)/2;

    if (minWhiteNoiseFreq < inArrays[KNEEFREQ_XVALUES][i0]) {
      i_top = i0;
    } else {
      i_bot = i0;
    }
  }
  minWhiteNoiseIndex = i_top;

  //
  // verify calculated indices
  //

  if ( !(maxOneOverFIndex > 0) ||
       !(minWhiteNoiseIndex >= maxOneOverFIndex) ||
       !(minWhiteNoiseIndex < inArraysLength-1) ) {
    return -2; // input error. frequenc(y/ies) are invalid.
  }

  double sumY = 0.0;
  double sumY2 = 0.0;
  double yBar;
  double ySigma;
  double sumLnXLnY = 0.0;
  double sumLnX = 0.0;
  double sumLnY = 0.0;
  double sumLnX2 = 0.0;

  //
  // calculate white noise limit
  //

  for (i = minWhiteNoiseIndex; i < inArraysLength; i++) {
    yi = inArrays[KNEEFREQ_YVALUES][i];
    sumY    +=  yi;
    sumY2   +=  pow(yi, 2.0);
  }

  yBar = sumY/(inArraysLength - minWhiteNoiseIndex);
  ySigma = sqrt((sumY2 - 2.0*yBar*sumY + pow(yBar,2.0)*(inArraysLength - minWhiteNoiseIndex))/(inArraysLength - minWhiteNoiseIndex));

  //
  // fit 1/f noise
  //

  for (i = 0; i < maxOneOverFIndex; i++) {
    xi = inArrays[KNEEFREQ_XVALUES][i];
    yi = inArrays[KNEEFREQ_YVALUES][i];

    if (!(yi-yBar > 0.0)) {
      return -2; // input error: maxOneOverFFreq too large?
    }

    if (xi > 0.0) {
      //-ybar to isolate 1/f noise
      sumLnXLnY += log(xi)*log(yi-yBar);
      sumLnX    += log(xi);
      sumLnY    += log(yi-yBar);
      sumLnX2   += pow(log(xi),2.0);
    }
  }

  double a;
  double b;
  double kneeFreq;

  //
  // calculate knee frequency
  //

  a = (maxOneOverFIndex*sumLnXLnY - sumLnX*sumLnY)/(maxOneOverFIndex*sumLnX2 - pow(sumLnX,2.0));
  b = exp((sumLnY - a*sumLnX)/maxOneOverFIndex);
  kneeFreq = pow(yBar*whiteNoiseC/b, 1.0/a); 

  //
  // output the fitted data
  //

  for (i = 0; i < maxOneOverFIndex; i++) {
    outArrays[KNEEFREQ_YFIT][i] = b * pow(inArrays[KNEEFREQ_XVALUES][i],a) + yBar;
    outArrays[KNEEFREQ_YRESIDUALS][i] = inArrays[KNEEFREQ_YVALUES][i] - outArrays[KNEEFREQ_YFIT][i];
  }

  // zeroes for the unfitted region
  for (i = maxOneOverFIndex; i < minWhiteNoiseIndex; i++) { 
    outArrays[KNEEFREQ_YFIT][i] = 0.0;
    outArrays[KNEEFREQ_YRESIDUALS][i] = 0.0;
  }

  for (i = minWhiteNoiseIndex; i < inArraysLength; i++) {
    outArrays[KNEEFREQ_YFIT][i] = yBar;
    outArrays[KNEEFREQ_YRESIDUALS][i] = inArrays[KNEEFREQ_YVALUES][i] - yBar;
  }

  outArrays[KNEEFREQ_PARAMETERS][0] = yBar;
  outArrays[KNEEFREQ_PARAMETERS][1] = ySigma;
  outArrays[KNEEFREQ_PARAMETERS][2] = b;
  outArrays[KNEEFREQ_PARAMETERS][3] = -a;
  outArrays[KNEEFREQ_PARAMETERS][4] = kneeFreq;

  return 0;
}

int parameterName(int iIndex, char** pName) {
  int iRetVal = 0;

  switch (iIndex) {
    case 0:
      *pName = strdup("White Noise Limit");
      iRetVal = 1;
      break;
    case 1:
      *pName = strdup("White Noise Sigma");
      iRetVal = 1;
      break;
    case 2:
      *pName = strdup("1/f^a Amplitude");
      iRetVal = 1;
      break;
    case 3:
      *pName = strdup("1/f^a Power Law a");
      iRetVal = 1;
      break;
    case 4:
      *pName = strdup("Knee Frequency");
      iRetVal = 1;
      break;
  }

  return iRetVal;
}
