/***************************************************************************
                          kstfitcurve.cpp: holds info for a fit curve
                             -------------------
    begin                : Wed Apr 28 2004
    copyright            : (C) 2004 The University of British Columbia
    email                :
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <klocale.h>
#include <kdebug.h>

#include "kstcolorsequence.h"
#include "kstcurvedialog_i.h"
#include "kstdatacollection.h"
#include "kstdebug.h"
#include "kstdoc.h"
#include "kstplugin.h"
#include "kstfitcurve.h"
#include "kstfitdialog_i.h"

static const QString CURVE_XVECTOR = "X";
static const QString CURVE_YVECTOR = "Y";
static const QString WVECTOR = "W";

KstFitCurve::KstFitCurve(const QString &in_tag, KstVectorPtr in_X, KstVectorPtr in_Y, KstVectorPtr in_W, const QColor &in_color)
: KstBaseCurve() {
  setHasPoints(false);
  setHasLines(true);
  setLineWidth(0);
  setLineStyle(0);
  
  commonConstructor(in_tag, in_color);
  VX = in_X;
  VY = in_Y;
  VW = in_W;
  update();
}


KstFitCurve::KstFitCurve(QDomElement &e)
: KstBaseCurve(e) {
  QString in_tag, xname, yname, wname;
  QColor in_color(KstColorSequence::next());

  setHasPoints(false);
  setHasLines(true);

  /* parse the DOM tree */
  QDomNode n = e.firstChild();
  while( !n.isNull() ) {
    QDomElement e = n.toElement(); // try to convert the node to an element.
    if( !e.isNull() ) { // the node was really an element.
      if (e.tagName() == "tag") {
        in_tag = e.text();
      } else if (e.tagName() == "xvectag") {
        xname = e.text();
      } else if (e.tagName() == "yvectag") {
        yname = e.text();
      } else if (e.tagName() == "wvectag") {
        wname = e.text();
      } else if (e.tagName() == "color") {
        in_color.setNamedColor(e.text());
      } else if (e.tagName() == "hasLines") {
        HasLines = (e.text() != "0");
      } else if (e.tagName() == "hasPoints") {
        HasPoints = (e.text() != "0");
      } else if (e.tagName() == "pointType") {
        Point.setType(e.text().toInt());
      } else if (e.tagName() == "lineWidth") {
        LineWidth = e.text().toInt();
      } else if (e.tagName() == "lineStyle") {
        LineStyle = e.text().toInt();
      }

    }
    n = n.nextSibling();
  }

  _inputVectorLoadQueue.append(qMakePair(CURVE_XVECTOR, xname));
  _inputVectorLoadQueue.append(qMakePair(CURVE_YVECTOR, yname));
  _inputVectorLoadQueue.append(qMakePair(WVECTOR, wname));
  commonConstructor(in_tag, in_color);
}

void KstFitCurve::commonConstructor(const QString &in_tag, const QColor &in_color) {
  _typeString = i18n("Fit");
  NumUsed = 0;
  Color = in_color;
  setTagName(in_tag);
}


KstFitCurve::~KstFitCurve() {
}


void KstFitCurve::setPlugin(KstPluginPtr plugin) {
  _plugin = plugin;
}


KstPluginPtr KstFitCurve::getPlugin() {
  return _plugin;
}


bool KstFitCurve::loadInputs() {
  QValueList<QPair<QString,QString> >::Iterator i;
  KST::vectorList.lock().readLock();
  for (i = _inputVectorLoadQueue.begin(); i != _inputVectorLoadQueue.end(); ++i) {
    if ((*i).first == CURVE_XVECTOR) {
      VX = *KST::vectorList.findTag((*i).second);
      if (!VX.data()) {
        KstDebug::self()->log(i18n("Unable to find X vector for %2: [%1]").arg((*i).second).arg(tagName()), KstDebug::Warning);
        KST::vectorList.lock().readUnlock();
        return false;
      }
    } else if ((*i).first == CURVE_YVECTOR) {
      VY = *KST::vectorList.findTag((*i).second);
      if (!VY.data()) {
        KstDebug::self()->log(i18n("Unable to find Y vector for %2: [%1]").arg((*i).second).arg(tagName()), KstDebug::Warning);
        KST::vectorList.lock().readUnlock();
        return false;
      }
    } else if ((*i).first == WVECTOR) {
      VW = *KST::vectorList.findTag((*i).second);
    }
  }
  KST::vectorList.lock().readUnlock();

  VX->readLock();
  VY->readLock();
  if (VW) {
    VW->readLock();
  }
  update(); // FIXME: this update happens in the wrong thread
  VX->readUnlock();
  VY->readUnlock();
  if (VW) {
    VW->readUnlock();
  }

  return true;
}

KstObject::UpdateType KstFitCurve::update(int update_counter) {
  if (!VX || !VY || KstObject::checkUpdateCounter(update_counter)) {
    return NO_CHANGE;
  }

  VX->update(update_counter);
  VY->update(update_counter);

  if (hasWeight()) {
    VW->update(update_counter);
  }

  MaxX = VX->max();
  MinX = VX->min();
  MeanX = VX->mean();
  MinPosX = VX->minPos();
  _ns_maxx = VX->ns_max();
  _ns_minx = VX->ns_min();

  if (MinPosX > MaxX) {
    MinPosX = 0;
  }

  MaxY = VY->max();
  MinY = VY->min();
  MeanY = VY->mean();
  MinPosY = VY->minPos();
  _ns_maxy = VY->ns_max();
  _ns_miny = VY->ns_min();

  if (MinPosY > MaxY) {
    MinPosY = 0;
  }

  if (VX->sampleCount() > VY->sampleCount()) {
    NS = VX->sampleCount();
  } else {
    NS = VY->sampleCount();
  }

  return UPDATE;
}

void KstFitCurve::getPoint(int i, double &x, double &y) {
  x = VX->interpolate(i, NS);
  y = VY->interpolate(i, NS);
}

QString KstFitCurve::getXVTag() const{
  return VX->tagName();
}

QString KstFitCurve::getYVTag() const{
  return VY->tagName();
}

QString KstFitCurve::getWVTag() const{
  if (hasWeight()) {
    return VW->tagName();
  } else {
    return i18n("<None>");
  }
}

bool KstFitCurve::hasWeight() const {
  bool bRetVal = false;

  if (VW) {
    if (VW.data()) {
      bRetVal = true;
    }
  }
  
  return bRetVal;
}

/** Save curve information */
void KstFitCurve::save(QTextStream &ts) {
  ts << " <fit>" << endl;
  ts << "  <tag>" << tagName() << "</tag>" << endl;
  ts << "  <xvectag>" << VX->tagName() << "</xvectag>" << endl;
  ts << "  <yvectag>" << VY->tagName() << "</yvectag>" << endl;
  if (hasWeight()) {
    ts << "  <wvectag>" << VW->tagName() << "</wvectag>" << endl;
  }
  ts << "  <color>" << Color.name() << "</color>" << endl;
  ts << "  <hasLines>" << HasLines << "</hasLines>" << endl;
  ts << "  <hasPoints>" << HasPoints << "</hasPoints>" << endl;
  ts << "  <pointType>" << Point.getType() << "</pointType>" << endl;
  ts << "  <lineWidth>" << LineWidth << "</lineWidth>" << endl;
  ts << "  <lineStyle>" << LineStyle << "</lineStyle>" << endl;
  ts << " </fit>" << endl;
}

void KstFitCurve::setXVector(KstVectorPtr new_vx) {
  VX = new_vx;
}

void KstFitCurve::setYVector(KstVectorPtr new_vy) {
  VY = new_vy;
}

void KstFitCurve::setWVector(KstVectorPtr new_vw) {
  VW = new_vw;
}

QString KstFitCurve::getXLabel() const {
  return VX->label();
}

QString KstFitCurve::getYLabel() const {
  return VY->label();
}

QString KstFitCurve::getTopLabel() const {
  return VY->fileLabel();
}

KstCurveType KstFitCurve::type() const {
  return KST_FITCURVE;
}

QString KstFitCurve::propertyString() const {
  return i18n("%1 vs %2").arg(getYVTag()).arg(getXVTag());
}

void KstFitCurve::_showDialog() {
  KstFitDialogI::globalInstance()->show_I(tagName());
}

int KstFitCurve::samplesPerFrame() const {
  const KstRVector *rvp = dynamic_cast<const KstRVector*>(VY.data());
  return rvp ? rvp->samplesPerFrame() : 1;
}

// vim: ts=2 sw=2 et
