/***************************************************************************
                              kst2dplot.h
                             -------------
    begin                : Mar 28, 2004
    copyright            : (C) 2004 The University of Toronto
    email                :
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KST2DPLOT_H
#define KST2DPLOT_H

#include "kstbackbuffer.h"
#include "kstbasecurve.h"
#include "kstlabel.h"
#include "kstplotbase.h"
#include "kstviewwidget.h"

#include <qguardedptr.h>

class KstLegend;
class Kst2DPlot;
typedef KstObjectList<KstSharedPtr<Kst2DPlot> > Kst2DPlotList;

typedef enum { AUTO, AC, FIXED, AUTOUP, NOSPIKE } KstScaleModeType;

struct KstPlotScale {
  double xmin;
  double xmax;
  double ymin;
  double ymax;
  KstScaleModeType xscalemode;
  KstScaleModeType yscalemode;
  bool xlog, ylog;
};

typedef enum { INACTIVE, XY_ZOOMBOX, Y_ZOOMBOX, X_ZOOMBOX, LABEL_TOOL, LAYOUT_TOOL } KstMouseModeType;
struct KstMouse {
  KstMouse() { mode = INACTIVE; label = -1; minMove = 2; }
  KstMouseModeType mode;
  int label, minMove;
  QPoint lastLocation, pressLocation;
  QRect plotGeometry;
  bool zooming() const;
  void zoomStart(KstMouseModeType t, const QPoint& location);
  void zoomUpdate(KstMouseModeType t, const QPoint& location);
  QRect mouseRect() const;
  bool rectBigEnough() const;
};


class Kst2DPlot : public KstPlotBase {
  Q_OBJECT
public:
  Kst2DPlot(const QString& in_tag = "SomePlot",
          KstScaleModeType yscale = AUTO,
          KstScaleModeType xscale = AUTO,
          double xmin = 0, double ymin = 0,
          double xmax = 1, double ymax = 1);
  Kst2DPlot(QDomElement& e);
  virtual ~Kst2DPlot();

  static Kst2DPlotList globalPlotList();

  virtual UpdateType update();
  virtual void save(QTextStream& ts);
  virtual void saveTag(QTextStream& ts);

  virtual bool popupMenu(KPopupMenu *menu, const QPoint& pos, KstViewObjectPtr topLevelParent);
  virtual bool layoutPopupMenu(KPopupMenu *menu, const QPoint& pos, KstViewObjectPtr topLevelParent);

  void drawDotAt(QPainter& p, double x, double y);
  double xInternalAlignment();
  void addCurve(KstBaseCurvePtr curve, bool set_dirty = true);
  void removeCurve(KstBaseCurvePtr curve);
  void setXScaleMode(KstScaleModeType scalemode);
  void setYScaleMode(KstScaleModeType scalemode);

  /** Set the scale */
  void setScale(double xmin, double ymin,
                double xmax, double ymax);
  /** Set the X scale */
  void setXScale(double xmin, double xmax);
  /** Set the y scale */
  void setYScale(double ymin, double ymax);

  /** Set the log_scale */
  void setLScale(double xmin, double ymin,
                double xmax, double ymax);
  /** Set the X log_scale */
  void setLXScale(double xmin, double xmax);
  /** Set the y log_scale */
  void setLYScale(double ymin, double ymax);
  /** Push the scale setting (X,Y, mode) onto scale list */
  void pushScale();
  /** Pop the scale settings off of scale list: true if stack not empty */
  bool popScale();

  void setLog(bool x_log, bool y_log);
  bool isXLog() const;
  bool isYLog() const;

  void getScale(double& xmin, double& ymin, double& xmax, double& ymax) const;
  void getLScale(double& xmin, double& ymin, double& xmax, double& ymax) const;

  KstScaleModeType getXScaleMode() const;
  KstScaleModeType getYScaleMode() const;

  void setXLabel(const QString& xlabel);
  void setYLabel(const QString& ylabel);
  void setTopLabel(const QString& toplabel);
  void initFonts(const QFont& font, int font_size = 0);

  QRect GetPlotRegion() const;
  QRect GetWinRegion() const;
  QRect GetTieBoxRegion() const;
  QRect GetPlotAndAxisRegion() const;

  bool isTied() const;
  void toggleTied();
  void setTied(bool is_tied);

  /** Labels */
  KstLabel *XLabel;
  KstLabel *YLabel;
  KstLabel *TopLabel;
  KstLabel *TickLabel;
  KstLegend *Legend;

  double _width;
  double _height;
  double _pos_x;
  double _pos_y;

  /** Arbitrary Labels */
  KstLabelList labelList;

  KstBaseCurveList Curves;

  void GenerateDefaultLabels();

  /* kstview tells kstplot where to offset the plot to */
  void setpixrect();

  virtual void resize(const QSize&);
  virtual void parentResized();

  virtual bool mouseHandler() const;
  virtual void mouseMoveEvent(QWidget *view, QMouseEvent *e);
  virtual void mousePressEvent(QWidget *view, QMouseEvent *e);
  virtual void mouseReleaseEvent(QWidget *view, QMouseEvent *e);
  virtual void keyPressEvent(QWidget *view, QKeyEvent *e);
  virtual void keyReleaseEvent(QWidget *view, QKeyEvent *e);
  virtual void dragEnterEvent(QWidget *view, QDragEnterEvent *e);
  virtual void dropEvent(QWidget *view, QDropEvent *e);
  virtual void dragMoveEvent(QWidget *view, QDragMoveEvent *e);
  virtual void wheelEvent(QWidget *view, QWheelEvent *e);

  void cancelZoom(QWidget *view);
  void moveSelfHorizontal(bool left);
  void moveSelfVertical(bool up);
  void zoomSelfHorizontal(bool in);
  void zoomSelfVertical(bool in);
  double xLeft() const;
  void setHasFocus(bool hasFocus);
  void removeFocus(QPainter& p);
  void moveUp(KstViewWidget *);
  void moveDown(KstViewWidget *);
  void moveLeft(KstViewWidget *);
  void moveRight(KstViewWidget *);
  void xZoomIn(KstViewWidget *);
  void yZoomIn(KstViewWidget *);
  void xZoomOut(KstViewWidget *);
  void yZoomOut(KstViewWidget *);
  void xZoomMax(KstViewWidget *);
  void yZoomMax(KstViewWidget *);
  void zoomMax(KstViewWidget *);
  void xLogSlot(KstViewWidget *);
  void yLogSlot(KstViewWidget *);
  void zoomPrev(KstViewWidget *);
  void yZoomAc(KstViewWidget *);
  void zoomSpikeInsensitiveMax(KstViewWidget *);

public slots:
  void copy();

  virtual void edit();
  void draw(); // draw into back buffer
  void draw(QPainter &p, double resolutionEnhancement = 1); // This actually does the drawing
  virtual void paint(KstPaintType type, QPainter& p);
  void editCurve(int id);
  void matchAxis(int id);
  void fitCurve(int id);
  void removeCurve(int id);
  void setDirty();

protected slots:
  void menuMoveUp();
  void menuMoveDown();
  void menuMoveLeft();
  void menuMoveRight();
  void menuXZoomIn();
  void menuXZoomOut();
  void menuYZoomIn();
  void menuYZoomOut();
  void menuXZoomMax();
  void menuYZoomMax();
  void menuZoomMax();
  void menuXLogSlot();
  void menuYLogSlot();
  void menuZoomPrev();
  void menuYZoomAc();
  void menuZoomSpikeInsensitiveMax();

protected:
  virtual KstViewObjectFactoryMethod factory() const;

private:
  template<class T>
  void updateTiedPlots(void (Kst2DPlot::*method)(T), T arg);

  void updateTieBox(QPainter&);
  int labelNumber(QMouseEvent *e);
  bool legendUnder(QMouseEvent *e);
  KstMouseModeType globalZoomType() const;
  void setCursorForMode(QWidget *view, KstMouseModeType mode);
  void setCursorForMode(QWidget *view); // gets the mode from the mouse
  void zoomRectUpdate(QWidget *view, KstMouseModeType t, int x, int y);
  inline void commonConstructor(const QString& in_tag,
                                KstScaleModeType yscale,
                                KstScaleModeType xscale,
                                double xmin,
                                double ymin,
                                double xmax,
                                double ymax,
                                bool x_log = false,
                                bool y_log = false);
  void setBorders(double& xleft_bdr_px, double& xright_bdr_px,
                  double& ytop_bdr_px, double& ybot_bdr_px,
                  double YTick, double Yorg,
                  QPainter &p);
  void setTicks(double& tick, double& org,
                double max, double min, bool is_log);

  /** range and domain of plot: not plot dimentions */
  double XMin, XMax, YMin, YMax;
  double _copy_x, _copy_y;

  bool _xLog : 1, _yLog : 1;
  bool _isTied : 1;
  bool _hasFocus : 1;
  bool _dirty : 1;
  bool _zoomPaused : 1;

  int _draggableLabel : 15; // I think this should be enough

  KstScaleModeType _xScaleMode, _yScaleMode;

  QPtrList<KstPlotScale> _plotScaleList;

  /** Stores border limits.  Set by KstPlot::paint().
      Stored here to be Used to determine mouse mode */
  QRect PlotRegion;
  QRect RelPlotRegion;
  QRect WinRegion;
  QRect RelWinRegion;
  QRect PlotAndAxisRegion;
  QRect RelPlotAndAxisRegion;

  void updateScale();

  KstMouse _mouse;
  QPoint _draggablePoint;

  QMap<int, QString> _curveEditMap, _curveFitMap, _curveRemoveMap;
  QMap<int, QGuardedPtr<Kst2DPlot> > _plotMap;

  KstBackBuffer _buffer;
  QGuardedPtr<KstViewWidget> _menuView;
  QSize _oldSize;
  double _oldXAlignment;
  double _m_X, _b_X, _m_Y, _b_Y;

  /** Store background and foreground colours for this plot **/
  QColor _backColor;
  QColor _foreColor;
};

typedef KstSharedPtr<Kst2DPlot> Kst2DPlotPtr;
typedef KstObjectMap<Kst2DPlotPtr> Kst2DPlotMap;

template<class T>
void Kst2DPlot::updateTiedPlots(void (Kst2DPlot::*method)(T), T arg) {
  if (isTied()) {
    Kst2DPlotList pl = globalPlotList();

    for (Kst2DPlotList::Iterator i = pl.begin(); i != pl.end(); ++i) {
      Kst2DPlotPtr p = *i;
      if (p->isTied() && p.data() != this) {
        (p->*method)(arg);
        p->pushScale();
        p->setDirty();
      }
    }
  }
}

#endif
// vim: ts=2 sw=2 et
