/***************************************************************************
 *
 * knetworkmanager-encryption.h - A NetworkManager frontend for KDE 
 *
 * Copyright (C) 2006 Novell, Inc.
 *
 * Author: Timo Hoenig        <thoenig@suse.de>, <thoenig@nouse.net>
 *         Will Stephenson    <wstephenson@suse.de>, <wstephenson@kde.org>
 *         Valentine Sinitsyn <e_val@inbox.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#ifndef KNETWORKMANAGER_ENCRYPTION_H
#define KNETWORKMANAGER_ENCRYPTION_H

#include "knetworkmanager.h"
#include <NetworkManager/NetworkManager.h>

#include <iwlib.h>

#include <NetworkManager/cipher.h>
#include <NetworkManager/cipher-wep-ascii.h>
#include <NetworkManager/cipher-wep-hex.h>
#include <NetworkManager/cipher-wep-passphrase.h>
#include <NetworkManager/cipher-wpa-psk-hex.h>
#include <NetworkManager/cipher-wpa-psk-passphrase.h>

class Encryption;
class Network;
class CredentialsRequest;

typedef QValueList<IEEE_802_11_Cipher*> CipherList;
typedef QMap<QString, QString> SecretMap;

class Encryption : public QObject
{
	Q_OBJECT
	public:
	  Encryption ();
	  virtual ~Encryption ();

	  /* either passphrase or key */
	  void      setSecrets (const SecretMap&);
	  SecretMap getSecrets (void) const;

	  bool    hasStoredKey (void) const;
	  void    restoreKeyAsync (void);
	  bool    restoreKey (void);

	  /* for all sub-classes but EncryptionNone */
	  virtual bool isValid (const QString &);
	  virtual bool serialize (DBusMessage*, const QString &) = 0;
	  virtual bool deserialize( DBusMessageIter *, int we_cipher ) = 0;
	  virtual void setDefaults (void) = 0;
	  virtual void persist( KConfigBase *, bool withKey = true ) const = 0;
	  virtual void restore( KConfigBase *, const char* version, bool withKey = false)  = 0;
	  
	  bool    isModified (void) const;
	  int     type(void) const;
	  int     getWeCipher(void) const;
	  void    setWeCipher(int we_cipher);
	public slots:
	  void slotCredentialsLoaded(QString, QMap<QString, QString>, bool);

	signals:
	  void    keyRestored (bool, bool);

	protected:
	  friend class Network;

	  void setNetwork(Network *network);
          void clearCipherList();
	  bool persistKey() const;
	  
	  SecretMap           _secret;
	  IEEE_802_11_Cipher* _currentCipher;
	  CipherList*         _cipherList;
	  Network*            _network;
	  mutable bool        _dirty;
	  int                 _we_cipher;
	  CredentialsRequest* _credentialsRequest;
};

/* EncryptionNone */

class EncryptionNone : public Encryption
{
	public:
	  EncryptionNone  ();
	  ~EncryptionNone ();
	  virtual void persist( KConfigBase *, bool withKey = true ) const;
	  virtual void restore( KConfigBase *, const char* version, bool withKey = false);
	  
	  /* overwrite Encryption::isValid () */
	  bool isValid     (const QString & ssid);
	  bool serialize   (DBusMessage*, const QString &);
	  virtual bool deserialize( DBusMessageIter *, int we_cipher );
	  //virtual bool serialize (DBusMessageIter*);
	  void setDefaults (void);
};

/* EncryptionWEP */

enum WEPType {
	 WEP_ASCII      = 0,
	 WEP_HEX        = 1,
	 WEP_PASSPHRASE = 3
};

enum WEPMethod {
	WEP_OPEN_SYSTEM = IW_AUTH_ALG_OPEN_SYSTEM,
	WEP_SHARED_KEY  = IW_AUTH_ALG_SHARED_KEY
};

class EncryptionWEP : public Encryption
{
	public:
	  EncryptionWEP  (WEPType);
	  ~EncryptionWEP ();

	  virtual void persist( KConfigBase *, bool withKey = true ) const;
	  virtual void restore( KConfigBase *, const char* version, bool withKey = false);

	  void      setMethod (WEPMethod);
	  WEPMethod getMethod (void);
	  void      setType( WEPType );

	  bool serialize   (DBusMessage*, const QString &);
	  virtual bool deserialize( DBusMessageIter *, int we_cipher );
	  void setDefaults (void);

	private:
	  WEPType   _type;
	  WEPMethod _method;
};

/* Used by both, Personal and Enterprise */

enum WPAProtocol {
	 WPA_AUTO     = NM_AUTH_TYPE_WPA_PSK_AUTO, /* WPA Personal */
	 WPA_TKIP     = NM_AUTH_TYPE_WPA_PSK_TKIP, /* WPA Personal */
	 WPA_CCMP_AES = NM_AUTH_TYPE_WPA_PSK_CCMP, /* WPA Personal */
	 WPA_EAP      = NM_AUTH_TYPE_WPA_EAP       /* WPA Enterprise */
};

enum WPAVersion {
	WPA1 = IW_AUTH_WPA_VERSION_WPA, /* WPA Personal and Enterprise */
	WPA2 = IW_AUTH_WPA_VERSION_WPA2 /* WPA Personal and Enterprise */
};


/* EncryptionWPAPersonal */

class EncryptionWPAPersonal : public Encryption
{
	public:
	  EncryptionWPAPersonal  ();
	  ~EncryptionWPAPersonal ();

	  void         setProtocol (WPAProtocol);
	  WPAProtocol  getProtocol (void);
	  void         setVersion (WPAVersion);
	  WPAVersion   getVersion (void);
	  virtual void persist( KConfigBase *, bool withKey = true ) const;
	  virtual void restore( KConfigBase *, const char* version,  bool withKey = false);

	  bool serialize   (DBusMessage*, const QString &);
	  virtual bool deserialize( DBusMessageIter *, int we_cipher );
	  void setDefaults (void);

	private:
	  WPAProtocol _protocol;
	  WPAVersion  _version;
};

/* EncryptionWPAEnterprise */

enum EAPMethod {
	EAP_PEAP = NM_EAP_METHOD_PEAP,
	EAP_TLS  = NM_EAP_METHOD_TLS,
	EAP_TTLS = NM_EAP_METHOD_TTLS,
	EAP_LEAP = 0x00000999 /* instead of '#define NM_EAP_METHOD_LEAP 0x00000999' */
};

#ifdef HAVE_PHASE2_AUTH
enum EAPPhaseTwo {
	PHASE2_NONE     = NM_PHASE2_AUTH_NONE,
	PHASE2_PAP      = NM_PHASE2_AUTH_PAP,
	PHASE2_MSCHAP   = NM_PHASE2_AUTH_MSCHAP,
	PHASE2_MSCHAPV2 = NM_PHASE2_AUTH_MSCHAPV2,
	PHASE2_GTC      = NM_PHASE2_AUTH_GTC
};
#else
enum EAPPhaseTwo {
	PHASE2_NONE     = 0x00000000,
	PHASE2_PAP      = 0x00010000,
	PHASE2_MSCHAP   = 0x00020000,
	PHASE2_MSCHAPV2 = 0x00030000,
	PHASE2_GTC      = 0x00040000
};
#endif

class EncryptionWPAEnterprise : public Encryption
{
	public:
	  EncryptionWPAEnterprise();
	  ~EncryptionWPAEnterprise ();

	  /* overwrite Encryption::isValid () */
	  bool isValid (const QString & ssid);
	  static QString IdPasswordKey;
	  static QString CertPrivatePasswordKey;

	  void         setIdentity (const QString &);
	  QString      getIdentify (void) const;
 	  void         setAnonIdentity (const QString &);
	  QString      getAnonIdentity (void) const;
	  void         setCertClient (const QString &);
	  QString      getCertClient (void) const;
	  void         setCertCA (const QString &);
	  QString      getCertCA (void) const;
	  void         setCertPrivate (const QString &);
	  QString      getCertPrivate (void) const;

	  void         setProtocol (WPAProtocol);
	  WPAProtocol  getProtocol (void);
	  void         setVersion (WPAVersion);
	  WPAVersion   getVersion (void);
	  void         setMethod (EAPMethod);
	  EAPMethod    getMethod (void);
	  void         setPhaseTwoAuth (EAPPhaseTwo);
	  EAPPhaseTwo  getPhaseTwoAuth (void);
	  void         setLeapMethod (const QString &);
	  QString      getLeapMethod (void);

	  virtual void persist( KConfigBase *, bool withKey = true ) const;
	  virtual void restore( KConfigBase *, const char* version, bool withKey = false);

	  bool serialize (DBusMessage*, const QString &);
	  virtual bool deserialize( DBusMessageIter *, int we_cipher );
	  void setDefaults (void);

	private:
	  QString _identity;
	  QString _anonIdentity;
	  QString _certClient;
	  QString _certCA;
	  QString _certPrivate;

	  WPAProtocol _protocol;
	  WPAVersion  _version;
	  EAPMethod   _method;
	  EAPPhaseTwo _phaseTwo;
	  QString     _leapmethod;
};

#endif /* KNETWORKMANAGER_ENCRYPTION_H */
