/* -*- c++ -*-
 *
 * optioneditor.cpp
 *
 * Copyright (C) 2003 Petter Stokke <ummo@hellokitty.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#include <kdebug.h>
#include <klocale.h>

#include <donkeyprotocol.h>

#include "optioneditor.h"
#include "kmldonkey.h"



MyDonkeyOption::MyDonkeyOption()
    : m_option()
    , m_value()
    , m_dirty(false)
{
}

MyDonkeyOption::MyDonkeyOption(const DonkeyOption& opt, const QString& value)
    : m_option(opt)
    , m_value(value)
    , m_dirty(false)
{
}

MyDonkeyOption::MyDonkeyOption(const MyDonkeyOption& opt) {
    m_option = opt.m_option;
    m_value = opt.m_value;
    m_dirty = opt.m_dirty;
}

void MyDonkeyOption::setValue(const QString& value)
{
    m_value = value;
    m_dirty = true;
}

bool MyDonkeyOption::isDirty() const
{
    return m_dirty;
}

void MyDonkeyOption::setDirty(bool dirty)
{
    m_dirty = dirty;
}

void MyDonkeyOption::applyOption(DonkeyProtocol* target)
{
    if (!m_dirty) return;
    target->setOption(m_option.optionName(), m_value);
    m_dirty = false;
    kdDebug() << "Option \"" << m_option.optionName() << "\" set to \"" << m_value << "\"" << endl;
}

const DonkeyOption& MyDonkeyOption::option() const
{
    return m_option;
}



OptionEditorWidget::OptionEditorWidget(QWidget* parent, const char* name)
    : QTable(0, 2, parent, name)
    , QToolTip(viewport())
{
    setSelectionMode(NoSelection);
    setColumnReadOnly(0, true);
    setColumnStretchable(1, true);
    verticalHeader()->hide();
    setLeftMargin(0);
    horizontalHeader()->setLabel(0, i18n("Option"));
    horizontalHeader()->setLabel(1, i18n("Value"));
    connect(this, SIGNAL(valueChanged(int, int)), SLOT(optionChanged(int, int)));
}

void OptionEditorWidget::maybeTip(const QPoint& vp)
{
    QPoint p(viewportToContents(vp));
    int row = rowAt(p.y()), col = columnAt(p.x());
    if (col < 0 || row < 0)
	return;
    
    QString help = m_options[row].option().optionHelp();
    if (help.isEmpty() && col)
	help = m_options[row].option().optionLabel();
    if (help.isEmpty())
	return;
    
    QRect r = cellGeometry(row, col);
    r.moveTopLeft(contentsToViewport(r.topLeft()));
    tip(r, help);
}

void OptionEditorWidget::addOption(const DonkeyOption& option)
{
    int row = numRows();
    setNumRows(row + 1);
    setText(row, 0, option.optionLabel());
    QString value = KMLDonkey::App->donkey->getOption(option.optionName());
    if (option.optionType() == "Bool") {
	QStringList list;
	list << "true" << "false";
	QComboTableItem* item;
	setItem(row, 1, item = new QComboTableItem(this, list));
	item->setCurrentItem(value);
    }
    else
	setText(row, 1, value);
    adjustColumn(0);
    m_options.append(MyDonkeyOption(option, value));
}

void OptionEditorWidget::optionChanged(int row, int col)
{
    m_options[row].setValue(text(row,col));
    emit listIsDirty();
}

void OptionEditorWidget::applyChangedOptions(DonkeyProtocol* target)
{
    QValueListIterator<MyDonkeyOption> it;
    for (it = m_options.begin(); it != m_options.end(); ++it)
	if ((*it).isDirty())
	    (*it).applyOption(target);
}


#include "optioneditor.moc"
