/***************************************************************************
                          fitdlg.h  -  description
                             -------------------
    begin                : Tue May 25 1999
    copyright            : (C) 2001 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef FITDLG_H
#define FITDLG_H

#include <qdialog.h>
#include <qlist.h>
#include "kplnamespace.h"

class QLineEdit;
class QLabel;
class QListBox;
class QCheckBox;
class KConfig;
class KURL;
class KplDoc;
class ArrayItem;
class FunItem;
class LMFit;
class SourceWidget;
class KplDoubleEdit;
class KplSpinBox;

/**
  * Correlation matrix dialog class. Provides a dialog for displaying
  * the parameter correlation matrix.
  *@author Werner Stille
  */
class CorrDlg : public QDialog {
  Q_OBJECT

public:
  /**
   * Constructor.
   * @param parent pointer to parent widget.
   * @param np number of fitted parameters.
   * @param ip array with parameter indices.
   * @param corr np * np matrix with correlation coefficients.
   */
	CorrDlg(QWidget* parent, int np, const int* ip, const double* corr);
  /** Destructor. */
  ~CorrDlg();
};

/**
  * Error model dialog class. Provides a dialog for setting
  * the calculation of the error bars for nonlinear fits.
  * @author Werner Stille
  */
class ErrModDlg : public QDialog {
  Q_OBJECT

public:
  /**
   * Constructor.
   * @param parent pointer to parent widget.
   * @param model pointer to KplDoc model object.
   * @param err0 pointer to structure containing the data error settings.
   */
	ErrModDlg(QWidget* parent, KplDoc* model, Kpl::DataErrorStruct* err0);
  /** Destructor. */
	~ErrModDlg();

protected:
  KplDoc* m;
  Kpl::DataErrorStruct *err, *errt;
  QLineEdit *fileName, *func;

protected slots:
  /** Displays the dialog for selection of a library file to load. */
  void slotFile();
  /** Displays the dialog for selection of a function. */
  void slotFunc();
  /** Displays the dialog for selection of a parameter file to read. */
  void slotGetParameter();
  /** Displays the dialog for parameter editing. */
  void slotEditParameter();
  /** Displays the dialog for selection of a parameter file to save. */
  void slotSaveParameter();
  /** Accepts changes and leaves the dialog. */
  void slotOK();
  /** Displays help for the function settings dialog. */
  void slotHelp();
};

/**
  * Fit dialog class. Provides a dialog for performing a parameter fit
  * of an ordinary function to a data set of an array.
  *@author Werner Stille
  */
class FitDlg : public QDialog {
  Q_OBJECT

public:
  /**
   * Constructor.
   * @param parent pointer to parent widget.
   * @param model pointer to KplDoc model object.
   * @param ad pointer to array item list.
   * @param fd pointer to function item list.
   * @param mode combination of ShowDlg and AutoFit.
   */
	FitDlg(QWidget* parent, KplDoc* model, QList<ArrayItem>* ad,
         QList<FunItem>* fd, int mode);
  /** Destructor. */
	~FitDlg();
  /** Dialog mode flags. */
  enum DlgMode{ShowDlg = 1, AutoFit = 2, SavePar = 4, Follow = 8};

public slots:
  /** Performs Levenberg-Marquardt fit. */
  void slotFit();
  /** Applies settings without leaving the dialog (if displayed). */
  void slotApply();

protected:
  /**
   * Checks and copies settings and parameters.
   * @param accept true for leaving the dialog, false for applying.
   */
  void getValues(bool accept);
  int ip[KPL_NPMAX];
  int np, dlgMode;
  double corr[KPL_NPMAX * KPL_NPMAX];
  KplDoc* m;
  QList<ArrayItem>* ad;
  QList<FunItem>* fd;
  Kpl::AutoStruct* aut;
  LMFit* lm;
  QListBox* results;
  KplDoubleEdit* eTol;
  QLineEdit* par[KPL_NPMAX];
  QLabel* lArr;
  QLabel* epar[KPL_NPMAX];
  QCheckBox* enFit[KPL_NPMAX];
  QCheckBox *errCol, *nonLin;
  KplSpinBox *sArr, *sItmax;
  QPushButton* showCorr;
  QPushButton* errMod;
  QList<QWidget> widgetList;
  Kpl::DataErrorStruct* err;
  KConfig* config;
  SourceWidget* sw;

protected slots:
  /** Displays the dialog for selection of a parameter file to read. */
  void slotGetParameter();
  /** Displays the dialog for selection of a parameter file to save. */
  void slotSaveParameter();
  /**
   * Disables/enables error model button.
   * @param state error column state, disables button when true.
   */
  void errColToggled(bool state);
  /**
   * Disables/enables widgets for nonlinear fits.
   * @param on enables widgets when true.
   */
  void enableNonLin(bool on);
  /**
   * updates display for selected array in data error settings.
   * @param i index of array in list of arrays to be fitted.
   */
  void updArray(int i);
  /** Accepts changes and leaves the dialog. */
  void slotOK();
  /** Stops fitting or leaves the dialog without applying the changes. */
  void slotCancel();
  /** Displays help for the function settings dialog. */
  void slotHelp();
  /** Displays the dialog for setting the calculation of error bars. */
  void slotErrMod();
  /**
   * Displays message line in listbox.
   * @param msg message text.
   */
  void slotMessage(const QString& msg);
  /** Displays parameter correlation matrix. */
  void slotCorr();
};

#endif
