/***************************************************************************
                          kgraph.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sun Apr 25 1999
    copyright            : (C) 2001 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdio.h>
#include <math.h>
#include <qpaintdevicemetrics.h>
#include <qpixmap.h>
#include "kgraph.h"

//#define NO_CLIPPING

const char* KGraph::face[] = {"Helvetica", "Symbol", "itc zapf dingbats"};

KGraph::KGraph()
{
}

KGraph::~KGraph()
{
}

void KGraph::r2i(double x, double y, int* ix, int* iy)
{
  xcur = x;
  ycur = y;
  *ix = minx + int(xscal * (x - xmin));
  *iy = miny + int(yscal * (y - ymin));
}

bool KGraph::i2cm(int ix, int iy, double* xcm, double* ycm) const
{
  *xcm = (ix - minx0) / (scalx * factor);
  *ycm = (iy - miny0) / (scaly * factor);
  return ((ix < minx) || (ix > maxx) || (iy > miny) || (iy < maxy));
}

void KGraph::i2r(int ix, int iy, double fxn, double fyn,
                 double* xr, double* yr) const
{
  *xr = ((ix - minx) / xscal + xmin) / fxn;
  if (logx)
    *xr = pow(10, *xr);
  *yr = ((iy - miny) / yscal + ymin) / fyn;
  if (logy)
    *yr = pow(10, *yr);
}

void KGraph::cm2r(double xcm, double ycm, double fxn, double fyn,
                  double* xr, double* yr) const
{
  i2r(int(xcm * scalx * factor) + minx0,
      int(ycm * scaly * factor) + miny0, fxn, fyn, xr, yr);
}

void KGraph::move(double x, double y)
{
  r2i(x, y, &kxcur, &kycur);
  moveTo(kxcur, kycur);
}

void KGraph::draw(double x, double y)
{
  if ((x != xcur) || (y != ycur)) {
    if (dash[1] == dash[0]) {
      r2i(x, y, &kxcur, &kycur);
      lineTo(kxcur, kycur);
    } else {
      dcur = dcur - dash[3] * (int) (dcur / dash[3]);
      double x1 = (x - xcur) * xscal / scalx;
      double y1 = (y - ycur) * yscal / scaly;
      double d0 = sqrt(x1 * x1 + y1 * y1);
      double d = d0;
      x1 = xcur;
      y1 = ycur;
      int i = 0;
      for (int j = 0; j < 3; j++)
        if (dcur >= dash[j])
          i = j + 1;
      while ((dcur + d) > dash[i]) {
        if (double f = (dash[i] - dcur) / d) {
          d = d * (1.0 - f);
          xcur = x - d * (x - x1) / d0;
          ycur = y - d * (y - y1) / d0;
          dcur = dash[i];
          if (i % 2)
            move(xcur,ycur);
          else {
            r2i(xcur, ycur, &kxcur, &kycur);
            lineTo(kxcur, kycur);
          }
        }
        if (i == 3)
          dcur = 0.0;
        i = ++i % 4;
      }
      xcur = x;
      ycur = y;
      if (i % 2)
        move(xcur, ycur);
      else {
        r2i(xcur, ycur, &kxcur, &kycur);
        lineTo(kxcur, kycur);
      }
      dcur += d;
    }
  }
}

void KGraph::line(double x1, double y1, double x2, double y2)
{
  move(x1, y1);
  draw(x2, y2);
}

void KGraph::plInit(const QPaintDevice* pd, int offLeft, int offTop)
{
  pFactor = 1;
  QPaintDeviceMetrics pdm(pd);
  begin(pd);
  eraseRect(window());
  if (pd->devType() == QInternal::Printer) {
    QRect r = window();
    pFactor = 10;
    r.setSize(pFactor * r.size());
    setWindow(r);
  }
  logx = logy = false;
  scalx = 10.0 * pFactor * pdm.width() / pdm.widthMM();
  scaly = -10.0 * pFactor * pdm.height() / pdm.heightMM();
  xscal = scalx;
  yscal = scaly;
  minx0 = minx = -offLeft;
  maxx = pFactor * pdm.width() + minx;
  miny0 = miny = pFactor * pdm.height() + offTop;
  maxy = offTop;
  dlen = (maxx - minx) / xscal;
  dhite = (maxy - miny) / yscal;
  factor = 1.0;
  xmin = ymin = 0.0;
  xcur = xmin;
  ycur = ymin;
  xmax = dlen;
  ymax = dhite;
  relang = 0.0;
  relsin = 0.0;
  relcos = 1.0;
  relSize = 1.0;
  textdir = 0.0;
  ndigx = ndigy = -1;
  linetype(1);
  pen1.setWidth((maxx - minx) / 240);
  pen2.setWidth((maxx - minx) / 480);
  pen3.setWidth((maxx - minx) / 339);
  setPen(pen3);
  QFont fnt = font();
  fnt.setFamily(face[0]);
  fnt.setPointSize(QMAX(int(0.0355 * maxx), 1));
  setFont(fnt);
  move(xcur, ycur);
}

void KGraph::format(double x, double y)
{
  xmin = ymin = 0.0;
  xmax = x;
  ymax = y;
  minx = minx0;
  miny = miny0;
  xscal = scalx * factor;
  yscal = scaly * factor;
  maxx = minx + int(xscal * xmax);
  maxy = miny + int(yscal * ymax);
  xcur = xmin;
  ycur = ymin;
#ifndef NO_CLIPPING
  setClipRect(0, maxy / pFactor, (maxx - minx) / pFactor,
              (miny - maxy) / pFactor);
  setClipping(false);
#endif
}

void KGraph::setColFrame(unsigned icol1)
{
  pen1.setColor(QColor(icol1));
}

void KGraph::setColGrid(unsigned icol2)
{
  pen2.setColor(QColor(icol2));
}

void KGraph::setColData(unsigned icol3)
{
  pen3.setColor(QColor(icol3));
  setPen(pen3);
}

void KGraph::setCol(unsigned icol1, unsigned icol2, unsigned icol3)
{
  setColFrame(icol1);
  setColGrid(icol2);
  setColData(icol3);
}

double KGraph::setRelSize(double r)
{
  double old = relSize;
  relSize = r;
  pen1.setWidth(int(0.00416667 * relSize * (maxx - minx)));
  pen2.setWidth(int(0.00208333 * relSize * (maxx - minx)));
  pen3.setWidth(int(0.00295 * relSize * (maxx - minx)));
  setPen(pen3);
  QFont fnt = font();
  fnt.setPointSize(QMAX(int(0.0355 * relSize * (maxx - minx)), 1));
  setFont(fnt);
  return old;
}

void KGraph::Window(double xmi, double w, double ymi, double h)
{
  xmin = xmi;
  xmax = xmi + w;
  ymin = ymi;
  ymax = ymi + h;
  minx = minx0 + int(xscal * xmin);
  maxx = minx + int(xscal * w);
  miny = miny0 + int(yscal * ymin);
  maxy = miny + int(yscal * h);
  xcur = xmin;
  ycur = ymin;
  move(xcur, ycur);
  dcur = 0.0;
  int dx = int(0.00416667 * relSize * (maxx - minx));
  pen1.setWidth(dx);
  pen2.setWidth(int(0.00208333 * relSize * (maxx - minx)));
  pen3.setWidth(int(0.00295 * relSize * (maxx - minx)));
  setPen(pen3);
  QFont fnt = font();
  fnt.setPointSize(QMAX(int(0.0355 * relSize * (maxx - minx)), 1));
  setFont(fnt);
#ifndef NO_CLIPPING
  setClipRect((minx + (dx >> 1)) / pFactor , (maxy + (dx >> 1)) / pFactor,
              (maxx - minx - dx) / pFactor, (miny - maxy - dx) / pFactor);
  setClipping(false);
#endif
}

void KGraph::scale(double xmi, double xma, double ymi, double yma,
                   bool lgx, bool lgy)
{
  logx = lgx;
  logy = lgy;
  xmin = logx ? log10(xmi) : xmi;
  xmax = logx ? log10(xma) : xma;
  ymin = logy ? log10(ymi) : ymi;
  ymax = logy ? log10(yma) : yma;
  xscal = (maxx - minx) / (xmax - xmin);
  yscal = (maxy - miny) / (ymax - ymin);
  xcur = xmin;
  ycur = ymin;
  move(xcur, ycur);
}

void KGraph::setDig(int nx, int ny)
{
  ndigx = nx;
  ndigy = ny;
}

void KGraph::frame()
{
  setPen(pen1);
  drawRect(minx, maxy, maxx - minx, miny - maxy);
  setPen(pen3);
}

void KGraph::raster(double xtic, double ytic, int intx, int inty, int mode)
{
  if (logx) {
    if (xtic < 2.5)
      xtic = 2.0;
    else
      if (xtic < 5.5)
        xtic = 3.0;
      else
        if (xtic < 99.0)
          xtic = 10.0;
  }
  if (logy) {
    if (ytic < 2.5)
      ytic = 2.0;
    else
      if (ytic < 5.5)
        ytic = 3.0;
      else
        ytic = 10.0;
  }
  double dx = 0.0, dy, x, y;
  if (mode) {
    setPen(pen2);
    dx = 0.99999 * (xmax - xmin);
    dy = 0.99999 * (ymax - ymin);
    if (mode <= 2) {
      dx *= 0.02;
      dy = -dx * xscal / yscal;
    }
    for (x = xmin; x < xmax;)
      if (logx) {
        double xa = x;
        double mant = incLog(&x, xtic);
        if (x < xmax)
          line(x, ymin, x, ymin + dy);
        if (intx > 1) {
          double xd = pow(10.0, xa - mant);
          for (;;) {
            xa = log10(pow(10.0, xa) + xd);
            if ((xa >= x) || (xa >= xmax))
              break;
            line(xa, ymin, xa, ymin + (mode <= 2 ? 0.5 * dy : dy));
          }
        }
      } else {
        if (mode <= 2  && intx > 1) {
          double xm = x;
          for (int i = 1; i < intx; i++)
            if ((xm += xtic / intx) < xmax)
              line(xm, ymin, xm, ymin + 0.5 * dy);
        }
        if ((x += xtic) < xmax)
          line(x, ymin, x, ymin + dy);
      }
    for (y = ymin; y < ymax;)
      if (logy) {
        double ya = y;
        double mant = incLog(&y, ytic);
        if (y < ymax)
          line(xmin, y, xmin + dx, y);
        if (inty > 1) {
          double yd = pow(10.0, ya - mant);
          for (;;) {
            ya = log10(pow(10.0, ya) + yd);
            if ((ya >= y) || (ya >= ymax))
              break;
            line(xmin, ya, xmin + (mode <= 2 ? 0.5 * dx : dx), ya);
          }
        }
      } else {
        if (mode <= 2  && intx > 1) {
          double ym = y;
          for (int i = 1; i < inty; i++)
            if ((ym += ytic / inty) < ymax)
              line(xmin, ym, xmin + 0.5 * dx, ym);
        }
        if ((y += ytic) < ymax)
          line(xmin, y, xmin + dx, y);
      }
  }
  setPen(pen1);
  if ((mode == 2) || (mode == 4)) {
    char label[256];
    y = ymin + 0.03225 * relSize * (xmax - xmin) * xscal / yscal;
    int ix, iy;
    bool e = (logx && (xtic > 9.9) && ((xmax > 2.999999) || (xmin < -2.999999)));
    int nd = ndigx;
    for (x = xmin; x <= (xmax + 1.0e-3 * dx);) {
      if (x > xmax)
         x = xmax;
      r2i(x, y, &ix, &iy);
      if (e) {
        sprintf(label, "10#nu%d", qRound(x));
        text(label, ix, iy, 2);
      } else {
        if (logx)
          if (x < 0)
            nd = 1 - int(x + 0.01);
          else
            nd = -1;
        int l = prenum(logx ? pow(10.0, x) : x, nd, label);
        drawText(ix, iy, 0, 0, AlignCenter | DontClip, label, l);
      }
      if (logx)
        incLog(&x, xtic);
      else
        x += xtic;
    }
    x = xmin - 0.025 * relSize * (xmax - xmin);
    dy = -0.0125 * relSize * (xmax - xmin) * xscal / yscal;
    e = (logy && (ytic > 9.9) && ((ymax > 2.999999) || (ymin < -2.999999)));
    nd = ndigy;
    for (y = ymin; y <= (ymax + 1.0e-3 * dy);) {
      if (y > ymax)
        y = ymax;
      r2i(x, y, &ix, &iy);
      if (e) {
        sprintf(label, "10#nu%d", qRound(y));
        text(label, ix, iy, 3);
      } else {
        if (logy)
          if (y < 0)
            nd = 1 - int(y + 0.01);
          else
            nd = -1;
        int l = prenum(logy ? pow(10.0, y) : y, nd, label);
        drawText(ix, iy - 1, 0, 0, AlignRight | AlignVCenter | DontClip, label,
                 l);
      }
      if (logy)
        incLog(&y, ytic);
      else
        y += ytic;
    }
  }
  frame();
  setPen(pen3);
}

void KGraph::dir(double ang)
{
  textdir = (float) ang;
}

void KGraph::text(const char* str, int ix, int iy, int iorg)
{
  if (int l = strlen(str)) {
    QWMatrix wm = worldMatrix();
    translate(float(ix), float(iy));
    if (textdir)
      rotate(-textdir);
    setPen(pen1);
    int tf = 0;
    if (!strchr(str, '#')) {
      switch (iorg) {
        case 1:
          tf = AlignLeft | AlignVCenter | DontClip;
          break;
        case 2:
          tf = AlignCenter | DontClip;
          break;
        case 3:
          tf = AlignRight | AlignVCenter | DontClip;
      }
      drawText(0, 0, 0, 0, tf, str, l);
    } else {
      const char fontcode[] = "NHBOSZI", levcode[] = "NUD";
      const int iface[7] = {0, 0, 0, 0, 1, 2, 0};
      const int weight[7] = {QFont::Normal, QFont::Normal, QFont::Bold,
                             QFont::Normal, QFont::Normal, QFont::Normal,
                             QFont::Bold};
      const bool italic[7] = {false, false, false, true, false, false, true};
      const double ys[2] = {0.6, -0.4};
      const char *c;
      ix = 0;
      tf = AlignLeft | AlignVCenter | DontClip;
      double dr = 0.025 * relSize * (xmax - xmin) * xscal;
      int j0 = (iorg == 1) ? 1 : 0;
      int idx = 0;
      int idy;
      for (int j = j0; j < 2; j++) {
        if (j) {
          if (iorg == 2)
            idx >>= 1;
          if (iorg != 1)
            ix = -idx;
        }
        const char* s = str;
        int ilev = 0;
        while (*s) {
          if ((l = ((c = strchr(s, '#'))) ? (int) (c - s) : strlen(s))) {
            QRect r;
            if (j == 0) {
              r = boundingRect(0, 0, maxx, maxy,
                                     AlignLeft | AlignVCenter, s, l);
              idx += r.width();
            } else {
              idy = (ilev) ? -int(dr * ys[ilev - 1]) : 0;
              drawText(ix, idy, 0, 0, tf, s, l, &r);
              ix += r.width();
            }
          }
          if (c) {
            if (strlen(c) < 3)
              break;
            s = c + 3;
            char code;
            if ((code = *(++c)) > 96)
              code -= (char) 32;
            const char* f;
            if (!(f = strchr(fontcode, code)))
              break;
            int ifnt = (int) (f - fontcode);
            if ((code = *(++c)) > 96)
              code -= (char) 32;
            if (!(f = strchr(levcode, code)))
              break;
            setFont(QFont(face[iface[ifnt]],
                    QMAX(int(((ilev = (int) (f - levcode)) ? 0.0251 : 0.0355)
                         * relSize * (maxx - minx)), 1),
                    weight[ifnt], italic[ifnt], QFont::AnyCharSet));
          } else
            s += l;
        }
        setFont(QFont(face[iface[0]],
                QMAX(int(0.0355 * relSize * (maxx - minx)), 1),
                weight[0], italic[0]));
      }
    }
    setPen(pen3);
    setWorldMatrix(wm);
  }
}

void KGraph::textcm(const char* str, double x, double y, int)
{
  double xa, ya;
  cm2r(x, y + 0.18, 1.0, 1.0, &xa, &ya);
  int ix, iy;
  r2i(logx ? log10(xa) : xa, logy ? log10(ya) : ya, &ix, &iy);
  text(str, ix, iy, 1);
}

void KGraph::letH(const char* str)
{
  int ix, iy;
  r2i(0.5 * (xmin + xmax),
      ymax - 0.04275 * relSize * (xmax - xmin) * xscal / yscal, &ix, &iy);
  text(str, ix, iy, 2);
}

void KGraph::letX(const char* str, int iex)
{
  letxy(str, iex, 1);
}

void KGraph::letY(const char* str, int iex)
{
  letxy(str, iex, 2);
}

void KGraph::setSymbol(int is)
{
  kSymb = is;
}

void KGraph::circle(double x0, double y0, double r)
{
  int ix = int(factor * scalx * x0);
  int iy = miny0 + int(factor * scaly * y0);
  int ir = int(factor * scalx * r);
  drawEllipse(ix - ir, iy - ir, ir + ir, ir + ir);
}

void KGraph::arcrel(double x, double y, double r, double angst, double angend)
{
  int x0, y0;
  r2i(x, y, &x0, &y0);
  int ir = int(xscal * fabs(r));
  double phi = 1.74532925e-2 * angst;
  int xstart = int(ir * cos(phi));
  int ystart = int(ir * sin(phi));
  if (r > 0.0) {
    x0 -= xstart;
    y0 += ystart;
  }
  drawArc(x0 - ir, y0 - ir, ir + ir, ir + ir,
          int(16.0 * angst), int(16.0 * angend));
}

void KGraph::plArray(const double *x, const double *y, double fx, double fy,
                     int n)
{
  if (kSymb == -17)
    return;
#ifndef NO_CLIPPING
  setClipping(true);
#endif
  double xa, ya;
  if (kSymb > 0) {
    linetype(kSymb);
    xa = fx * x[0];
    ya = fy * y[0];
    bool b = false;
    if (((!logx) || (xa > 0)) && ((!logy) || (ya > 0))) {
      move(logx ? log10(xa) : xa, logy ? log10(ya) : ya);
      b = true;
    }
    for (int i = 1; i < n; i++) {
      xa = fx * x[i];
      ya = fy * y[i];
      if (((!logx) || (xa > 0)) && ((!logy) || (ya > 0)))
        if (b)
          draw(logx ? log10(xa) : xa, logy ? log10(ya) : ya);
        else {
          move(logx ? log10(xa) : xa, logy ? log10(ya) : ya);
          b = true;
        }
    }
    linetype(1);
  } else {
    int k = (-kSymb) % 17;
    QPointArray ap(10);
    if ((k > 10) || (k == 1)) {
      setBrush(pen3.color());
      setPen(NoPen);
    }
    const int ax[48] = {0, -2, 0, 2, 0, 2, -2, -2, 2, 2, 0, -2, 2, 0,
                        2, -2, 2, -2, 2, 2, 1, -1, -2, -2, -1, 1, 2, 2,
                        2, -2, 0, -2, 2, 0, 0, 0, -2, 2, 2, 0, -2, 0, 0,
                        0, -2, 2, 0, 0},
              ay[48] = {2, 0, -2, 0, 2, 2, 2, -2, -2, 2, 2, -1, -1, 2,
                        2, 2, -2, -2, 2, 1, 2, 2, 1, -1, -2, -2, -1, 1,
                        2, -2, 0, 2, -2, 2, -2, 0, 0, 0, 2, 0, 2, 0, -2,
                        2, 0, 0, 2, -2},
              as[10] = {0, 5, 10, 14, 19, 28, 33, 38, 43, 48};
    int j, l = 0, r;
    for (int i = 0; i < n; i++) {
      double f = 0.005 * relSize * (maxx - minx);
      xa = fx * x[i];
      ya = fy * y[i];
      if (((!logx) || (xa > 0)) && ((!logy) || (ya > 0))) {
        move(logx ? log10(xa) : xa, logy ? log10(ya) : ya);
        switch (k) {
          case 0:
          case 11:
            r = int(f + f);
            drawEllipse(kxcur - r, kycur - r, r + r, r + r);
            break;
          case 1:
            r = int(0.4 * f);
            drawEllipse(kxcur - r, kycur - r, r + r, r + r);
            break;
          default:
            r = (k < 12) ? k : k - 10;
            for (j = as[r - 2]; j < as[r - 1]; j++) {
              l = j - as[r - 2];
              if (k < 12)
                if (l)
                  lineTo(int(kxcur + f * ax[j]), int(kycur - f * ay[j]));
                else
                  moveTo(int(kxcur + f * ax[j]), int(kycur - f * ay[j]));
              else
                ap[l] = QPoint(int(kxcur + f * ax[j]), int(kycur - f * ay[j]));
            }
            if (k >= 12)
              drawPolygon(ap, false, 0, l);
        }
      }
    }
    if ((k > 10) || (k == 1)) {
      setBrush(NoBrush);
      setPen(pen3);
    }
  }
#ifndef NO_CLIPPING
  setClipping(false);
#endif
}

void KGraph::plError(const double* x, const double* y, const double* e,
                     double fx, double fy, int n)
{
#ifndef NO_CLIPPING
  setClipping(true);
#endif
  int idy = int(0.01 * relSize * (maxx - minx));
  int idx = idy >> 1;
  for (int i = 0; i < n; i++)
    if (((!logx) || (x[i] > 0)) && ((!logy) || (y[i] > 0))) {
      int ie;
      if (logy)
        ie = int(-0.4343 * e[i] * yscal / y[i]);
      else
        ie = int(-fy * e[i] * yscal);
      double xa = fx * x[i];
      if (logx)
        xa = log10(xa);
      double ya = fy * y[i];
      if (logy)
        ya = log10(ya);
      if ((xa >= xmin) && (xa <= xmax) &&
          (ya >= ymin) && (ya <= ymax) && (ie >= idy)) {
        move(xa, ya);
        moveTo(kxcur, kycur + idy);
        lineTo(kxcur, kycur + ie);
        moveTo(kxcur - idx, kycur + ie);
        lineTo(kxcur + idx, kycur + ie);
        moveTo(kxcur, kycur - idy);
        lineTo(kxcur, kycur - ie);
        moveTo(kxcur - idx, kycur - ie);
        lineTo(kxcur + idx, kycur - ie);
      }
    }
#ifndef NO_CLIPPING
  setClipping(false);
#endif
}

void KGraph::linetype(int is)
{
  const int pat[9][2] = {{8, 8}, {7, 0}, {2, 0}, {1, 0}, {9, 0},
                         {8, 1}, {7, 2}, {4, 4}, {1, 1}};

  int j = QMAX(0, (is - 1) % 9);
  double f = relSize * (maxx - minx) / (scalx * 320.0);
  double dsp = (double) ((16 - pat[j][0] - pat[j][1]) >> 1);
  dash[0] = f * pat[j][0];
  dash[1] = dash[0] + f * dsp;
  dash[2] = dash[1] + f * pat[j][1];
  dash[3] = dash[2] + f * dsp;
}

void KGraph::letxy(const char* str, int iex, int iax)
{
  if (strlen(str) || iex) {
    double x, y;
    char label[256];
    double dx = 0.025 * relSize * (xmax - xmin);
    if (iax == 2) {
      int l = 3;
      if (logy)
        if ((ymax > 2.999999) || (ymin < -2.999999))
          l += (ymin < 0) ? 4 : 3;
        else
          l += 1 + int(QMAX(ymax, 1.99 - ymin));
      else
        l += QMAX(prenum(ymin, ndigy, label), prenum(ymax, ndigy, label));
      x = xmin - (0.75 * l + 0.71) * dx;
      y = 0.5 * (ymin + ymax);
      dir(90.0);
    } else {
      x = 0.5 * (xmin + xmax);
      y = ymin + 3.79 * dx * xscal / yscal;
    }
    int ix, iy;
    r2i(x, y, &ix, &iy);
    if (iex) {
      sprintf(label, "10#nu%d#nn %s", iex, str);
      text(label, ix, iy, 2);
    } else
      text(str, ix, iy, 2);
    if (iax == 2)
      dir(0.0);
  }
}

int prenum(double a, int ndig, char* label)
{
  sprintf(label, "%.*f", QMAX(0, ndig), a);
  if (ndig == 0)
    strcat(label, ".");
  return strlen(label);
}

double incLog(double* x, double f)
{
  double mant = *x - int(*x + 1.0e-6 * fabs(*x));
  if (fabs(mant) < 1.0e-9)
    mant = 0.0;
  if (mant < 0.0)
    mant += 1.0;
  if (mant < 0.1)
    *x += log10(f);
  else
    if (mant < 0.4)
      *x += log10(2.5);
    else
      if (mant < 0.6)
        *x += log10(10.0 / 3.0);
      else
        *x += log10(2.0);
  return mant;
}
