/***************************************************************************
                          arrayitem.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sat Aug 28 1999
    copyright            : (C) 2001 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdlib.h>
#include <math.h>
#include <qfileinfo.h>
#include <qurl.h>
#include <klocale.h>
#include <ksimpleconfig.h>
#include <kmessagebox.h>
#include <kio/netaccess.h>
#include "arrayitem.h"
#include "arraydlg.h"
#include "kpldoc.h"
#include "kplgraph.h"
#include "kplchecklistitem.h"
#include "utils.h"

ArrayItem::ArrayItem() : ix(0), iy(0), ie(0), errbars(0)
{
  init();
}

ArrayItem::ArrayItem(const ArrayItem& a) :
 ScaledItem(a), ix(a.ix), iy(a.iy), ie(a.ie), istart(a.istart), n(a.n),
 errbars(a.errbars), nrows(a.nrows), ncols(a.ncols), x(a.x), url(a.url)
{
  if (a.x)
    x = a.copyX();
}

ArrayItem::ArrayItem(Kpl::AutoStruct* aut) : ScaledItem(aut),
 ix(aut->ixAuto), iy(aut->iyAuto), ie(aut->ieAuto), errbars(aut->autoErr)
{
  init();
}

ArrayItem::ArrayItem(KSimpleConfig* plo, Kpl::AutoStruct* aut,
                     const KURL& uPlo) : ScaledItem(plo, aut)
{
  init();
  QString s = plo->readEntry("path", "");
  url = QUrl::isRelativeUrl(s) ? (uPlo.directory(false) + s) : s;
  if (!readFile()) {
    KMessageBox::error(0, url.url());
    return;
  }
  int icolmax = ncols - 1;
  ix = plo->readNumEntry("ix", QMIN(aut->ixAuto, icolmax));
  iy = plo->readNumEntry("iy", QMIN(aut->iyAuto, icolmax));
  ie = plo->readNumEntry("ie", QMIN(aut->ieAuto, icolmax));
  istart = plo->readNumEntry("istart");
  n = plo->readNumEntry("n", nrows - istart);
  errbars = plo->readNumEntry("errbars", aut->autoErr);
}

ArrayItem::ArrayItem(bool act, int sym, const QString& col,
                     double xn, double yn, int i_x, int i_y, int i_e,
                     int i_start, int np, int errb, const KURL& u)
 : ScaledItem(act, sym, col, xn, yn), ix(i_x), iy(i_y), ie(i_e),
 errbars(errb), url(u)
{
  init();
  if (!readFile()) {
    KMessageBox::error(0, url.url());
    return;
  }
  istart = i_start;
  n = np;
}

ArrayItem::~ArrayItem()
{
  freeX();
}

ArrayItem& ArrayItem::operator=(const ArrayItem& a)
{
  if (this != &a) {
    *(ScaledItem*)this = a;
    ix = a.ix;
    iy = a.iy;
    ie = a.ie;
    istart = a.istart;
    n = a.n;
    errbars = a.errbars;
    nrows = a.nrows;
    ncols = a.ncols;
    url = a.url;
    freeX();
    x = a.copyX();
  }
  return *this;
}

KplItem::ItemTypes ArrayItem::iType() const
{
  return Array;
}

void ArrayItem::init()
{
  istart = n = nrows = ncols = 0;
  x = 0;
}

void ArrayItem::freeX()
{
  if (x) {
    delete x[0];
    delete [] x;
    x = 0;
  }
}

int ArrayItem::readFile()
{
  freeX();
  ncols = nrows = 0;
  QString fn;
  if (url.isLocalFile())
    fn = url.path();
  else
    if (!KIO::NetAccess::download(url, fn))
      return nrows;
  if (FILE* f = fopen(fn, "r")) {
    int i;
    char s[256];
    do
      if ((i = (fgets(s, 256, f) != 0)))
        i = (s[0] == '#') ? 1 : 2;
    while (i == 1);
    if (i) {
      const char delim[] = " ,\t";
      char* s1 = strtok(s, delim);
      while (s1) {
        ncols++;
        s1 = strtok(0, delim);
      }
      if (ncols) {
        QArray<double> x2;
        rewind(f);
        int np = 0;
        int k = 0;
        while (fgets(s, 256, f))
          if (s[0] != '#') {
            nrows++;
            np += ncols;
            x2.resize(np);
            x2[k++] = ((s1 = strtok(s, delim))) ? atof(s1) : 0.0;
            for (i = 1; i < ncols; i++)
              x2[k++] = ((s1 = strtok(0, delim))) ? atof(s1) : 0.0;
          }
        double* x1 = new double[np];
        x = new double*[ncols];
        for (i = 0; i < ncols; i++)
          x[i] = &x1[i * nrows];
        k = 0;
        for (int j = 0; j < nrows; j++)
          for (i = 0; i < ncols; i++)
            x[i][j] = x2[k++];
      }
    }
    fclose(f);
  }
  if (!url.isLocalFile())
    KIO::NetAccess::removeTempFile(fn);
  return nrows;
}

void ArrayItem::draw(KplGraph* g) const
{
  if (x && active) {
    setProperties(g);
    g->plArray(&x[ix][istart], &x[iy][istart], fx, fy, n);
    if (errbars)
      g->plError(&x[ix][istart], &x[iy][istart], &x[ie][istart], fx, fy, n);
  }
}

void ArrayItem::writePlo(FILE* f, const KURL& uPlo, bool abs) const
{
  fprintf(f, "Type=ARRAYITEM\n");
  ScaledItem::writePlo(f, url, abs);
  fprintf(f, "ix=%i\niy=%i\nie=%i\nistart=%i\nn=%i\nerrbars=%i\npath=%s\n",
          ix, iy, ie, istart, n, errbars,
          Utils::relPath(uPlo, url, abs).latin1());
}

void ArrayItem::setText(KplCheckListItem* it, bool* arrays, bool*) const
{
  *arrays = true;
  it->setText(1, i18n("Array"));
  it->setText(2, i18n("file") + " " + url.fileName() + ", " + i18n("columns")
              + " " + QString::number(ix) + ", " + QString::number(iy));
}

int ArrayItem::editItem(QWidget* parent, KplDoc* m, int)
{
  ArrayDlg dlg(parent, m, this);
  return dlg.exec();
}

double** ArrayItem::copyX() const
{
  int i = ncols * nrows;
  double* x1 = new double[i];
  memcpy(x1, x[0], i * sizeof(double));
  double** x2 = new double*[ncols];
  for (i = 0; i < ncols; i++)
    x2[i] = &x1[i * nrows];
  return x2;
}

KplItem* ArrayItem::copy() const
{
  return new ArrayItem(*this);
}

void ArrayItem::expoItem(int* iext, int* ieyt, double* fxt, double* fyt) const
{
  if (x) {
    double xmin, xmax, ymin, ymax;
    minMax(&xmin, &xmax, &ymin, &ymax);
    Utils::expo(QMAX(fabs(xmin), fabs(xmax)), iext, fxt);
    Utils::expo(QMAX(fabs(ymin), fabs(ymax)), ieyt, fyt);
  }
}

void ArrayItem::minMax(double* xmi, double* xma, double* ymi, double* yma) const
{
  if (x) {
    Utils::minMaxFile(xmi, xma, &x[ix][istart], n);
    Utils::minMaxFile(ymi, yma, &x[iy][istart], n);
  }
}
