/*
 * peoplespace.cpp
 *
 * Copyright (C) 1999 Mike Pilone <mpilone@slac.com>
 */
// Local includes
#include "peoplespace.h"
#include "psentryviewer.h"
#include "psassociate.h"
#include "psdatabasemanager.h"
#include "errordialog.h"
#include "resource.h"
#include "psentry.h"
#include "startupwizard.h"
#include "databaseresource.h"
#include "dbopentypedialog.h"
#include "lookupdialog.h"
#include "configdialog.h"

// QT includes
#include <qkeycode.h>
#include <qpopupmenu.h>
#include <qmessagebox.h>

// KDE includes
#include <kfm.h>
#include <kiconloader.h>
#include <ktoolbar.h>
#include <kstatusbar.h>
#include <kmenubar.h>
#include <kconfig.h>
#include <kmsgbox.h>
#include <kfiledialog.h>
#include <ksock.h>
#include <kprocess.h>

// Stdlib includes
#include <iostream.h> 
#include <stdlib.h>        

PeopleSpaceApp::PeopleSpaceApp()
{
   fPSEntryViewer = NULL;
  fPSDatabaseManager = NULL;
  fPSAssociate = NULL;
  fLookupDialog = NULL;

  // Set the caption
  QString caption = "PeopleSpace v";
  caption += VERSION;
  
  setCaption(caption);

  // Complete the initialization
  init();

  thereIsOpenDB( false );
  thereIsSelectedEntry( false );

  // Open the database iff needed
  slotOpenDatabase( true );
}

PeopleSpaceApp::~PeopleSpaceApp()
{
  delete(fToolBar1);
  delete(fMenuBar);
  delete(fStatusBar);

  delete(fFilePopMenu);
  delete(fEditPopMenu); 
  delete(fHelpPopMenu);
  delete(fContactPopMenu);
  delete(fLookupPopMenu);
  delete(fViewPopMenu);
  delete(fViewGroupPopMenu);

  delete(fPSEntryViewer);       
  delete(fPSDatabaseManager);
  delete(fPSAssociate);

  // delete the error dialog last in case anything wants to report a final error
  delete(gErrorDialog);                
}

void PeopleSpaceApp::init()
{
 KConfig* config = kapp->getConfig();

  // check to see if the welcome dialog needs to be run
 config->setGroup("APPEARANCE");
 if (!config->hasKey("MainGeometry"))
   {
     StartupWizard wizard(this, "StartupWizard");
     wizard.exec();
   }
 
  config->sync();

  // Read any opotions from the config that will be needed
  // for the widget initialization
  readOptions();

  // Set the geometry of this window before creating the insides, like the toolbar
  // statusbar and menubar.
  this->setGeometry(fMainGeomRect);
  
  // create a DropZone over the entire window and connect it
  // to the slotDropEvent
  //connect(new KDNDDropZone(this, DndURL), 
//	  SIGNAL(dropAction(KDNDDropZone *)), 
//	  SLOT(slotDropEvent(KDNDDropZone *)));

  // Create the lookupDialog
  fLookupDialog = new LookupDialog(this, "LookupDialog");
  fLookupDialog->hide();
  connect(fLookupDialog, SIGNAL( lookupButtonClicked(int, QString, QString) ), this, SLOT( slotLookupButtonClicked(int, QString, QString)));

  // setup the databse manager first so it can be given to the other classes
  initPSDatabaseManager();
  initAssociate();

  // create each of the main window components
  initMenu();
  initToolBar();
  initStatusBar();
  initPSEntryViewer();

 resizeEvent(NULL);
}

void PeopleSpaceApp::initPSEntryViewer()
{
  fPSEntryViewer = new PSEntryViewer(fPSDatabaseManager, this, "PSEntryViewer");
  connect(fPSEntryViewer, SIGNAL(unsavedDataExists()), this, SLOT( slotUnsavedDataExists()));

  setView(fPSEntryViewer);
}

void PeopleSpaceApp::initToolBar()
{
  /////////////////////////////////////////////////
  // Toolbar 1

  fToolBar1 = new KToolBar(this, "ToolBar1");
  fToolBar1->setBarPos(fToolBar1Pos);

  fToolBar1->insertButton(Icon("start.xpm"), ID_VIEW_FIRST, true, i18n("First Contact"));
  fToolBar1->insertButton(Icon("back.xpm"), ID_VIEW_PREV, true, i18n("Previous Contact"));
  fToolBar1->insertButton(Icon("forward.xpm"), ID_VIEW_NEXT, true, i18n("Next Contact"));
  fToolBar1->insertButton(Icon("finish.xpm"), ID_VIEW_LAST, true, i18n("Last Contact"));

  fToolBar1->insertSeparator();
  
  fToolBar1->insertButton(Icon("fileopen.xpm"), ID_FILE_OPEN, true, i18n("Open Database..."));
  fToolBar1->insertButton(Icon("delete.xpm"), ID_FILE_CLOSE, true, i18n("Close Database"));
  fToolBar1->insertButton(Icon("filefloppy.xpm"), ID_FILE_SAVE, true, i18n("Save Changes"));

  fToolBar1->insertSeparator();
  
  fToolBar1->insertButton(Icon("filenew.xpm"), ID_CONTACT_NEW, true, i18n("New Contact"));
  fToolBar1->insertButton(Icon("duplicate.xpm"), ID_CONTACT_DUPLICATE, true, i18n("Duplicate Contact"));
  fToolBar1->insertButton(Icon("filedel.xpm"), ID_CONTACT_DELETE, true, i18n("Delete Contact"));

  fToolBar1->insertSeparator();
  fToolBar1->insertButton(Icon("exit.xpm"), ID_APP_EXIT, true, i18n("Exit"));
 
 fToolBar1->insertSeparator();
  fToolBar1->insertButton(Icon("listview.xpm"), ID_VIEW_LIST, true, i18n("List View"));
  fToolBar1->insertSeparator();
  
  fToolBar1->insertCombo(i18n("Personal"), ID_TOOLBAR1_COMBO, FALSE, SIGNAL( activated(int) ), this, SLOT( slotGroupFilterChanged(int) ), 
			 TRUE, i18n("Contact Filter"), 125);
  fToolBar1->insertComboItem(ID_TOOLBAR1_COMBO, i18n("Business"), PSEntry::groupBusiness);
  fToolBar1->insertComboItem(ID_TOOLBAR1_COMBO, i18n("Unfiled"), PSEntry::groupUnfiled);
  fToolBar1->insertComboItem(ID_TOOLBAR1_COMBO, i18n("All"), PSEntry::groupUnfiled+1);
  fToolBar1->insertComboItem(ID_TOOLBAR1_COMBO, i18n("Search Results"), ID_SEARCH_RESULTS);
  fToolBar1->setCurrentComboItem(ID_TOOLBAR1_COMBO, fGroupFilter);
  
  CONNECT_TOOLBAR(fToolBar1);

  addToolBar(fToolBar1);
}

void PeopleSpaceApp::initMenu()
{
  fMenuBar = new KMenuBar(this, "MenuBar");
  
  // Set up the menu items

  ///////////////////////////////////////////////
  // File menu
  fFilePopMenu = new QPopupMenu();
  fFilePopMenu->insertItem(Icon("fileopen.xpm"), i18n("&Open Database..."), ID_FILE_OPEN);
  fFilePopMenu->insertItem(Icon("delete.xpm"), i18n("&Close Database"), ID_FILE_CLOSE);
  fFilePopMenu->insertItem(Icon("filefloppy.xpm"), i18n("&Save Database"), ID_FILE_SAVE);
  fFilePopMenu->insertSeparator();
  fFilePopMenu->insertItem(Icon("exit.xpm"), i18n("E&xit"), ID_APP_EXIT);
  
  fFilePopMenu->setAccel(CTRL+Key_S, ID_FILE_SAVE);
  fFilePopMenu->setAccel(CTRL+Key_Q, ID_APP_EXIT);

  ////////////////////////////////////////////////
  // Edit menu
  fEditPopMenu = new QPopupMenu();
  fEditPopMenu->insertItem(i18n("&Preferences"), ID_EDIT_PREFS);
  
  fEditPopMenu->setAccel(CTRL+Key_P, ID_EDIT_PREFS);

  ////////////////////////////////////////////////
  // View menu (and sub menus)
  fViewGroupPopMenu = new QPopupMenu();
  fViewGroupPopMenu->insertItem(i18n("&Personal"), ID_VIEW_GROUP_PERSONAL);
  fViewGroupPopMenu->insertItem(i18n("&Business"), ID_VIEW_GROUP_BUSINESS);
  fViewGroupPopMenu->insertItem(i18n("&Unfiled"), ID_VIEW_GROUP_UNFILED);
  fViewGroupPopMenu->insertItem(i18n("&All"), ID_VIEW_GROUP_ALL);
  fViewGroupPopMenu->insertItem(i18n("&Search Results"), ID_VIEW_GROUP_SEARCHRESULTS);
  fViewGroupPopMenu->setCheckable( TRUE );
  
  fViewPopMenu = new QPopupMenu();
  fViewPopMenu->insertItem(i18n("&Group"), fViewGroupPopMenu, ID_VIEW_GROUP);
  fViewPopMenu->insertSeparator();
  fViewPopMenu->insertItem(Icon("start.xpm"), i18n("&First"), ID_VIEW_FIRST);
  fViewPopMenu->insertItem(Icon("back.xpm"), i18n("&Prev"), ID_VIEW_PREV);
  fViewPopMenu->insertItem(Icon("forward.xpm"), i18n("&Next"), ID_VIEW_NEXT);
  fViewPopMenu->insertItem(Icon("finish.xpm"), i18n("&Last"), ID_VIEW_LAST);
  fViewPopMenu->insertSeparator();
  fViewPopMenu->insertItem(Icon("listview.xpm"), i18n("L&ist View"), ID_VIEW_LIST);

  ////////////////////////////////////////////////
  // Contact menu
  fContactPopMenu = new QPopupMenu();
  fContactPopMenu->insertItem(Icon("filenew.xpm"), i18n("&New"), ID_CONTACT_NEW);
  fContactPopMenu->insertItem(Icon("duplicate.xpm"), i18n("D&uplicate"), ID_CONTACT_DUPLICATE);
  fContactPopMenu->insertItem(Icon("filedel.xpm"), i18n("&Delete"), ID_CONTACT_DELETE);

  fContactPopMenu->setAccel(CTRL+Key_N, ID_CONTACT_NEW);
  fContactPopMenu->setAccel(CTRL+Key_D, ID_CONTACT_DELETE);

  ////////////////////////////////////////////////
  // Lookup Menu
  fLookupPopMenu = new QPopupMenu();
  fLookupAddressPopMenu = new QPopupMenu();
  fLookupPersonalPopMenu = new QPopupMenu();
  fLookupDatePopMenu = new QPopupMenu();
  fLookupRecordPopMenu = new QPopupMenu();

  fLookupPersonalPopMenu->insertItem(i18n("&First Name"), ID_LOOKUP_FIRSTNAME);
  fLookupPersonalPopMenu->insertItem(i18n("&Last Name"), ID_LOOKUP_LASTNAME);
  fLookupPersonalPopMenu->insertItem(i18n("&Title"), ID_LOOKUP_TITLE);
  fLookupPersonalPopMenu->insertItem(i18n("&Company"), ID_LOOKUP_COMPANY);
  fLookupPersonalPopMenu->insertItem(i18n("&Phone..."), ID_LOOKUP_PHONE);
  //fLookupPopMenu->insertSeparator();
  fLookupAddressPopMenu->insertItem(i18n("&Address"), ID_LOOKUP_ADDRESS);
  fLookupAddressPopMenu->insertItem(i18n("C&ity"), ID_LOOKUP_CITY);
  fLookupAddressPopMenu->insertItem(i18n("&State"), ID_LOOKUP_STATE);
  fLookupAddressPopMenu->insertItem(i18n("&Zip"), ID_LOOKUP_ZIP);
  fLookupAddressPopMenu->insertItem(i18n("C&ountry"), ID_LOOKUP_COUNTRY);
  //fLookupPopMenu->insertSeparator();
  fLookupRecordPopMenu->insertItem(i18n("C&ustom..."), ID_LOOKUP_CUSTOM);
  fLookupRecordPopMenu->insertItem(i18n("&Referred By"), ID_LOOKUP_REFERREDBY);
  fLookupRecordPopMenu->insertItem(i18n("R&ecord Creator"), ID_LOOKUP_RECORDCREATOR);
  //fLookupPopMenu->insertSeparator();
  fLookupDatePopMenu->insertItem(i18n("Edit Date"), ID_LOOKUP_EDITDATE);
  fLookupDatePopMenu->insertItem(i18n("Merge Date"), ID_LOOKUP_MERGEDATE);
  fLookupDatePopMenu->insertItem(i18n("Last Reach"), ID_LOOKUP_LASTREACH);
  fLookupDatePopMenu->insertItem(i18n("Last Attempt"), ID_LOOKUP_LASTATTEMPT);
  fLookupDatePopMenu->insertItem(i18n("Last Meeeting"), ID_LOOKUP_LASTMEETING);
  
  fLookupPopMenu->insertItem(i18n("&Personal"), fLookupPersonalPopMenu, ID_LOOKUP_PERSONALMENU);
  fLookupPopMenu->insertItem(i18n("&Address"), fLookupAddressPopMenu, ID_LOOKUP_ADDRESSMENU);
  fLookupPopMenu->insertItem(i18n("&Date"), fLookupDatePopMenu, ID_LOOKUP_DATEMENU);
  fLookupPopMenu->insertItem(i18n("&Record"), fLookupRecordPopMenu, ID_LOOKUP_RECORDMENU);

  ////////////////////////////////////////////////
  // Help menu
  QString caption = "PeopleSpace v";
  caption += VERSION;
  caption += "\n\nCopyright (c) 1998 by Mike Pilone <mpilone@slac.com>\n\nThis program is licensed uner the GNU General Piblic License (GPL).\nPeopleSpace comes with ABSOLUTELY NO WARRANTY to the extent permitted by applicable law.";
  fHelpPopMenu = kapp->getHelpMenu(TRUE, i18n(caption));
  
  // Now the menubar
  fMenuBar->insertItem(i18n("&File"), fFilePopMenu);
  fMenuBar->insertItem(i18n("&Edit"), fEditPopMenu);
  fMenuBar->insertItem(i18n("&Contact"), fContactPopMenu);
  fMenuBar->insertItem(i18n("&Lookup"), fLookupPopMenu);
  fMenuBar->insertItem(i18n("&View"), fViewPopMenu);
  fMenuBar->insertSeparator();
  fMenuBar->insertItem(i18n("&Help"), fHelpPopMenu);
  
  CONNECT_CMD(fFilePopMenu);
  CONNECT_CMD(fEditPopMenu);
  CONNECT_CMD(fContactPopMenu);
  CONNECT_CMD(fLookupPopMenu);
  CONNECT_CMD(fLookupRecordPopMenu);
  CONNECT_CMD(fLookupAddressPopMenu);
  CONNECT_CMD(fLookupDatePopMenu);
  CONNECT_CMD(fLookupPersonalPopMenu);
  CONNECT_CMD(fViewPopMenu);
  CONNECT_CMD(fViewGroupPopMenu);

  setMenu(fMenuBar);
  
}

void PeopleSpaceApp::initStatusBar()
{
  fStatusBar = new KStatusBar(this, "StatusBar");
  fStatusBar->setInsertOrder(KStatusBar::RightToLeft);
  
  fStatusBar->insertItem(i18n(MSG_DB_TYPE_READWRITE), 0);
  fStatusBar->insertItem(i18n(MSG_NO_DB_OPEN), 1);
  fStatusBar->changeItem(i18n(MSG_DB_TYPE_NOTOPEN), 0);
 

  setStatusBar(fStatusBar);
}

void PeopleSpaceApp::initPSDatabaseManager()
{
  fPSDatabaseManager = new PSDatabaseManager;
   connect(fPSDatabaseManager, SIGNAL(groupFilterApplied(int, 
							QList<QString>*,  
							QList<QString>*,  
							QList<QString>*,  
							QList<QString>*)),
	  this, SLOT( slotGroupFilterApplied(int, 
					     QList<QString>*,  
					     QList<QString>*,  
					     QList<QString>*,  
					     QList<QString>*)));
  connect(fPSDatabaseManager, SIGNAL( databaseHasOpened(QString, int) ), this, SLOT( slotDatabaseHasOpened(QString, int) ));
  connect(fPSDatabaseManager, SIGNAL( databaseHasClosed() ), this, SLOT( slotDatabaseHasClosed() ));
  //  connect(fPSDatabaseManager, SIGNAL( setViewedEntry(PSEntry*, bool, bool) ), this, SLOT( slotSetViewedEntry(PSEntry*, bool, bool) ));
}

void PeopleSpaceApp::initAssociate()
{
  fPSAssociate = new PSAssociate(fPSDatabaseManager, NULL, "PSAssociate");
  fPSAssociate->setGeometry(fPSAssociateGeomRect);

  connect(fPSAssociate, SIGNAL( entrySelected(int, bool, bool)), this, SLOT( slotSetViewedEntry(int,bool,bool) ));

  fPSAssociate->show();
}

bool PeopleSpaceApp::queryExit()
{
  return checkForUnsavedData();
}

void PeopleSpaceApp::saveOptions()
{

 // write the config file entries
  KConfig *config = kapp->getConfig();
  KCombo *combo = fToolBar1->getCombo(ID_TOOLBAR1_COMBO); 
 
  config->setGroup("APPEARANCE");
  config->writeEntry("MainGeometry", this->geometry());
  config->writeEntry("PSAssociateGeometry", fPSAssociate->geometry());
  config->writeEntry("ToolBar1Pos", (int)fToolBar1->barPos());
  config->writeEntry("GroupFilter", combo->currentItem());

  fPSEntryViewer->saveYourself();

  config->sync();

}

void PeopleSpaceApp::readOptions()
{
  // read the config file entries
  KConfig *config = kapp->getConfig();

  config->setGroup("APPEARANCE");
  fMainGeomRect = config->readRectEntry("MainGeometry",&QRect(0,0,675,380));
  fPSAssociateGeomRect = config->readRectEntry("PSAssociateGeometry", &QRect(0,0,400,175));
  fToolBar1Pos = (KToolBar::BarPosition)config->readNumEntry("ToolBar1Pos", KToolBar::Top);
  fGroupFilter = config->readNumEntry("GroupFilter", PSEntry::groupUnfiled+1);
}

void PeopleSpaceApp::saveProperties(KConfig *)
{
	// the 'config' object points to the session managed
	// config file.  anything you write here will be available
	// later when this app is restored
	
	// e.g., config->writeEntry("key", var); 
}

void PeopleSpaceApp::readProperties(KConfig *)
{
	// the 'config' object points to the session managed
	// config file.  this function is automatically called whenever
	// the app is being restored.  read in here whatever you wrote
	// in 'saveProperties'

	// e.g., var = config->readEntry("key"); 
}

bool PeopleSpaceApp::checkForUnsavedData()
{
  if ((!gUnsavedData) || (fPSDatabaseManager->openDBType() != dbOpenReadWrite))
    return true; 
  
  switch(QMessageBox::warning(this,
			      i18n("Modified Contact List Warning"),
			      i18n("Changes have been made to this Contact List and they haven't been\nwritten to disk. What would you like to do?"),
			      i18n("&Ignore"),
			      i18n("&Save"),
			      i18n("&Cancel"),
			      2)) {
  case 0: // Ignore
    break;
    
  case 1:
    slotSaveLoadedEntryList();
    break;
    
  case 2:
    return false;
    break;
  } 
  
  return TRUE;
}

void PeopleSpaceApp::thereIsOpenDB(bool state)
{
  if (state)  // TRUE, There is a DB open
    {
      fToolBar1->setItemEnabled(ID_CONTACT_NEW, TRUE);
      fToolBar1->setItemEnabled(ID_TOOLBAR1_COMBO, true);
      fToolBar1->setItemEnabled(ID_FILE_CLOSE, true);

      fMenuBar->setItemEnabled(3, TRUE);
      fMenuBar->setItemEnabled(2, TRUE);
      //  fFilePopMenu->setItemEnabled(ID_FILE_SAVE, TRUE);
      fViewPopMenu->setItemEnabled(ID_VIEW_GROUP, TRUE);
      fFilePopMenu->setItemEnabled(ID_FILE_CLOSE, TRUE);
      
    }

  else
    {
      fToolBar1->setItemEnabled(ID_CONTACT_NEW, FALSE);
      fToolBar1->setItemEnabled(ID_CONTACT_DUPLICATE, FALSE);
      fToolBar1->setItemEnabled(ID_CONTACT_DELETE, FALSE);
      fToolBar1->setItemEnabled(ID_VIEW_FIRST, FALSE);
      fToolBar1->setItemEnabled(ID_VIEW_NEXT, FALSE);
      fToolBar1->setItemEnabled(ID_VIEW_PREV, FALSE);
      fToolBar1->setItemEnabled(ID_VIEW_LAST, FALSE);
      fToolBar1->setItemEnabled(ID_FILE_SAVE, FALSE);
      fToolBar1->setItemEnabled(ID_TOOLBAR1_COMBO, FALSE);
      fToolBar1->setItemEnabled(ID_FILE_CLOSE, false);

      fMenuBar->setItemEnabled(3, FALSE);
      fMenuBar->setItemEnabled(2, FALSE);
      fFilePopMenu->setItemEnabled(ID_FILE_SAVE, FALSE);
      fViewPopMenu->setItemEnabled(ID_VIEW_GROUP, FALSE);
      fViewPopMenu->setItemEnabled(ID_VIEW_FIRST, FALSE);
      fViewPopMenu->setItemEnabled(ID_VIEW_NEXT, FALSE);
      fViewPopMenu->setItemEnabled(ID_VIEW_PREV, FALSE);
      fViewPopMenu->setItemEnabled(ID_VIEW_LAST, FALSE);
      fViewPopMenu->setItemEnabled(ID_FILE_SAVE, FALSE);
      fFilePopMenu->setItemEnabled(ID_FILE_CLOSE, FALSE);
    }
      
}

void PeopleSpaceApp::thereIsSelectedEntry(bool state)
{
  if (state) // TRUE, There is a selected entry
    {
      fContactPopMenu->setItemEnabled(ID_CONTACT_DUPLICATE, TRUE);
      fContactPopMenu->setItemEnabled(ID_CONTACT_DELETE, TRUE);
      fToolBar1->setItemEnabled(ID_CONTACT_DUPLICATE, TRUE);
      fToolBar1->setItemEnabled(ID_CONTACT_DELETE, TRUE);
    
    }

  else
    {
      fToolBar1->setItemEnabled(ID_CONTACT_DUPLICATE, FALSE);
      fToolBar1->setItemEnabled(ID_CONTACT_DELETE, FALSE);
 
      fContactPopMenu->setItemEnabled(ID_CONTACT_DUPLICATE, FALSE);
      fContactPopMenu->setItemEnabled(ID_CONTACT_DELETE, FALSE);
      fViewPopMenu->setItemEnabled(ID_VIEW_FIRST, FALSE);
      fViewPopMenu->setItemEnabled(ID_VIEW_NEXT, FALSE);
      fViewPopMenu->setItemEnabled(ID_VIEW_PREV, FALSE);
      fViewPopMenu->setItemEnabled(ID_VIEW_LAST, FALSE);
      fViewPopMenu->setItemEnabled(ID_VIEW_CONTACT, FALSE);
    }
}

void PeopleSpaceApp::openDatabase(bool isStartup)
{
  //fComManager->sendCommand(SOCK_CMD_OPENDB);
}

void PeopleSpaceApp::updateGroupFilterWidgets(int filter)
{
   KCombo* combo = fToolBar1->getCombo(ID_TOOLBAR1_COMBO);

  // We don't know where this came from, so we will disconnect everything and
  // Make sure everything is up to date
  DISCONNECT_CMD(fViewGroupPopMenu);
  disconnect(combo, SIGNAL( activated(int) ), this, SLOT(slotGroupFilterChanged(int) ));

  fViewGroupPopMenu->setItemChecked(ID_VIEW_GROUP_UNFILED, false);
  fViewGroupPopMenu->setItemChecked(ID_VIEW_GROUP_PERSONAL, false); 
  fViewGroupPopMenu->setItemChecked(ID_VIEW_GROUP_BUSINESS, false);
  fViewGroupPopMenu->setItemChecked(ID_VIEW_GROUP_ALL, false);
  fViewGroupPopMenu->setItemChecked(ID_VIEW_GROUP_SEARCHRESULTS, false);

    switch(filter)
      {
      case ID_VIEW_GROUP_UNFILED:
      case PSEntry::groupUnfiled:
	fViewGroupPopMenu->setItemChecked(ID_VIEW_GROUP_UNFILED, true);
	fToolBar1->setCurrentComboItem(ID_TOOLBAR1_COMBO, PSEntry::groupUnfiled);
	break;
	
      case ID_VIEW_GROUP_PERSONAL:
      case PSEntry::groupPersonal:
	fViewGroupPopMenu->setItemChecked(ID_VIEW_GROUP_PERSONAL, true);
	fToolBar1->setCurrentComboItem(ID_TOOLBAR1_COMBO, PSEntry::groupPersonal);
	break;
	
      case ID_VIEW_GROUP_BUSINESS:
      case PSEntry::groupBusiness:
	fViewGroupPopMenu->setItemChecked(ID_VIEW_GROUP_BUSINESS, true);
	fToolBar1->setCurrentComboItem(ID_TOOLBAR1_COMBO, PSEntry::groupBusiness);
	break;
	
      case ID_VIEW_GROUP_ALL:
      case (PSEntry::groupUnfiled+1):
	fViewGroupPopMenu->setItemChecked(ID_VIEW_GROUP_ALL, true);
	fToolBar1->setCurrentComboItem(ID_TOOLBAR1_COMBO, PSEntry::groupUnfiled+1);
	break;

      case ID_VIEW_GROUP_SEARCHRESULTS:
      case ID_SEARCH_RESULTS:
	fViewGroupPopMenu->setItemChecked(ID_VIEW_GROUP_SEARCHRESULTS, true);
	fToolBar1->setCurrentComboItem(ID_TOOLBAR1_COMBO, ID_SEARCH_RESULTS);
	break;
	
    } 

    CONNECT_CMD(fViewGroupPopMenu);
    connect(combo, SIGNAL( activated(int) ), this, SLOT(slotGroupFilterChanged(int) ));
}


///////////////////////////////////////////////////////////////
//////////////////////  SLOTS  ////////////////////////////////
/*
void PeopleSpaceApp::slotDropEvent(KDNDDropZone *zone)
{
	// the user dropped something on our window.
	QString url, temp_file;

	// get the URL pointing to the dropped file
	url = zone->getURLList().first();

	// let KFM grab the file
	if (KFM::download(url, temp_file))
	{
		// 'temp_file' now contains the absolute path to a temp file
		// with the contents of the the dropped file.  You would presumably
		// handle it right now.

		// after you are done handling it, let KFM delete the temp file
		KFM::removeTempFile(temp_file);
	}
}
*/
void PeopleSpaceApp::slotAppExit()
{
  ///////////////////////////////////////////////////////////////////
  // exits the Application
  if(this->queryExit())
    {
      fPSDatabaseManager->closeDB();
      saveOptions();
      KTMainWindow::deleteAll();
      kapp->quit();
    }
  else return;
  ///////////////////////////////////////////////////////////////////
}

void PeopleSpaceApp::slotGroupFilterChanged(int filter)
{
   switch(filter)
      {
      case ID_VIEW_GROUP_UNFILED:
      case PSEntry::groupUnfiled:
	fPSDatabaseManager->applyGroupFilter( PSEntry::groupUnfiled);
	break;
	
      case ID_VIEW_GROUP_PERSONAL:
      case PSEntry::groupPersonal:
	fPSDatabaseManager->applyGroupFilter( PSEntry::groupPersonal); 
	break;
	
      case ID_VIEW_GROUP_BUSINESS:
      case PSEntry::groupBusiness:
	fPSDatabaseManager->applyGroupFilter( PSEntry::groupBusiness );
	break;
	
      case ID_VIEW_GROUP_ALL:
      case (PSEntry::groupUnfiled+1):
	fPSDatabaseManager->applyGroupFilter( PSEntry::groupUnfiled+1 );
	break;

      case ID_VIEW_GROUP_SEARCHRESULTS:
      case ID_SEARCH_RESULTS:
	fPSDatabaseManager->applyGroupFilter( ID_SEARCH_RESULTS );
	break;
	
    } 
}

void PeopleSpaceApp::slotCreateNewEntry()
{
  int newKey;
  
  newKey = fPSDatabaseManager->createNewEntry();

  fPSDatabaseManager->applyGroupFilter( PSEntry::groupUnfiled );

  kapp->getKApplication()->processEvents();

  //cout << "PeopleSpaceApp::slotCreateNewEntry -- NewKey: " << newKey << endl;

  fPSAssociate->changeSelectedEntry(newKey);
}

void PeopleSpaceApp::slotDuplicateEntry()
{
 KCombo *combo = fToolBar1->getCombo(ID_TOOLBAR1_COMBO); 

  fPSDatabaseManager->storeEntry(fPSEntryViewer->currentlyViewedEntry()->duplicateSelf());

  fPSDatabaseManager->applyGroupFilter( combo->currentItem() );
}

void PeopleSpaceApp::slotSaveLoadedEntryList()
{
  KCombo *combo = fToolBar1->getCombo(ID_TOOLBAR1_COMBO); 

  if (fPSDatabaseManager->openDBType() == dbNotOpen)
    return;

  int currentKey = fPSAssociate->selectedEntry();

  fPSEntryViewer->saveCurrentlyViewedEntry();

  fPSDatabaseManager->storeEntryList();
  gUnsavedData = FALSE;
  slotUnsavedDataExists(FALSE);  

  fPSDatabaseManager->applyGroupFilter( combo->currentItem() );
  fPSAssociate->changeSelectedEntry(currentKey);
}

void PeopleSpaceApp::slotDeleteViewedEntry()
{
  fPSEntryViewer->setCurrentlyViewedEntryDeleted();
}

void PeopleSpaceApp::slotDatabaseHasOpened(QString pathWithFile, int type)
{
  KCombo *combo = fToolBar1->getCombo(ID_TOOLBAR1_COMBO); 
  KConfig* config = kapp->getConfig();

  thereIsOpenDB( TRUE );
  fStatusBar->changeItem(pathWithFile, 1);
  switch(type)
    {
    case dbOpenReadWrite:
       fStatusBar->changeItem(i18n(MSG_DB_TYPE_READWRITE), 0);
       break;
    default:
       fStatusBar->changeItem(i18n(MSG_DB_TYPE_READ), 0);
       break;
    }

  config->setGroup("DATABASE");
  config->writeEntry("LastOpened", pathWithFile);
  config->sync();

  // Make sure everything is up to date before we apply the group filter
  kapp->getKApplication()->processEvents();

  fPSDatabaseManager->applyGroupFilter( combo->currentItem() );
}

void PeopleSpaceApp::slotDatabaseHasClosed()
{
  fStatusBar->changeItem(i18n(MSG_NO_DB_OPEN), 1);
  fStatusBar->changeItem(i18n(MSG_DB_TYPE_NOTOPEN), 0);

  fPSEntryViewer->setCurrentlyViewedEntry(NULL);
  thereIsOpenDB( false );
  thereIsSelectedEntry( false );
}

void PeopleSpaceApp::slotUnsavedDataExists(bool state)
{
  if (fPSDatabaseManager->openDBType() != dbOpenReadWrite)
    {
      KMsgBox::message(this, i18n("PeopleSpace Warning"), i18n("You are modifying a Read Only database.\nAny changes you make will not be saved."), KMsgBox::STOP, i18n("Ok"));
      return;
    }

  if (state)
    {
    setCaption(i18n("PeopleSpace (modified)"));
    fToolBar1->setItemEnabled(ID_FILE_SAVE, TRUE);
    fFilePopMenu->setItemEnabled(ID_FILE_SAVE, TRUE);
    }
  
  else
    {
    setCaption(i18n("PeopleSpace"));
    fToolBar1->setItemEnabled(ID_FILE_SAVE, FALSE);
    fFilePopMenu->setItemEnabled(ID_FILE_SAVE, FALSE);
    }
}

void PeopleSpaceApp::slotDirectionButtonClicked(int buttonId)
{
  fPSAssociate->changeSelectedEntry(buttonId);
}

void PeopleSpaceApp::slotGroupFilterApplied(int filter, 
					    QList<QString>*,  
					    QList<QString>*,  
					    QList<QString>*,  
					    QList<QString>*)
{
  updateGroupFilterWidgets(filter);
}

void PeopleSpaceApp::slotConnectionBroken()
{
  cout << "ERROR: Connection loast!" << endl;
}

void PeopleSpaceApp::slotReceiveCommand(int command, QString param)
{
 
}

void PeopleSpaceApp::slotOpenDatabase(bool isStartup)
{
   QString filename;
  int local = 0, openType = 0;
  KConfig* config = kapp->getConfig();
  config->setGroup("STARTUP");

  if (fPSDatabaseManager->openDBType() != dbNotOpen)
    {
      if (checkForUnsavedData())
	fPSDatabaseManager->closeDB();
      else
	return;
    }

  if (isStartup)
    {
      // Check the config for how we should open the DB.
      // 0 = open last used
      // 1 = never auto open
      // 2 = open particular file
      config->setGroup("STARTUP");
      switch(config->readNumEntry("OpenDB", 1))
	{
	case 0:
	  config->setGroup("DATABASE");
	  filename = config->readEntry("LastOpened", "");
	  break;

	case 1:
	  return;
	  
	case 2:
	  config->setGroup("STARTUP");
	  filename = config->readEntry("OpenDBPath", "");
	  break;

	default:
	  ERROR("Couldn't determine autoload DB options");
	  return;
	}
    }

  else
    {
      // Show the file selection dialog
      filename = KFileDialog::getSaveFileName(getenv("HOME"), "*.gdbm", this, "FileDialog");
      
      // make sure the user entered the .db extension
      if ((!filename.isEmpty()) && (filename[filename.length()-1] != '/'))
	{
	  local = filename.findRev('.', filename.length()-1, FALSE);
	  if (local > -1)
	    {
	      if (filename.right(filename.length() - local) != ".gdbm") 
		filename += ".gdbm";
	    }
	  
	  else
	    {
	      filename += ".gdbm";
	    }
	}
      else
	return;
    }
    
  config->setGroup("DATABASE");
  openType = config->readNumEntry("OpenType", 3);
  
  switch (openType)
    {
    case 0:
      fPSDatabaseManager->openDB(filename, dbOpenReadWrite);
      break;
      
    case 1:
      fPSDatabaseManager->openDB(filename, dbOpenRead);
      break;
	  
    case 2:
      if (!fPSDatabaseManager->openDB(filename, dbOpenReadWrite))
	fPSDatabaseManager->openDB(filename, dbOpenRead);
      break;
      
    case 3:
    default:
      DBOpenTypeDialog dbOpenTypeDialog(this, "DBOpenTypeDialog");
      
      if (dbOpenTypeDialog.defaultCheckBoxIsChecked())
	{
	  //cout << "Setting default" << endl;
	  config->setGroup("DATABASE");
	  config->writeEntry("OpenType", dbOpenTypeDialog.selectedOption());
	  config->sync();
	}
      
      fPSDatabaseManager->openDB(filename, (dbOpenType)dbOpenTypeDialog.selectedOption());
      break;
    }
}

void PeopleSpaceApp::slotCloseDatabase()
{
  if (checkForUnsavedData())
    fPSDatabaseManager->closeDB();
}

void PeopleSpaceApp::slotTogglePSAssociateVisible()
{
  if (fPSAssociate->isVisible())
    fPSAssociate->hide();

  else
    fPSAssociate->show();
}

void PeopleSpaceApp::slotSetViewedEntry(int key, bool first, bool last)
{
  //cout << "Setting view to Key: " << key << endl;
  //   << "First is: " << first << endl
  //   << "Last is: " << last << endl;

  if (key == -1)
    {
     thereIsSelectedEntry(false);
     fPSEntryViewer->setCurrentlyViewedEntry(NULL);
    }
  
  else
    {
      
      fPSEntryViewer->setCurrentlyViewedEntry(fPSDatabaseManager->fetchEntry(key));
      
      thereIsSelectedEntry(true);
      
      if (first)
	{  
	  fViewPopMenu->setItemEnabled(ID_VIEW_FIRST, FALSE);
	  fViewPopMenu->setItemEnabled(ID_VIEW_PREV, FALSE);
	  fToolBar1->setItemEnabled(ID_VIEW_FIRST, FALSE);
	  fToolBar1->setItemEnabled(ID_VIEW_PREV, FALSE);
	}
      else
	{
	  fViewPopMenu->setItemEnabled(ID_VIEW_FIRST, true);
	  fViewPopMenu->setItemEnabled(ID_VIEW_PREV, true);
	  fToolBar1->setItemEnabled(ID_VIEW_FIRST, true);
	  fToolBar1->setItemEnabled(ID_VIEW_PREV, true);
	}
      
      
      if (last)
	{
	  fViewPopMenu->setItemEnabled(ID_VIEW_NEXT, false);
	  fViewPopMenu->setItemEnabled(ID_VIEW_LAST, false);
	  fToolBar1->setItemEnabled(ID_VIEW_LAST, false);
	  fToolBar1->setItemEnabled(ID_VIEW_NEXT, false);
	}
      else
	{
	  fViewPopMenu->setItemEnabled(ID_VIEW_NEXT, true);
	  fViewPopMenu->setItemEnabled(ID_VIEW_LAST, true);
	  fToolBar1->setItemEnabled(ID_VIEW_LAST, true);
	  fToolBar1->setItemEnabled(ID_VIEW_NEXT, true);
	}
    }
}

void PeopleSpaceApp::slotDisplayLookupDialog(int keyOffset)
{
  fLookupDialog->setLookupField(keyOffset);
  fLookupDialog->show();
}

void PeopleSpaceApp::slotLookupButtonClicked(int fieldId, QString param1, QString param2)
{
  fPSDatabaseManager->applySearchFilter(fieldId, param1, param2);
}

void PeopleSpaceApp::slotShowConfigDialog()
{
  ConfigDialog* configDialog = new ConfigDialog(this, "ConfigDialog");
  configDialog->exec();
}

/////////////////////////////////////////////////////////////////////////////////
// THESE MACROS CONSTRUCT THE SWITCH FOR THE SIGNAL commandCallback(int)

BEGIN_CMD(PeopleSpaceApp)
  ON_CMD(ID_APP_EXIT,                 slotAppExit())

  ON_CMD(ID_VIEW_GROUP_ALL,           slotGroupFilterChanged(ID_VIEW_GROUP_ALL))
  ON_CMD(ID_VIEW_GROUP_PERSONAL,      slotGroupFilterChanged(ID_VIEW_GROUP_PERSONAL))
  ON_CMD(ID_VIEW_GROUP_BUSINESS,      slotGroupFilterChanged(ID_VIEW_GROUP_BUSINESS))
  ON_CMD(ID_VIEW_GROUP_UNFILED,       slotGroupFilterChanged(ID_VIEW_GROUP_UNFILED))
  ON_CMD(ID_VIEW_GROUP_SEARCHRESULTS, slotGroupFilterChanged(ID_VIEW_GROUP_SEARCHRESULTS))
  ON_CMD(ID_VIEW_LIST,                slotTogglePSAssociateVisible())
  ON_CMD(ID_VIEW_FIRST,               slotDirectionButtonClicked(ID_VIEW_FIRST))
  ON_CMD(ID_VIEW_PREV,                slotDirectionButtonClicked(ID_VIEW_PREV))
  ON_CMD(ID_VIEW_NEXT,                slotDirectionButtonClicked(ID_VIEW_NEXT))
  ON_CMD(ID_VIEW_LAST,                slotDirectionButtonClicked(ID_VIEW_LAST))

  ON_CMD(ID_CONTACT_NEW,              slotCreateNewEntry())
  ON_CMD(ID_CONTACT_DELETE,           slotDeleteViewedEntry())
  ON_CMD(ID_CONTACT_DUPLICATE,        slotDuplicateEntry())

  ON_CMD(ID_FILE_SAVE,                slotSaveLoadedEntryList())
  ON_CMD(ID_FILE_OPEN,                slotOpenDatabase())
  ON_CMD(ID_FILE_CLOSE,               slotCloseDatabase())

  ON_CMD(ID_EDIT_PREFS,               slotShowConfigDialog())
  
  ON_CMD(ID_LOOKUP_FIRSTNAME,         slotDisplayLookupDialog(LookupDialog::combo_firstname))
  ON_CMD(ID_LOOKUP_LASTNAME,          slotDisplayLookupDialog(LookupDialog::combo_lastname))
  ON_CMD(ID_LOOKUP_TITLE,             slotDisplayLookupDialog(LookupDialog::combo_title))
  ON_CMD(ID_LOOKUP_COMPANY,           slotDisplayLookupDialog(LookupDialog::combo_company))
  ON_CMD(ID_LOOKUP_ADDRESS,           slotDisplayLookupDialog(LookupDialog::combo_address))
  ON_CMD(ID_LOOKUP_CITY,              slotDisplayLookupDialog(LookupDialog::combo_city))
  ON_CMD(ID_LOOKUP_STATE,             slotDisplayLookupDialog(LookupDialog::combo_state))
  ON_CMD(ID_LOOKUP_ZIP,               slotDisplayLookupDialog(LookupDialog::combo_zip))
  ON_CMD(ID_LOOKUP_COUNTRY,           slotDisplayLookupDialog(LookupDialog::combo_country))
  ON_CMD(ID_LOOKUP_PHONE,             slotDisplayLookupDialog(LookupDialog::combo_phone))
  ON_CMD(ID_LOOKUP_CUSTOM,            slotDisplayLookupDialog(LookupDialog::combo_custom))
  ON_CMD(ID_LOOKUP_REFERREDBY,        slotDisplayLookupDialog(LookupDialog::combo_referredby))
  ON_CMD(ID_LOOKUP_EDITDATE,          slotDisplayLookupDialog(LookupDialog::combo_editdate))
  ON_CMD(ID_LOOKUP_MERGEDATE,         slotDisplayLookupDialog(LookupDialog::combo_mergedate))
  ON_CMD(ID_LOOKUP_LASTREACH,         slotDisplayLookupDialog(LookupDialog::combo_lastreach))
  ON_CMD(ID_LOOKUP_LASTATTEMPT,       slotDisplayLookupDialog(LookupDialog::combo_lastattempt))
  ON_CMD(ID_LOOKUP_LASTMEETING,       slotDisplayLookupDialog(LookupDialog::combo_lastmeeting))
  ON_CMD(ID_LOOKUP_RECORDCREATOR,     slotDisplayLookupDialog(LookupDialog::combo_recordcreator))
  ON_CMD(ID_LOOKUP_CREATEDATE,        slotDisplayLookupDialog(LookupDialog::combo_createdate))

  default:
     ERROR_FORCE_DISPLAY("Item not implemented yet");
     break;

END_CMD()

#include "peoplespace.moc"
