// ----------------------------------------------------------------------------
// Filename:    KTimeclockListItem.h
// $Date: 1998/11/16 07:30:07 $
// $Revision: 1.3 $
// ----------------------------------------------------------------------------

#ifndef KTimeclockListItem_included
#define KTimeclockListItem_included

#include <qlistview.h>
#include <kapp.h>

/**
 * This is the KTimeclockListItem class.  It is used for displaying items in a
 * @ref QListView.  I've derived from the @ref QListViewItem class, and simply
 * added in the extra functionality which was required (primarily just
 * formatting of text strings, and some constants for defining columns).
 *
 * @short   KTimeclockListItem class
 * @author  Graham TerMarsch (gtermars@home.com)
 * @version $Revision: 1.3 $
 * @see     QListViewItem
 */
class KTimeclockListItem : public QListViewItem
{
public:
    /**
     * Constant stating that the item is a Project or a Sub-project.  Should be
     * used when setting the type of item that this is.
     */
    static QString ITEM_PROJECT;

    /**
     * Constant stating that the item is a task.  Should be used when setting
     * the type of item that this is.
     */
    static QString ITEM_TASK;

    /**
     * Constructor. Passes arguments through to our base class, and sets up our
     * own custom data as well as look and feel.
     *
     * @param   parent      Handle to our parent widget (the tree view)
     * @param   type        Type of item (ITEM_PROJECT or ITEM_TASK)
     * @param   desc        Description of the item (name of task/project)
     * @param   totalsecs   Total seconds spent on this item so far
     */
    KTimeclockListItem ( QListView* parent, const char* type, const char* desc, int totalsecs )
        : QListViewItem( parent )
    {
        this->setDefaults( type, desc, totalsecs );
    }

    /**
     * Constructor. Passes arguments through to our base class, and sets up our
     * own custom data as well as look and feel.
     *
     * @param   parent      Handle to our parent widget (parent project)
     * @param   type        Type of item (ITEM_PROJECT or ITEM_TASK)
     * @param   desc        Description of the item (name of task/project)
     * @param   totalsecs   Total seconds spent on this item so far
     */
    KTimeclockListItem ( KTimeclockListItem* parent, const char* type, const char* desc, int totalsecs )
        : QListViewItem( parent )
    {
        this->setDefaults( type, desc, totalsecs );
    }

    /**
     * Sets the description for the list item.
     *
     * @param   desc    Description to set for the item.
     */
    void description (const char* desc);

    /**
     * Gets the description for the list item and returns it.
     *
     * @return  Description of the item.
     */
    const char* description ();

    /**
     * Sets the total time spent (in seconds)
     *
     * @param   seconds Total time spent on the item, in seconds.
     */
    void totalTime (int seconds);

    /**
     * Returns the total time spent (in seconds)
     *
     * @return  Total time spent on the item, in seconds.
     */
    int totalTime ();

    /**
     * Sets the time spent in this session (in seconds)
     *
     * @param   seconds Time spent on this item in this session (in seconds).
     */
    void sessionTime (int seconds);

    /**
     * Gets the time spent during this session (in seconds)
     *
     * @return  Time spent on this item in this session (in seconds).
     */
    int sessionTime ();

    /**
     * Sets the type for this list item.  The value provided should be one of
     * the defined constants; ITEM_PROJECT, ITEM_TASK.
     *
     * @param   type    Type of item we are.
     */
    void type (QString);

    /**
     * Gets the type for this list item.  The value returned will be the value
     * as set by type(QString).
     *
     * @return  Type of item we are.
     */
    QString type ();

    /**
     * Overridden repaint method.
     * This version of repaint will first take the values for the number of
     * seconds spent in this session and in total and update the text columns
     * for the item.  We'll then call the base class to do the repainting once
     * we've got this set up.
     */
    void repaint ();

private:
    /**
     * Enumerated constants outlining what the columns will be.  When accessing
     * data from a KTimeclockListItem, use this as the column number.
     *
     * e.g.
     * <pre>KTimeclockListItem* item = new KTimeclockListItem;
     * item->setText( COL_DESCRIPTION, "Description" )</pre>
     */
    enum COLUMN { COL_DESCRIPTION=0, COL_TEXTSESSION, COL_TEXTTOTAL, COL_TYPE,
                  COL_SESSION, COL_TOTAL
                };

    /**
     * Converts a given uint to a string in the format of 'hh:mm:ss'.  Inlined
     * method (so we don't have another .cpp file kicking around somewhere).
     *
     * @param   timestamp   time to convert to formatted text string
     * @return  QString containing formatted time value
     */
    QString timeString (int timestamp);

    /**
     * Sets default values for our columns and provides us a pixmap depending
     * as to what type of item we are.  Internal method only.
     */
    void setDefaults ( const char* type, const char* desc, int totalsecs );
};

#endif // KTimeclockListItem_included
