/***************************************************************************
                          knd_load.cpp  -  description                              
                             -------------------                                         
    begin                : Thu Mar 25 14:26:46 GMT 1999
                                           
    copyright            : (C) 1999 by Mike Richardson                         
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#include	"knd_load.h"


/*  checkBounds	: Check value against bounds				*/
/*  value	: int		: Value to be checked			*/
/*  upper	: int &		: Current/modified upper bound		*/
/*  lower	: int &		: Current/modified lower bound		*/
/*  minval	: int		: Minimum scale				*/
/*  (returns)	: bool		: TRUE if bounds changed.		*/

bool	checkBounds
	(	int	value,
		int	&upper,
		int	&lower,
		int	minval
	)
{
	int	p10	;

	assert	((value >= 0) && (value < 2000000)) ;
	assert	((upper >= 0) && (upper < 2000000)) ; 
	assert	((lower >= 0) && (lower < 2000000)) ;

	/* See if the value has moved outside the current scale range,	*/
	/* in which case readjust the scale appropriately, keeping to	*/
	/* 1..2..5..10.. intervals.					*/
	if ( ( value  > upper) ||
	     ((value <= lower) && (upper > minval)) )
	{
		for (upper = p10 = minval ;; p10 *= 10)
		{
			lower	= upper   ;
			upper  = p10	   ;
			if (upper > value) break ;

			lower	= upper   ;
			upper	= p10 * 2  ;
			if (upper > value) break ;

			lower	= upper   ;
			upper	= p10 * 5  ;
			if (upper > value) break ;
		}

		return	true	;
	}

	return	false	;
}

/*  KNDLoadHist								*/
/*  KNDLoadHist	: Constructor for load histogram ovbject		*/
/*  parent	: QWidget *	: Parent				*/
/*  _x, _y	: int, int	: Position relative to parent		*/
/*  _w, _h	: int, int	: Size					*/
/*  _adjust	: bool		: Automatically adjust scale		*/
/*  (returns)	: KNDLoadHist	:					*/

KNDLoadHist::KNDLoadHist
	(	QWidget	*_parent,
		int	_x,
		int	_y,
		int	_w,
		int	_h,
		int	_nsecs,
		bool	_adjust
	)
	:
	QFrame	(_parent),
	x	(_x),
	y	(_y),
	w	(_w),
	h	(_h),
	nsecs	(_nsecs),
	adjust	(_adjust)
{
	setLineWidth      (1) ;
	setFrameStyle     (QFrame::Box|QFrame::Plain) ;
	setBackgroundMode (NoBackground) ;
	setGeometry       (x, y, w, h) ;

	w      -= 1	   ;
	h      -= 1	   ;
	offsetp	= 0	   ;
	offsetd	= 0	   ;
	insert	= w - 1	   ;
	scpref	= MINLOAD  ;
	sccurr	= MINLOAD  ;
	scdown	= MINLOAD  ;
	maxval	= MINLOAD  ;
	tickno	= _nsecs   ;
	frozen	= false	   ;
	pktcnt	= new LoadPair[w] ;
	dispval	= new LoadPair[w] ;

	for (int slot = 0 ; slot < w ; slot += 1)
	{	pktcnt [slot].pktlen = 0 ;
		pktcnt [slot].datlen = 0 ;
		dispval[slot].pktlen = h ;
		dispval[slot].datlen = h ;
	}

	repaint	() ;
}

/*  KNDLoadHist								*/
/*  ~KNDLoadHist: Destructor for load histogram object			*/
/*  (returns)	:		:					*/

KNDLoadHist::~KNDLoadHist ()
{
	if (pktcnt  != NULL) delete pktcnt  ;
	if (dispval != NULL) delete dispval ;
}

/*  KNDLoadHist								*/
/*  addPacket	: Add latest packet information				*/
/*  pktinfo	: PktInfo *	  : Packet information			*/
/*  (returns)	: void		  :					*/

void	KNDLoadHist::addPacket
	(	PktInfo	*pktinfo
	)
{
	int	slot	;

	if (pktinfo->ago < w)
	{
		if ((slot = insert + w - pktinfo->ago) >= w) slot -= w ;
		assert ((slot >= 0) && (slot < w)) ;

		/* Add in the packet and data lengths. At this stage,	*/
		/* the upper and lower display bounds remain unchanged	*/
		/* as they are set on the next timer tick when not	*/
		/* frozen						*/
		pktcnt[slot].pktlen += pktinfo->pktlen ;
		pktcnt[slot].datlen += pktinfo->datlen ;
	}
}

/*  KNDLoadHist								*/
/*  reScale	: Rescale display bounds				*/
/*  (returns)	: void		:					*/

void	KNDLoadHist::reScale ()
{
	for (int slot = 0 ; slot < w ; slot += 1)
	{	dispval[slot].pktlen = boundary(pktcnt[slot].pktlen) ;
		dispval[slot].datlen = boundary(pktcnt[slot].datlen) ;
	}

	char	tip[32]	;
	sprintf	      (tip, "Scale = %d bytes/sec", sccurr) ;
	QToolTip::add (this, tip) ;
}

/*  KNDLoadHist								*/
/*  timerTick	: Handle timer tick					*/
/*  _second	: long		: Second				*/
/*  (returns)	: int		: Current scale				*/

int	KNDLoadHist::timerTick
	(	long	_second
	)
{
	int	slot	  ;

	second	= _second ;

	/* Decrement the tick count. If it falls to zero then reset it	*/
	/* and process the tick, otherwise return. This gives us an	*/
	/* n-seconds time constant.					*/
	if ((tickno -= 1) > 0) return scpref ;
	tickno = nsecs ;

	/* Scan the loding table to find the maximum byte count. We	*/
	/* could do better if we kept track of it ...			*/
	for (slot = 0, maxval = -1 ; slot < w ; slot += 1)
	{	assert (pktcnt[slot].pktlen < 2000000) ;
		if (pktcnt[slot].pktlen > maxval)
			maxval	= pktcnt[slot].pktlen ;
	}

	/* The next chunk of code relates to adjusting the upper and	*/
	/* lower display points. If we are frozen then these are not	*/
	/* altered, so skip it.						*/
	if (!frozen)
	{
		/* See if the maximum value has moved outside the	*/
		/* current scale range, in which case readjust the	*/
		/* scale and then recalculate all display bounds.	*/
		if (checkBounds (maxval, scpref, scdown, MINLOAD))
		{
			if (adjust)
			{	sccurr	= scpref ;
				reScale ()	 ;
			}
		}
		else
		{	/* No change in scale so calculate the display	*/
			/* bounds for the timer interval that has just	*/
			/* completed.					*/
			if ((slot = offsetp + w - 1) >= w) slot -= w ;
			dispval[slot].pktlen = boundary (pktcnt[slot].pktlen) ;
			dispval[slot].datlen = boundary (pktcnt[slot].datlen) ;
		}

		dispval[offsetd].pktlen  = h ;
		dispval[offsetd].datlen  = h ;
	}

	/* The loading slot which is currently the first displayed will	*/
	/* be reused, so reset the values, then advance the offsets by	*/
	/* one. If we are not frozen then the display offset follows	*/
	/* this.							*/
	pktcnt[offsetp].pktlen = 0 ;
	pktcnt[offsetp].datlen = 0 ;

	if ((offsetp += 1) >= w) offsetp = 0 ;
	if ((insert  += 1) >= w) insert  = 0 ;
	if (!frozen) offsetd = offsetp ;

	return	scpref	;
}

/*  KNDLoadHist								*/
/*  setScale	: Specifically set the scale				*/
/*  scale	: int		: New scale value			*/
/*  (returns)	: void		:					*/

void	KNDLoadHist::setScale
	(	int	scale
	)
{
	if (sccurr != scale)
	{	sccurr	= scale ;
		reScale	()	;
	}
}

/*  KNDLoadHist								*/
/*  drawContents: Handle redrawing display				*/
/*  painter	: QPainter *	: Paint context				*/
/*  (returns)	: void		:					*/

void	KNDLoadHist::drawContents
	(	QPainter	*painter
	)
{
	int	posn	;
	int	slot	;

	/* The painting is done in three passes of the loading array,	*/
	/* to paint the upper gray area, the middle red (network	*/
	/* traffic overhead) area, and the lower blue (network traffic	*/
	/* data) area. Doing it this way means that we only need to set	*/
	/* the pen three times, rather than (width*3) times.		*/
	posn	= 0	 	;
	slot	= offsetd	;
	painter->setPen (gray)	;

	while (posn < w)
	{
		painter->drawLine (posn, 0,
				   posn, dispval[slot].pktlen) ;
		posn += 1 ;
		if ((slot += 1) >= w) slot = 0 ;
	}


	posn	= 0		;
	slot	= offsetd	;
	painter->setPen (QColor (255, 0, 0)) ;

	while (posn < w)
	{
		painter->drawLine (posn, dispval[slot].pktlen,
				   posn, dispval[slot].datlen) ;
		posn += 1 ;
		if ((slot += 1) >= w) slot = 0 ;
	}

	posn	= 0		;
	slot	= offsetd	;
	painter->setPen (QColor (0, 0, 255)) ;

	while (posn < w)
	{
		painter->drawLine (posn, dispval[slot].datlen,
				   posn, h) ;
		posn += 1 ;
		if ((slot += 1) >= w) slot = 0 ;
	}
}

/*  KNDLoadHost								*/
/*  mouseDoubleClickEvent	: Handle mouse double click		*/
/*  e		: QMouseEvent *	: Mouse event				*/
/*  (returns)	:		:					*/

void	KNDLoadHist::mouseDoubleClickEvent
	(	QMouseEvent	*e
	)
{
	emit doubleClick () ;
}

/*  KNDLoadHist								*/
/*  freeze	: Set frozen or unfrozen				*/
/*  _frozen	: bool		: New state				*/
/*  (returns)	: void		:					*/

void	KNDLoadHist::freeze
	(	bool	_frozen
	)
{
	/* If we are becoming unfrozen then we need to calculate	*/
	/* display bounds for all samples, since those that arrived	*/
	/* while we were frozen will not have been processed. This is	*/
	/* true whether the scale changes or not.			*/
	if (!(frozen = _frozen))
	{
		checkBounds (maxval, scpref, scdown, MINLOAD) ;

		if (adjust)
		{	sccurr	= scpref ;
			reScale	() ;
		}

		repaint	    () ;
	}
}



/*  KNDLoadBar								*/
/*  KNDLoadBar	: Constructor for load bar object			*/
/*  parent	: QWidget *	: Parent				*/
/*  _x, _y	: int, int	: Position relative to parent		*/
/*  _w, _h	: int, int	: Size					*/
/*  _nsec	: int		: Time constant				*/
/*  (returns)	: KNDLoadHist	:					*/


KNDLoadBar::KNDLoadBar
	(	QWidget	*_parent,
		int	_x,
		int	_y,
		int	_w,
		int	_h,
		int	_nsecs
	)
	:
	QFrame	(_parent),
	nsecs	(_nsecs),
	x	(_x),
	y	(_y),
	w	(_w),
	h	(_h)
{
	setLineWidth      (1) ;
	setFrameStyle     (QFrame::Box|QFrame::Plain) ;
	setBackgroundMode (NoBackground) ;
	setGeometry       (x, y, w, h) ;

	w      -= 1	  ;
	h      -= 1	  ;
	offset	= 0	  ;
	insert	= nsecs-1 ;
	scale	= MINLOAD ;
	frozen	= false   ;

	pktcnt	= new LoadPair[nsecs] ;
	memset	(pktcnt, 0, nsecs * sizeof(LoadPair)) ;
	dispval.pktlen = 0 ;
	dispval.datlen = 0 ;
	repaint	()  ;
}

/*  KNDLoadBar								*/
/*  ~KNDLoadBar	: Destructor for load bar object			*/
/*  (returns)	:		:					*/

KNDLoadBar::~KNDLoadBar ()
{
	if (pktcnt != NULL) delete pktcnt ;
}

/*  KNDLoadBar								*/
/*  addPacket	: Add latest packet information				*/
/*  pktinfo	: PktDta *	  : Packet data				*/
/*  (returns)	: void		  :					*/

void	KNDLoadBar::addPacket
	(	PktInfo	*pktinfo
	)
{
	int	slot	;

	if (pktinfo->ago < nsecs)
	{
		if ((slot = insert + nsecs - pktinfo->ago) >= nsecs)
			slot -= nsecs ;
		assert ((slot >= 0) && (slot < nsecs)) ;

		/* Add in the packet and data lengths. The entire	*/
		/* array will be summed when displayed.			*/
		pktcnt[insert].pktlen += pktinfo->pktlen ;
		pktcnt[insert].datlen += pktinfo->datlen ;
	}
}

/*  KNDLoadBar								*/
/*  reScale	: Rescale display bounds				*/
/*  (returns)	: void		:					*/

void	KNDLoadBar::reScale ()
{
	int	slot	;
	int	pktlen	;
	int	datlen	;

	/* Sum the packet and data lengths, and calculate the upper and	*/
	/* lower bar boundaries. We do this here rather than when the	*/
	/* bar is redrawn, as typically "timerTick" is called before	*/
	/* "drawContents", are we would otherwise loose one sample.	*/
	for (slot = pktlen = datlen = 0 ; slot < nsecs ; slot += 1)
	{	pktlen	+= pktcnt[slot].pktlen ;
		datlen	+= pktcnt[slot].datlen ;
	}

	dispval.pktlen	 = boundary (pktlen / nsecs) ;
	dispval.datlen	 = boundary (datlen / nsecs) ;

	char	tip[32]	;
	sprintf	      (tip, "Scale = %d bytes/sec", scale) ;
	QToolTip::add (this, tip) ;
}

/*  KNDLoadBar								*/
/*  timerTick	: Handle timer tick					*/
/*  _second	: long		: Second				*/
/*  (returns)	: void		:					*/

void	KNDLoadBar::timerTick
	(	long	_second
	)
{
	second	= _second ;

	if (!frozen) reScale () ;

	/* The loading slot which is currently the earliest will be	*/
	/* reused, so reset the values, then advance the offset by one.	*/
	pktcnt[offset].pktlen = 0 ;
	pktcnt[offset].datlen = 0 ;

	if ((offset += 1) >= nsecs) offset = 0 ;
	if ((insert += 1) >= nsecs) insert = 0 ;
}

/*  KNDLoadBar								*/
/*  drawContents: Handle redrawing display				*/
/*  painter	: QPainter *	: Paint context				*/
/*  (returns)	: void		:					*/

void	KNDLoadBar::drawContents
	(	QPainter	*painter
	)
{
	painter->fillRect (0,     	   0, w, dispval.pktlen, gray) ;
	painter->fillRect (0, dispval.pktlen, w, dispval.datlen, red ) ;
	painter->fillRect (0, dispval.datlen, w,     	      h, blue) ;
}

/*  KNDLoadBar								*/
/*  mouseDoubleClickEvent	: Handle mouse double click		*/
/*  e		: QMouseEvent *	: Mouse event				*/
/*  (returns)	:		:					*/

void	KNDLoadBar::mouseDoubleClickEvent
	(	QMouseEvent	*e
	)
{
	emit doubleClick () ;
}

/*  KNDLoadBar								*/
/*  freeze	: Set frozen or unfrozen				*/
/*  _frozen	: bool		: New state				*/
/*  (returns)	: void		:					*/

void	KNDLoadBar::freeze
	(	bool	_frozen
	)
{
	if (!(frozen = _frozen))
	{	reScale () ;
		repaint () ;
	}
}



/*  KNDPktHist								*/
/*  KNDPktHist	: Constructor for packet histogram ovbject		*/
/*  parent	: QWidget *	: Parent				*/
/*  _x, _y	: int, int	: Position relative to parent		*/
/*  _w, _h	: int, int	: Size					*/
/*  _adjust	: bool		: Automatically adjust scale		*/
/*  (returns)	: KNDPktHist	:					*/

KNDPktHist::KNDPktHist
	(	QWidget	*_parent,
		int	_x,
		int	_y,
		int	_w,
		int	_h,
		int	_nsecs,
		bool	_adjust
	)
	:
	QFrame	(_parent),
	x	(_x),
	y	(_y),
	w	(_w),
	h	(_h),
	nsecs	(_nsecs),
	adjust	(_adjust)
{
	setLineWidth      (1) ;
	setFrameStyle     (QFrame::Box|QFrame::Plain) ;
	setBackgroundMode (NoBackground) ;
	setGeometry       (x, y, w, h) ;

	w      -= 1	   ;
	h      -= 1	   ;
	offsetp	= 0	   ;
	offsetd	= 0	   ;
	insert	= w - 1	   ;
	scpref	= MINPCNT  ;
	sccurr	= MINPCNT  ;
	scdown	= MINPCNT  ;
	maxval	= MINPCNT  ;
	tickno	= _nsecs   ;
	frozen	= false	   ;
	pktcnt	= new int[w] ;
	dispval	= new int[w] ;

	for (int slot = 0 ; slot < w ; slot += 1)
	{	pktcnt [slot]	= 0 ;
		dispval[slot]	= h ;
	}

	repaint	() ;
}

/*  KNDPktHist								*/
/*  ~KNDPktHist: Destructor for load histogram object			*/
/*  (returns)	:		:					*/

KNDPktHist::~KNDPktHist ()
{
	if (pktcnt  != NULL) delete pktcnt  ;
	if (dispval != NULL) delete dispval ;
}

/*  KNDPktHist								*/
/*  addPacket	: Add latest packet information				*/
/*  pktinfo	: PktInfo *	  : Packet information			*/
/*  (returns)	: void		  :					*/

void	KNDPktHist::addPacket
	(	PktInfo		*pktinfo
	)
{
	int	slot	;

	if (pktinfo->ago < w)
	{
		if ((slot = insert + w - pktinfo->ago) >= w) slot -= w ;
		assert ((slot >= 0) && (slot < w)) ;

		/* Increment. At this stage, the upper and lower	*/
		/* display bounds remain unchanged - they are set on	*/
		/* the next timer tick when not frozen			*/
		pktcnt[slot] += 1 ;
	}
}

/*  KNDPktHist								*/
/*  reScale	: Rescale display bounds				*/
/*  (returns)	: void		:					*/

void	KNDPktHist::reScale ()
{
	for (int slot = 0 ; slot < w ; slot += 1)
		dispval[slot] = boundary(pktcnt[slot]) ;

	char	tip[32]	;
	sprintf	      (tip, "Scale = %d packets/sec", sccurr) ;
	QToolTip::add (this, tip) ;
}

/*  KNDPktHist								*/
/*  timerTick	: Handle timer tick					*/
/*  _second	: long		: Second				*/
/*  (returns)	: int		: Current scale				*/

int	KNDPktHist::timerTick
	(	long	_second
	)
{
	int	slot	  ;

	second	= _second ;

	/* Decrement the tick count. If it falls to zero then reset it	*/
	/* and process the tick, otherwise return. This gives us an	*/
	/* n-seconds time constant.					*/
	if ((tickno -= 1) > 0) return scpref ;
	tickno = nsecs ;

	/* Scan the loding table to find the maximum byte count. We	*/
	/* could do better if we kept track of it ...			*/
	for (slot = 0, maxval = -1 ; slot < w ; slot += 1)
		if (pktcnt[slot] > maxval)
			maxval	= pktcnt[slot] ;

	/* The next chunk of code relates to adjusting the display	*/
	/* value. If we are frozen then these are not altered, so skip	*/
	/* it.								*/
	if (!frozen)
	{
		/* See if the maximum value has moved outside the	*/
		/* current scale range, in which case readjust the	*/
		/* scale and then recalculate all display bounds.	*/
		if (checkBounds (maxval, scpref, scdown, MINPCNT))
		{
			if (adjust)
			{	sccurr	= scpref ;
				reScale ()	 ;
			}
		}
		else
		{	/* No change in scale so calculate the display	*/
			/* bounds for the timer interval that has just	*/
			/* completed.					*/
			if ((slot = offsetp + w - 1) >= w) slot -= w ;
			dispval[slot] = boundary (pktcnt[slot]) ;
		}

		dispval[offsetd]  = h ;
	}

	/* The loading slot which is currently the first displayed will	*/
	/* be reused, so reset the values, then advance the offset by	*/
	/* one. If we are not frozen then the display offset follows	*/
	/* this.							*/
	pktcnt[offsetp] = 0 ;

	if ((offsetp += 1) >= w) offsetp = 0 ;
	if ((insert  += 1) >= w) insert  = 0 ;
	if (!frozen) offsetd = offsetp ;

	return	scpref	;
}

/*  KNDPktHist								*/
/*  setScale	: Specifically set the scale				*/
/*  scale	: int		: New scale value			*/
/*  (returns)	: void		:					*/

void	KNDPktHist::setScale
	(	int	scale
	)
{
	if (sccurr != scale)
	{	sccurr	= scale ;
		reScale	()	;
	}
}

/*  KNDPktHist								*/
/*  drawContents: Handle redrawing display				*/
/*  painter	: QPainter *	: Paint context				*/
/*  (returns)	: void		:					*/

void	KNDPktHist::drawContents
	(	QPainter	*painter
	)
{
	int	posn	;
	int	slot	;

	/* The painting is done in two passes of the loading array, to	*/
	/* paint the upper gray area, and the lower blue (network	*/
	/* traffic data) area. Doing it this way means that we only	*/
	/* need to set the pen twice, rather than (width2) times.	*/
	posn	= 0	 	;
	slot	= offsetd	;
	painter->setPen (gray)	;

	while (posn < w)
	{
		painter->drawLine (posn, 0, posn, dispval[slot]) ;
		posn += 1 ;
		if ((slot += 1) >= w) slot = 0 ;
	}


	posn	= 0		;
	slot	= offsetd	;
	painter->setPen (QColor (0, 0, 255)) ;

	while (posn < w)
	{
		painter->drawLine (posn, dispval[slot], posn, h) ;
		posn += 1 ;
		if ((slot += 1) >= w) slot = 0 ;
	}
}

/*  KNDLoadHost								*/
/*  mouseDoubleClickEvent	: Handle mouse double click		*/
/*  e		: QMouseEvent *	: Mouse event				*/
/*  (returns)	:		:					*/

void	KNDPktHist::mouseDoubleClickEvent
	(	QMouseEvent	*e
	)
{
	emit doubleClick () ;
}

/*  KNDPktHist								*/
/*  freeze	: Set frozen or unfrozen				*/
/*  _frozen	: bool		: New state				*/
/*  (returns)	: void		:					*/

void	KNDPktHist::freeze
	(	bool	_frozen
	)
{
	/* If we are becoming unfrozen then we need to calculate	*/
	/* display bounds for all samples, since those that arrived	*/
	/* while we were frozen will not have been processed. This is	*/
	/* true whether the scale changes or not.			*/
	if (!(frozen = _frozen))
	{
		checkBounds (maxval, scpref, scdown, MINPCNT) ;

		if (adjust)
		{	sccurr	= scpref ;
			reScale	() ;
		}

		repaint	    () ;
	}
}



