/***************************************************************************
                          knd_confrep.cpp  -  description                              
                             -------------------                                         
    begin                : Thu Mar 25 14:26:46 GMT 1999
                                           
    copyright            : (C) 1999 by Mike Richardson                         
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include	<sys/types.h>
#include	<sys/stat.h>
#include	<fcntl.h>
#include	<unistd.h>

#include	"knd_confrep.h"


static	void	AddTime
	(	TVAL	&tval,
		TVAL	&incr
	)
{
	tval.tv_sec	+= incr.tv_sec	;
	tval.tv_usec	+= incr.tv_usec	;

	if (tval.tv_usec >= 1000000)
	{	tval.tv_sec  += 1	;
		tval.tv_usec -= 1000000 ;
	}
}

/*  KNDSpeedBox								*/
/*  KNDSpeedBox	: Replay speed spin control constructor			*/
/*  minVal	: int		: Minimum value				*/
/*  maxVal	: int		: Maximum value				*/
/*  iniVal	: int		: Initial value				*/
/*  (returns)	: KNDSpeedBox	:					*/

KNDSpeedBox::KNDSpeedBox
	(	QWidget	*parent,
		int	minVal,
		int	maxVal,
		int	iniVal
	)
	:
	QSpinBox (parent)
{
	setRange (minVal, maxVal) ;
	setValue (speed = iniVal) ;

	connect	 (this, SIGNAL(valueChanged(int)), this, SLOT(speedChanged(int))) ;
}

/*  KNDSpeedBox								*/
/*  speedChanged: Handle change on speed box				*/
/*  newval	: int		: New value				*/
/*  (returns)	: void		:					*/

void	KNDSpeedBox::speedChanged
	(	int	newval
	)
{
	/* The purpose of this code is to arrange that we skip zero and	*/
	/* minus one, ie., going down from 1 goes to -2, and up from -2	*/
	/* goes to one. The code is arranged so that we don't fall foul	*/
	/* of nested "valueChanged" signals.				*/
	if (newval != speed)
		if ((newval == 0) || (newval == -1))
			if (newval > speed)
				setValue (speed =  1) ;
			else	setValue (speed = -2) ;
}

/*  KNDSpeedBox								*/
/*  mapValueToText: Map spin value to display text			*/
/*  value	  : int		: Current value				*/
/*  (returns)	  : QString	: Display text				*/

QString	KNDSpeedBox::mapValueToText
	(	int	value
	)
{
	QString	res	;

	if	(value == 1)
		res.sprintf ("Original speed") ;
	else if (value >  0)
		res.sprintf ("%d times speed-up",   value) ;
	else	res.sprintf ("%d times slow-down", -value) ;

	return	res	;
}

/*  KNDConfRep								*/
/*  KNDConfRep	: Constructor for configuration and control object	*/
/*  view	: KNDView *	: Parent object				*/
/*  graphic	: KNDGraphic *	: Associated graphic display object	*/
/*  packets	: KNDPacket *	: Associated packet display object	*/
/*  multi	: KNDMulti *	: Multiple graph object			*/
/*  slot	: int		: Unique slot number			*/
/*  name	: char *	: Initial monitor name			*/
/*  (returns)	: KNDConfRep	:					*/

KNDConfRep::KNDConfRep
	(	KNDView		*view,
		KNDGraphic	*graphic,
		KNDPacket	*packets,
		KNDMulti	*multi,
		int		slot,
		char		*name
	)
	:
	KNDConf	(view, graphic, packets, multi, slot, name),
	t_logb	(this),
	l_if	(this),
	t_if	(this),
	t_prog	(this),
	s_speed	(this, -50, 50, 1)
{
	l_if   .setGeometry	  ( 10,  55,  60,  25) ;
	l_if   .setText		  ("Interface") ;
	t_if   .setGeometry	  ( 80,  55, 100,  25) ;
	t_if   .setFrameStyle	  (QFrame::Panel|QFrame::Sunken) ;
	t_if   .setLineWidth	  (2) ;
	t_if   .setBackgroundMode (PaletteBase) ;

	QToolTip::add (&t_if,   i18n ("Network interface monitored")) ;

	QToolTip::add (&t_name, i18n ("Replay name")) ;
	QToolTip::add (&b_go,   i18n ("Start/stop replay")) ;

	t_logb .setGeometry	  (515,  85, 260,  25) ;
	t_logb .setText		  (v_logb = v_name + ".kndb") ;
	QToolTip::add (&t_logb, i18n ("Name of binary logging file")) ;

	s_speed.setGeometry	  (610,  55, 165,  25) ;

	t_prog .setGeometry	  ( 10, 200, 400,  25) ;
	t_prog .setFrameStyle	  (QFrame::Panel|QFrame::Sunken) ;
	t_prog .setLineWidth	  (2) ;
	t_prog .setBackgroundMode (PaletteBase) ;

	b_res  .setEnabled	  (false) ;
	b_res  .hide		  () ;

	logfd	= -1	;
	going	= false	;
	frozen	= false	;
}

/*  KNDConfRep								*/
/*  ~KNDConfRep	: Destructor for configuration object			*/
/*  (returns)	: void		:					*/

KNDConfRep::~KNDConfRep ()
{
	if (logfd >= 0) ::close (logfd) ;
}

/*  KNDConfRep								*/
/*  clickSet	: Handle a click of the set button			*/
/*  (returns)	: void		:					*/

void	KNDConfRep::clickSet ()
{
	int	len		;
	char	buff	[512]	;
	int	what		;
	bool	magic		= false ;
	bool	error		= false ;

	d_multi	= c_multi.isChecked () ;
	v_speed	= s_speed.value () ;
	v_logb	= t_logb .text  () ;

	d_time	= c_time .isChecked  () ;
	d_proto	= c_proto.isChecked  () ;
	d_srce	= c_srce .isChecked  () ;
	d_dest	= c_dest .isChecked  () ;
	d_info	= c_info .isChecked  () ;
	d_size	= c_size .isChecked  () ;
	d_multi = c_multi.isChecked  () ;
	v_load1	= s_load1.value	     () ;
	v_bar1 	= s_bar1 .value	     () ;
	v_bar2 	= s_bar2 .value	     () ;

	/* Attempt to open the specified log file, reporting an error	*/
	/* if it does not exist or cannot be opened.			*/
	if ((logfd = open (v_logb, O_RDONLY)) < 0)
	{
		Error	("Ksnuffle file error", strerror(errno)) ;
		return	;
	}

	/* Read the header section, which should give us the inteface	*/
	/* name and the filter program.					*/
	for (what = -1 ; (what != (int)ItemLast) && !error ; )
	{
		/* Read the next item marker. Check for an error,	*/
		/* otherwise see if it is the end-of-header marker.	*/
		if (read (logfd, &what, sizeof(int)) != sizeof(int))
		{	error	= true ;
			break	;
		}

		if ((LogItem)what == ItemLast) break ;

		/* Read in the item length followed by the item value,	*/
		/* again checking for errors. We set an arbitrary limit	*/
		/* on the item length.					*/
		if ( (read (logfd, &len,  sizeof(int)) != sizeof(int )) ||
		     ((unsigned int)len >= 		  sizeof(buff)) ||
		     (read (logfd, buff,  len) 	       != len	      ) )
		{
			error	= true ;
			break	;
		}

		switch ((LogItem)what)
		{
			case ItemMagic	:
				/* Magic marker. Either flag an error	*/
				/* if it is wrong, or flag that we have	*/
				/* found it if it is OK.		*/
				if (strcmp (buff, MAGIC) != 0)
					error	= true ;
				else	magic	= true ;
				break	;

			case ItemIface	:
				/* Set the interface name ...		*/
				t_if	.setText (buff) ;
				break	;

			case ItemProg	:
				/* Set the filter program ...		*/
				t_prog	.setText (buff) ;
				break	;

			default	:
				/* Anything else is unrecognised and is	*/
				/* treated as an error.			*/
				error	= true ;
				break	;
		}
	}

	/* Report an error if either the error flag has been set	*/
	/* explicitely or if we failed to find the magic markder.	*/
	if (error || !magic)
	{
		Error	("Ksnuffle file error",
			 "File does not appear to be a binary logging file") ;
		::close	(logfd) ;
		return	;
	}

	/* Note the offset of the first record, then check that the	*/
	/* file actually appears to contain some logging data.		*/
	frec	= lseek (logfd, 0, SEEK_CUR) ;

	if (lseek (logfd, 0, SEEK_END) <= frec)
	{
		Error	("Ksnuffle file error", "Logging file is empty") ;
		::close	(logfd) ;
		return	;
	}

	/* Set the time increment to be used on each clock tick		*/
	/* depending on the speed-up or slow-down.			*/
	if (v_speed > 0)
		inctime.tv_sec = v_speed, inctime.tv_usec = 0 ;
	else	inctime.tv_sec = 0,	  inctime.tv_usec = 1000000/(-v_speed) ;

	/* Pass the display options to the graphic and packet display	*/
	/* objects, and then anable the start button so that the user	*/
	/* can start the replay.					*/
	packets->setOptions ( (d_time  ? PO_TIME  : 0) |
			      (d_proto ? PO_PROTO : 0) |
			      (d_srce  ? PO_SRC   : 0) |
			      (d_dest  ? PO_DST   : 0) |
			      (d_info  ? PO_INFO  : 0) |
			      (d_size  ? PO_SIZE  : 0) ) ;

	graphic->setOptions (v_load1, v_bar1, v_bar2) ;
	b_go	.setEnabled (true)	;
}

/*  KNDConfRep								*/
/*  clickReset	: Handle a click of the reset button			*/
/*  (returns)	: void		:					*/

void	KNDConfRep::clickReset ()
{
}

/*  KNDConfRep								*/
/*  clickClear	: Handle a click of the clear button			*/
/*  (returns)	: void		:					*/

void	KNDConfRep::clickClear ()
{
	c_time .setChecked	(true)	  ;
	c_proto.setChecked	(true)	  ;
	c_srce .setChecked	(true)	  ;
	c_dest .setChecked	(true)	  ;
	c_info .setChecked	(true)	  ;
	c_size .setChecked	(true)	  ;
	c_multi.setChecked	(true)	  ;

	s_load1.setValue	( 1) ;
	s_bar1 .setValue	( 5) ;
	s_bar2 .setValue	(20) ;

	halt	() ;
}

/*  KNDConfRep								*/
/*  halt	: Halt replay						*/
/*  (returns)	: void		:					*/

void	KNDConfRep::halt ()
{
	if (going)
	{
		b_go	 .setText   ("Start" ) ;
		b_freeze .setText   ("Freeze") ;
		b_freeze.setEnabled (false)    ;
		going	= false	;
		frozen	= false ;

		multi  ->unRegister(slot) ;
	}
}

/*  KNDConfRep								*/
/*  clickGo	: Handle a click of the go button			*/
/*  (returns)	: void		:					*/

void	KNDConfRep::clickGo ()
{
	if (logfd <= 0)
		return ;

	/* If we are currently running then stop, resetting the start	*/
	/* button and disabling the freeze button.			*/
	if (going)
	{	halt	() ;
		return	;
	}

	/* In case we are restarting the log, seek back to the offset	*/
	/* of the first record, then read the first record in. Neither	*/
	/* of these should fail, unless the logging file has been	*/
	/* corrupted.							*/
	if (lseek (logfd, frec, SEEK_SET) != frec)
	{
		Error	("Ksnuffle file error", strerror(errno)) ;
		return	;
	}

	if (read (logfd, &pktdata, sizeof(pktdata)) != sizeof(pktdata))
	{
		Error	("Ksnuffle file error", "Log file is truncated") ;
		return	;
	}

	/* Note the whole-second time of the first packet for use in	*/
	/* timeing packet replay, then adjust the start and freeze	*/
	/* buttons.							*/
	reptime.tv_sec	= pkttime = pktdata.tval.tv_sec ;
	reptime.tv_usec	= 0 ;
	b_go	.setText    ("Stop") ;
	b_freeze.setEnabled (true)   ;

	/* Notigy the packet and graphic display objects that we are	*/
	/* now running ...						*/
	if (d_multi) multi->Register (slot, v_name) ;
	packets->execute    (true) ;
	graphic->execute    (true) ;
	going	= true	;
}

/*  KNDConfRep								*/
/*  clickFreeze	: Handle a click of the freeze button			*/
/*  (returns)	: void		:					*/

void	KNDConfRep::clickFreeze ()
{
	if (going)
	{	frozen	= !frozen ;
		b_freeze.setText (frozen ? "Resume" : "Freeze") ;
	}
		
}

/*  KNDConfRep								*/
/*  clickLog	: Handle a click of the log browse button		*/
/*  (returns)	: void		:					*/

void	KNDConfRep::clickLog ()
{
	QString	name = KFileDialog::getSaveFileName (NULL, "*.kndb", this) ;

	if (!name.isNull() && !name.isEmpty()) t_logb.setText (name) ;
}

/*  KNDConfRep								*/
/*  timerTick	: Handle a timer tick					*/
/*  second	: long		: Second				*/
/*  (returns)	: void		:					*/

void	KNDConfRep::timerTick
	(	long	second
	)
{
	PktInfo	*pktinfo ;

	/* Rather then have an additional replay object, equivalent to	*/
	/* the capture object, so spit out packets from the log, it is	*/
	/* easire to do it here.					*/
	if (going && !frozen)
	{
		/* Increment the replay time by the increment. For	*/
		/* speed ups, this will be greater than one second,	*/
		/* while for slow-downs it will be less.		*/
		AddTime (reptime, inctime) ;

		/* The loop now displays packets until we either reach	*/
		/* the end of the logging file, or we get to a packet	*/
		/* which was logged beyond the replay time.		*/
		while (timercmp (&pktdata.tval, &reptime, <))
		{
			/* Spit out the requisite number of timer ticks	*/
			/* to the graphic display to bring it to the	*/
			/* correct time for this next backup.		*/
			while (pktdata.tval.tv_sec > pkttime)
			{	graphic->timerTick (pkttime) ;
				pkttime += 1 ;
			}

			/* Decode the packet. Note that here we set the	*/
			/* time-ago value to zero as we know that we	*/
			/* are synchronised ...				*/
			pktinfo = getPktInfo () ;
			pktinfo->inuse = false  ;

			PktDecode (&pktdata, pktinfo, 0xffffff) ;
			pktinfo->ago = 0 ;

			packets->addPacket (pktinfo) ;
			graphic->addPacket (pktinfo) ;
			multi  ->addPacket (pktinfo, slot) ;

			if (!pktinfo->inuse) freePktInfo (pktinfo) ;

			/* Preload the next packet, checking for the	*/
			/* end of the recording.			*/
			if (read (logfd, &pktdata, sizeof(pktdata)) != sizeof(pktdata))
			{	halt () ;
				break	;
			}
		}

		/* We may now need to pass some more timer ticks to	*/
		/* the graphics object to bring it up to the simulated	*/
		/* replay time.						*/
		while (reptime.tv_sec > pkttime)
		{	graphic->timerTick (pkttime) ;
			pkttime += 1 ;
		}
	}
}

/*  KNDConfRep								*/
/*  replayStop	: Stop replay execution					*/
/*  (returns)	: void		:					*/

void	KNDConfRep::replayStop ()
{
	if ( going) clickGo () ;
}

/*  KNDConfRep								*/
/*  replayStart	: Start replay execution				*/
/*  (returns)	: void		:					*/

void	KNDConfRep::replayStart ()
{
	if (!going)
	{	clickSet () ;
		clickGo  () ;
	}
}

/*  KNDConfRep								*/
/*  replayFreeze: Freeze replay execution				*/
/*  (returns)	 : void		:					*/

void	KNDConfRep::replayFreeze ()
{
	if (!frozen) clickFreeze () ;
}

/*  KNDConfRep								*/
/*  replayResume: Resumereplay execution				*/
/*  (returns)	 : void		:					*/

void	KNDConfRep::replayResume ()
{
	if ( frozen) clickFreeze () ;
}

