#ifndef _PROTOCOL_H_
#define _PROTOCOL_H_

#include <qstring.h>
#include <qstrlist.h>
#include <qregexp.h>
#include "ProgressDlg.h"
#include <kurl.h>

#include "fileinfo.h"
#include "mimemagic.h"
#include "kmime.h"
#include "protinter.h"

//----------------------------------------------------------------------------------------------
// This class is used for subprotocols, that is
// the protocol used for archive files.
//
// Formats supported : Tar, Zip, Rpm

class SubProtocol
{
public:
	SubProtocol(int _type, const char *str) : type(_type), extension(str,TRUE,TRUE)
	{
		switch (type) {
		   case 9: protocol = "tar"; break;
		   case 10: protocol = "zip"; break;
		   case 11: protocol = "rpm"; break;
		   case 12: protocol = "deb"; break;
		}
	}
	bool matchFileName(const char *name) { return (extension.match(name) != -1);}

	int		type;
	QRegExp		extension;
	QString		protocol;
};

typedef QList<SubProtocol> 	SubProtList;
typedef QListIterator<SubProtocol>	SubProtListIterator;

SubProtocol* matchFileName(const char *str, SubProtList *list);

//----------------------------------------------------------------------------------------------

#include <qevent.h>
#define	Event_Message	1002
class QMessageEvent : public QEvent
{
public:
	QMessageEvent(const char *msg) : QEvent(Event_Message), Msg(msg) {}
	const char* message() const { return Msg.data();}
private:
	QString	Msg;
};

//----------------------------------------------------------------------------------------------

class Protocol
{
public:
	enum TypeSpec { File, Mount, Ftp, Trash, Smb, SmbMgr, SmbMach, SmbShare, Arch, Tar, Zip, Rpm, Deb};

	Protocol(TypeSpec type = File);
	virtual ~Protocol();

	int type() const;
	void setupMime(KMimeExtDict *extDict = 0, KMimeExtList *nameDict = 0, MimeMagic *magic = 0, bool use = TRUE);
	KMimeExtension* getMimeTypeFromExtension(FileInfo *fi);
	KMimeExtension* getMimeTypeFromName(FileInfo *fi);
	KMimeExtension* getMimeTypeFromMagic(FileInfo *fi);
	KMimeExtension* getMimeTypeFromNode(FileInfo *fi);
	void setupSubProtList(SubProtList *list = 0);
	void setManager(QObject *mgr = 0);
	void sendMessage(const char *msg);

	virtual const FileInfoList* entryInfoList(bool hidden, bool dirsOnly, bool showArchive = FALSE) = 0;
	virtual const QStrList* entryList(bool hidden, bool dirsOnly, bool showArchive = FALSE) = 0;
	virtual bool mkdir(const char *pathname) = 0;
	virtual QString absPath() = 0;
	virtual bool setPath(const char *pathname) = 0;
	virtual QString dirName() = 0;
	virtual bool remove(const char *filename, ProgressDlg *dlg = 0) = 0;
	virtual bool rmdir(const char *dirname, ProgressDlg *dlg = 0) = 0;
	virtual bool isRoot() = 0;
	virtual bool rename(const char *src, const char *dest) = 0;
	virtual bool exists(const char *filename) = 0;
	virtual bool chmod(const char *filename, int perm) = 0;
	virtual bool chown(const char *filename, const char *owner, const char *group) = 0;

	virtual bool copyToProtocol(const char *filename, const char *targetDir = 0, bool move = FALSE, ProgressDlg *dlg = 0) = 0;
	virtual bool copyFromProtocol(const char *filename, const char *targetDir, bool move = FALSE, ProgressDlg *dlg = 0) = 0;
	virtual bool init() = 0;
	virtual bool finish() = 0;
	virtual bool cleanup() = 0;
	virtual bool matchURL(KURL url) = 0;
	virtual const char* currentURL() = 0;
	virtual bool isLocal() = 0;
	int score() { return Prefix.length();}
	QString prefix() { return Prefix;}
	const char* errorMsg() { return ErrorMsg.data();}
	virtual unsigned long long freeSize(const char *dirname) = 0;
	virtual bool hasChilds(const char *dirname) = 0;
	virtual FileInfo* getInfo(KURL& url) = 0;
	virtual bool shutDown();
	virtual bool isAccesible();
	virtual const char* menuEntry();	// menu entry to be added to popup menu in kexplorer
	virtual const char* passwd();
	virtual const char* user();
	virtual bool needLogin();
	virtual void setLogin(const char* user, const char* pwd);
	virtual void loadImage(FileInfo *fi, int save, int size);
	virtual void refresh();
	virtual bool check() { return true;}	// check if protocol is still available

	virtual const char* getTempFile(KURL *url, ProgressDlg *dlg = 0);

	bool canBeMounted();

protected:
	TypeSpec	Type;
	KMimeExtDict	*ExtDict;
	KMimeExtList	*NameDict;
	MimeMagic	*Magic;
	bool		UseMagic;
	QString		Prefix;
	QString		ErrorMsg;
	SubProtList	*ProtList;
	QObject		*Manager;

};

inline Protocol::Protocol(TypeSpec type) : Type(type), ExtDict(0), NameDict(0), Manager(0)
{}

inline Protocol::~Protocol()
{}

inline int Protocol::type() const
{ return Type;}

inline void Protocol::setupMime(KMimeExtDict *extDict, KMimeExtList *nameDict, MimeMagic *magic, bool use)
{
	ExtDict = extDict;
	NameDict = nameDict;
	Magic = magic;
	UseMagic = use;
}

inline void Protocol::setupSubProtList(SubProtList *list)
{ ProtList = list;}

inline const char* Protocol::passwd()
{ return 0;}

inline const char* Protocol::user()
{ return 0;}

inline bool Protocol::needLogin()
{ return false;}

inline void Protocol::setLogin(const char *user, const char *pwd)
{}

inline void Protocol::loadImage(FileInfo*, int, int)
{}

inline void Protocol::setManager(QObject *mgr)
{ Manager = mgr;}

inline bool Protocol::canBeMounted()
{ return (Type == Mount || Type == Smb);}

inline void Protocol::refresh()
{}

#endif
