/***************************************************************************
                          usrmodem.cpp  -  description                              
                             -------------------                                         
    begin                : Wed Apr 7 1999                                           
    copyright            : (C) 1999 by Torsten Uhlmann                         
    email                : TUhlmann@gmx.de                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#include <termios.h>
#include <unistd.h>
#include <fcntl.h>

#include <errno.h>
#include <sys/signal.h>
#include <sys/time.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <qstring.h>
#include <qprogressdialog.h>
#include <qfile.h>
#include <qdatastream.h>
#include <qmsgbox.h>
#include <kapp.h>
#include "usrmodem.h"




//#define MODEM_INIT							"ATE0V1+MCS=1\r\n"
#define MODEM_CLEAR_MEMORY      "AT+MEM\r\n"
#define MODEM_GET_MEMORY_INFO   "AT+MSR=%i\r\n"
#define MODEM_GET_MEMORY_INFO2  "AT+MSI\r\n"
#define MODEM_GET_MEMORY        "AT+MTM\r\n"
#define MODEM_RESET             ""


#define MODEM_USR_PAGE          32768

#define MODEM_ETX	              0x03
#define MODEM_DLE	              0x10
#define MODEM_SUB	              0x1A

#define MODEM_NEW_PAGE          100

USRModem::USRModem(const QString &device, const QString &baudrate,
		const QString &lockfile) : ModemConnector(device, baudrate, lockfile)
{
  memoryFile 				= 0;
  faxFile    				= 0;
  WasDLE      			= 0;
  tempFile    			= true;
  tempVoiceFile 		= true;
  _memoryInfoValid	= false;
  modemType					= unknownModem;
}

USRModem::~USRModem()
{
  deleteMemoryFile();
  deleteUserMemoryFile();
  if (tempFile)
    deleteFaxFiles();

  if (tempVoiceFile)
    deleteVoiceFiles();

  resetModem();
}

/** get fax from memoryFile and store separated pages in
    faxFile-<nr> */
bool USRModem::getFax(int id, const QString &filename = 0)
{
  char *tmpStr  = 0;

  if (memoryFile.isEmpty())
    return false;

  if (tempFile)
    deleteFaxFiles();

	if (!filename)
	{
	  tempFile = true;
    // get temp filename for whole modem memory
	  if (!(tmpStr = tempnam(0,"fxfax")))
  	  return false;
	  faxFile.setStr(tmpStr);
  	free(tmpStr);
  } else {
    tempFile = false;
	  faxFile.setStr(filename.data());
  }
  if (!ExtractMessage(id, Fax))
    return false;

  return true;
}

/** get voice message from memoryFile and store in
    voiceFile.gsm */
bool USRModem::getVoice(int id, const QString &filename = 0)
{
	char *tmpStr  = 0;

  if (memoryFile.isEmpty())
    return false;

  if (tempVoiceFile)
    deleteVoiceFiles();

	if (!filename)
	{
	  tempVoiceFile = true;
    // get temp filename for whole modem memory
	  if (!(tmpStr = tempnam(0,"fxmsg")))
  	  return false;
	  voiceFile.setStr(tmpStr);
  	free(tmpStr);
  } else {
    tempVoiceFile = false;
	  voiceFile.setStr(filename.data());
  }
 	return (ExtractMessage(id, Voice));
}

/** get voice message from userMemoryFile and store in
    voiceFile.gsm */
bool USRModem::getUserPageVoice(int id, const QString &filename = 0)
{
	char *tmpStr  = 0;

  if (userMemoryFile.isEmpty())
    return false;

  if (tempVoiceFile)
    deleteVoiceFiles();

	if (!filename)
	{
	  tempVoiceFile = true;
    // get temp filename for whole modem memory
	  if (!(tmpStr = tempnam(0,"fxmsg")))
  	  return false;
	  voiceFile.setStr(tmpStr);
  	free(tmpStr);
  } else {
    tempVoiceFile = false;
	  voiceFile.setStr(filename.data());
  }
  return (ExtractMessageUserSector(id));
}

/** get used memory and store it in memoryFile */
bool USRModem::getMemory()
{
  FILE *tmpFile    		= 0;
  int c			= 0;
  bool 	loop        	= true;
  int 	byteCount    	= 0;
  int 	usrPageInput 	= MODEM_USR_PAGE + 2;
  char 	*tmpStr     	= 0;
  int		bytesTotal 		= 0;
  int 	kBytesMemUsed	= 0;

	// get KBytes of Memory used in Modem
	kBytesMemUsed = (modemInfo.memSize * 1024 * modemInfo.memUsed) / 100;

  deleteMemoryFile();

  // get temp filename for whole modem memory
  if (!(tmpStr = tempnam(0,"fxmem")))
    return false;

  memoryFile.setStr(tmpStr);
  free(tmpStr);
  /* terminal settings done, now handle in/ouput */
  tmpFile  = fopen(memoryFile.data(),"wb");
  flushSerialLine();
  if (!writeStr(MODEM_GET_MEMORY))
    return false;

	QProgressDialog progress( i18n("Downloading modem memory"), 0,
														kBytesMemUsed, kapp->topWidget(), 0, true );
	progress.setProgress( 0 );
	
	//set wait cursor
	kapp->setOverrideCursor(waitCursor);
	
  while (loop)
  {
    if ((c = readC()) < 0)
      loop = false;
    else
    {
    	bytesTotal++;
    	if ((bytesTotal % 1024) == 0)
    	{
    		progress.setProgress((int)(bytesTotal / 1024));
    	}
    		
      if (byteCount >= usrPageInput) {
	      byteCount = 0;
      }
      if (byteCount > 1) {
        fputc((unsigned char)c,tmpFile);
      }
      byteCount++;
    }
  }

  // restore cursor
	kapp->restoreOverrideCursor();

  fclose(tmpFile);
	progress.setProgress( kBytesMemUsed );
	progress.close();
  return true;
}

/** get used memory and store it in memoryFile */
bool USRModem::getMemoryPage(int start, int stop)
{
  FILE *tmpFile    		= 0;
  int 	c							= 0;
  bool 	loop        	= true;
  int 	byteCount    	= 0;
  int 	usrPageInput 	= MODEM_USR_PAGE + 2;
  char 	*tmpStr     	= 0;
  int		bytesTotal 		= 0;
  int 	kBytesMemUsed	= 0;
  QString cmd					= 0;

	// get KBytes of Memory used in Modem
	kBytesMemUsed = (32 * (stop - start + 1));

  deleteUserMemoryFile();

  // get temp filename for whole modem memory
  if (!(tmpStr = tempnam(0,"fxumem")))
    return false;

  userMemoryFile.setStr(tmpStr);
  free(tmpStr);
  /* terminal settings done, now handle in/ouput */
  tmpFile  = fopen(userMemoryFile.data(),"wb");
  flushSerialLine();

	QProgressDialog progress( i18n("Downloading modem memory"), 0,
														kBytesMemUsed, kapp->topWidget(), 0, true );
	progress.setProgress( 0 );
	//set wait cursor
	kapp->setOverrideCursor(waitCursor);

	for (int i = start;i <= stop; i++)
	{	
		// get pages
		cmd.sprintf("AT+MTP=%i\r\n",i);
  	if (!writeStr(cmd.data())) return false;
		usleep(1000000);
    	
		loop = true;
  	while (loop)
  	{
    	if ((c = readC()) < 0)
      	loop = false;
    	else
    	{
    		bytesTotal++;
    		if ((bytesTotal % 1024) == 0)
    		{
    			progress.setProgress((int)(bytesTotal / 1024));
    		}
    		
      	if (byteCount >= usrPageInput) {
	      	byteCount = 0;
	      	loop = false;
      	}
      	if (byteCount > 1) {
        	fputc((unsigned char)c,tmpFile);
      	}
      	byteCount++;
    	}
  	}
  }

  // restore cursor
	kapp->restoreOverrideCursor();

  fclose(tmpFile);
	progress.setProgress( kBytesMemUsed );
	progress.close();
  return true;
}


/** get memory info about stored faxes */
bool USRModem::getMemoryInfo(ModemGeneralInfo &moGeneral,
														 ModemMessageList &moMsgList)
{
  int maxIndex      = 0;
  int i            	= 0;
	QString cmdout 		= 0;
	QString modemStr 	= 0;
	QString memInfo  	= 0;
	QString listInfo 	= 0;
	
  ModemMsgInfoStruct	*moInfo = 0;

  // new memory info means new files
  deleteMemoryFile();
  if (tempFile)
    deleteFaxFiles();

  _memoryInfoValid = false;

	modemStr.sprintf(MODEM_GET_MEMORY_INFO, maxIndex);
  debug("Command: %s",modemStr.data());
	if (cmdout = executeCommand(modemStr,"\nOK"))
	{
		debug("success");
		memInfo = cmdout.right(cmdout.length() - 2).copy();
		// remove trailing \n OK
		memInfo.setStr(memInfo.left(memInfo.find('\n')).data());
		debug("Result: %s",memInfo.data());
	} else
		return false;

  // now we have a string who says how many messages are stored
	maxIndex = memInfo.mid(8,3).toInt() + memInfo.mid(16,3).toInt();
  //debug("%s + %s: maxIndex -> %i",memInfo.mid(8,3).data(),
  //																memInfo.mid(16,3).data(),maxIndex);

  // fill ModemGeneralInfo structure
	moGeneral.memSize 					= memInfo.mid(0,3).toInt();
	modemInfo.memSize 					= memInfo.mid(0,3).toInt();
	moGeneral.memUsed 					= memInfo.mid(4,3).toInt();
	modemInfo.memUsed 					= memInfo.mid(4,3).toInt();
	moGeneral.noVoiceMsg 				= memInfo.mid(8,3).toInt();
	modemInfo.noVoiceMsg 				= memInfo.mid(8,3).toInt();
	moGeneral.noVoiceUnreleased	= memInfo.mid(12,3).toInt();
	modemInfo.noVoiceUnreleased	= memInfo.mid(12,3).toInt();
	moGeneral.noFaxMsg 					= memInfo.mid(16,3).toInt();
	modemInfo.noFaxMsg 					= memInfo.mid(16,3).toInt();
	moGeneral.noFaxUnreleased		= memInfo.mid(20,3).toInt();
	modemInfo.noFaxUnreleased		= memInfo.mid(20,3).toInt();
	if (memInfo.mid(24,3).toInt() == 1)
	{
		moGeneral.outMsgPresent = true;
		modemInfo.outMsgPresent = true;
	} else {
		moGeneral.outMsgPresent = false;
		modemInfo.outMsgPresent = false;
	}
	if (memInfo.mid(28,3).toInt() == 1)
	{
		moGeneral.fullMsgPresent = true;
		modemInfo.fullMsgPresent = true;
	} else {
		moGeneral.fullMsgPresent = false;
		modemInfo.fullMsgPresent = false;
	}

  if (!maxIndex)
  {
		QMessageBox::information(0,i18n("Get Memory Info"),
			i18n("There are no messages stored\nin modem memory!"),
			QMessageBox::Ok | QMessageBox::Default);

    return true;
  }

  for (i=1;i<=maxIndex;i++)
  {
    listInfo 	= "";
    modemStr 	= "";
    cmdout    = 0;
    moInfo 		= 0;

		modemStr.sprintf(MODEM_GET_MEMORY_INFO, i);
  	debug("Command: %s",modemStr.data());
		if (cmdout = executeCommand(modemStr,"\nOK"))
		{
			debug("success");
			listInfo = cmdout.right(cmdout.length() - 2).copy();
			// remove trailing \n OK
			listInfo = listInfo.left(listInfo.find('\n')).copy();
			debug("Result: %s",listInfo.data());
		} else
			return false;

    // now create list with all available faxes
		moInfo = new struct ModemMsgInfoStruct;
		CHECK_PTR(moInfo);

		// fill moInfo
		moInfo->msgIndex =	listInfo.mid(0,3).toInt();
		switch (listInfo.mid(4,3).toInt())
		{
			case 1	: moInfo->msgType = Fax; 			break;
			case 2	: moInfo->msgType = Voice; 		break;
			case 3	: moInfo->msgType = Data; 		break;
			default	: moInfo->msgType = unknown;
		}
		moInfo->msgSize =	listInfo.mid(8,3).toInt();
		switch (listInfo.mid(12,3).toInt())
		{
			case 255	: moInfo->msgAttribute = UnreleasedUnchecked;	break;
			case 253	: moInfo->msgAttribute = UnreleasedChecked; 		break;
			case 252	: moInfo->msgAttribute = ReleasedChecked; 			break;
			default		: moInfo->msgAttribute = Atunknown;
		}
		moInfo->msgStatus =	listInfo.mid(16,3).toInt();
		if (listInfo.mid(20,3).toInt() == 255)
		{
			moInfo->clockValid 	= false;
			moInfo->msgDay			= 0;
			moInfo->msgHour			= 0;
			moInfo->msgMinute		= 0;
		} else {
			moInfo->clockValid = true;
			moInfo->msgDay			= listInfo.mid(20,3).toInt();
			moInfo->msgHour			= listInfo.mid(24,3).toInt();
			moInfo->msgMinute		= listInfo.mid(28,3).toInt();
		}
		moInfo->callerId 			= listInfo.mid(32,20).copy();
		moInfo->callerId			= moInfo->callerId.stripWhiteSpace();
		if (moInfo->callerId.isEmpty()) moInfo->callerId = i18n("unknown");
		moInfo->msgStartPage 	= listInfo.mid(53,3).toInt();
		moInfo->msgAddress		=	listInfo.mid(57,3).toInt();
		moInfo->msgAddress		=	moInfo->msgAddress << 8;
		moInfo->msgAddress		+= listInfo.mid(61,3).toInt();
		
		// add entry to ModemMessageList
		moMsgList.append(moInfo);
  }
  _memoryInfoValid = true;
  return true;
}

bool USRModem::clearMemory()
{
	debug("clear modem memory");
	if (executeSimpleCommand(MODEM_CLEAR_MEMORY,"\nOK"))
	{
		debug("success");
		return true;
	}
	debug("error!");
	return false;
}

QString USRModem::getFaxFileName()
{
  if (!faxFile.isEmpty())
    return faxFile;
  else
    return 0;
}

QString USRModem::getVoiceFileName()
{
  if (!voiceFile.isEmpty())
    return voiceFile;
  else
    return 0;
}

QString USRModem::getMemoryFileName()
{
  if (!memoryFile.isEmpty())
    return memoryFile;
  else
    return 0;
}

QString USRModem::getUserMemoryFileName()
{
  if (!userMemoryFile.isEmpty())
    return userMemoryFile;
  else
    return 0;
}

int USRModem::getPagesWritten()
{
  return pagesWritten;
}

bool USRModem::initModem()
{

 	modemType = detectModemType();
 	
	bool success = false;
		
	debug("init modem");
	if (!(success = executeSimpleCommand("ATZ\r\n","\nOK"))) {
		debug("ATZ: error");
		return false;
	}
/*	if (!(success = executeSimpleCommand("AT&F1 E0 H0 Q0 &D2 &H1&R2&I0\r\n","\nOK"))) {
		debug("AT&F1...: error");
		return false;
	}
	if (!(success = executeSimpleCommand("ATM2\r\n","\nOK"))) {
		debug("ATM2: error");
		return false;
	}
	if (!(success = executeSimpleCommand("ATS0=0\r\n","\nOK"))) {
		debug("ATS0=0: error");
		return false;
	}
	if (!(success = executeSimpleCommand("ATS31=128\r\n","\nOK"))) {
		debug("ATS31=128: error");
		return false;
	}
	if (!(success = executeSimpleCommand("AT#CLS=8\r\n","\nOK"))) {
		debug("AT#CLS=8: error");
		return false;
	}*/
//	if (!(success = executeSimpleCommand("#VSS=3 #VSD=0 #VBT=2 #VRA=0 #VRN=0 #VSP=30\r\n","\nOK"))) {
//		debug("#VSS=3 #VSD=0 #VBT=2 #VRA=0 #VRN=0 #VSP=30: error");
//		return false;
//	}
	
	switch (modemType)
	{
		case MessageModem:
			success = executeSimpleCommand("ATE0V1+MCS=1\r\n","\nOK");
			break;
		case ProfessionalModem:
			success = executeSimpleCommand("ATE0V1+MCA=1\r\n","\nOK");
			break;
		default: debug("Modem not initialized");
	}
	if (success)
		debug("success");
	else
		debug("error!");
		
	return success;
}

bool USRModem::resetModem()
{
	if (strlen(MODEM_RESET) == 0)
		return true;
	
	debug("reset modem");
	if (executeSimpleCommand(MODEM_RESET,"\nOK"))
	{
		debug("success");
		return true;
	}
	debug("error!");
	return false;
}

QString USRModem::faxFileName(int pageNo)
{
	QString s;
  s.sprintf("%s-%i.g3",faxFile.data(),pageNo);
  return s;
}

QString USRModem::voiceFileName()
{
	QString s;
  s.sprintf("%s.gsm",voiceFile.data());
  return s;
}

bool USRModem::ExtractMessageUserSector(int id)
{
  FILE *msgFH, *memFH;
  bool 					found     = false;
  unsigned char c     		= 0;
  QString 			fileName	= "";

  pagesWritten = 0;

 	fileName = voiceFileName();

  if (!(memFH = fopen(userMemoryFile.data(),"rb"))) {
    return false;
  }
  if (!(msgFH = fopen(fileName.data(),"wb"))) {
    return false;
  }
  /* message starts at the beginning of page */
  found = false;
  while ((!found) && (!feof(memFH)))
  {
  	fread(&c,1,1,memFH);
    /* fwrite(&c,1,1,msgFH); */
    if (WritePage(msgFH,c) == MODEM_NEW_PAGE)
    {
     	found = true;
     	break;
    }
  }
  fclose(memFH);
  fclose(msgFH);
  	
  return true;
}

bool USRModem::ExtractMessage(int msgNo, MessageType msgType)
{
  FILE *msgFH, *memFH;
  USRMemHdr 		hdr;
  int 					bytesRead	= 0;
  unsigned long bytePos 	= 0;
  unsigned long adr   		= 0;
  bool 					found     = false;
  bool 					readError = false;
  unsigned char c     		= 0;
  int 					pageNo		= 1;
  QString 			fileName	= "";

  pagesWritten = 0;

  if (msgType == Fax)
  	fileName = faxFileName(pageNo);
  else
  	fileName = voiceFileName();

  if (!(memFH = fopen(memoryFile.data(),"rb"))) {
    return false;
  }
  if (!(msgFH = fopen(fileName.data(),"wb"))) {
    return false;
  }
  /* find appropriate message */
  while ((!found) && (!readError) && (!feof(memFH)))
  {
    /* read header */
    bytesRead=fread(&hdr,34,1,memFH);
    if (bytesRead == 1) {
      if (hdr.mIndex == msgNo) {
	      found = true;
      } else {
	      adr = hdr.mNextAddr1 * 256;
        adr += hdr.mNextAddr2;
	      /* add pages */
	      adr += (hdr.mNextPage - 4) * MODEM_USR_PAGE;
	      fseek(memFH,adr,SEEK_SET);
      }
    } else {
      readError = true;
    }
  }
  /* now file pointer just read message header of right message */
  if (found) {
    debug("position found");
    /* get address of next message and read up to this point into msgFH */
    adr = hdr.mNextAddr1 * 256;
    adr += hdr.mNextAddr2;
    /* add pages */
    adr += (hdr.mNextPage - 4) * MODEM_USR_PAGE;
    bytePos = ftell(memFH);
    found = false;
    while ((!found) && (!feof(memFH)))
    {
      if (bytePos >= adr) {
	      found = true;
      } else {
	      fread(&c,1,1,memFH);
	      /* fwrite(&c,1,1,msgFH); */
	      if (WritePage(msgFH,c) == MODEM_NEW_PAGE)
	      {
  				if (msgType == Fax)
  				{
	        	if ((bytePos + 1) < adr)
	        	{
            	fclose(msgFH);
	          	pageNo++;
            	fileName = faxFileName(pageNo);
            	if (!(msgFH = fopen(fileName.data(),"wb")))
            	{
              	return false;
            	}
	        	}
	        } else
	        	break;
	      }
	      bytePos++;
      }
    }
  }
  fclose(memFH);
  fclose(msgFH);
  if (msgType == Fax)
  	pagesWritten = pageNo;
  	
  return true;
}

int USRModem::WritePage(FILE *msgFile, unsigned char c)
{
  int EndOfPage = 0;

  if ( !WasDLE )
  {
    if ( c == MODEM_DLE ) WasDLE = 1;
    else fputc( c, msgFile );
  }
  else
  {
    if ( c == MODEM_DLE )
      fputc( c, msgFile );	/* DLE DLE -> DLE */
    else
    {
      if ( c == MODEM_SUB )				/* DLE SUB -> 2x DLE */
      {						                /* (class 2.0) */
        fputc( MODEM_DLE, msgFile ); fputc( MODEM_DLE, msgFile );
      }
      else
      {
        if ( c == MODEM_ETX ) EndOfPage = MODEM_NEW_PAGE; /* DLE ETX -> end */
      }
    }
    WasDLE = 0;
  }
  return EndOfPage;
}

void USRModem::deleteMemoryFile()
{
  if (!memoryFile.isEmpty())
  {
    remove(memoryFile.data());
    memoryFile = "";
  }
}

void USRModem::deleteUserMemoryFile()
{
  if (!userMemoryFile.isEmpty())
  {
    remove(userMemoryFile.data());
    userMemoryFile = "";
  }
}

void USRModem::deleteFaxFiles()
{
  int  		i	=0;
  QString fname;

  if ((!faxFile.isEmpty()) && (pagesWritten))
  {
    for (i=1; i<=pagesWritten;i++)
    {
      fname = faxFileName(i);
      remove(fname.data());
    }
    faxFile = 0;
    pagesWritten = 0;
  }
}

void USRModem::deleteVoiceFiles()
{
  QString fname;

  if (!voiceFile.isEmpty())
  {
  	fname = voiceFileName();
		remove(fname.data());
	}
  voiceFile = 0;
}


bool USRModem::playVoiceIntern(int index)
{
	QString cmd = 0;
	debug("play message index %d",index);
	if (setStandaloneMode())
	{
		cmd.sprintf("AT+MVP=%d\r\n",index);
		if (executeSimpleCommand(cmd,(QString)0))
		{
			return true;
		}
	}
	return false;
}

bool USRModem::setStandaloneMode()
{
	bool success = false;
		
	debug("switch standalone mode on");
	switch (modemType)
	{
		case MessageModem:
			success = executeSimpleCommand("AT+MCS=1\r\n","\nOK");
			break;
		case ProfessionalModem:
			success = executeSimpleCommand("AT+MCA=1\r\n","\nOK");
			break;
		default: debug("Modem unknown!");
	}
	if (success)
		debug("success");
	else
		debug("error!");
		
	return success;
}

bool USRModem::unsetStandaloneMode()
{
	bool success = false;
		
	debug("switch standalone mode off");
	switch (modemType)
	{
		case MessageModem:
			success = executeSimpleCommand("AT+MCS=0\r\n","\nOK");
			break;
		case ProfessionalModem:
			success = executeSimpleCommand("AT+MCA=0\r\n","\nOK");
			break;
		default: debug("Modem unknown!");
	}
	if (success)
		debug("success");
	else
		debug("error!");
		
	return success;
}

bool USRModem::resetClock()
{
	debug("reset modem clock");
	if (executeSimpleCommand("AT+MCC\r\n","\nOK"))
	{
		debug("success");
		return true;
	}
	debug("error!");
	return false;
}

bool USRModem::getFaxReception()
{
	QString cmdout = 0;

	debug("get Fax Reception");
	if (cmdout = executeCommand("AT+MCF?\r\n","\nOK"))
	{
		debug("success");
		if (cmdout.mid(2,1).toInt() == 1)
			return true;
		else
			return false;
	}
	debug("error!");
	return false;
}

bool USRModem::setFaxReception()
{
	debug("switch fax reception on");
	if (executeSimpleCommand("AT+MCF=1\r\n","\nOK"))
	{
		debug("success");
		return true;
	}
	debug("error!");
	return false;
}

bool USRModem::unsetFaxReception()
{
	debug("switch fax reception off");
	if (executeSimpleCommand("AT+MCF=0\r\n","\nOK"))
	{
		debug("success");
		return true;
	}
	debug("error!");
	return false;
}

bool USRModem::getVoiceReception()
{
	QString cmdout= 0;

	debug("get Voice Reception");
	if (cmdout = executeCommand("AT+MCV?\r\n","\nOK"))
	{
		debug("success");
		if (cmdout.mid(2,1).toInt() == 1)
			return true;
		else
			return false;
	}
	debug("error!");
	return false;
}

bool USRModem::setVoiceReception()
{
	debug("switch voice reception on");
	if (executeSimpleCommand("AT+MCV=1\r\n","\nOK"))
	{
		debug("success");
		return true;
	}
	debug("error!");
	return false;
}

bool USRModem::unsetVoiceReception()
{
	debug("switch voice reception off");
	if (executeSimpleCommand("AT+MCV=0\r\n","\nOK"))
	{
		debug("success");
		return true;
	}
	debug("error!");
	return false;
}

bool USRModem::getControlMonitor()
{
	QString cmdout = 0;

	debug("get Control Monitor Status");
	if (cmdout = executeCommand("AT+MCM?\r\n","\nOK"))
	{
		debug("success");
		if (cmdout.mid(2,1).toInt() == 1)
			return true;
		else
			return false;
	}
	debug("error!");
	return false;
}

bool USRModem::setControlMonitor()
{
	debug("switch control monitor on");
	if (executeSimpleCommand("AT+MCM=1\r\n","\nOK"))
	{
		debug("success");
		return true;
	}
	debug("error!");
	return false;
}

bool USRModem::unsetControlMonitor()
{
	debug("switch control monitor off");
	if (executeSimpleCommand("AT+MCM=0\r\n","\nOK"))
	{
		debug("success");
		return true;
	}
	debug("error!");
	return false;
}

bool USRModem::setSetupRings(int rings)
{
	QString tmp;
	if ((rings < 3) || (rings > 6))
		return false;
		
	tmp.sprintf("AT+MCR=%i\r\n",rings);
	debug("set number of rings with %s",tmp.data());
	if (executeSimpleCommand(tmp.data(),"\nOK"))
	{
		debug("success");
		return true;
	}
	debug("error!");
	return false;
}

int  USRModem::getSetupRings()
{
	QString cmdout = 0;

	debug("get Number of Rings");
	if (cmdout = executeCommand("AT+MCR?\r\n","\nOK"))
	{
		debug("success");
		return cmdout.mid(2,1).toInt();
	}
	debug("error!");
	return 0;
}

bool USRModem::setVoiceDuration(int sec)
{
	QString tmp;
	if ((sec < 0) || (sec > 255))
		return false;
		
	tmp.sprintf("AT+MVD=%i\r\n",sec);
	debug("set voice duration: %s",tmp.data());
	if (executeSimpleCommand(tmp.data(),"\nOK"))
	{
		debug("success");
		return true;
	}
	debug("error!");
	return false;
}

int  USRModem::getVoiceDuration()
{
	QString cmdout = 0;
	QString str;

	debug("get voice duration");
	if (cmdout = executeCommand("AT+MVD?\r\n","\nOK"))
	{
		debug("success");
		str = cmdout.right(cmdout.length() - 2);
		// remove rtailing \n OK
		str = str.left(str.find('\n')).copy();
		debug("duration %s",str.data());
		return str.toInt();
	}
	debug("error!");
	return 0;
}

bool USRModem::setSetupFaxId(QString id)
{
	QString tmp = "";
	tmp.sprintf("AT+MFI=\"%s\"\r\n",id.data());
	
	debug("set Fax Id with %s",tmp.data());
	if (executeSimpleCommand(tmp.data(),"\nOK"))
	{
		debug("success");
		return true;
	}
	debug("error!");
	return false;
}

QString USRModem::getSetupFaxId()
{
	QString cmdout = 0;

	debug("get Fax Id");
	if (cmdout = executeCommand("AT+MFI?\r\n","\nOK"))
	{
		debug("success");
		cmdout = cmdout.right(cmdout.length() - 3);
		// remove rtailing " OK
		cmdout = cmdout.left(cmdout.findRev('"'));
		return cmdout;
	}
	debug("error!");
	cmdout = 0;
	return cmdout;
}

bool	USRModem::setMessageInternal(bool start, bool outgoing)
{
	QString tmp = "";
	if (start)
	{
		tmp.sprintf("AT+MVR=%i\r\n",((outgoing) ? 0 : 1));
		debug("start recording: %s",tmp.data());
		writeStr(tmp);
		return true;
	} else {
		// stop recording
		tmp = "AT\r\n";
		debug("stop recording: %s",tmp.data());
		if (executeSimpleCommand(tmp.data(),"\nOK"))
		{
			debug("success");
			return true;
		}
		debug("error!");
		return false;
	}
}

bool USRModem::dialVoiceNumber(const QString number)
{
	bool success = false;
	QString cmd; // "AT#VTS=" + number + "]\r\n";

	debug("dial number %s",number.data());

	if (!(success = executeSimpleCommand("AT&F1\r\n","\nOK"))) {
		debug("AT&F1: error");
		return false;
	}
	if (!(success = executeSimpleCommand("AT #CLS=8 #VRN=0 #VLS=6 #VBT=1 E0 V1 M0\r\n","\nOK"))) {
		debug("AT ...: error");
		return false;
	}
	if (!(success = executeSimpleCommand("ATA\r\n","\nOK"))) {
		debug("ATA: error");
		return false;
	}
	usleep(200000);
	for (unsigned int i=0; i<number.length();i++)
	{
		cmd = "AT#VTS=" + number.mid(i,1) + "\r\n";
		debug("dial: %s",cmd.data());
		writeStr(cmd);
		usleep(300000);
	}
	debug("done");
	return true;
}

bool USRModem::hangupPhone()
{
	bool success = false;
	
	if (!(success = executeSimpleCommand("ATH0#CLS=8\r\n","\nOK"))) {
		debug("ATH#CLS=8: error");
		return false;
	}
	initModem();
	return true;
}

bool USRModem::setSpeakerMikeVolume(int speaker,int mike)
{
	bool success = false;
	QString cmd;
	
	cmd.sprintf("AT#VGR=%d\r\n",speaker);
	//debug("Speaker: %s",cmd.data());
	if (!(success = executeSimpleCommand(cmd.data(),"\nOK"))) {
		debug("AT#VGR=%d: error",speaker);
		return false;
	}
	cmd.sprintf("AT#VGT=%d\r\n",mike);
	//debug("Mike: %s",cmd.data());
	if (!(success = executeSimpleCommand(cmd.data(),"\nOK"))) {
		debug("AT#VGT=%d: error",mike);
		return false;
	}
	return true;
}

bool USRModem::getSpeakerMikeVolume(int &speaker, int &mike)
{
	QString cmdout = 0;

	debug("get Volume of Speaker");
	if (cmdout = executeCommand("AT#VGR?\r\n","\nOK"))
	{
		debug("success: %s",cmdout.mid(2,3).data());
		speaker = cmdout.mid(2,3).toInt();
	} else {
		debug("error!");
		return false;
	}
	debug("get Volume of Mike");
	if (cmdout = executeCommand("AT#VGT?\r\n","\nOK"))
	{
		debug("success: %s",cmdout.mid(2,3).data());
		mike = cmdout.mid(2,3).toInt();
	} else {
		debug("error!");
		return false;
	}
	return true;
}


bool USRModem::uploadVoiceMessage(QString file,bool outgoing)
{
	QString 			tmp 		= "";
	int 					oneByte = 0;
	unsigned char oneChar = 0;
	char 					end[]		= {0x10 , 0x03};

	tmp.sprintf("AT+MVC=%i\n\r",((outgoing) ? 0 : 1));
	
	debug("capture message command %s",tmp.data());
	
	if (!executeSimpleCommand("ATZ\n\r","\nOK"))
	{
		debug("ATZ: error");
		return false;
	}
	if (!executeSimpleCommand("AT+MCS=1\n\r","\nOK"))
	{
		debug("AT+MCS=1: error");
		return false;
	}
	if (!executeSimpleCommand(tmp.data(),"\nCONNECT"))
	{
		debug("error");
		return false;
	}
	debug("connected");
	if (!executeSimpleCommand("AT &F1 E0 F1 V1 &N0 &I0 &H1 &R2 &W0 Y0\n\r","\nOK"))
	{
		debug("AT&W0Y0Z: error");
		return false;
	}
	if (!executeSimpleCommand("AT+MCS=0\n\r","\nOK"))
	{
		debug("AT+MCS=0: error");
		return false;
	}
	
	FILE *fd=fopen(file.data(),"rb");
	if(!fd)
	{
		debug("could not open!");
		return false;
	}
	//flushSerialLine();

	debug("start loop!");
	while((oneByte = fgetc(fd)) != EOF)
	{
		oneChar = (unsigned char) oneByte;
		write(_fd,&oneChar,1);
		if(oneChar == 0x10) write(_fd,&oneChar,1);
	}
	debug("end of loop!");
	write(_fd,end,2);
	fclose(fd);
	usleep(5000000);
	//flushSerialLine();
	
	if (!executeSimpleCommand("AT+MCS=1\n\r","\nOK"))
	{
		debug("AT+MCS=1: error");
		return false;
	}
	
	
	
	return true;

/*	
	QFile f(file);
	if (!f.open(IO_ReadOnly | IO_Raw)) return false;
	QDataStream stream(&f);
	debug("allocate stream");
	stream.readRawBytes(&oneByte,1);
	//debug("%c",oneByte);
	debug("write stream");
	writeStr(byteArray,len);
	debug ("end of loop");
	writeStr(end,2);
	f.close();
	if (byteArray) delete byteArray;
	return true;
*/
}

bool USRModem::commitSetup()
{
	debug("commit setup");
	if (executeSimpleCommand("AT+MCW\r\n","\nOK"))
	{
		debug("success");
		return true;
	}
	debug("error!");
	return false;
}

QString USRModem::rawCommand(const QString &command)
{
	QString cmdout = 0;
	QString s = command.copy();
	
	s += "\r";
	
	debug("send raw command: %s",s.data());
	if (cmdout = executeCommand(s,"\nOK"))
		debug("success");
	else
		debug("error!");
		
	return cmdout;
}

USRModem::ModemType USRModem::detectModemType()
{
	QString cmdout = 0;
	
	debug("get Modem identity");
	if (!(cmdout = executeCommand("ATI3\r\n","\nOK")))
	{
		debug("error!");
		return unknownModem;
	}
	debug("we are a %s",cmdout.data());
	if ((!cmdout.contains("message",false)) && (!cmdout.contains("msg",false)))
	{
		debug("Neither Message nor Message Professional");
		return unknownModem;
	}
	if (cmdout.contains("pro",false))
	{
		debug("Professional Message Modem");
		return ProfessionalModem;
	}
	debug("Normal Message Modem");
	return MessageModem;
}
















