/*
    KMLOFax
    
    A utility to process facsimile received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 1999-2000 Oliver Gantz <o.gantz@tu-bs.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
    PostScript(R) is a registered trademark of Adobe Systems Incorporated.
*/

#include "printdlg.h"
#include "printdlg.moc"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif 

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <qwidget.h>
#include <qdialog.h>
#include <qlayout.h>
#include <qradiobutton.h>
#include <qgroupbox.h>
#include <qbuttongroup.h>
#include <qlineedit.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qframe.h>
#include <qcursor.h>
#include <qspinbox.h>
#include <qcheckbox.h>
#include <qtooltip.h>
#include <qsize.h>
#include <qimage.h>
#include <qcolor.h>

#include <kapp.h>
#include <kmsgbox.h>

#include "global.h"
#include "mlofile.h"
#include "preferences.h"



PrintDlg * printdlg = 0;



MLO2PSFilter::MLO2PSFilter( const char * infile )
{
	mlofile.setName( infile );

	ps = 0;
	setFormat( PAPER_A4 );
	interpolate = FALSE;
	first_page = 0;
	last_page = 0;
	doc_copies = 1;
	l_margin = 0;
	r_margin = 0;
	t_margin = 0;
	b_margin = 0;

	resetImageData();
}


MLO2PSFilter::~MLO2PSFilter()
{
}


void MLO2PSFilter::setFormat( int format )
{
	/* A4, B4, B5, Letter, Legal, Executive */
	static int papers_x[] = { 595,  729, 516, 612,  612, 540 };
	static int papers_y[] = { 842, 1032, 729, 792, 1008, 720 };

	paper_x = papers_x[format];
	paper_y = papers_y[format];
}


void MLO2PSFilter::setInterpolate( bool interp )
{
	interpolate = interp;
}


void MLO2PSFilter::setRange( int first, int last )
{
	first_page = first;
	last_page = last;
}


void MLO2PSFilter::setCopies( int copies )
{
	doc_copies = copies;
}


void MLO2PSFilter::setMargins( int left, int right, int top, int bottom )
{
	l_margin = left;
	r_margin = right;
	t_margin = top;
	b_margin = bottom;
}


bool MLO2PSFilter::convertFile( FILE * stream, const char * sender )
{
	page_info_t info;
	int page, ps_pages, sub_page;
	int * size_y_buff;
	int maxlines;
	time_t t;

	ps = stream;

	/* Size of viewport (in pixel, 1/72 dpi) */
	view_x = paper_x - (int)((double)(l_margin + r_margin) * 2.8346);
	view_y = paper_y - (int)((double)(t_margin + b_margin) * 2.8346);
	
	/* Bottom-left border (in pixel, 1/72 dpi) */
	trans_x = (int)((double)l_margin * 2.8346);
	trans_y = (int)((double)b_margin * 2.8346);
	
	/* Calculate real number of pages */
	if (!mlofile.open()) {
		return FALSE;
	}

	ps_pages = mlofile.pages();
	
	if (first_page == 0) {
		first_page = 1;
		last_page = ps_pages;
	}

	if ((first_page > last_page) || (first_page > ps_pages)) {
		mlofile.close();
		return FALSE;
	}

	if (last_page > ps_pages)
		last_page = ps_pages;

	ps_pages = 0;
	size_y_buff = (int *)malloc((last_page - first_page + 1) * sizeof(int));
	for (page = first_page; page <= last_page; page++) {
		mlofile.readPageInfo( page, &info );

		/* Horizontal size of image (in pixel, 204 dpi) */
		size_x = info.width;
		/* Vertical size of image (in pixel, 98 or 196 lpi) */
		size_y = mlofile.pageUsedHeight( page );
		size_y_buff[page-first_page] = size_y;

		scale_x = size_x * 72 / 204;
		maxlines = (view_y * info.lpi) / 72;

		if (scale_x > view_x)
			maxlines = (maxlines * scale_x) / view_x;

		while (size_y > maxlines) {
			size_y -= maxlines;
			ps_pages++;
		}
		ps_pages++;
	}

	fprintf(ps, "%%!PS-Adobe-2.0\n");
	fprintf(ps, "%%%%BoundingBox: %d %d %d %d\n", trans_x, trans_y, trans_x + view_x, trans_y + view_y);
	fprintf(ps, "%%%%Creator: KMLOFax Version %s\n", VERSION);
	if (time(&t) != -1)
		fprintf(ps, "%%%%CreationDate: %s", ctime(&t));
	fprintf(ps, "%%%%DocumentData: Clean7Bit\n");
	fprintf(ps, "%%%%LanguageLevel: 2\n");
	fprintf(ps, "%%%%Orientation: Portrait\n");
	fprintf(ps, "%%%%Pages: %d\n", ps_pages);
	fprintf(ps, "%%%%PageOrder: Ascend\n");
	fprintf(ps, "%%%%Title: ");
	fprintf(ps, i18n("Facsimile from %s"), sender);
	fprintf(ps, "\n%%%%EndComments\n");
	fprintf(ps, "%%%%EndProlog\n");
	fprintf(ps, "%%%%BeginSetup\n");
	fprintf(ps, "/#copies %d def\n", doc_copies);
	fprintf(ps, "%%%%EndSetup\n");

	ps_pages = 0;
	for (page = first_page; page <= last_page; page++) {
		mlofile.readPageInfo( page, &info );

		size_x = info.width;
		size_y = size_y_buff[page-first_page];

		scale_x = (size_x * 72) / 204;
		scale_y = (size_y * 72) / info.lpi;
		maxlines = (view_y * info.lpi) / 72;

		if (scale_x > view_x) {
			scale_y = (scale_y * view_x) / scale_x;
			maxlines = (maxlines * scale_x) / view_x;
			scale_x = view_x;
		}

		mlofile.gotoPage(page);

		sub_page = 0;
		while (size_y > maxlines) {
			if (!convertPage( page, ++sub_page, ++ps_pages, maxlines, view_y )) {
				mlofile.close();
				return FALSE;
			}
			size_y -= maxlines;
			scale_y -= view_y;
		}
		if (!convertPage( page, ++sub_page, ++ps_pages, size_y, scale_y )) {
			mlofile.close();
			return FALSE;
		}
	}

	free(size_y_buff);

	fprintf(ps, "%%%%EOF\n");

	mlofile.close();

	return TRUE;
}


bool MLO2PSFilter::convertPage( int fax_page, int sub_page, int ps_page, int lines, int strech_y )
{
	int bpl, i;
	uchar in_buff[MAX_HUF_BUFF];
	
	if (fprintf(ps, "%%%%Page: %d.%d %d\n", fax_page, sub_page, ps_page) <= 0)
		return FALSE;

	fprintf(ps, "%%%%BeginPageSetup\n/pagelevel save def\n%%%%EndPageSetup\n");

	fprintf(ps, "%d %d translate\n", trans_x, trans_y + view_y - strech_y);
	fprintf(ps, "%d %d scale\n", scale_x, strech_y);

	fprintf(ps, "{\n");
	fprintf(ps, "/G3DICT <<\n");
	fprintf(ps, " /Columns %d\n", size_x);
	fprintf(ps, " /K 0\n");
	fprintf(ps, " /Uncompressed false\n");
	fprintf(ps, " /EndOfLine true\n");
	fprintf(ps, " /EndOfBlock true\n");
	fprintf(ps, " /BlackIs1 false\n");
	fprintf(ps, ">> def\n");
	fprintf(ps, "/Data1 currentfile /ASCII85Decode filter def\n");
	fprintf(ps, "/Data2 Data1 G3DICT /CCITTFaxDecode filter def\n");
	fprintf(ps, "<<\n");
	fprintf(ps, " /ImageType 1\n");
	fprintf(ps, " /Width %d\n", size_x);
	fprintf(ps, " /Height %d\n", lines);
	fprintf(ps, " /BitsPerComponent 1\n");
	fprintf(ps, " /Decode [ 0 1 ]\n");
	fprintf(ps, " /ImageMatrix [ %d 0 0 %d 0 %d ]\n", size_x, -lines, lines);
	fprintf(ps, " /Interpolate %s\n", interpolate ? "true" : "false");
	fprintf(ps, " /DataSource Data2\n");
	fprintf(ps, ">> image\n");
	fprintf(ps, "Data1 flushfile\n");
	fprintf(ps, "} exec\n");

	writeBase85( 0x00 );
	writeBase85( 0x01 );

	while (lines--) {
		bpl = mlofile.readHuffLine( in_buff );
		for (i=0; i < bpl; i++)
			writeBase85( in_buff[i] );
	}

	writeBase85( 0x00 ); /* Write End-Of-Block */
	writeBase85( 0x01 );
	writeBase85( 0x00 );
	writeBase85( 0x10 );
	writeBase85( 0x01 );
	writeBase85( 0x00 );
	writeBase85( 0x10 );
	writeBase85( 0x01 );

	flushBase85();
	flushImageData();
	resetImageData();

	fprintf(ps, "~>\n");
	fprintf(ps, "pagelevel restore\n");
	fprintf(ps, "showpage\n");

	return TRUE;
}


void MLO2PSFilter::writeBase85( uchar c )
{
	base85_buff[base85_ind] = c;

	if (base85_ind)
		base85_ind--;
	else
		flushBase85();
}


void MLO2PSFilter::flushBase85()
{
	static ulong power85[5] = { 85*85*85*85, 85*85*85, 85*85, 85, 1 };
	ulong v, c;
	int i;

	if (base85_ind == 3)
		return;

	v = *((ulong *)base85_buff);

	if (v)
		for (i=0; i < 5; i++) {
			c = v / power85[i];
			writeImageData('!' + (char)c);
			v -= c * power85[i];
		}
	else
		writeImageData('z');

	*((ulong *)base85_buff) = 0;
	base85_ind = 3;
}


void MLO2PSFilter::writeImageData( char c )
{
	image_buff[image_ind] = c;

	if (++image_ind == 253)
		flushImageData();
}


void MLO2PSFilter::flushImageData()
{
	if (!image_ind)
		return;

	image_buff[image_ind] = 0;
	
	if ((image_buff[0] == '%') && (image_buff[1] == '%'))
		fprintf(ps, "%% %s\n", &image_buff[1]);
	else
		fprintf(ps, "%s\n", image_buff);

	image_ind = 0;
}


void MLO2PSFilter::resetImageData()
{
	*((ulong *)base85_buff) = 0;
	base85_ind = 3;
	
	image_ind = 0;
}




PrintDlg::PrintDlg( QWidget * parent, const char * name ) : QDialog ( parent, name, TRUE )
{
	QBoxLayout * vbox, * hbox, * fvbox, * fhbox;
	QRadioButton * rb;
	QGroupBox * gb;
	QPushButton * ok, * cancel;
	QLabel * l1, * l2, * l3;
	QFrame * frame;
	QSize size;

	gotToolTips = FALSE;

	to_printer = TRUE;
	paper_num = 0;

	fname = 0;
	salias = 0;

	vbox = new QVBoxLayout( this , 8 );

	destination = new QButtonGroup( this );
	destination->setTitle( i18n("Print Destination") );
	fvbox = new QVBoxLayout( destination, 12 );
	fvbox->addSpacing( 8 );
	rb = new QRadioButton( i18n("Print to Printer:"), destination );
	rb->setMinimumSize( rb->sizeHint() );
	rb->setChecked( TRUE );
	fvbox->addWidget( rb );
	fhbox = new QHBoxLayout();
	fvbox->addLayout( fhbox );
	fhbox->addSpacing( 19 );
	printer = new QLineEdit( destination );
	printer->setMinimumWidth( printer->sizeHint().width() );
	printer->setFixedHeight( printer->sizeHint().height() );
	fhbox->addWidget( printer );
	rb = new QRadioButton( i18n("Print to File:"), destination );
	rb->setMinimumSize( rb->sizeHint() );
	fvbox->addWidget( rb );
	fhbox = new QHBoxLayout();
	fvbox->addLayout( fhbox );
	fhbox->addSpacing( 19 );
	file = new QLineEdit( destination );
	file->setMinimumWidth( file->sizeHint().width() );
	file->setFixedHeight( file->sizeHint().height() );
	file->setText( "facsimile.ps" );
	file->setEnabled( FALSE );
	
	fhbox->addWidget( file );
	fvbox->activate();
	vbox->addWidget( destination );

	hbox = new QHBoxLayout( 8 );
	vbox->addLayout( hbox );

	paper_format = new QButtonGroup( i18n("Paper Format"), this );
	fvbox = new QVBoxLayout( paper_format, 10 );
	fvbox->addSpacing( 8 );
	rb = new QRadioButton( i18n("A4 (210297 mm)"), paper_format );
	rb->setMinimumSize( rb->sizeHint() );
	rb->setChecked( TRUE );
	fvbox->addWidget( rb );
	rb = new QRadioButton( i18n("B4 (257364 mm)"), paper_format );
	rb->setMinimumSize( rb->sizeHint() );
	fvbox->addWidget( rb );
	rb = new QRadioButton( i18n("B5 (182257 mm)"), paper_format );
	rb->setMinimumSize( rb->sizeHint() );
	fvbox->addWidget( rb );
	rb = new QRadioButton( i18n("Letter (811 in.)"), paper_format );
	rb->setMinimumSize( rb->sizeHint() );
	fvbox->addWidget( rb );
	rb = new QRadioButton( i18n("Legal (814 in.)"), paper_format );
	rb->setMinimumSize( rb->sizeHint() );
	fvbox->addWidget( rb );
	rb = new QRadioButton( i18n("Executive (710 in.)"), paper_format );
	rb->setMinimumSize( rb->sizeHint() );
	fvbox->addWidget( rb );
	fvbox->activate();

	hbox->addWidget( paper_format );

	gb = new QGroupBox( i18n("Options"), this );
	fvbox = new QVBoxLayout( gb, 12 );
	fvbox->addSpacing( 8 );
	fhbox = new QHBoxLayout();
	fvbox->addLayout( fhbox );
	l1 = new QLabel( i18n("From Page"), gb );
	fhbox->addWidget( l1 );
	from_page = new QSpinBox( 1, 999, 1, gb );
	from_page->setValue( 1 );
	from_page->setMinimumWidth( from_page->sizeHint().width() );
	from_page->setFixedHeight( from_page->sizeHint().height() );
	fhbox->addWidget( from_page, 1 );
	fhbox = new QHBoxLayout();
	fvbox->addLayout( fhbox );
	l2 = new QLabel( i18n("To Page"), gb );
	fhbox->addWidget( l2 );
	to_page = new QSpinBox( 1, 999, 1, gb );
	to_page->setValue( 999 );
	to_page->setMinimumWidth( to_page->sizeHint().width() );
	to_page->setFixedHeight( to_page->sizeHint().height() );
	fhbox->addWidget( to_page, 1 );

	frame = new QFrame( gb );
	frame->setFrameStyle( QFrame::HLine | QFrame::Sunken );
	frame->setFixedHeight( 6 );
	fvbox->addWidget( frame, 1 );
	
	fhbox = new QHBoxLayout();
	fvbox->addLayout( fhbox );
	l3 = new QLabel( i18n("Number of Copies"), gb );
	fhbox->addWidget( l3 );
	copies = new QSpinBox( 1, 99, 1, gb );
	copies->setValue( 1 );
	copies->setMinimumWidth( copies->sizeHint().width() );
	copies->setFixedHeight( copies->sizeHint().height() );
	fhbox->addWidget( copies, 1 );

	frame = new QFrame( gb );
	frame->setFrameStyle( QFrame::HLine | QFrame::Sunken );
	frame->setFixedHeight( 6 );
	fvbox->addWidget( frame, 1 );
	
	interpolate = new QCheckBox( i18n("Interpolate"), gb );
	interpolate->setMinimumSize( interpolate->sizeHint() );
	fvbox->addWidget( interpolate );

	size = l1->sizeHint().expandedTo( l2->sizeHint() ).expandedTo( l3->sizeHint() );
	l1->setMinimumSize( size );
	l2->setMinimumSize( size );
	l3->setMinimumSize( size );

	fvbox->activate();

	hbox->addWidget( gb );

	hbox = new QHBoxLayout( 8 );
	vbox->addLayout( hbox );

	hbox->addStretch( 1 );

	ok = new QPushButton( i18n("OK"), this );
	ok->setDefault( true );
	cancel = new QPushButton( i18n("Cancel"), this );
	size = ok->sizeHint().expandedTo( cancel->sizeHint() );

	ok->setFixedSize( size );
	cancel->setFixedSize( size );

	hbox->addWidget( ok );
	hbox->addWidget( cancel );

	vbox->activate();

	printer->setText( prefs->prtCommand() );
	paper_format->setButton( prefs->prtPaper() );
	interpolate->setChecked( prefs->prtInterpolate() );

	connect( destination, SIGNAL(clicked(int)), SLOT(destinationSelected(int)) );
	connect( paper_format, SIGNAL(clicked(int)), SLOT(formatSelected(int)) );
	connect( ok, SIGNAL(clicked()), SLOT(okClicked()) );
	connect( cancel, SIGNAL(clicked()), SLOT(reject()) );

	setCaption( i18n("KMLOFax Print") );

	resize( sizeHint() );
}


PrintDlg::~PrintDlg()
{
	if (fname)
		free(fname);

	if (salias)
		free(salias);
}


void PrintDlg::addToolTips( bool on )
{
	if (on == gotToolTips)
		return;

	gotToolTips = on;

	if (gotToolTips) {
		QToolTip::add( printer, i18n("Command to invoke for printing") );
		QToolTip::add( file, i18n("Name of file to print to") );
		QToolTip::add( paper_format, i18n("Paper format used by the printer") );
		QToolTip::add( from_page, i18n("First page to print") );
		QToolTip::add( to_page, i18n("Last page to print") );
		QToolTip::add( copies, i18n("Number of copies to print") );
		QToolTip::add( interpolate, i18n("Interpolate for smoother output") );
	}
	else {
		QToolTip::remove( printer );
		QToolTip::remove( file );
		QToolTip::remove( paper_format );
		QToolTip::remove( from_page );
		QToolTip::remove( to_page );
		QToolTip::remove( copies );
		QToolTip::remove( interpolate );
	}
}	


void PrintDlg::printFax( const char * name, const char * alias )
{
	MLOFile mlofile;

	if (fname)
		free(fname);
	fname = strdup(name);

	if (salias)
		free(salias);
	salias = strdup(alias);

	mlofile.setName( expand_path(name) );

	if (!mlofile.open()) {
		KMsgBox::message( 0, i18n("File Error"), i18n("Cannot open facsimile file."), KMsgBox::EXCLAMATION);
		return;
	}

	from_page->setRange( 1, mlofile.pages() );
	from_page->setValue( 1 );
	to_page->setRange( 1, mlofile.pages() );
	to_page->setValue( mlofile.pages() );
	mlofile.close();

	if (isVisible())
		raise();
	else
		show();
}


void PrintDlg::okClicked()
{
	MLO2PSFilter * filter;
	FILE * ps;

	filter = new MLO2PSFilter( expand_path(fname) );
	filter->setFormat( paper_num );
	filter->setInterpolate( interpolate->isChecked() );
	filter->setRange( from_page->value(), to_page->value() );
	filter->setCopies( copies->value() );
	filter->setMargins( prefs->prtLeftMargin(), prefs->prtRightMargin(), prefs->prtTopMargin(), prefs->prtBottomMargin() );

	mykapp->setOverrideCursor( waitCursor );
	mykapp->processEvents( 100 );

	if (to_printer) {
		if ((ps = popen(printer->text(), "w")) <= 0) {
			KMsgBox::message( 0, i18n("Printer Error"), i18n("Cannot execute print command."), KMsgBox::EXCLAMATION);
		} else {
			filter->convertFile( ps, salias );
			pclose(ps);
		}
	} else {
		if ((ps = fopen(file->text(), "w")) == 0) {
			KMsgBox::message( 0, i18n("Printer Error"), i18n("Cannot create file for output."), KMsgBox::EXCLAMATION);
		} else {
			filter->convertFile( ps, salias );
			fclose(ps);
		}
	}		

	mykapp->restoreOverrideCursor();

	delete filter;

	hide();
}


void PrintDlg::destinationSelected( int id )
{
	to_printer = (id == 0);
	printer->setEnabled ( to_printer );
	file->setEnabled ( !to_printer );
}


void PrintDlg::formatSelected( int id )
{
	paper_num = id;
}
