/*
	This file is part of `klp', a KDE Line Printer queue manager

	Copyright (C) 1998
	Frans van Dorsselaer
	<dorssel@MolPhys.LeidenUniv.nl>

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <stdlib.h>

#include <qpushbutton.h>
#include <qlistbox.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qradiobutton.h>
#include <qbuttongroup.h>
#include <qcombobox.h>

#include <kapp.h>

#include "add_edit.h"
#include "common.h"
#include "inter.h"


const Printer empty_printer = { 0, 0, 0, true, 0 };


AddEditDialog::AddEditDialog(QWidget *parent, const char *n) : QDialog(parent, n, true)
{
	setCaption(i18n("Add printer"));
	edit = false;
	printer = empty_printer;
	setupWindow();
}


AddEditDialog::AddEditDialog(const Printer &p, QWidget *parent, const char *n) : QDialog(parent, n, true)
{
	setCaption(i18n("Edit printer"));
	edit = true;
	printer = p;
	setupWindow();
}


void AddEditDialog::setupWindow()
{
	description = new QLineEdit(this);
	description->setText(printer.description);
	descr_label = new QLabel(description, i18n("&Description"), this);

	server_type = new QComboBox(this);
	server_label = new QLabel(server_type, i18n("&Server type"), this);

	for (int i = 0 ; i < SERVER_COUNT ; ++i)
		server_type->insertItem(i18n(servers[i].description));

	name = new QLineEdit(this);
	name_label = new QLabel(name, i18n("&Printer name"), this);

	name_combo = new QComboBox(this);
	name_combo->hide();
	have_list = false;

	group = new QButtonGroup(i18n("Connection"), this);
	connect(group, SIGNAL(clicked(int)), SLOT(radioClicked()));
	local = new QRadioButton(i18n("&Local"), group);
	remote = new QRadioButton(i18n("&Remote"), group);

	host = new QLineEdit(this);

	ok = new QPushButton(i18n("Ok"), this);
	connect(ok, SIGNAL(clicked()), SLOT(accept()));

	cancel = new QPushButton(i18n("Cancel"), this);
	connect(cancel, SIGNAL(clicked()), SLOT(reject()));

	server_type->setCurrentItem(printer.type);
	serverActivated(printer.type);

	if (have_list) {
		for (int i = 0 ; i < name_combo->count() ; ++i)
			if (printer.name == name_combo->text(i)) {
				name_combo->setCurrentItem(i);
				break;
			}
	} else
		name->setText(printer.name);

	host->setText(printer.host);

	if (printer.local) {
		local->setChecked(true);
		host->setEnabled(false);
	} else {
		remote->setChecked(true);
		host->setEnabled(true);
	}

	setMinimumWidth(320);
	validate();
	description->selectAll();
	description->setFocus();
}


void AddEditDialog::resizeEvent(QResizeEvent *event)
{
	descr_label->move(10, 10);
	descr_label->adjustSize();

	server_label->move(10, 30 + descr_label->height());
	server_label->adjustSize();

	name_label->move(10, 20 + server_label->y() + server_label->height());
	name_label->adjustSize();

	int x = descr_label->width();
	if (name_label->width() > x)
		x = name_label->width();
	if (server_label->width() > x)
		x = server_label->width();
	x += 20;
	description->move(x, descr_label->y());
	description->adjustSize();
	description->resize(event->size().width() - x - 10, description->height());
	name->move(x, name_label->y());
	name->adjustSize();
	name->resize(event->size().width() - x - 10, name->height());
	name_combo->move(name->x(), name->y());
	name_combo->adjustSize();
	name_combo->resize(name->width(), name_combo->height());
	server_type->move(x, server_label->y());
	server_type->adjustSize();
	server_type->resize(event->size().width() - x - 10, server_type->height());

	group->move(10, name->y() + name->height() + 20);
	local->move(20,30);
	local->adjustSize();
	remote->move(20, 20 + local->height() + 20);
	remote->adjustSize();
	host->move(40 + remote->width(), group->y() + remote->y());
	host->adjustSize();
	host->resize(event->size().width() - 30 - host->x(), host->height());
	group->resize(event->size().width() - 20, remote->y() + remote->height() + 20);

	ok->setGeometry(40, group->y() + group->height() + 20, 100, 30);
	cancel->setGeometry(event->size().width() - 140, ok->y(), 100, 30);

	connect(group, SIGNAL(clicked(int)), SLOT(validate()));
	connect(description, SIGNAL(textChanged(const char *)), SLOT(validate()));
	connect(server_type, SIGNAL(activated(int)), SLOT(serverActivated(int)));
	connect(name, SIGNAL(textChanged(const char *)), SLOT(validate()));
	connect(host, SIGNAL(textChanged(const char *)), SLOT(validate()));

	setFixedHeight(ok->y() + 40);
}


void AddEditDialog::radioClicked()
{
	if (local->isChecked())
		host->setEnabled(false);
	else {
		host->setEnabled(true);
		host->setFocus();
	}
	validate();
}


void AddEditDialog::serverActivated(int i)
{
	local->setEnabled(servers[i].local);
	remote->setEnabled(servers[i].remote);
	host->setEnabled(servers[i].remote);
	if (!servers[i].local)
		remote->setChecked(true);
	else if (!servers[i].remote)
		local->setChecked(true);
	char **printers = NULL;
	if (servers[i].printer_list && (printers = servers[i].printer_list())) {
		char **p = printers;
		name->hide();
		name->clear();
		name_combo->clear();
		while (*p) {
			name_combo->insertItem(*p);
			free(*p);
			++p;
		}
		free(printers);
		name_combo->show();
		have_list = true;
	} else {
		name_combo->hide();
		name_combo->clear();
		name->show();
		have_list = false;
	}
	validate();
}


void AddEditDialog::validate()
{
	QString s;
	s = description->text();
	s.stripWhiteSpace();
	if (s.isEmpty())
		goto not_ok;

	if (have_list) {
		if (name_combo->currentItem() < 0)
			goto not_ok;
	} else {
		s = name->text();
		s.stripWhiteSpace();
		if (s.isEmpty())
			goto not_ok;
	}
	if (remote->isChecked()) {
		s = host->text();
		s.stripWhiteSpace();
		if (s.isEmpty())
			goto not_ok;
	}

	ok->setEnabled(true);
	ok->setDefault(true);
	return;

not_ok:
	ok->setEnabled(false);
	ok->setDefault(false);
}


void AddEditDialog::done(int r)
{
	if (r == Accepted) {
		printer.description = description->text();
		printer.description.stripWhiteSpace();
		printer.type = server_type->currentItem();
		if (have_list)
			printer.name = name_combo->currentText();
		else
			printer.name = name->text();
		printer.name.stripWhiteSpace();
		printer.host = host->text();
		printer.host.stripWhiteSpace();
		printer.local = local->isChecked();
	}
	QDialog::done(r);
}


#include "add_edit.moc"
