/*
 *            klm: A lm_sensors front end for the KDE project
 *
 * $Id: VertGraph.cpp,v 1.32 1999/01/25 21:29:25 humphrey Exp $
 *
 *            Copyright (C) 1998 Brendon Humphrey
 *                   brendy@swipnet.se
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <math.h>

#include <qframe.h>
#include <qpixmap.h>
#include <qlayout.h>
#include <qpainter.h>
#include <qrect.h>
#include <qfont.h>
#include <qbrush.h>
#include <qdrawutil.h>
#include <kmsgbox.h>
#include <qmsgbox.h>
#include <kapp.h>
#include "VertGraph.h"
#include "Sensors.h"

VertGraph::VertGraph( 
  QWidget* parent, 
  const char* name,
  double min_val,
  double max_val,
  bool lowAlarmSettable,
  bool highAlarmSettable,
  bool scalable,
  bool hystGraph	) : QFrame( parent, name )
{	
  setGeometry( 5, 5, 30, 150 );
  setMinimumSize( 30, 150 );
  setFrameStyle( QFrame::WinPanel | QFrame::Sunken );
  setLineWidth( 2 );
  
  // create colors and color groups	
  dark_red = red.dark(200);
  dark_green = green.dark(200);
  dark_yellow = yellow.dark(200);
  
  redCg = new QColorGroup(
      black,
      white,
      dark_red.dark(200),
      dark_red.light(),
      dark_red,
      black,
      white );
  
  greenCg = new QColorGroup(
      black,
      white,
      dark_green.dark(200),
      dark_green.light(),
      dark_green,
      black,
      white );
  
  // initialise class data
  isHystGraph = hystGraph;
  data=0;
  last_data=LM_NO_DATA;
  max=max_val;
  min=min_val;
  configuring = false;	
  selItem = NONE;
  lowAlarmIsSettable=lowAlarmSettable;
  highAlarmIsSettable=highAlarmSettable;
  isScalable=scalable;
}

VertGraph::~VertGraph()
{
  delete redCg;
  delete greenCg;
}

void VertGraph::plot( 
  double GrMin, 
  double GrMax, 
  double lwAlarm, 
  double hiAlarm, 
  double value,
  bool sensorAlarm	)
{	
  min = GrMin;
  max = GrMax;
  alarmSounding = sensorAlarm;
  
  low_alarm = fabs(lwAlarm);
  high_alarm = fabs(hiAlarm);
  checkMinMax( min, max );
  checkMinMax( low_alarm, high_alarm );
  
  if (low_alarm < min )
  {
    low_alarm = min;
  }
  
  if (high_alarm > max )
  {
    high_alarm = max;
  }
  
  last_data = data;
  data = fabs( value );
  
  repaint(2,2,width()-4, height()-4, false);	
}

void VertGraph::paintEvent( QPaintEvent *e )
{
  if( isHystGraph )
  {
    drawHystGraph();	
  }
  else
  {
    drawNormalGraph();
  }
}

void VertGraph::drawHystGraph()
{
  QString s;
  QPainter p;
  QPixmap pm(width() - 4, height() - 4);
  QBrush b;
  QPen pen;
  int y1, dy;
  
  pm.fill( white );
  p.begin( &pm );
  
  // top (!ok zone)
  b.setColor(dark_red);
  b.setStyle(Dense5Pattern);
  
  y1 = dataToY( pm.height(), max, min, high_alarm );
  dy	= dataToY( pm.height(), max, min, low_alarm );
  
  low_alarm_min = dy;
  high_alarm_min = y1;
  
  p.fillRect(
    0, 
    0,
    pm.width(),
    pm.height() - y1,
    b);
  
  // bottom (ok zone)
  b.setColor(dark_green);
  b.setStyle(Dense5Pattern);
  
  p.fillRect(
    0, 
    pm.height() - dy,
    pm.width(),
    dy,
    b);
  
  // middle (hysteresis zone)
  b.setColor(dark_yellow);
  b.setStyle(Dense5Pattern);
  
  p.fillRect(
    0, 
    pm.height() - y1,
    pm.width(),
    y1 - dy,
    b);
  
  // if	configuring, plot bars at top and bottom of ok zone
  if (configuring)
  {		
    b.setColor(dark_red);
    b.setStyle(SolidPattern);
    
    // draw lower drag bar
    if (lowAlarmIsSettable)
    {
      qDrawShadePanel( 
        &p, 
        0, pm.height()-dy-2,
        pm.width(), 4,
        *redCg,
        true,
        1, &b);
    }
    
    // draw upper drag bar			
    if(highAlarmIsSettable)
    {	
      qDrawShadePanel( 
        &p, 
        0, pm.height()-y1,
        pm.width(), 4,
        *redCg,
        true,
        1, &b);
    }	
  }
  
  
  // plot reading value	
  if ( data )
  {
    if ( alarmSounding )
    {
      b.setColor(dark_red);
      b.setStyle(SolidPattern);
    }
    else
    {						
      b.setColor(dark_green);
      b.setStyle(SolidPattern);
    }
    
    y1 = dataToY( pm.height(), max, min, data );
    
    qDrawShadePanel( 
      &p, 
      0, pm.height()-y1,
      configuring ? pm.width()/2 : pm.width(), y1,
      alarmSounding ? *redCg : *greenCg,
      true,
      2, &b);		
  }
  
  bitBlt(this, 2, 2, &pm, 0, 0, pm.width(), pm.height(), CopyROP);
  p.end();
  
  p.begin(this);	
  drawFrame( &p );
  p.end();		
}

void VertGraph::drawNormalGraph()
{
  QString s;
  QPainter p;
  QPixmap pm(width() - 4, height() - 4);
  QBrush b;
  QPen pen;
  int y1, dy;
  
  pm.fill( white );
  p.begin( &pm );
  
  // top (!ok zone)
  b.setColor(dark_red);
  b.setStyle(Dense5Pattern);
  
  y1 = dataToY( pm.height(), max, min, high_alarm );
  dy	= dataToY( pm.height(), max, min, low_alarm );
  
  low_alarm_min = dy;
  high_alarm_min = y1;
  
  p.fillRect(
    0, 
    pm.height() - dy,
    pm.width(),
    dy,
    b);
  
  p.fillRect(
    0, 
    0,
    pm.width(),
    pm.height() - y1,
    b);
  
  // middle (ok zone)
  b.setColor(dark_green);
  b.setStyle(Dense5Pattern); //Dense6Pattern);
  
  p.fillRect(
    0, 
    pm.height() - y1,
    pm.width(),
    y1 - dy,
    b);
  
  // if configuring, plot bars at top and bottom of ok zone
  if (configuring)
  {
    b.setColor(dark_red);
    b.setStyle(SolidPattern);
    
    // draw lower drag bar
    if (lowAlarmIsSettable)
    {
      qDrawShadePanel( 
        &p, 
        0, pm.height()-dy-2,
        pm.width(), 4,
        *redCg,
        true,
        1, &b);
    }
    
    // draw upper drag bar			
    if(highAlarmIsSettable)
    {	
      qDrawShadePanel( 
        &p, 
        0, pm.height()-y1,
        pm.width(), 4,
        *redCg,
        true,
        1, &b);
    }	
  }
  
  
  // plot reading value	
  if ( data )
  {
    if ( alarmSounding )
    {
      b.setColor(dark_red);
      b.setStyle(SolidPattern);
    }
    else
    {			
      b.setColor(dark_green);
      b.setStyle(SolidPattern);
    }
    
    y1 = dataToY( pm.height(), max, min, data );
    
    qDrawShadePanel( 
      &p, 
      0, pm.height()-y1,
      configuring ? pm.width()/2 : pm.width(), y1,
      alarmSounding ? *redCg : *greenCg,
      true,
      2, &b);
  }
  
  bitBlt(this, 2, 2, &pm, 0, 0, pm.width(), pm.height(), CopyROP);
  p.end();
  
  p.begin(this);	
  drawFrame( &p );
  p.end();	
  
}

int VertGraph::dataToY( int graph_height, double max, double min, double value )
{
  int y;
  
  if((value < min) || (value>max))
  {
    value = 0;
  }		
  
  y = (int)((float)(graph_height*value)/max);
  
  return (y);
}

double VertGraph::yToData( int graph_height, double max, int y )
{
  return (((float)(graph_height) - y)*max)/graph_height;	
}

void VertGraph::mousePressEvent( QMouseEvent *e ) 
{
  switch( e->button() )
  {
    case LeftButton: // may have started dragging a limit marker.
      selItem = whatClicked(e->y());
      break;
    case MidButton: // entered/left config mode 
      emit configurationRequest();
      break;
    case RightButton:
    default:
      QFrame::mousePressEvent( e );
      break;
  }
}

void VertGraph::configureYourself( bool config ) 
{
  configuring = config;
  repaint(2, 2, width(), height(), false);	
}

void VertGraph::checkMinMax( double &min, double &max)
{
  double tmp;
  
  if (min > max)
  {
    tmp=min;
    min=max;
    max=tmp;
  }
}

int VertGraph::whatClicked( int y )
{
  
  if (!configuring) 
  {
    return NONE;
  }
  else
  {		
    y = height() - y;
    
    if ( lowAlarmIsSettable && 
        (y>=low_alarm_min-4) && 
        (y<=low_alarm_min+4) )
    {
      return LOWER;
    }
    
    if ( highAlarmIsSettable &&
        (y>=high_alarm_min-4) && 
        (y<=high_alarm_min+4) )
    {
      return UPPER;
    }
    
    return NONE;
  }
}

void VertGraph::forceInRange( double &val )
{
  if (val > max)
  {
    val = max;
  }
  
  if (val < min)
  {
    val = min;
  }
}


void VertGraph::mouseMoveEvent( QMouseEvent *e )
{
  switch ( selItem )
  {
    case LOWER:
      low_alarm = yToData( height(), max, e->y() );
      forceInRange( low_alarm );
      checkMinMax( low_alarm, high_alarm );			
      emit newLimits( low_alarm, high_alarm );
      repaint(2,2,width(), height(), false);				
      break;
      
    case UPPER:
      high_alarm = yToData( height(), max, e->y() );
      forceInRange( high_alarm );
      checkMinMax( low_alarm, high_alarm );			
      emit newLimits( low_alarm, high_alarm );
      repaint(2,2,width(), height(), false);				
      break;
      
    case NONE:
    default:
      break;			
  }
}

void VertGraph::mouseReleaseEvent( QMouseEvent *e )
{
  selItem = NONE;
  emit mouseReleased();
}

void VertGraph::getConfig( 
  double &minVal, 	
  double &maxVal,
  double &minAlarmVal,
  double &maxAlarmVal )
{
  minVal = min;
  maxVal = max;
  minAlarmVal = low_alarm;
  maxAlarmVal = high_alarm;	
}

#include <VertGraph.moc.cpp>


