/*
 *            klm: A lm_sensors front end for the KDE project
 *
 * $Id: Sensors.cpp,v 1.1 1999/01/30 19:48:50 humphrey Exp $
 *
 *            Copyright (C) 1998 Brendon Humphrey
 *                   brendy@swipnet.se
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "qregexp.h"
#include "Sensors.h"

// definition of keys within the config file
#define MAX "_MAX"
#define MIN "_MIN"
#define LOW_ALARM "_LOW_ALARM"
#define HIGH_ALARM "_HIGH_ALARM"
#define NAME "_NAME"
#define SCALE "_SCALE"
#define DISABLED "_DISABLED"
#define OFFSET "_OFFSET"


Sensor::Sensor( 
  int sysc1, 
  int sysc2, 
  int sysc3,
  int sysc4,	
  double grMin, 
  double grMax)
{
  sysctl_name[0] = sysc1;
  sysctl_name[1] = sysc2;
  sysctl_name[2] = sysc3;	
  sysctl_name[3] = sysc4;
  graph_min = grMin;
  graph_max = grMax;
  configurable = true;
}

Sensor::~Sensor() 
{
}

bool Sensor::testAlarm( 
  int sysctl_addr2, 
  int sysctl_addr3, 
  long alarmMask )
{
  int address[4] = { CTL_DEV, DEV_SENSORS, sysctl_addr2, sysctl_addr3 };
  long alarms;
  
  size_t bufferSize = (size_t)sizeof(alarms);
  
  if	(sysctl( address, 4, (void *)&alarms, &bufferSize, (void*)0, 0)==0) 
  {	
    printf("sysctl(): %s in %s (%d)\n",strerror(errno), __FILE__, __LINE__);		
    return false;
  }
  else
  {
    printf("sysctl(): %s in %s (%d)\n",strerror(errno), __FILE__, __LINE__);		
    return (bool)(alarms & alarmMask ); 	
  }
}

bool Sensor::getVolt( SensorValue &sensor )
{
  long val[3];
  size_t len = sizeof(val);
  
  if (sysctl( sysctl_name, 4, (void *)val, &len, 0, 0)) 
  {		
    printf("sysctl(): %s in %s (%d)\n",strerror(errno), __FILE__, __LINE__);		
    sensor.min = LM_NO_DATA;
    sensor.value = LM_NO_DATA;
    sensor.max = LM_NO_DATA;
    
    return (false);
  }
  else 
  {	
    sensor.min = (double)(val[0]); 
    sensor.max = (double)(val[1]); 
    sensor.value = (double)(val[2]);
    
    return (true);	
  }
}

bool Sensor::getVidVolt( SensorValue &sensor )
{
  long val[1];
  size_t len = sizeof(val);
  
  if (sysctl( sysctl_name, 4, (void *)val, &len, 0, 0)) 
  {		
    printf("sysctl(): %s in %s (%d)\n",strerror(errno), __FILE__, __LINE__);		
    sensor.min = LM_NO_DATA;
    sensor.value = LM_NO_DATA;
    sensor.max = LM_NO_DATA;
    
    return (false);
  }
  else 
  {	
    sensor.min = LM_NO_DATA;
    sensor.max = LM_NO_DATA;
    sensor.value = (double)(val[0]);
    
    return (true);	
  }
}

bool Sensor::getFan( SensorValue &sensor )
{
  long val[2];
  size_t len = sizeof(val);
  
  if (sysctl( sysctl_name, 4, (void *)val, &len, 0, 0)) 
  {		
    printf("sysctl(): %s in %s (%d)\n",strerror(errno), __FILE__, __LINE__);		
    sensor.min = LM_NO_DATA;
    sensor.value = LM_NO_DATA;
    sensor.max = LM_NO_DATA;
    
    return (false);
  }
  else 
  {	
    sensor.min = val[0]; 
    sensor.max = graph_max;
    sensor.value = val[1];  
    
    return ( true );
  }
}

bool Sensor::getTemp( SensorValue &sensor )
{
  long val[3];
  size_t len = sizeof(val);
  
  if (sysctl( sysctl_name, 4, (void *)val, &len, 0, 0)) 
  {		
    printf("sysctl(): %s in %s (%d)\n",strerror(errno), __FILE__, __LINE__);		
    sensor.min = LM_NO_DATA;
    sensor.max = LM_NO_DATA;
    sensor.value = LM_NO_DATA;
    
    return (false);
  }
  else 
  {	
    sensor.min = val[0]; 
    sensor.max = val[1];
    sensor.value = val[2]; 
    
    return (true);
  }
}

bool Sensor::writeVolt( SensorValue &sensor )
{
  long val[3];
  size_t len = sizeof(val);
  
  val[0]=(long)(sensor.min);
  val[1]=(long)(sensor.max);
  val[2]=(long)(sensor.value);
  
  if (sysctl( sysctl_name, 4, 0, 0, (void *)val, len)) 
  {	
    printf("sysctl(): %s in %s (%d)\n",strerror(errno), __FILE__, __LINE__);		
    configurable = false;
    return (false);
  }
  else 
  {	
    return (true);
  }
}

bool Sensor::writeFan( SensorValue &sensor )
{
  long val[1];
  size_t len = sizeof(val);
  
  val[0]=(long)(sensor.min);
  
  if (sysctl( sysctl_name, 4, 0, 0, val, len) ) 
  {		
    printf("sysctl(): %s in %s (%d)\n",strerror(errno), __FILE__, __LINE__);		
    configurable = false;
    return (false);
  }
  else 
  {	
    return (true);
  }
}

bool Sensor::writeTemp( SensorValue &sensor )
{
  long val[3];
  size_t len = sizeof(val);
  
  val[0]=(long)(sensor.min);
  val[1]=(long)(sensor.max);
  val[2]=(long)(sensor.value);
  
  if (sysctl( sysctl_name, 4, 0, 0, (void *)val, len)) 
  {		
    printf("sysctl(): %s in %s (%d)\n",strerror(errno), __FILE__, __LINE__);		
    configurable = false;
    return (false);
  }
  else 
  {	
    return (true);
  }
}

bool Sensor::read( SensorValue &val )
{
  return false;
}

bool Sensor::write( SensorValue &val )
{
  return false;
}

double Sensor::scale()
{
  return (double)0;
}

bool Sensor::setScale( double &scaleFactor )
{
  return false;
}

int * Sensor::getAddr()
{
  return sysctl_name;
}

void Sensor::getScaleFactor( 
  int sysctl_addr3, 
  int &numScales, 
  int bufferLen, 
  long *scaleBuffer )
{
  int address[4] = { CTL_DEV, DEV_SENSORS, sysctl_name[2], sysctl_addr3 };
  size_t bufferSize = (size_t)bufferLen;
  
  if	(!sysctl( address, 4, scaleBuffer, &bufferSize, 0, 0)) 
  {	
    numScales = bufferSize/sizeof(long);
  }
  else
  {
    numScales = 0;
  }
}

bool Sensor::setScaleFactor( 
  int sysctl_addr3, 
  int offset, 
  double factor )
{
  long scaleBuffer[20];
  int numNames;
  int address[4] = { CTL_DEV, DEV_SENSORS, sysctl_name[2], sysctl_addr3 };
  size_t bufferSize;
  
  
  // read all of the labels into a buffer and tokenise
  getScaleFactor( 
    sysctl_addr3, 
    numNames, 
    sizeof(scaleBuffer),
    scaleBuffer );
  
  // substitute the new scale factor into the array.	
  scaleBuffer[sysctl_name[3]-offset] = (long)factor;
  
  // write to lm_sensors
  bufferSize = (size_t)numNames*sizeof(long);
  
  if( sysctl( address, 4, 0, 0, scaleBuffer, bufferSize) != 0 )
  {
    configurable = false;
    return false;
  }
  else
  {
    return true;
  }
}

bool Sensor::increaseScale()
{
  return false;
}

bool Sensor::decreaseScale()
{
  return false;
}

bool Sensor::increaseGraphRange()
{
  return false;
}

bool Sensor::decreaseGraphRange()
{
  return false;
}

bool Sensor::setGraphRange( double min, double max )
{
  graph_min = min;
  graph_max = max;
  
  return true;
}

bool Sensor::isTemp( )
{
  return false;
}

void Sensor::name( QString &name )
{
  name="";
}

bool Sensor::setName( QString &name )
{
  return false;
}

void Sensor::graphRange( double &min, double &max )
{
  min = graph_min;
  max = graph_max;
}

bool Sensor::lowAlarmSettable()
{
  return true;
}

bool Sensor::highAlarmSettable()
{
  return true;
}

bool Sensor::scalable()
{
  return true;
}

bool Sensor::nameSettable()
{
  return true;
}

bool Sensor::displayDecimals()
{
  return true;
}

bool Sensor::displayUnits()
{
  return true;
}

void Sensor::units( QString &unitStr )
{
  unitStr="";
}

bool Sensor::save()
{
  return false;
}

bool Sensor::saveSettings( SensorValue &val, const QString &name, double scale )
{
  QString group;
  KConfig *config;
  
  // gain access to the applications KConfig object.
  config = kapp->getConfig();
  
  if( config == (KConfig *)NULL )
  {
    return false;
  }
  
  // switch to a group based on the sensors identity.
  group.sprintf( 
    "klm-%d-%d-%d-%d", 
    sysctl_name[0],
    sysctl_name[1],
    sysctl_name[2], 
    sysctl_name[3] );
  
  config->setGroup( group );
  
  // write the configuration data.
  config->writeEntry( MAX, graph_max );
  config->writeEntry( MIN, graph_min );
  config->writeEntry( LOW_ALARM, val.min );
  config->writeEntry( HIGH_ALARM, val.max );
  config->writeEntry( SCALE, scale );	
  config->writeEntry( DISABLED, disabled );
  config->writeEntry( NAME, name );
  config->writeEntry( OFFSET, sensorOffset );
  
  return true;
}

bool Sensor::getSettings( SensorValue &val, QString &name, double &scale )
{
  QString group;
  KConfig *config;
  double tmp_max;
  double tmp_min;
  
  // gain access to the applications KConfig object.
  config = kapp->getConfig();
  
  // switch to a group based on the sensors identity.
  group.sprintf( 
    "klm-%d-%d-%d-%d", 
    sysctl_name[0],
    sysctl_name[1],
    sysctl_name[2],
    sysctl_name[3]);
  
  config->setGroup( group );
  
  // read values for the sensor
  tmp_max	= config->readDoubleNumEntry( MAX, LM_NO_DATA );
  tmp_min	= config->readDoubleNumEntry( MIN, LM_NO_DATA );
  val.min = config->readDoubleNumEntry( LOW_ALARM, LM_NO_DATA );
  val.max = config->readDoubleNumEntry( HIGH_ALARM, LM_NO_DATA );	
  scale = config->readDoubleNumEntry( SCALE, LM_NO_DATA );
  sensorOffset = config->readDoubleNumEntry( OFFSET, 0 );	
  disabled = config->readBoolEntry( DISABLED, false );	
  name = config->readEntry( NAME, "" );
  
  val.value = 0;	
  
  if ( 
      ( tmp_max == LM_NO_DATA ) ||
      ( tmp_min == LM_NO_DATA ) ||
      ( val.min == LM_NO_DATA ) ||
      ( val.max == LM_NO_DATA ) ||
      ( scale == LM_NO_DATA ) ||
      ( name.length() == 0 ) )
  {
    return false;
  }
  else
  {
    val.value = 0;	
    graph_min = tmp_min;
    graph_max = tmp_max;		
    return true;
  }
}

bool Sensor::isDisabled()
{
  return (disabled);
}

void Sensor::setDisabled( bool d )
{
  disabled = d;
}

bool Sensor::isConfigurable()
{
  return configurable;
}

bool Sensor::isAlarm()
{
  return false;
}

bool Sensor::isOsHystTemp()
{
  return false;
}

void Sensor::setConfigurable( bool cfg )
{
  configurable = cfg;
}

bool Sensor::increaseOffset()
{
  return false;
}

bool Sensor::decreaseOffset()
{
  return false;
}

double Sensor::getOffset()
{
  return sensorOffset;
}

bool Sensor::setOffset( double offset )
{
  sensorOffset = offset;
  return true;
}

