/*
 * klistedit.cpp
 *
 * Copyright (c) 1998 Michael Kropfberger <mkropfbe@edu.uni-klu.ac.at>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <qtooltip.h>

#include <kstdaccel.h>
#include <kiconloader.h>
#include <kapp.h>
#include "klocale.h"

#include "klistedit.h"
#include "klistedit.moc"

#define BORDER_SIZE 5
#define ELEM_HEIGHT 24
#define ICON_HEIGHT 24
#define SEPARATOR 6

/************************************************************************
* real Constructor-init
*/
void KStrListEdit::init() 
{
 elems = new QStrList();
 elems->append("");
 actIndex=0;

 stdAcc = new KStdAccel(kapp->getConfig() );
 qaccel = new QAccel(this);

 upKey=qaccel->insertItem(qaccel->insertItem( Key_Up ) );
 upKey=qaccel->findKey(Key_Up);
 qaccel->insertItem(qaccel->insertItem( Key_Down ) );
 downKey=qaccel->findKey(Key_Down);
 qaccel->insertItem(qaccel->insertItem( stdAcc->prior() ) );
 topKey=qaccel->findKey( stdAcc->prior() );
 qaccel->insertItem( qaccel->insertItem( stdAcc->next()) );
 bottomKey=qaccel->findKey( stdAcc->next() );
 connect(qaccel,SIGNAL(activated(int)),this,SLOT(accKeyActivatedSlot(int)));

 connect(this, SIGNAL( buttonUp() ), this, SLOT( upSlot()) );
 connect(this, SIGNAL( buttonDown() ), this, SLOT( downSlot()) );
 connect(this, SIGNAL( buttonTop() ), this, SLOT( topSlot()) );
 connect(this, SIGNAL( buttonBottom() ), this, SLOT( bottomSlot()) );
 connect(this, SIGNAL( buttonNewEntry() ), this, SLOT( newSlot()) );
 connect(this, SIGNAL( buttonDelEntry() ), this, SLOT( delSlot()) );
 connect(this, SIGNAL( returnPressed() ), this, SLOT( returnPressedSlot() ) );

 upperElem = new QLabel(this); CHECK_PTR(upperElem);
 upperElem->setFrameStyle( QFrame::Panel | QFrame::Sunken );

 actElemBox = new QGroupBox("unknown Title",this); CHECK_PTR(actElemBox);
 actElemBox->setAlignment(AlignHCenter);
 actElem = new QLineEdit(this); CHECK_PTR(actElem);
 connect(actElem,SIGNAL(returnPressed()),this,SIGNAL(returnPressed()) );

 lowerElem = new QLabel(this); CHECK_PTR(lowerElem);
 lowerElem->setFrameStyle( QFrame::Panel | QFrame::Sunken );

 KIconLoader *loader = kapp->getIconLoader();

 upBtn = new QPushButton("up",this); CHECK_PTR(upBtn);
   connect(upBtn,SIGNAL(clicked()),this,SIGNAL(buttonUp()) );
   QToolTip::add(upBtn,klocale->translate("go one line up") );
   upBtn->setAutoRepeat(TRUE);
 downBtn = new QPushButton("down",this); CHECK_PTR(downBtn);
   connect(downBtn,SIGNAL(clicked()),this,SIGNAL(buttonDown()) );
   QToolTip::add(downBtn,klocale->translate("go one line down") );
   downBtn->setAutoRepeat(TRUE);
 topBtn = new QPushButton("top",this); CHECK_PTR(topBtn);
   connect(topBtn,SIGNAL(clicked()),this,SIGNAL(buttonTop()) );
   QToolTip::add(topBtn,klocale->translate("go to the first line") );
 bottomBtn = new QPushButton("bottom",this); CHECK_PTR(bottomBtn);
   connect(bottomBtn,SIGNAL(clicked()),this,SIGNAL(buttonBottom()) );
   QToolTip::add(bottomBtn,klocale->translate("go to the last line") );
 newBtn = new QPushButton("new",this); CHECK_PTR(newBtn);
   connect(newBtn,SIGNAL(clicked()),this,SIGNAL(buttonNewEntry()) );
   QToolTip::add(newBtn,klocale->translate("insert a new line") );
 delBtn = new QPushButton("del",this); CHECK_PTR(delBtn);
   connect(delBtn,SIGNAL(clicked()),this,SIGNAL(buttonDelEntry()) );
   QToolTip::add(delBtn,klocale->translate("delete this line") );
 
 upBtn->setPixmap(loader->loadIcon("up.xpm") );
 downBtn->setPixmap(loader->loadIcon("down.xpm") );
 topBtn->setPixmap(loader->loadIcon("top.xpm") );
 bottomBtn->setPixmap(loader->loadIcon("bottom.xpm") );
 newBtn->setPixmap(loader->loadIcon("idea.xpm") );
 delBtn->setPixmap(loader->loadIcon("exit.xpm") );

 upperElem->move(0,SEPARATOR);
 actElemBox->move(0,ICON_HEIGHT+SEPARATOR);
 actElem->move(SEPARATOR,actElemBox->y()+16);
 lowerElem->move(0,actElemBox->y()+2*ELEM_HEIGHT+1*SEPARATOR);

 upBtn->setFixedSize(ICON_HEIGHT,ICON_HEIGHT);
 downBtn->setFixedSize(ICON_HEIGHT,ICON_HEIGHT);
 topBtn->setFixedSize(ICON_HEIGHT,ICON_HEIGHT);
 bottomBtn->setFixedSize(ICON_HEIGHT,ICON_HEIGHT);
 newBtn->setFixedSize(ICON_HEIGHT,ICON_HEIGHT);
 delBtn->setFixedSize(ICON_HEIGHT,ICON_HEIGHT);
 
  setFixedHeight(2*ELEM_HEIGHT+2*ICON_HEIGHT+3*SEPARATOR);

}


/************************************************************************
* Constructor with a given QStrList
*/
KStrListEdit::KStrListEdit(QStrList *data,QWidget *parent,const char* name)
                                  : QWidget(parent,name) 
{
  init(); 
  if ( data->count() > 0 ) {
    elems=data;
    actElem->setText(elems->first());
  }
}

/************************************************************************
* Constructor without a given QStrList
*/
KStrListEdit::KStrListEdit(QWidget *parent,const char* name)
                                  : QWidget(parent,name) 
{
  init(); 
}

/************************************************************************
  * resizeEvent redraws all elements(only changes width)
  **/
void KStrListEdit::resizeEvent(QResizeEvent *) {
  //debug("KStrListBox::resizeEvent: %dx%d",width(),height()); 
  upperElem->resize(width() - 2*ICON_HEIGHT - 3*SEPARATOR
                    ,ELEM_HEIGHT);
  actElemBox->resize(width(),2*ELEM_HEIGHT);
  actElem->resize(upperElem->width()-SEPARATOR,ELEM_HEIGHT);
  lowerElem->resize(upperElem->width(),ELEM_HEIGHT);
  
  upBtn->move(width() - ICON_HEIGHT - SEPARATOR, SEPARATOR);
  downBtn->move(upBtn->x(), height() - ICON_HEIGHT - SEPARATOR);
  topBtn->move(upBtn->x() - ICON_HEIGHT - SEPARATOR, upBtn->y());
  bottomBtn->move(topBtn->x(),downBtn->y());
  delBtn->move(topBtn->x(), actElem->y());
  newBtn->move(delBtn->x() + ICON_HEIGHT+ SEPARATOR, actElem->y());
};

/************************************************************************
  * paintEvent 
  **/
void KStrListEdit::paintEvent(QPaintEvent *) {
  // debug("KStrListEdit::paintEvent at: %d",actIndex);
  QString s;
  s.sprintf("(%d/%d)",actIndex+1,elems->count());
  actElemBox->setTitle(s);

  if ( actIndex >= 1 ) {
      upperElem->setText(elems->at(actIndex-1));
      upperElem->show();
      upBtn->setEnabled(TRUE);
      topBtn->setEnabled(TRUE);
  } else {
      upperElem->setText("");
      upperElem->hide();
      upBtn->setEnabled(FALSE);
      topBtn->setEnabled(FALSE);
  }

  if ( actIndex < elems->count()-1 ) {
      lowerElem->setText(elems->at(actIndex+1));
      lowerElem->show();
      downBtn->setEnabled(TRUE);
      bottomBtn->setEnabled(TRUE);
  } else {
      lowerElem->setText("");
      lowerElem->hide();
      downBtn->setEnabled(FALSE);
      bottomBtn->setEnabled(FALSE);
  }
  
  // dots for more elems
 if ( actIndex >= 2 ) 
    drawText(0,1,"...");
 if ( (int)actIndex <= (int)elems->count()-3 ) 
    drawText(0,height(),"...");
 

}


void KStrListEdit::upSlot() {
  //debug("KStrListEdit::upSlot");
 if ( actIndex  == 0 ) return;
  elems->remove( actIndex ); // removes the current
  elems->insert( actIndex, actElem->text() );
 actElem->setText( elems->at(--actIndex) );
 repaint();
}

void KStrListEdit::downSlot() {
  //debug("KStrListEdit::downSlot");
 if ( actIndex == elems->count()-1 ) return;
  elems->remove( actIndex ); // removes the current
  elems->insert( actIndex, actElem->text() );
 actElem->setText( elems->at(++actIndex) );
 repaint();
}

void KStrListEdit::topSlot() {
 elems->remove( actIndex ); // removes the current
 elems->insert( actIndex, actElem->text() );
 actIndex=0;
 actElem->setText( elems->first() );
 repaint();
}

void KStrListEdit::bottomSlot() {
 elems->remove( actIndex ); // removes the current
 elems->insert( actIndex, actElem->text() );
 actIndex=elems->count()-1;
 actElem->setText( elems->last() );
 repaint();
}

void KStrListEdit::newSlot() {
 actElem->setText("");
 elems->insert(actIndex,actElem->text());
 repaint(); 
}

void KStrListEdit::delSlot() {
 elems->remove( actIndex ); // removes the current
 if ( elems->count() == 0 ) {
   actElem->setText("");
   elems->append(actElem->text());
   return;
 }
 if (actIndex >= elems->count()-1)
   actIndex=elems->count()-1;
 // if (--actIndex == -1)
 //   actIndex=0;
 actElem->setText(elems->at(actIndex));
 repaint();
}

void KStrListEdit::returnPressedSlot() {
  //  debug(" KStrListEdit::returnPressedSlot");

  QString inputS=actElem->text();
  /*  if ( inputS.isEmpty() ) {
      kapp->beep(); return;
  }
  */

  QString s=elems->at(actIndex);
  elems->remove( actIndex ); // removes the current
  elems->insert( actIndex, actElem->text() );

  if ( actIndex == elems->count()-1 ) {  // is was the last field
      elems->append("");
  }    

  actIndex++;
  actElem->setText(elems->at(actIndex));
  repaint();
}

void KStrListEdit::accKeyActivatedSlot(int k_id) {
  //debug("KStrListEdit::accKeyActivatedSlot(int %d)",k_id);
  if ( hasFocus() || actElem->hasFocus() ) { 
    //debug("and has focus");
    if (k_id == upKey) { emit buttonUp(); return; };
    if (k_id == downKey) { emit buttonDown(); return; };
    if (k_id == topKey) { emit buttonTop(); return; };
    if (k_id == bottomKey) { emit buttonBottom(); return; };
 
  }
}

bool KStrListEdit::isEmpty() {
  return ( this.count() == 0 );
}

unsigned int KStrListEdit::count() {
 elems->remove( actIndex ); // removes the current
 elems->insert( actIndex, actElem->text() );
 unsigned int i = elems->count();
 QString s = elems->last();
 if ( s.isEmpty() ) i--;
 return i;
}

void KStrListEdit::clear() {
 elems->clear();
 actElem->setText("");
 actIndex=0;
 elems->append("");
 repaint();
}

QStrList* KStrListEdit::data() {
 elems->remove( actIndex ); // removes the current
 elems->insert( actIndex, actElem->text() );
 QStrList *pureElems = new QStrList();
 for (QString i=elems->first(); i != 0;i= elems->next())
   pureElems->append(i);   // make a deep copy of the QStrList

 QString s=pureElems->at(pureElems->count()-1); // get last entry
 if ( s.isEmpty() ) { //omit last entry if empty
   pureElems->removeLast();
 }
 return pureElems;
}

void KStrListEdit::setData(QStrList *data) {
 elems=data;
}

unsigned int KStrListEdit::index() {
  return actIndex;
}

void KStrListEdit::setIndex(unsigned int index) {
  actIndex=index;
}
