/*****************************************************************************
 * This software is distributed under the terms of the General Public License.
 *
 * Program : kfstab
 * Authors : Dirk Holtwick & Andreas Reuter
 * E-Mail  : holtwick@uni-duisburg.de
 *           Andreas.Reuter@munich.netsurf.de
*****************************************************************************/

// Common Header Files
#include <fstream.h>

// QT Header Files
#include <qlayout.h>
#include <qkeycode.h> // needed for shortkeys
#include <qprinter.h>
#include <qpaintdevicemetrics.h>

// KDE Header Files
#include <kapp.h>
#include <kbuttonbox.h>
#include <kprocess.h>
#include <kfiledialog.h> // for filedialogs
#include <kmsgbox.h>

// Application Header Files
#include "kfstab.h"

#define LOC i18n

void KFstab::setIcon( QListViewItem *lvi )
{
  QString fsys = lvi->text( 3 );

  if(!strcmp( fsys, "iso9660" ))
    lvi->setPixmap( 0, QPixmap( KApplication::kde_icondir() + 
       "/mini/cdrom_unmount.xpm" ));
  else if( !strcmp( fsys, "ext2" ))
    lvi->setPixmap( 0, QPixmap( KApplication::kde_icondir() + 
       "/mini/folder.xpm" ));
  else if( !strcmp( fsys, "umsdos" ))
    lvi->setPixmap( 0, QPixmap( KApplication::kde_icondir() + 
       "/mini/3floppy_mount.xpm" ));
  else if( !strcmp( fsys, "msdos" ))
    lvi->setPixmap( 0, QPixmap( KApplication::kde_icondir() + 
       "/mini/3floppy_unmount.xpm" ));
  else if( !strcmp( fsys, "nfs" ))
    lvi->setPixmap( 0, QPixmap( KApplication::kde_icondir() + 
       "/mini/network_settings.xpm" ));
  else if( !strcmp( fsys, "proc" ))
    lvi->setPixmap( 0, QPixmap( KApplication::kde_icondir() + 
       "/mini/pipe.xpm" ));
  else 
    lvi->setPixmap( 0, QPixmap( KApplication::kde_icondir() + 
       "/mini/exec.xpm" ));
}

void KFstab::openFile()
{
  char 
    text[1024],
    *dev, 
    *mpt, 
    *fsys, 
    *opt, 
    *dump, 
    *fsck,
    *tmpcomment,
    comment[1024];

  ifstream in;
  static bool first_run = TRUE;
  
  if ( ( first_run == TRUE) || ( KMsgBox::yesNo( this,
     LOC( "Kfstab Warning" ), 
         LOC( "You are about to overwrite your current changes.\n"
	      "If you are continue your changes will be lost." ),
	      KMsgBox::EXCLAMATION | KMsgBox::DB_SECOND,
	      LOC( "&Yes, restore them!" ), LOC( "&No, continue." )
	      ) == 1 ) ){

  first_run = FALSE;
  list->clear();

  in.open( FSTAB );
  while( in )
    {
      in.get( text, 1024 ).ignore( 128, '\n' );
      
      // As a comment is allways placed before an entry we need to read it and
      // save it in 'comment' for later use
      if ( ( tmpcomment = strstr( text, "####" )))
        {
	  // increase pointer to get first character after '####'
	  tmpcomment += 4 * sizeof( char );
	  strcpy( comment, tmpcomment );
	}
      // Now real fstab entries are read
      else
        {
          dev  = strtok( text, "\n\t\r " );
          mpt  = strtok( NULL, "\n\t\r " );      
      
          if( dev )
	    {
	      if( *dev != '#' )
	        {
	          fsys = strtok( NULL, "\n\t\r " );
	          opt  = strtok( NULL, "\n\t\r " );		      
	          dump = strtok( NULL, "\n\t\r " );		      
	          fsck = strtok( NULL, "\n\t\r " );
	      
	          QListViewItem *lvi = 
	            new QListViewItem( list, "", dev, mpt, fsys, opt, dump, 
		      fsck, comment );
	          CHECK_PTR( lvi );

	          setIcon( lvi );
	        }
	    }
	  // As there is no comment clear it
	  *comment = 0;
	}
    }
  in.close();

  list->setSorting( 2, TRUE ); // sorting by moint points
  list->triggerUpdate();
  list->setSelected( list->firstChild(), TRUE );
// for future use
/*  actualChild = 0;*/
  itemsChanged = FALSE;

// for future use
/*  // There we get the amount of entries
  childCount = list->childCount();
  // We use a pointer to set disabled entries
  disabledPtr = new int[ childCount ];
  for( int i = 0; i < childCount; i++)
     disabledPtr[i] = false;*/
  }
}

void KFstab::closeFile()
{
  QString tmpStr;
  // An entry beginning with '##' is part of the header
  // An entry beginning with '###' is a disabled entry
  // An entry beginning with '####' followed by a regular fstab-entry is a
  // comment
  // Disabling entries is not yet implemented
  tmpStr.sprintf( LOC( "You are about to overwrite your existing %s.\n"
     "If you are continue your old %s "
     "is no longer available." ), FSTAB, FSTAB );
  if ( KMsgBox::yesNo( this, LOC( "Kfstab Warning" ),
     tmpStr, KMsgBox::EXCLAMATION | KMsgBox::DB_SECOND,
     LOC( "&Yes, commit changes!" ),
     LOC( "&No, dont save changes." ) ) == 1 ) {

  // Create backup
  system( "mv " FSTAB " " FSTAB "~" );

  // Write a new /etc/fstab (/etc/vfstab)
  ofstream *out = new ofstream( FSTAB, ios::out );
  CHECK_PTR( out );

  item = list->firstChild(); 

  // Write Header
  *out << "## Modified by Kfstab " << VERSION <<
          "\n## Copyright Dirk Holtwick, Andreas Reuter, 1998, 1999\n\n";

  int i = 0;
  while( item )
    {  
      // Write comment in the line above the entry
      if ( item->text( 7 ) != NULL )  
         *out << "####" << item->text( 7 ) << "\n";
// for future use
/*      // Write a disabled entry
      if ( disabledPtr[ i ] == true )  *out << "###";*/
      // Now write the entry
      *out
        << item->text( 1 ) << "\t"
	<< item->text( 2 ) << "\t"
	<< item->text( 3 ) << "\t"
	<< item->text( 4 );
      // dump & fsck can have NULL-terminated strings. Avoid this.
      if ( ( item->text( 5 ) != NULL ) && ( item->text( 6 ) != NULL ) )
         *out 
	   << "\t"
	   << item->text( 5 ) << "\t"
	   << item->text( 6 ) << "\n";
      else *out << "\n";
      item = item->nextSibling(); 
      i++;
    }
  delete out;
  itemsChanged = FALSE;
  }
}

void KFstab::itemChanged( QListViewItem *cur )
{
  item = 0;

  str_device->setText( cur->text( 1 ));
  if ( cur->height() != 0 ) {
// for future use
/*     actualChild = cur->itemPos()/cur->height();
     if ( disabledPtr[ actualChild ] == true ) {
	disableEntry( true );
     }
     else {
        disableEntry( false );
     }*/
  }
  str_mpoint->setText( cur->text( 2 ));
  str_filesys->setEditText( cur->text( 3 ));
  str_options->setText( cur->text( 4 ));
  str_dump->setEditText( cur->text( 5 ));
  str_fsck->setEditText( cur->text( 6 ));
  str_comment->setText( cur->text( 7 ));
  
  item = cur;

}

void KFstab::filesysChanged()
{
  int changesRun = TRUE;
  if ( strstr( str_filesys->currentText(), "proc" ))
  {
     switch( KMsgBox::yesNo( this, LOC( "proc filesystem" ),
        LOC ("You changed to filesystem proc\n"
	         "As this filesystem has limitations, should I make them for you ?" ),
	      KMsgBox::QUESTION | KMsgBox::DB_FIRST,
	      LOC( "&Continue" ), LOC ("&Abort" ) ) ) {
     case 1: // Continue
	str_device->setText( "none" );
	str_mpoint->setText( "/proc" );
	str_options->setText( "defaults" );
	item->setText( 5, "0" );
	item->setText( 6, "0" );
	item->setText( 7, LOC( "This is the proc filesystem" ));
	itemChanged( item );
        break;
     case 0: // Abort
	changesRun = FALSE;
	break;
     }
  }
  if ( strstr( str_filesys->currentText(), "swap" ))
  {
     switch( KMsgBox::yesNo( this, LOC ("swap filesystem" ),
        LOC( "You changed to filesystem swap\n"
	         "As this filesystem has limitations, should I make them for you ?" ),
	      KMsgBox::QUESTION| KMsgBox::DB_FIRST,
	      LOC( "&Continue" ), LOC( "&Abort" ) ) ) {
     case 1: // Continue
	str_device->setText( "/" );
	str_mpoint->setText( "none" );
	str_options->setText( "sw" );
	item->setText( 5, "" );
	item->setText( 6, "" );
	item->setText( 7, LOC( "This is the system's swap space" ));
	itemChanged( item );
	devicePreferences();
        break;
     case 0: // Abort
	changesRun = FALSE;
	break;
     }
  }
  if ( changesRun ) contentChanged();
  else itemChanged( item );
}

void KFstab::contentChanged()
{
  if( item )
    {
      item->setText( 1, str_device->text());
      item->setText( 2, str_mpoint->text());
      item->setText( 3, str_filesys->currentText());
      item->setText( 4, str_options->text());
      item->setText( 5, str_dump->currentText());
      item->setText( 6, str_fsck->currentText());
      item->setText( 7, str_comment->text());
      setIcon( item );
      itemsChanged = TRUE;
    }
}

void KFstab::addItem()
{
// We need to work on disabledPtr the added Pointer
// Problem: where's the inserted Entry placed ?

  int kmb;
  kmb = KMsgBox::yesNoCancel( this, LOC( "Item added" ),
           LOC( "You added a new item, should I help you to finish it ?" ),
           KMsgBox::QUESTION | KMsgBox::DB_FIRST,
           LOC( "&Yes" ), LOC( "&No" ), LOC( "&Cancel" ) );
  if ( kmb != 3 ) { // yes or no selected
     QListViewItem *lvi = new QListViewItem( list, "", "/dev/", "/mnt/", 
        "ext2", "defaults", "0", "0");  
     CHECK_PTR( lvi );
     list->setSelected( lvi, TRUE );
     itemChanged( lvi );
     
     setIcon( lvi );
     itemsChanged = TRUE;
     if ( kmb == 1 ) { // yes: start assistants
        kmb = KMsgBox::yesNoCancel( this, LOC( "Select primary filesystem" ),
              LOC( "Please select your primary filesystem\n\n"
	               "Standard - partition mounted to a directory\n"
	               "proc - proc filesystem\n"
	               "swap - swap filesystem" ),
	            KMsgBox::INFORMATION | KMsgBox::DB_FIRST,
	            LOC( "Standard" ), LOC( "proc" ), LOC( "swap" ) );
        switch( kmb ) {
	         case 1: // Standard selected
	            devicePreferences();
              mountPointPreferences();
              optionsPreferences();
	            break;
	         case 2: // proc selected
	            str_filesys->setEditText( "proc" );
	            filesysChanged();
	            break;
	         case 3: // swap selected
	            str_filesys->setEditText( "swap" );
	            filesysChanged();
	            break;
	      }
     }
  itemChanged( lvi );
  } 
}

void KFstab::deleteItem()
{
// We need to work on disabledPtr the deleted Pointer
  QString tmpStr;
  tmpStr.sprintf( LOC( "You are about to delete a mount point in %s.\n"
     "If you are continue this mount point is no longer accessable." ), FSTAB );
  if ( KMsgBox::yesNo( this, LOC( "Kfstab Warning" ),
          tmpStr,
	        KMsgBox::EXCLAMATION | KMsgBox::DB_SECOND,
	        LOC( "&Yes, commit deletion!" ), LOC( "&No, dont delete it." )
	        ) == 1 ) {
  if( item )
    delete item;
  item = 0;
  list->setSelected( list->firstChild(), TRUE );
  itemsChanged = TRUE;
  }
}

KFstab::KFstab( const char *name ) : KTMainWindow( name )
{
  item = 0;
  // needed for config stuff
  extern QString ConfigWidth;
  extern QString ConfigHeight;
  extern int itisroot;
  
  // Menu stuff
  menuBar = new KMenuBar( this );
  CHECK_PTR( menuBar );
  {
    QString tmpStr;
    QPopupMenu *file = new QPopupMenu();
    CHECK_PTR( file );
    file->insertItem( LOC( "&Reload" ), this, SLOT( openFile()), CTRL+Key_R );
    menu_file_saveID = file->insertItem( LOC( "&Save" ), this, 
       SLOT( closeFile()), CTRL+Key_S );
    file->insertSeparator();
    tmpStr.sprintf( LOC( "Print %s ..." ), FSTAB );
    file->insertItem( tmpStr, this, SLOT( printFile()),
       CTRL+Key_P);
    file->insertSeparator();
    file->insertItem( LOC( "&Exit" ), this, SLOT( closeWindow()), CTRL+Key_Q );
    menuBar->insertItem( LOC( "&File" ), file );
    
    file->setItemEnabled( menu_file_saveID, itisroot );
  }

  {
    QPopupMenu *edit = new QPopupMenu();
    CHECK_PTR( edit );
    menu_edit_addID = edit->insertItem( LOC( "&Add" ), this, 
       SLOT( addItem()), CTRL+Key_A );
    menu_edit_deleteID = edit->insertItem( LOC( "&Delete" ), this, 
       SLOT( deleteItem()), CTRL+Key_D );
    menuBar->insertItem( LOC( "&Edit" ), edit);
    edit->setItemEnabled( menu_edit_addID, itisroot );
    edit->setItemEnabled( menu_edit_deleteID, itisroot );
  }
  
  {
    QPopupMenu *QPMTools = new QPopupMenu();
    CHECK_PTR( QPMTools );
    QPMTools->insertItem( LOC( "Start kdf..." ), this, SLOT( callKdf() ));
    QPMTools->insertItem( LOC( "Start kdu..." ), this, SLOT( callKdu() ));
    menuBar->insertItem( LOC( "&Tools" ), QPMTools );
  }

  menuBar->insertSeparator();

    {
    QString tmpStr;
    tmpStr.sprintf( LOC( "KFstab %s"
		         "\nEasy editing of %s\n"
			 "\n(C)opyright by Dirk Holtwick, "
			 "Andreas Reuter 1998, 1999\n"
			 "\nHomepage: "
			 "http://homepages.munich.netsurf.de/Andreas.Reuter/kfstab/\n\n"
			  ), VERSION, FSTAB );
    QPopupMenu *help = kapp->getHelpMenu( this, tmpStr );
    menuBar->insertItem( LOC( "&Help" ), help );
  }

  setMenu( menuBar );
   
  // Vertical Box (List + Input + Buttons)
  {
    QWidget *w = new QWidget( this );
    CHECK_PTR( w );
    QBoxLayout *vbox = new QVBoxLayout( w );
    CHECK_PTR( vbox );
    
    // Listview
    {  
      list = new CListView( w );
      CHECK_PTR( list );
    
      vbox->addWidget( list );
    }

    // Separator
    {
      QFrame *sep = new QFrame( w );
      CHECK_PTR( sep );
      sep->setFrameStyle( QFrame::HLine | QFrame::Sunken ); 
      sep->setFixedHeight( sep->sizeHint().height() );
      vbox->addWidget( sep );    
    }
        
    // InputBox
    {
      QWidget *inp = new QWidget( w );
      CHECK_PTR( inp );
      QGridLayout *ibox = new QGridLayout( inp, 8, 3);
      CHECK_PTR( ibox );

      // Row: Device
      {
	QPushButton *pPreferences = new QPushButton( LOC( "Preferences..." ) , 
	   inp );
	CHECK_PTR( pPreferences );
	pPreferences->setGeometry(0, 0, 25, sizeHint().height());
	connect( pPreferences, SIGNAL( clicked() ), 
	   SLOT( devicePreferences() ) );
	ibox->addWidget( new QLabel( LOC( "Device" ), inp), 0, 0);
	ibox->addWidget( pPreferences, 0, 1 );
	ibox->addWidget( str_device = new QLineEdit( inp ), 0, 2);
	CHECK_PTR( str_device );
	str_device->setFixedHeight( str_device->sizeHint().height());
	QPBDevPreferences = pPreferences;
      }

      // Row: Mount Point
      {
	QPushButton *pPreferences = new QPushButton( LOC( "Preferences..." ) , 
	   inp );
	CHECK_PTR( pPreferences );
	pPreferences->setGeometry( 0, 0, 25, sizeHint().height());
	connect( pPreferences, SIGNAL( clicked() ), 
	   SLOT( mountPointPreferences()) );
	ibox->addWidget(new QLabel( LOC( "Mount Point" ), inp ), 1, 0 );
	ibox->addWidget( pPreferences, 1, 1 );
	ibox->addWidget( str_mpoint = new QLineEdit( inp ), 1, 2 );
	CHECK_PTR( str_mpoint );
	str_mpoint->setFixedHeight( str_mpoint->sizeHint().height());
	QPBMPointPreferences = pPreferences;
      }

      // Row: Filesystem
      {
	ibox->addWidget( new QLabel( LOC( "Filesystem" ), inp ), 2, 0 );
	ibox->addWidget( str_filesys = new QComboBox( TRUE, inp ), 2, 2 );
	CHECK_PTR( str_filesys );
	// all currently available filesystems are in /proc/filesystems
	// here: listed in alphabetic order
	str_filesys->insertItem( "affs" );
	str_filesys->insertItem( "ext" ); // obsolete since 2.1.21
	str_filesys->insertItem( "ext2" );
	str_filesys->insertItem( "hpfs" );
	str_filesys->insertItem( "iso9660" );
	str_filesys->insertItem( "minix" );
	str_filesys->insertItem( "msdos" );
	str_filesys->insertItem( "ncpfs" );
	str_filesys->insertItem( "nfs" );
	str_filesys->insertItem( "proc" );
	str_filesys->insertItem( "romfs" );
	str_filesys->insertItem( "smbfs" );
	str_filesys->insertItem( "swap" );
	str_filesys->insertItem( "sysv" ); // also for xenix & coherent
	str_filesys->insertItem( "ufs");
	str_filesys->insertItem( "umsdos" );
	str_filesys->insertItem( "vfat" );
	str_filesys->insertItem( "xiafs" ); // obsolete since 2.1.21
	str_filesys->setAutoCompletion( TRUE );
	str_filesys->setFixedHeight( str_filesys->sizeHint().height() );
      }

      // Row: Options
      {
	QPushButton *pPreferences = new QPushButton( LOC( "Preferences..." ) ,
	   inp );
	CHECK_PTR( pPreferences );
	pPreferences->setGeometry( 0, 0, 25, sizeHint().height());
	connect( pPreferences, SIGNAL( clicked() ), 
	   SLOT( optionsPreferences() ) );
	ibox->addWidget( new QLabel( LOC( "Options" ), inp ), 3, 0 );
	ibox->addWidget( pPreferences, 3, 1 );
	ibox->addWidget(str_options = new QLineEdit( inp ), 3, 2);
	CHECK_PTR( str_options );
	str_options->setFixedHeight( str_options->sizeHint().height());
	QPBOptPreferences = pPreferences;
      }

      // Row: Dump
      {
	ibox->addWidget( new QLabel( LOC( "Dump" ), inp ), 4, 0 );
	ibox->addWidget( str_dump = new QComboBox( TRUE, inp ), 4, 2 );
	CHECK_PTR( str_dump );
	// str_dump = new QComboBox(TRUE, inp);
       	str_dump->insertItem( "0" );
	str_dump->insertItem( "1" );
	str_dump->insertItem( "2" );
	str_dump->insertItem( "3" );
	str_dump->insertItem( "4" );
	str_dump->insertItem( "5" );
	str_dump->insertItem( "6" );
	str_dump->insertItem( "7" );
	str_dump->insertItem( "8" );
	str_dump->insertItem( "9" );
	str_dump->setFixedHeight( str_dump->sizeHint().height());
      }

      // Row: Fsck
      {
	ibox->addWidget( new QLabel( LOC( "Fsck" ), inp ), 5, 0 );
	ibox->addWidget( str_fsck = new QComboBox( TRUE, inp ), 5, 2 );
	CHECK_PTR( str_fsck );
	//str_fsck = new QComboBox(TRUE, inp);
	str_fsck->insertItem( "0" );
	str_fsck->insertItem( "1" );
	str_fsck->insertItem( "2" );
	str_fsck->insertItem( "3" );
	str_fsck->insertItem( "4" );
	str_fsck->insertItem( "5" );
	str_fsck->insertItem( "6" );
	str_fsck->insertItem( "7" );
	str_fsck->insertItem( "8" );
	str_fsck->insertItem( "9" );
	str_fsck->setFixedHeight( str_fsck->sizeHint().height());
      }

      // Row: Comment
      {
	ibox->addWidget( new QLabel( LOC( "Comment" ), inp ), 6, 0 );
	ibox->addWidget( str_comment = new QLineEdit( inp ), 6, 2 );
	CHECK_PTR( str_comment );
	str_comment->setFixedHeight( str_comment->sizeHint().height() );
      }
// for future use      
      // Row: Entry disabled
      {
        QLabel *qLabel = new QLabel( LOC( "Disable Entry" ), inp );
	CHECK_PTR( qLabel );
	QLabel *qLabel2 = new QLabel( LOC( "for future use, please wait." ), inp );
	CHECK_PTR( qLabel2 );
	ibox->addWidget( qLabel, 7, 0 );
	ibox->addWidget( QCBDisabled = new QCheckBox( inp ), 7, 1 );
	ibox->addWidget( qLabel2, 7, 2 );
	CHECK_PTR( QCBDisabled );
	QCBDisabled->setFixedHeight( QCBDisabled->sizeHint().height() );
	qLabel->setEnabled( false );
	QCBDisabled->setEnabled( false );
	qLabel2->setEnabled( false );
      }

      vbox->addWidget( inp );
    }
    
    // Separator
    {
      QFrame *sep = new QFrame( w );  
      CHECK_PTR( sep );
      sep->setFrameStyle( QFrame::HLine | QFrame::Sunken ); 
      sep->setFixedHeight( sep->sizeHint().height() );
      vbox->addWidget( sep );    
    }

    // ButtonBox
    {
      KButtonBox *bbox = new KButtonBox( w );
      CHECK_PTR( bbox );
      but_rel = bbox->addButton( LOC( "Reload" ));
      but_sav = bbox->addButton( LOC( "Save" ));
      but_prt = bbox->addButton( LOC( "Print" ));
		bbox->addStretch();
      but_add = bbox->addButton( LOC( "Add" ));
      but_del = bbox->addButton( LOC( "Delete" ));
		bbox->addStretch();
      but_hlp = bbox->addButton( LOC( "Help" ));
      but_ext = bbox->addButton( LOC( "Exit" ));
      bbox->setFixedHeight( bbox->sizeHint().height());
      vbox->addWidget( bbox );
     but_sav->setEnabled( itisroot );
     but_add->setEnabled( itisroot );
     but_del->setEnabled( itisroot );
    }

    setView( w );
  }

  connect( list,        SIGNAL( currentChanged( QListViewItem *)),  
	  this,        SLOT( itemChanged( QListViewItem * )));
  connect( str_device,  SIGNAL( textChanged( const char * )),  
	  this,        SLOT( contentChanged()));  
  connect( str_mpoint,  SIGNAL( textChanged( const char * )),  
	  this,        SLOT( contentChanged()));  
  connect( str_filesys, SIGNAL( activated( const char *)),  
	  this,        SLOT( filesysChanged()));
  connect( str_options, SIGNAL( textChanged( const char *)),  
	  this,        SLOT( contentChanged()));
  connect( str_dump,    SIGNAL( activated( const char *)),  
	  this,        SLOT( contentChanged()));
  connect( str_fsck,    SIGNAL( activated( const char *)),  
	  this,        SLOT( contentChanged()));
  connect( but_add,     SIGNAL( clicked()),  
	  this,        SLOT( addItem()));
  connect( but_del,     SIGNAL( clicked()),  
	  this,        SLOT( deleteItem()));
  connect( but_rel,     SIGNAL( clicked()),  
 	  this,        SLOT( openFile()));
  connect( but_sav,     SIGNAL( clicked()),  
 	  this,        SLOT( closeFile()));
  connect( but_prt,     SIGNAL( clicked()),  
 	  this,        SLOT( printFile()));
  connect( but_hlp,     SIGNAL( clicked()),  
 	  this,        SLOT( helpSelected()));
  connect( but_ext,     SIGNAL( clicked()),  
 	  this,        SLOT( closeWindow()));
  connect( str_comment, SIGNAL( textChanged( const char * )),
          this,        SLOT( contentChanged()));
// for future use
/*  connect( QCBDisabled, SIGNAL( clicked()),
          this,        SLOT( disableButtonPressed()));*/

  openFile();
  setMinimumSize( 637 , 285 );
  resize( ConfigWidth.toInt(), ConfigHeight.toInt() );
  
  op = 0;
}

void KFstab::closeWindow()
{
   closeEvent( 0 );
}

void KFstab::callKdf()
{
   if ( kdfProc.isRunning() ) {
      KMsgBox::message( this, LOC( "Warning" ),
         LOC( "You have already started kdf (KDiskFree)" ),
         KMsgBox::EXCLAMATION );
      return;
   }
   kdfProc << "kdf" << "";
   if( !kdfProc.start() )
      KMsgBox::message( this, LOC( "Error" ), LOC( "Could not start kdf"
         " (KDiskFree)"), KMsgBox::EXCLAMATION );
}

void KFstab::callKdu()
{
   if ( kduProc.isRunning() ) {
      KMsgBox::message( this, LOC( "Warning" ),
         LOC( "You have already started kdu" ),
         KMsgBox::EXCLAMATION );
   }
   kduProc << "kdu" << "";
   if( !kduProc.start() )
      KMsgBox::message( this, LOC( "Error" ), LOC( "Could not start kdu"),
         KMsgBox::EXCLAMATION );
}

void KFstab::helpSelected()
{
  KApplication::getKApplication()->invokeHTMLHelp( "" , "" );
}

void KFstab::closeEvent( QCloseEvent *e )
{
  (void) e;
  extern int itisroot;
  QString tmpStr;

  if ( itemsChanged == TRUE )
     if ( itisroot ) closeFile();
     else {
        tmpStr.sprintf( LOC( "Only root can save to %s.\n"
	         "As you are not root, all changes you made are not saved." ), FSTAB );
        KMsgBox::message( this,
           LOC( "Warning" ),
	         tmpStr, KMsgBox:: EXCLAMATION );
     }
  if ( kdfProc.isRunning() )
     if ( KMsgBox::yesNo( this, LOC( "Warning" ),
            LOC( "The application kdf (KDiskFree) you started has not "
            "yet been finished." ), KMsgBox::EXCLAMATION | KMsgBox::DB_FIRST,
            LOC( "Kill kdf" ),
            LOC( "Leave it running" ) ) == 1 ) {
        kdfProc.kill( 15 );
        }
  if ( kduProc.isRunning() )
     if ( KMsgBox::yesNo( this, LOC( "Warning" ),
            LOC( "The application kdu you started has not "
            "yet been finished." ),KMsgBox::EXCLAMATION | KMsgBox::DB_FIRST,
            LOC( "Kill kdu" ),
            LOC( "Leave it running" ) ) == 1 ) {
        kduProc.kill( 15 );
        }
  qApp->quit();
}

void KFstab::printFile()
{
   printer = new QPrinter;

   // Open printer dialog
   if ( printer->setup( this ) ) {
      // open FSTAB
      ifstream in;
      in.open( FSTAB );

      QPainter p;
      char text[ 1024 ];
      const int MARGIN = 10;
      // Prepare the job for printing
      p.begin( printer );
      p.setFont( QFont( "courier", 12, QFont::Normal ) );
      // yPos holds the y position for each line
      int yPos = 0;
      QFontMetrics fm = p.fontMetrics();
      // get width/height of printer surface
      QPaintDeviceMetrics metrics( printer );
      // output new line as long as there is one
      while( in ) {
         // If there's not enough spare place on this page
         // create new page and go back to top
         if ( MARGIN + yPos > metrics.height() - MARGIN ) {
            printer->newPage();
            yPos = 0;
         }
         in.get( text, 1024 ).ignore( 128, '\n' );
	 p.drawText( MARGIN, MARGIN + yPos,
            metrics.width(), fm.lineSpacing(),
            ExpandTabs | DontClip, text );
            yPos = yPos + fm.lineSpacing();
      }
      // close FSTAB
      in.close();
      // Now send the job to the printer
      p.end();
   }
}

void KFstab::devicePreferences()
{
   if ( strstr( str_filesys->currentText(), "proc" ) ) {
      KMsgBox::message( this, LOC( "Warning" ),
         LOC ( "proc filesystem device must be none" ), KMsgBox::EXCLAMATION );
	    str_device->setText( "none" );
   }
   else {
   dp = new DevPreferences( this, LOC( "KFstab - Device Preferences" ));
   CHECK_PTR( dp );
   connect( dp, SIGNAL( newDevice( char * )), 
      this, SLOT( devicePreferencesNew( char * )) );
   dp->show();
   }
}

void KFstab::devicePreferencesNew( char *strDev )
{
   str_device->setText( strDev );	// change text field and listbox
}

void KFstab::mountPointPreferences()
{
   if ( strstr( str_filesys->currentText(), "proc" ) )
      KMsgBox::message( this, LOC( "Warning" ),
         LOC( "proc filesystem cannot be mounted other than /proc" ),
         KMsgBox::EXCLAMATION );
   else {
   if ( strstr( str_filesys->currentText(), "swap" ) )
      KMsgBox::message( this, LOC( "Warning" ),
         LOC( "swap partitions cannot be mounted other than none" ),
         KMsgBox::EXCLAMATION );
   else {
   QString dirName = KFileDialog::getDirectory( str_mpoint->text(), this );
   if ( ! dirName.isNull() ) 
      str_mpoint->setText( dirName );
   }
   }
}

void KFstab::optionsPreferences()
{
   QString optStr;
   optStr = str_options->text();

   if ( strstr( str_filesys->currentText(), "swap" )) {
      KMsgBox::message( this, LOC( "Warning" ),
         LOC( "Swap partitions cannot be declared other than sw" ),
         KMsgBox::EXCLAMATION );
   }
   else {
      if ( strstr( str_filesys->currentText(), "proc" ) )
         KMsgBox::message( this, LOC( "Warning" ),
	    LOC( "proc filesystem cannot be declared other than defaults" ),
	    KMsgBox::EXCLAMATION );
      else {  
         op = new OptPreferences ( optStr, this, LOC( "KFstab - Options Preferences" ));
         CHECK_PTR( op );
	 connect( op, SIGNAL( newOptions ( char * )), 
            this, SLOT( optionsPreferencesNew( char * )) );
         op->show();
      }
   }
}

void KFstab::optionsPreferencesNew( char *strOpt )
{
   str_options->setText( strOpt );	// change text field and listbox
}
// for future use
/*
void KFstab::disableButtonPressed()
{
   disableEntry( !disabledPtr[ actualChild ] );
}

void KFstab::disableEntry( bool modus )
{
   // Place possibility to enable or disable an entry

   // As there is no member function `QListViewItem::setEnabled(bool)' defined
   // We must use other possibilities
   // 1) We will use another listview to view disabled entries.
   //    -- lots of time
   //    ++ looks nicer
   // 2) We mark the device with the ### prefix and disable the possibility to
   //    enter anything
   //    ++ fast to implement
   //    -- looks not very nice
   // As 2) seems to be easier we try this

   // if an entry is disabled we set a different icon in front of it
   if ( modus && !disabledPtr[ actualChild ] )
      item->setPixmap( 0, QPixmap( KApplication::kde_icondir() + 
         "/mini/mini.excl.xpm" ));
   if ( !modus && disabledPtr[ actualChild ] )
      setIcon( item );

   str_device->setEnabled( !modus );
   str_mpoint->setEnabled( !modus );
   str_filesys->setEnabled( !modus );
   str_options->setEnabled( !modus );
   str_dump->setEnabled( !modus );
   str_fsck->setEnabled( !modus );
   QPBDevPreferences->setEnabled( !modus );
   QPBMPointPreferences->setEnabled( !modus );
   QPBOptPreferences->setEnabled( !modus );
   QCBDisabled->setChecked( modus );
   disabledPtr[ actualChild ] = modus;
}
*/
#include "kfstab.moc"

