#include "setupdlg.h"
#include "setupdlg.moc"

#include <qlabel.h>
#include <qgrpbox.h>
#include <qfileinf.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <kiconloaderdialog.h>
#include <klocale.h>
#include <qmsgbox.h>
#include <qfont.h>

#include <qpalette.h>
#include <kfiledialog.h>

#include <kfm.h>

#include "kurllabel.h"

#define EDITHEIGHT QMAX(fontMetrics().lineSpacing() + (fontMetrics().descent() * 3), 25)
//////////////////////////////////////////////////////////////////////
// KBiffScramble
//////////////////////////////////////////////////////////////////////
const char* KBiffScramble::scramble(const char* text)
{
	return m_scramble(text, true);
}

const char* KBiffScramble::unscramble(const char* text)
{
	return m_scramble(text, false);
}

const char* KBiffScramble::m_scramble(const char* text, bool state)
{
	char *ptr = new char[strlen(text)];
	char *ret_ptr = ptr;

	while (*text)
		*ptr++ = state ? (*(text++) - 4) : (*(text++) + 4);
	*ptr = '\0';

	return ret_ptr;
}


//////////////////////////////////////////////////////////////////////
// KBiffMailboxInfo
//////////////////////////////////////////////////////////////////////
KBiffMailboxInfo::KBiffMailboxInfo(QWidget *parent)
	: QWidget(parent)
{
	int x = 10;

	// setup the radio boxes
	m_radioLocalMailbox = new QRadioButton(this);
	m_radioLocalMailbox->move(10, x);
	m_radioLocalMailbox->setText(klocale->translate("Monitor local mailbox"));
	m_radioLocalMailbox->adjustSize();

	m_radioRemoteMailbox = new QRadioButton(this);
	m_radioRemoteMailbox->setText(klocale->translate("Monitor remote mailbox"));
	m_radioRemoteMailbox->adjustSize();

	m_buttonGroup = new QButtonGroup();
	m_buttonGroup->insert(m_radioLocalMailbox);
	m_buttonGroup->insert(m_radioRemoteMailbox);
	connect(m_buttonGroup, SIGNAL(clicked(int)), SLOT(m_radioClicked(int)));

	x += m_radioLocalMailbox->height();

	// partition off the local mailbox
	QGroupBox* local_groupbox = new QGroupBox(this);
	local_groupbox->move(10, x);

	x += 10;

	// browse button
	m_buttonLocalMailbox = new QPushButton(this);
	m_buttonLocalMailbox->setGeometry(220, x, 75, EDITHEIGHT);
	m_buttonLocalMailbox->setText(klocale->translate("Browse"));
	connect(m_buttonLocalMailbox, SIGNAL(clicked()),
                                 SLOT(m_browseLocalMailbox()));

	// local mailbox
	m_editMailbox = new QLineEdit(this);
	m_editMailbox->setGeometry(20, x, 195, EDITHEIGHT);

	local_groupbox->resize(295, EDITHEIGHT + 20);

	x += EDITHEIGHT + 15;

	m_radioRemoteMailbox->move(10, x);

	x += m_radioRemoteMailbox->height();

	// group together the pop3 stuff
	QGroupBox* remote_groupbox = new QGroupBox(this);
	remote_groupbox->move(10, x);
	remote_groupbox->resize(295, (EDITHEIGHT * 4) + 10);

	x += 5;

	// the pop3 server
	QLabel* server_label = new QLabel(this);
	server_label->move(15, x + 3);
	server_label->setText(klocale->translate("POP3 Server:"));
	server_label->adjustSize();

	m_editPopServer = new QLineEdit(this);
	m_editPopServer->move(server_label->width() + 20, x);
	m_editPopServer->resize(295 - m_editPopServer->x(), EDITHEIGHT);

	x += EDITHEIGHT + 3;

	// the pop3 username (mailbox name, actually)
	QLabel* user_label = new QLabel(this);
	user_label->move(15, x + 3);
	user_label->setText(klocale->translate("Username:"));
	user_label->adjustSize();

	m_editUser = new QLineEdit(this);
	m_editUser->move(server_label->width() + 20, x);
	m_editUser->resize(295 - m_editUser->x(), EDITHEIGHT);

	x += EDITHEIGHT + 3;

	// the pop3 password
	QLabel* password_label = new QLabel(this);
	password_label->move(15, x + 3);
	password_label->setText(klocale->translate("Password:"));
	password_label->adjustSize();

	m_editPassword = new QLineEdit(this);
	m_editPassword->move(server_label->width() + 20, x);
	m_editPassword->resize(295 - m_editPassword->x(), EDITHEIGHT);
	m_editPassword->setEchoMode(QLineEdit::Password);

	x += EDITHEIGHT;

	// do we want to store the password?
	m_checkStorePassword = new QCheckBox(this);
	m_checkStorePassword->move(server_label->width() + 20, x);
	m_checkStorePassword->setText(klocale->translate("Store password"));
	m_checkStorePassword->adjustSize();
}

KBiffMailboxInfo::~KBiffMailboxInfo()
{
}

const char* KBiffMailboxInfo::getMailbox()
{
	return m_editMailbox->text();
}

void KBiffMailboxInfo::setMailbox(const char* mailbox)
{
	m_editMailbox->setText(mailbox);
}

const bool KBiffMailboxInfo::getReadLocalMailbox()
{
	return m_radioLocalMailbox->isChecked();
}

const char* KBiffMailboxInfo::getPopServer()
{
	return m_editPopServer->text();
}

const char* KBiffMailboxInfo::getUser()
{
	return m_editUser->text();
}

const char* KBiffMailboxInfo::getPassword()
{
	return m_editPassword->text();
}

void KBiffMailboxInfo::setReadLocalMailbox(bool local)
{
	m_radioRemoteMailbox->setChecked(!local);
	m_radioLocalMailbox->setChecked(local);

	if (local)
		m_radioClicked(0);
	else
		m_radioClicked(1);
}

void KBiffMailboxInfo::setPopServer(const char* pop_server)
{
	m_editPopServer->setText(pop_server);
}

void KBiffMailboxInfo::setUser(const char* username)
{
	m_editUser->setText(username);
}

void KBiffMailboxInfo::setPassword(const char* password)
{
	m_editPassword->setText(password);
}

void KBiffMailboxInfo::setStorePassword(bool store)
{
	m_checkStorePassword->setChecked(store);
}

const bool KBiffMailboxInfo::getStorePassword()
{
	return m_checkStorePassword->isChecked();
}

void KBiffMailboxInfo::m_browseLocalMailbox()
{
  QString mailbox_path = KFileDialog::getOpenFileName();
  if (!mailbox_path.isEmpty() && !mailbox_path.isNull())
    setMailbox(mailbox_path);
}

void KBiffMailboxInfo::m_radioClicked(int remote)
{
	// enable or disable the "other" mailboxes
	if (remote)
	{
		m_editMailbox->setEnabled(false);
		m_buttonLocalMailbox->setEnabled(false);
		m_editPopServer->setEnabled(true);
		m_editUser->setEnabled(true);
		m_editPassword->setEnabled(true);
		m_checkStorePassword->setEnabled(true);
	}
	else
	{
		m_editMailbox->setEnabled(true);
		m_buttonLocalMailbox->setEnabled(true);
		m_editPopServer->setEnabled(false);
		m_editUser->setEnabled(false);
		m_editPassword->setEnabled(false);
		m_checkStorePassword->setEnabled(false);
	}
}

//////////////////////////////////////////////////////////////////////
// KBiffAbout
//////////////////////////////////////////////////////////////////////
KBiffAbout::KBiffAbout(QWidget *parent)
	: QWidget(parent)
{
	int x, y;

	// load in the kbiff pixmap
	QPixmap logo_pixmap(KApplication::kde_icondir() + '/' + "kbiff.xpm");

	KURLLabel *logo = new KURLLabel(this);
	logo->setPixmap(logo_pixmap);
	logo->move(10,10);
	logo->setFont(QFont("helvetica", 24, QFont::Bold));
	logo->setURL("http://www.pobox.com/~kurt_granroth/kbiff");
	logo->setText(klocale->translate("KBiff"));
	logo->setTextAlignment(Right);
	logo->setUnderline(false);
	logo->setGlow(false);
	logo->setFloat(true);
	logo->setAutoResize(true);
	logo->adjustSize();

	connect(logo, SIGNAL(leftClickedURL(const char*)),
	                 SLOT(m_homepage(const char*)));

	x = logo_pixmap.width() + 15;
	y = 10;

	QFontMetrics big_fm(QFont("helvetica", 24, QFont::Bold));
	QFontMetrics small_fm(QFont("helvetica", 12));

	y += big_fm.lineSpacing();

	QLabel *version = new QLabel(this);
	version->setFont(QFont("helvetica", 12));
	version->setText(klocale->translate("Version 1.0.2"));
	version->move(x, y);
	version->adjustSize();
	y += small_fm.lineSpacing() * 2;

	QLabel *copyright = new QLabel(this);
	copyright->setFont(QFont("helvetica", 12));
	copyright->setText(klocale->translate("Copyright (C) 1998"));
	copyright->move(x, y);
	copyright->adjustSize();
	y += small_fm.lineSpacing();

	QLabel *name = new QLabel(this);
	name->setFont(QFont("helvetica", 12));
	name->setText("Kurt Granroth");
	name->move(x, y);
	name->adjustSize();
	y += small_fm.lineSpacing();

	KURLLabel *email = new KURLLabel(this);
	email->setText("granroth@kde.org");
	email->setURL("granroth@kde.org");
	email->setFont(QFont("helvetica", 12));
	email->setUnderline(false);
	email->setGlow(false);
	email->setFloat();
	email->move(x, y);
	email->setAutoResize(true);
	email->adjustSize();
	connect(email, SIGNAL(leftClickedURL(const char*)),
	                 SLOT(m_mailTo(const char*)));
}

KBiffAbout::~KBiffAbout()
{
}

void KBiffAbout::m_mailTo(const char* url)
{
	// this kludge is necessary until KFM handles mailto: urls
	char *mailer = getenv("MAILER");
	if (mailer)
	{
		char cmd[80];
		sprintf(cmd, "%s -s \"KBiff\" %s &", mailer, url);
		system(cmd);
	}
}

void KBiffAbout::m_homepage(const char* url)
{
	KFM kfm;
	kfm.openURL(url);
}

///////////////////////////////////////////////////////////////////////
// KBiffNewMailPrefs
///////////////////////////////////////////////////////////////////////
KBiffNewMailPrefs::KBiffNewMailPrefs(QWidget *parent)
	: QWidget(parent)
{
	// setup the Run Command stuff
	m_checkRunCommand = new QCheckBox(this);
	m_checkRunCommand->setText(klocale->translate("Run Command"));
	m_checkRunCommand->move(10, 10);
	m_checkRunCommand->adjustSize();
	m_editRunCommand = new QLineEdit(this);
	m_editRunCommand->setGeometry(10, 30, 170, EDITHEIGHT);
	m_buttonBrowseRunCommand = new QPushButton(this);
	m_buttonBrowseRunCommand->setGeometry(190, 30, 70, EDITHEIGHT);
	m_buttonBrowseRunCommand->setText(klocale->translate("Browse"));

	// setup the Play Sound stuff
	m_checkPlaySound = new QCheckBox(this);
	m_checkPlaySound->setText(klocale->translate("Play Sound"));
	m_checkPlaySound->move(10, 75);
	m_checkPlaySound->adjustSize();
	m_editPlaySound = new QLineEdit(this);
	m_editPlaySound->setGeometry(10, 95, 170, EDITHEIGHT);
	m_buttonBrowsePlaySound = new QPushButton(this);
	m_buttonBrowsePlaySound->setGeometry(190, 95, 70, EDITHEIGHT);
	m_buttonBrowsePlaySound->setText(klocale->translate("Browse"));

	// setup the System Sound stuff
	m_checkBeep = new QCheckBox(this);
	m_checkBeep->setText(klocale->translate("System Beep"));
	m_checkBeep->move(10, 140);
	m_checkBeep->adjustSize();

	// connect some slots and signals
	connect(m_buttonBrowsePlaySound, SIGNAL(clicked()),
	                                 SLOT(m_browsePlaySound()));
	connect(m_buttonBrowseRunCommand, SIGNAL(clicked()),
	                                  SLOT(m_browseRunCommand()));
	connect(m_checkPlaySound, SIGNAL(toggled(bool)),
	                          SLOT(m_enablePlaySound(bool)));
	connect(m_checkRunCommand, SIGNAL(toggled(bool)),
	                           SLOT(m_enableRunCommand(bool)));
}

KBiffNewMailPrefs::~KBiffNewMailPrefs()
{
}

bool KBiffNewMailPrefs::getRunCommand()
{
	return m_checkRunCommand->isChecked();
}

const char* KBiffNewMailPrefs::getRunCommandPath()
{
	return m_editRunCommand->text();
}

bool KBiffNewMailPrefs::getPlaySound()
{
	return m_checkPlaySound->isChecked();
}

const char* KBiffNewMailPrefs::getPlaySoundPath()
{
	return m_editPlaySound->text();
}

bool KBiffNewMailPrefs::getBeep()
{
	return m_checkBeep->isChecked();
}

void KBiffNewMailPrefs::setRunCommand(bool run)
{
	m_checkRunCommand->setChecked(run);
	m_enableRunCommand(run);
}

void KBiffNewMailPrefs::setRunCommandPath(const char* path)
{
	m_editRunCommand->setText(path);
}

void KBiffNewMailPrefs::setPlaySound(bool play)
{
	m_checkPlaySound->setChecked(play);
	m_enablePlaySound(play);
}

void KBiffNewMailPrefs::setPlaySoundPath(const char* path)
{
	m_editPlaySound->setText(path);
}

void KBiffNewMailPrefs::setBeep(bool beep)
{
	m_checkBeep->setChecked(beep);
}

void KBiffNewMailPrefs::m_enableRunCommand(bool enable)
{
	m_editRunCommand->setEnabled(enable);
	m_buttonBrowseRunCommand->setEnabled(enable);
}

void KBiffNewMailPrefs::m_enablePlaySound(bool enable)
{
	m_editPlaySound->setEnabled(enable);
	m_buttonBrowsePlaySound->setEnabled(enable);
}

void KBiffNewMailPrefs::m_browseRunCommand()
{
  QString command_path = KFileDialog::getOpenFileName();
  if (!command_path.isEmpty() && !command_path.isNull())
  {
    setRunCommandPath(command_path);
  }
}

void KBiffNewMailPrefs::m_browsePlaySound()
{
  QString sound_path = KFileDialog::getOpenFileName();
  if (!sound_path.isEmpty() && !sound_path.isNull())
  {
    setPlaySoundPath(sound_path);
  }
}

///////////////////////////////////////////////////////////////////////
// KBiffGeneralPrefs
///////////////////////////////////////////////////////////////////////
KBiffGeneralPrefs::KBiffGeneralPrefs(QWidget *parent)
	: QWidget(parent)
{
	// the poll time (in seconds)
	QLabel* poll_label = new QLabel(this);
	poll_label->setGeometry(10, 20, 65, 30);
	poll_label->setText(klocale->translate("Poll (sec):"));
	poll_label->adjustSize();

	// the command to run when clicked
	QLabel* command_label = new QLabel(this);
	command_label->setGeometry(10, 50, 65, 30);
	command_label->setText(klocale->translate("Mail client:"));
	command_label->adjustSize();

	int x = QMAX(poll_label->width() + 15, command_label->width() + 15);

	m_editPoll = new QLineEdit(this);
	m_editPoll->setGeometry(x, 15, 60, EDITHEIGHT);

	m_editCommand = new QLineEdit(this);
	m_editCommand->setGeometry(x, 45, 300 - x, EDITHEIGHT);

	// do we dock automatically?
	m_checkDock = new QCheckBox(this);
	m_checkDock->setText(klocale->translate("Dock in panel"));
	m_checkDock->setGeometry(x, 75, 150, 25);
	m_checkDock->adjustSize();

	// should we support session management?
	m_checkNoSession = new QCheckBox(this);
	m_checkNoSession->setText(klocale->translate("Use session management"));
	m_checkNoSession->setGeometry(x, 100, 190, 25);
	m_checkNoSession->adjustSize();

	// group box to hold the icons together
	QGroupBox* icons_groupbox = new QGroupBox(this);
	icons_groupbox->setTitle(klocale->translate("Icons:"));
	icons_groupbox->move(10, 125);

	x = 125 + fontMetrics().lineSpacing();

	// "no mail" pixmap button
	QLabel* nomail_label = new QLabel(this);
	nomail_label->setText(klocale->translate("No Mail:"));
	nomail_label->adjustSize();

	// "old mail" pixmap button
	QLabel* oldmail_label = new QLabel(this);
	oldmail_label->setText(klocale->translate("Old Mail:"));
	oldmail_label->adjustSize();

	// "new mail" pixmap button
	QLabel* newmail_label = new QLabel(this);
	newmail_label->setText(klocale->translate("New Mail:"));
	newmail_label->adjustSize();

	nomail_label->move(25, x);
	oldmail_label->move(158 - (oldmail_label->width()/2),x);
	newmail_label->move(290 - newmail_label->width(),x);

	x += nomail_label->height() + 5;

	m_buttonNoMail = new KIconLoaderButton(this);
	m_buttonNoMail->setGeometry(
		nomail_label->x() + (nomail_label->width()/2) - 25,
		x, 50, 50);

	m_buttonOldMail = new KIconLoaderButton(this);
	m_buttonOldMail->setGeometry(
		oldmail_label->x() + (oldmail_label->width()/2) - 25,
		x, 50, 50);

	m_buttonNewMail = new KIconLoaderButton(this);
	m_buttonNewMail->setGeometry(
		newmail_label->x() + (newmail_label->width()/2) - 25,
		x, 50, 50);

	icons_groupbox->resize(295,
		(fontMetrics().lineSpacing() * 2) + 65);
}

KBiffGeneralPrefs::~KBiffGeneralPrefs()
{
}

void KBiffGeneralPrefs::setSessionManagement(bool enable)
{
	m_checkNoSession->setChecked(enable);
}

const bool KBiffGeneralPrefs::getSessionManagement()
{
	return m_checkNoSession->isChecked();
}

void KBiffGeneralPrefs::setDock(bool dock)
{
	m_checkDock->setChecked(dock);
}

const bool KBiffGeneralPrefs::getDock()
{
	return m_checkDock->isChecked();
}

void KBiffGeneralPrefs::setButtonNewMail(const char* new_mail)
{
	m_buttonNewMail->setIcon(new_mail);
	m_buttonNewMail->setPixmap(kapp->getIconLoader()->loadIcon(new_mail));
}

void KBiffGeneralPrefs::setButtonOldMail(const char* old_mail)
{
	m_buttonOldMail->setIcon(old_mail);
	m_buttonOldMail->setPixmap(kapp->getIconLoader()->loadIcon(old_mail));
}

void KBiffGeneralPrefs::setButtonNoMail(const char* no_mail)
{
	m_buttonNoMail->setIcon(no_mail);
	m_buttonNoMail->setPixmap(kapp->getIconLoader()->loadIcon(no_mail));
}

/*
void KBiffGeneralPrefs::setMailbox(const char* mailbox)
{
	m_editMailbox->setText(mailbox);
}
*/

void KBiffGeneralPrefs::setPoll(int poll)
{
	QString poll_str;
	poll_str.sprintf("%d", poll);
	m_editPoll->setText(poll_str);
}

void KBiffGeneralPrefs::setCommand(const char* command)
{
	m_editCommand->setText(command);
}

const char* KBiffGeneralPrefs::getButtonOldMail()
{
	return m_buttonOldMail->icon();
}

const char* KBiffGeneralPrefs::getButtonNewMail()
{
	return m_buttonNewMail->icon();
}

const char* KBiffGeneralPrefs::getButtonNoMail()
{
	return m_buttonNoMail->icon();
}

/*
const char* KBiffGeneralPrefs::getMailbox()
{
	return m_editMailbox->text();
}
*/

const char* KBiffGeneralPrefs::getCommand()
{
	return m_editCommand->text();
}

const int KBiffGeneralPrefs::getPoll()
{
	return atoi(m_editPoll->text());
}

//////////////////////////////////////////////////////////////////////
// KBiffSetupDlg
//////////////////////////////////////////////////////////////////////
KBiffSetupDlg::KBiffSetupDlg(const char *name)
	: QDialog(0, 0, true, 0)
{
	int y;

	// set the name of the config file
	m_configFileName = KApplication::localconfigdir();
	m_configFileName += "/kbiffrc";

	// the name of the current profile can be passed along
	m_name = name;

	// we want the name gone when we delete it
	m_mailboxList.setAutoDelete(true);

	// my name is
	setCaption(klocale->translate("KBiff Setup"));

	// mailbox groupbox
	QGroupBox* mailbox_groupbox;
	mailbox_groupbox = new QGroupBox(this);
	mailbox_groupbox->setGeometry(10, 10, 305, 100);
	mailbox_groupbox->setTitle(klocale->translate("Mailbox:"));

	// combo box to hold the profile names
	m_comboMailbox = new QComboBox(false, this);
	m_comboMailbox->setGeometry(25, 30, 275, EDITHEIGHT);
	connect(m_comboMailbox, SIGNAL(highlighted(int)),
	                        SLOT(m_mailboxWasSelected(int)));
	m_comboMailbox->setSizeLimit(10);

	y = m_comboMailbox->height() + 40;

	// add new profile button
	QPushButton* add_button;
	add_button = new QPushButton(this);
	add_button->setGeometry(25, y, 85, 30);
	connect(add_button, SIGNAL(clicked()), SLOT(m_addNewMailbox()));
	add_button->setText(klocale->translate("&New..."));

	// rename current profile button
	QPushButton* rename_button = new QPushButton(this);
	rename_button->setGeometry(120, y, 85, 30);
	connect(rename_button, SIGNAL(clicked()), SLOT(m_renameMailbox()));
	rename_button->setText(klocale->translate("&Rename..."));

	// delete current profile button
	QPushButton* delete_button;
	delete_button = new QPushButton(this);
	delete_button->setGeometry(215, y, 85, 30);
	connect(delete_button, SIGNAL(clicked()), SLOT(m_deleteMailbox()));
	delete_button->setText(klocale->translate("&Delete"));

	y += 30;

	mailbox_groupbox->resize(305, y);

	// setup the tab control
	m_tab = new KTabCtl(this);
	m_generalPrefs = new KBiffGeneralPrefs(m_tab);
	m_newmailPrefs = new KBiffNewMailPrefs(m_tab);
	m_aboutPrefs = new KBiffAbout(m_tab);
	m_mailboxPrefs = new KBiffMailboxInfo(m_tab);

	// we need to use signals/slots here since m_generalPrefs is
	// buried within the tab dialog.
	connect(this, SIGNAL(signal_setButtonNewMail(const char*)),
	        m_generalPrefs, SLOT(setButtonNewMail(const char*)));
	connect(this, SIGNAL(signal_setButtonNoMail(const char*)),
	        m_generalPrefs, SLOT(setButtonNoMail(const char*)));
	connect(this, SIGNAL(signal_setButtonOldMail(const char*)),
	        m_generalPrefs, SLOT(setButtonOldMail(const char*)));
	connect(this, SIGNAL(signal_setPoll(int)),
	        m_generalPrefs, SLOT(setPoll(int)));
	connect(this, SIGNAL(signal_setCommand(const char*)),
	        m_generalPrefs, SLOT(setCommand(const char*)));
	connect(this, SIGNAL(signal_setDock(bool)),
	        m_generalPrefs, SLOT(setDock(bool)));
	connect(this, SIGNAL(signal_setSessionManagement(bool)),
	        m_generalPrefs, SLOT(setSessionManagement(bool)));

	// same for m_newmailPrefs
	connect(this, SIGNAL(signal_setRunCommand(bool)),
	        m_newmailPrefs, SLOT(setRunCommand(bool)));
	connect(this, SIGNAL(signal_setRunCommandPath(const char*)),
	        m_newmailPrefs, SLOT(setRunCommandPath(const char*)));
	connect(this, SIGNAL(signal_setPlaySound(bool)),
	        m_newmailPrefs, SLOT(setPlaySound(bool)));
	connect(this, SIGNAL(signal_setPlaySoundPath(const char*)),
	        m_newmailPrefs, SLOT(setPlaySoundPath(const char*)));
	connect(this, SIGNAL(signal_setBeep(bool)),
	        m_newmailPrefs, SLOT(setBeep(bool)));

	// and same for m_mailboxPrefs
	connect(this, SIGNAL(signal_setMailbox(const char*)),
	        m_mailboxPrefs, SLOT(setMailbox(const char*)));
	connect(this, SIGNAL(signal_setPopServer(const char*)),
	        m_mailboxPrefs, SLOT(setPopServer(const char*)));
	connect(this, SIGNAL(signal_setUser(const char*)),
	        m_mailboxPrefs, SLOT(setUser(const char*)));
	connect(this, SIGNAL(signal_setPassword(const char*)),
	        m_mailboxPrefs, SLOT(setPassword(const char*)));
	connect(this, SIGNAL(signal_setReadLocalMailbox(bool)),
	        m_mailboxPrefs, SLOT(setReadLocalMailbox(bool)));
	connect(this, SIGNAL(signal_setStorePassword(bool)),
	        m_mailboxPrefs, SLOT(setStorePassword(bool)));

	// add the tabs
	m_tab->addTab(m_generalPrefs, klocale->translate("General"));
	m_tab->addTab(m_newmailPrefs, klocale->translate("New Mail"));
	m_tab->addTab(m_mailboxPrefs, klocale->translate("Mailbox"));
	m_tab->addTab(m_aboutPrefs, klocale->translate("About"));

	y += 15;

	m_tab->setGeometry(5, y, 315, 280);

	y += 285;

	// help button
	m_help = new QPushButton(klocale->translate("Help"), this);
	m_help->setGeometry(10, y, 75, 30);
	connect(m_help, SIGNAL(clicked()), SLOT(showHelp()));

	// ok button
	m_ok = new QPushButton(klocale->translate("OK"), this);
	m_ok->setGeometry(155, y, 75, 30);
	m_ok->setDefault(true);
	connect(m_ok, SIGNAL(clicked()), SLOT(m_slotDone()));
	connect(this, SIGNAL(m_accept()), SLOT(accept()));

	// cancel button
	m_cancel = new QPushButton(klocale->translate("Cancel"), this);
	m_cancel->setGeometry(240, y, 75, 30);
	connect(m_cancel, SIGNAL(clicked()), SLOT(reject()));

	y += 40;

	// set the size.  we do NOT want the user to resize this
	resize(325, y);
	setMaximumSize(325, y);
	setMinimumSize(325, y);

	// read in the config
	m_readConfig();
}

KBiffSetupDlg::~KBiffSetupDlg()
{
}

void KBiffSetupDlg::showHelp()
{
	KApplication::getKApplication()->invokeHTMLHelp("kbiff/kbiff.html","");
}

void KBiffSetupDlg::m_mailboxWasSelected(int index)
{
	readMailboxConfig(m_comboMailbox->text(index));
	m_setWidgets();
}

void KBiffSetupDlg::m_addNewMailbox()
{
	KBiffNewMailboxDlg dlg;

	// popup the name chooser
	if (dlg.exec())
	{
		QString mailbox_name;
		mailbox_name = dlg.getMailboxName();

		// bail out if we already have this name
		for (int i = 0; i < m_comboMailbox->count(); i++)
		{
			if (mailbox_name == m_comboMailbox->text(i))
				return;
		}

		// continue only if we received a decent name
		if (mailbox_name.isNull() == false)
		{
			m_name = mailbox_name;
			m_comboMailbox->insertItem(mailbox_name, 0);
			m_mailboxList.append(mailbox_name);

			// setup some defaults
			QFileInfo mailbox(getenv("MAIL"));
			if (mailbox.exists() == false)
			{
				QString s("/var/spool/mail/");
				s += getlogin();
				mailbox.setFile(s);
			}

			m_mailboxPath = mailbox.absFilePath();

			m_oldMailPixmapPath = "oldmail.xpm";
			m_newMailPixmapPath = "newmail.xpm";
			m_noMailPixmapPath = "nomail.xpm";
			char *mailer = getenv("MAILER");
			if (mailer)
				m_command = mailer;
			else
				m_command = "kmail";
			m_poll = 60;
			m_dock = true;
			m_sessionManagement = true;

			m_runCommand = false;
			m_runCommandPath = "";
			m_playSound = false;
			m_playSound = "";
			m_beep = true;

			m_popServer = "";
			m_username = "";
			m_password = "";
			m_port     = 110;
			m_readLocalMailbox = true;
			m_storePassword = false;

			// now that the defaults are set, display them
			m_setWidgets();

			// and save the settings
			m_saveSettings();
		}
	}
}

void KBiffSetupDlg::m_renameMailbox()
{
	KBiffNewMailboxDlg dlg;

	dlg.setCaption(klocale->translate("New Profile Name"));
	// popup the name chooser
	if (dlg.exec())
	{
		QString mailbox_name;
		mailbox_name = dlg.getMailboxName();

		// bail out if we already have this name
		for (int i = 0; i < m_comboMailbox->count(); i++)
		{
			if (mailbox_name == m_comboMailbox->text(i))
				return;
		}

		// continue only if we received a decent name
		if (mailbox_name.isNull() == false)
		{
			// remove the old name from the list and append the new one
			m_mailboxList.remove(m_mailboxList.find(m_name));
			m_mailboxList.append(mailbox_name);

			// remove the old reference from the config file
			KSimpleConfig *config = new KSimpleConfig(m_configFileName);
			// write the mailboxes
			config->setGroup("General");
			config->writeEntry("Mailboxes", m_mailboxList);

			// nuke the group
			config->deleteGroup(m_name, true);
			delete config;

			// set the new name and save the current settings
			m_name = mailbox_name;
			m_saveSettings();

			// clear the combo box and add the updated list
			m_comboMailbox->clear();
			m_comboMailbox->insertStrList(&m_mailboxList);

			// select the last mailbox profile (our new one)
			m_comboMailbox->setCurrentItem(m_mailboxList.count()-1);
		}
	}
}

void KBiffSetupDlg::m_deleteMailbox()
{
	QString title, msg;

	title.sprintf(klocale->translate("Delete Mailbox Profile: %s"), (const char*)m_name);
	msg.sprintf(klocale->translate("Are you sure you wish to delete this profile?\n"),
	             (const char*)m_name);
	
	switch (QMessageBox::warning(this, title, msg,
	                             QMessageBox::Yes | QMessageBox::Default,
								 QMessageBox::No | QMessageBox::Escape))
	{
		case QMessageBox::Yes:
		{
			// remove the mailbox from the list
			m_mailboxList.remove(m_mailboxList.find(m_name));

			// remove the reference from the config file
			KSimpleConfig *config = new KSimpleConfig(m_configFileName);
			// write the mailboxes
			config->setGroup("General");
			config->writeEntry("Mailboxes", m_mailboxList);

			// nuke the group
			config->deleteGroup(m_name, true);

			delete config;

			// clear the combo box and add the updated list
			m_comboMailbox->clear();

			// if we no longer have any mailboxes, go with defaults
			if (m_mailboxList.count() > 0)
			{
				m_comboMailbox->insertStrList(&m_mailboxList);

				// arbitrarily select the first mailbox to select
				m_mailboxWasSelected(0);
			}
			else
				m_initDefaults();

			break;
		}
		case QMessageBox::No:
		default:
			break;
	}
}

const char* KBiffSetupDlg::getPopServer()
{
	return m_popServer;
}	

const int KBiffSetupDlg::getPort()
{
	return m_port;
}

const char* KBiffSetupDlg::getUser()
{
	return m_username;
}

const char* KBiffSetupDlg::getPassword()
{
	return m_password;
}

const bool KBiffSetupDlg::getReadLocalMailbox()
{
	return m_readLocalMailbox;
}

const bool KBiffSetupDlg::getRunCommand()
{
	return m_runCommand;
}

const char* KBiffSetupDlg::getRunCommandPath()
{
	return (const char*)m_runCommandPath;
}

const bool KBiffSetupDlg::getPlaySound()
{
	return m_playSound;
}

const char* KBiffSetupDlg::getPlaySoundPath()
{
	return (const char*)m_playSoundPath;
}

const bool KBiffSetupDlg::getBeep()
{
	return m_beep;
}

const bool KBiffSetupDlg::getSessionManagement()
{
	return m_sessionManagement;
}

const bool KBiffSetupDlg::getDock()
{
	return m_dock;
}

const char* KBiffSetupDlg::getOldMailPixmapPath()
{
	return m_generalPrefs->getButtonOldMail();
}

const char* KBiffSetupDlg::getNewMailPixmapPath()
{
	return m_generalPrefs->getButtonNewMail();
}

const char* KBiffSetupDlg::getNoMailPixmapPath()
{
	return m_generalPrefs->getButtonNoMail();
}

const char* KBiffSetupDlg::getMailboxPath()
{
	return m_mailboxPath;
}

const char* KBiffSetupDlg::getCommand()
{
	return m_command;
}

void KBiffSetupDlg::setName(const QString& name)
{
	m_name = name;
}

const char* KBiffSetupDlg::getName()
{
	return m_name;
}

const int KBiffSetupDlg::getPoll()
{
	return m_poll;
}

void KBiffSetupDlg::m_initDefaults()
{
	// setup some defaults first (just in case)
	QFileInfo mailbox(getenv("MAIL"));
	if (mailbox.exists() == false)
	{
		QString s("/var/spool/mail/");
		s += getlogin();
		mailbox.setFile(s);
	}

	m_mailboxPath = mailbox.absFilePath();

	m_oldMailPixmapPath = "oldmail.xpm";
	m_newMailPixmapPath = "newmail.xpm";
	m_noMailPixmapPath = "nomail.xpm";

	m_name = klocale->translate("Inbox");

	char *mailer = getenv("MAILER");
	if (mailer)
		m_command = mailer;
	else
		m_command = "kmail";
	m_poll = 60;

	m_dock = true;
	m_sessionManagement = true;

	m_runCommand = false;
	m_runCommandPath = "";
	m_playSound = false;
	m_playSoundPath = "";
	m_beep = true;

	m_popServer = "";
	m_username = "";
	m_password = "";
	m_readLocalMailbox = true;
	m_storePassword = false;

	// now that the defaults are set, display them
	m_setWidgets();

	// add the name to the combo box
	m_comboMailbox->insertItem(m_name);

	// add the name to the mailbox list
	m_mailboxList.append(m_name);

	// and save the settings
	m_saveSettings();
}

void KBiffSetupDlg::m_saveSettings()
{
	KSimpleConfig *config = new KSimpleConfig(m_configFileName);

	// write the mailboxes
	config->setGroup("General");
	config->writeEntry("Mailboxes", m_mailboxList);

	// now write the current mailbox (just to be safe)
	config->setGroup(m_name);
	config->writeEntry("NoMailPixmapPath", m_noMailPixmapPath);
	config->writeEntry("NewMailPixmapPath", m_newMailPixmapPath);
	config->writeEntry("OldMailPixmapPath", m_oldMailPixmapPath);
	config->writeEntry("MailboxPath", m_mailboxPath);
	config->writeEntry("Poll", m_poll);
	config->writeEntry("Command", m_command);
	config->writeEntry("Docked", m_dock);
	config->writeEntry("SessionManagement", m_sessionManagement);
	config->writeEntry("RunCommand", m_runCommand);
	config->writeEntry("RunCommandPath", m_runCommandPath);
	config->writeEntry("PlaySound", m_playSound);
	config->writeEntry("PlaySoundPath", m_playSoundPath);
	config->writeEntry("Beep", m_beep);
	config->writeEntry("PopServer", m_popServer);
	config->writeEntry("Username", m_username);
	config->writeEntry("ReadLocal", m_readLocalMailbox);
	config->writeEntry("StorePassword", m_storePassword);
	if (m_storePassword)
		config->writeEntry("Password", m_scramble.scramble(m_password));
	config->sync();
}

void KBiffSetupDlg::m_readConfig()
{
	// get the config spec
	KSimpleConfig *config = new KSimpleConfig(m_configFileName);

	// read in the general information
	config->setGroup("General");

	// read in the list of mailboxes
	int number_of_mailboxes = config->readListEntry("Mailboxes", m_mailboxList);

	delete config;

	// check if we have any mailboxes to read in
	if (number_of_mailboxes > 0)
	{
		// load up the combo box
		m_comboMailbox->insertStrList(&m_mailboxList);

		// read in the data from the first mailbox if we don't have a name
		if (m_name.isNull())
			readMailboxConfig(m_mailboxList.getFirst());
		else
		{
			readMailboxConfig(m_name);
			
			for (int i = 0; i < m_comboMailbox->count(); i++)
			{
				if (m_name == m_comboMailbox->text(i))
				{
					m_comboMailbox->setCurrentItem(i);
					break;
				}
			}
		}
		m_setWidgets();
	}
	else
		m_initDefaults();
}

bool KBiffSetupDlg::readMailboxConfig(const QString& name)
{
	KSimpleConfig *config = new KSimpleConfig(m_configFileName);
	config->setGroup(name);
	m_name = name;

	// read in the entries from the config file
	m_noMailPixmapPath = config->readEntry("NoMailPixmapPath");
	m_newMailPixmapPath = config->readEntry("NewMailPixmapPath");
	m_oldMailPixmapPath = config->readEntry("OldMailPixmapPath");
	m_mailboxPath = config->readEntry("MailboxPath");
	m_poll = config->readNumEntry("Poll");
	m_command = config->readEntry("Command");
	m_dock = config->readBoolEntry("Docked");
	m_sessionManagement = config->readBoolEntry("SessionManagement");
	m_playSound = config->readBoolEntry("PlaySound");
	m_playSoundPath = config->readEntry("PlaySoundPath");
	m_runCommand = config->readBoolEntry("RunCommand");
	m_runCommandPath = config->readEntry("RunCommandPath");
	m_beep = config->readBoolEntry("Beep");
	m_popServer = config->readEntry("PopServer");

	m_checkForPort();

	m_username = config->readEntry("Username");
	m_readLocalMailbox = config->readBoolEntry("ReadLocal");
	m_storePassword = config->readBoolEntry("StorePassword");
	if (m_storePassword)
		m_password = m_scramble.unscramble(config->readEntry("Password"));

	delete config;

	if (m_oldMailPixmapPath.isNull())
		return false;
	else
		return true;
}

void KBiffSetupDlg::m_checkForPort()
{
	// see if we have the port on the server
	if (m_popServer.contains(':') > 0)
	{
		// yes.  now extract it
		QString tmp_port(m_popServer);
		tmp_port = m_popServer.mid(m_popServer.find(':') + 1, m_popServer.length());
		m_port = tmp_port.toInt();
		m_popServer = m_popServer.left(m_popServer.find(':'));
	}
	else
		m_port = 110;
}

void KBiffSetupDlg::m_setWidgets()
{
	// set the appropriate entries
	emit(signal_setButtonNewMail(m_newMailPixmapPath));
	emit(signal_setButtonNoMail(m_noMailPixmapPath));
	emit(signal_setButtonOldMail(m_oldMailPixmapPath));
	emit(signal_setMailbox(m_mailboxPath));
	emit(signal_setPoll(m_poll));
	emit(signal_setCommand(m_command));
	emit(signal_setDock(m_dock));
	emit(signal_setSessionManagement(m_sessionManagement));
	emit(signal_setRunCommand(m_runCommand));
	emit(signal_setRunCommandPath(m_runCommandPath));
	emit(signal_setPlaySound(m_playSound));
	emit(signal_setPlaySoundPath(m_playSoundPath));
	emit(signal_setBeep(m_beep));

	// do special things for the port
	if (m_port != 110)
	{
		QString tmp_server;
		tmp_server.sprintf("%s:%d", (const char*)m_popServer, m_port);
		emit(signal_setPopServer(tmp_server));
	}
	else
		emit(signal_setPopServer(m_popServer));
	emit(signal_setUser(m_username));
	emit(signal_setPassword(m_password));
	emit(signal_setReadLocalMailbox(m_readLocalMailbox));
	emit(signal_setStorePassword(m_storePassword));
}

void KBiffSetupDlg::m_slotDone()
{
	// get the general settings
	m_poll = m_generalPrefs->getPoll();
	m_command = m_generalPrefs->getCommand();
	m_dock = m_generalPrefs->getDock();
	m_sessionManagement = m_generalPrefs->getSessionManagement();
	m_newMailPixmapPath = m_generalPrefs->getButtonNewMail();
	m_oldMailPixmapPath = m_generalPrefs->getButtonOldMail();
	m_noMailPixmapPath = m_generalPrefs->getButtonNoMail();

	// get the new mail settings
	m_runCommand = m_newmailPrefs->getRunCommand();
	m_runCommandPath = m_newmailPrefs->getRunCommandPath();
	m_playSound = m_newmailPrefs->getPlaySound();
	m_playSoundPath = m_newmailPrefs->getPlaySoundPath();
	m_beep = m_newmailPrefs->getBeep();

	// get the mailbox settings
	m_readLocalMailbox = m_mailboxPrefs->getReadLocalMailbox();

	if (m_readLocalMailbox)
		m_mailboxPath = m_mailboxPrefs->getMailbox();
	else
	{
		m_popServer = m_mailboxPrefs->getPopServer();
		m_username = m_mailboxPrefs->getUser();
		m_password = m_mailboxPrefs->getPassword();
		m_storePassword = m_mailboxPrefs->getStorePassword();
	}

	// save 'em
	m_saveSettings();

	// do special things for pop port
	if (!m_readLocalMailbox)
		m_checkForPort();

	// and be done with it
	emit m_accept();
}

////////////////////////////////////////////////////////////////////////
KBiffNewMailboxDlg::KBiffNewMailboxDlg(QWidget* parent, const char* name)
	: QDialog(parent, name, true, 0)
{
	int y = 10;

	QLabel* label1 = new QLabel(this);
	label1->setText(klocale->translate("Mailbox Name (i.e., Inbox):"));
	label1->setGeometry(10, 10, 180, 20);
	label1->adjustSize();

	y += label1->height() + 5;

	// mailbox profile edit box
	m_editMailbox = new QLineEdit(this);
	m_editMailbox->setGeometry(10, y,
		QMAX(label1->width(), 180),
		EDITHEIGHT);
	m_editMailbox->setFocus();

	y += m_editMailbox->height() + 5;

	// ok button
	QPushButton* button_ok;
	button_ok = new QPushButton(this);
	button_ok->setGeometry(10, y, 80, 30);
	connect(button_ok, SIGNAL(clicked()), SLOT(accept()));
	button_ok->setDefault(true);
	button_ok->setText(klocale->translate("OK"));

	// cancel button
	QPushButton* button_cancel;
	button_cancel = new QPushButton(this);
	button_cancel->setGeometry(m_editMailbox->width() - 70, y, 80, 30);
	connect(button_cancel, SIGNAL(clicked()), SLOT(reject()));
	button_cancel->setText(klocale->translate("Cancel"));

	y += button_ok->height() + 5;

	// set the size
	resize(m_editMailbox->width()+20, y);

	// set my name
	setCaption(klocale->translate("New Mailbox Profile Name"));
}


KBiffNewMailboxDlg::~KBiffNewMailboxDlg()
{
}

