/* -*- C++ -*-
 * This test creates a database of adresses and 
 * configuration values. It also reports any errors
 * that happen using the database functionality.
 * 
 * the Configuration Database library
 * copyright:  (C) Mirko Sucker, 1998
 * license:    GNU Public License, Version 2
 * mail to:    Mirko Sucker <mirko.sucker@hamburg.netsurf.de>
 *                          <mirko.sucker@unibw-hamburg.de>
 * requires:   C++-compiler, STL, string class, Qt
 *             NANA (for debugging)
 * $Revision: 1.2 $
 */
  
#ifndef ADDRESSBOOK_H
#define ADDRESSBOOK_H

#include "configDB.h"
#include <qdatetm.h>

class AddressBook : public ConfigDB
{
public:
  /** An object of the class {\em Entry} represents one entry.
    * More fields can easily be added: add a new string, find a
    * name for it in the entry-sections of the database that is 
    * not used until now, and change the {\tt makeEntryFromSection}-,
    * the {\tt add(..)}- and the {\tt change}-methods.
    */
  class Entry {
  public:
    // changed on April 4 1998
	 // (added talk-address-list)
    // ----------------------------------
    string name;
    string firstname;
    string additionalName;
    string namePrefix;
    string fn; // formatted name 
    string comment;
    string org; // organization -
    string orgUnit; 
    string orgSubUnit; 
    string title; 
    string role; 
    // QPixmap logo;
    QDate birthday; // now implemented
    list<string> talk; // list of talk addresses
    // string sound; // base64 coded wav - file
    // string agent; // contains a complete vCard possibly?
    string deliveryLabel; // fehlt!
    string email; 
    string email2; 
    string email3; 
    string address; 
    string town; 
    string telephone; 
    string fax; 
    string modem; 
    string URL; 
  };
protected:
  /** This method will be called by all REQUIRE and ENSURE statements. It
    * returns {\tt true} if no inconsistencies regarding the internal 
    * data structures can be found.
    * You need to be careful not to create endless looping here!
    * (See the Nana documentation, "eiffel.h", for details.)
    */
  bool invariant();
  /** The DB uses another map to access the entries in different orders.
    * This map has always the same number of entries as there are subsections
    * in the entry-section. The map contains strings describing the order of 
    * the entries (which can be reordered) and according to each string the 
    * {\em subsection name} of the entry in the entry-section. The 
    * {\tt current}-iterator is used to have one entry that is currently 
    * displayed.
    */
  typedef map<string, string, less<string> > StringStringMap;
  StringStringMap entries;
  StringStringMap::iterator current;
  bool isFirstEntry();
  bool isLastEntry();
  /** This method sets current to the element in entries at the given index.
    */
  bool setCurrent(int index); 
  /** This method sets the current key to the one with the given key 
    * in the ENTRIES-section. This is reverse to the method before!
    */
  bool setCurrent(const string& key); 
  /** This method creates the mirror map by inserting an entry for each 
    * (DB) entry in it. It sets current to the entry with the key <key>.
    */
  void updateEntriesMap(string key="");
  /// Convenience methods to retrieve frequently needed pointers and numbers.
  Section* configSection();
  Section* entrySection();
  unsigned int noOfEntries();
  /** See the implementation file addressbook.cc for the contents of the 
    * constants.
    */
  static const char* ConfigSection;
  static const char* EntrySection;
  static const char* DBFileName;
  static const char* Defaults[];
  /** This converts a subsection of the entry-section to an object of the 
    * {\em Entry}-class.
    */
  bool makeEntryFromSection(Section&, Entry&);
  /// Get a verbose name (human readable) for an entry:
  string getName(const string& key); 
  /// internal!
  string nextAvailEntryKey();  
  // creates a new database:
  bool createNew(string filename);
public:
  /// This methods retrieve the current entry.
  bool currentEntry(Section**);
  bool currentEntry(Entry&);
  /** The following methods are used to navigate through the entries.
    * They all change the current-iterator.
    */
  bool first();
  bool previous();
  bool next();
  bool last();
  // managing entries
  // add an entry, return the new key for further operations:
  bool add(const Entry&, string& key);
  inline bool add(string& key); // empty entry 
  bool change(const Entry&); // change the current entry
  bool remove(); // the current one
  // constructor
  AddressBook(bool readonly=false);
  // -----
  void restoreDefaults();
  void changed(); // virtual from ConfigDB
  virtual void currentChanged(); // for notifications
};

#endif // ADDRESSBOOK_H

