#include "progress.h"

#include <qpainter.h>
#include <qstring.h>
#include <qpixmap.h>
#include <kapp.h>

myProgress::myProgress(QWidget *parent, const char *name)	: QFrame(parent, name),
	QRangeControl(0, 100, 1, 10, 0),
	orient(Horizontal)
{
}

myProgress::myProgress(Orientation orientation, QWidget *parent, const char *name)	: QFrame(parent, name),
	
	QRangeControl(0, 100, 1, 10, 0),
	orient(orientation)
{
	initialize();
}

myProgress::myProgress(	int minValue,
											int maxValue,
											int value, 
											Orientation orientation,
											QWidget *parent, const char *name)	: QFrame(parent, name),
	
	QRangeControl(minValue, maxValue, 1, 10, value),
	orient(orientation)
{
	initialize();
}

myProgress::~myProgress()
{
}

void myProgress::setValueRange(int minValue, int maxValue)
{
	QRangeControl::setRange(minValue, maxValue);
}

void myProgress::advance(int offset)
{
	setValue(value() + offset);
}

void myProgress::initialize()
{
	bar_pixmap = 0;
	bar_style = Solid;
	bar_color = kapp->selectColor;
	bar_text_color = kapp->selectTextColor;
	text_color = kapp->textColor;
	setBackgroundColor( kapp->windowColor );
	setFont(QFont("helvetica", 12, QFont::Bold));
	text_enabled = TRUE;
	adjustStyle();
}


void myProgress::setBarPixmap(const QPixmap &pixmap)
{
	if (pixmap.isNull())
		return;
	if (bar_pixmap)
		delete bar_pixmap;
		
	bar_pixmap = new QPixmap(pixmap);
}

void myProgress::setBarColor(const QColor &color)
{
	bar_color = color;
	if (bar_pixmap) {
		delete bar_pixmap;
		bar_pixmap = 0;
	}
}

void myProgress::setBarStyle(BarStyle style)
{
	if (bar_style != style) {
		bar_style = style;
		update();
	}
}

void myProgress::setOrientation(Orientation orientation)
{
	if (orient != orientation) {
		orient = orientation;
		update();
	}
}
 
void myProgress::setValue(int value)
{
	QRangeControl::setValue(value);
}

void myProgress::setTextEnabled(bool enable)
{
	text_enabled = enable;
}

const QColor & myProgress::barColor() const
{
	return bar_color;
}

const QPixmap * myProgress::barPixmap() const
{
	return bar_pixmap;
}

bool myProgress::textEnabled() const
{
	return text_enabled;
}

QSize myProgress::sizeHint() const
{
  QSize s( size() );

  if(orientation() == myProgress::Vertical) {
    s.setWidth(24);
  } else {
    s.setHeight(24);
  }

  return s;
}


myProgress::Orientation myProgress::orientation() const
{
	return orient;
}

myProgress::BarStyle myProgress::barStyle() const
{
	return bar_style;
}

int myProgress::recalcValue(int range)
{
	int abs_value = value() - minValue();
	int abs_range = maxValue() - minValue();
	return range * abs_value / abs_range;
}

void myProgress::valueChange()
{
	repaint(contentsRect(), FALSE);
	emit percentageChanged(recalcValue(100));
}

void myProgress::rangeChange()
{
	repaint(contentsRect(), FALSE);
	emit percentageChanged(recalcValue(100));
}

void myProgress::styleChange(GUIStyle)
{
	adjustStyle();
}

void myProgress::adjustStyle()
{
	switch (style()) {
		case WindowsStyle:
			setFrameStyle(QFrame::WinPanel | QFrame::Sunken);
			break;
		case MotifStyle:
		default:
			setFrameStyle(QFrame::Panel | QFrame::Sunken);
			setLineWidth( 2 );
			break;
	}
	update();
}

void myProgress::paletteChange( const QPalette & )
{
	bar_color = kapp->selectColor;
	bar_text_color = kapp->selectTextColor;
	text_color = kapp->textColor;
	setBackgroundColor( kapp->windowColor );
}
		
void myProgress::drawText(QPainter *p)
{
	QRect r(contentsRect());
	QString s;
	
	s.sprintf("%i%%", recalcValue(100));
	p->setPen(text_color);
	p->drawText(r, AlignCenter, s);
	p->setClipRegion( fr );
	p->setPen(bar_text_color);
	p->drawText(r, AlignCenter, s);
	
}

void myProgress::drawContents(QPainter *p)
{
	QRect cr = contentsRect(), er = cr;
	fr = cr;
	QBrush fb(bar_color), eb(backgroundColor());

	if (bar_pixmap)
		fb.setPixmap(*bar_pixmap);

	if (backgroundPixmap())
		eb.setPixmap(*backgroundPixmap());

	switch (bar_style) {
		case Solid:
			if (orient == Horizontal) {
				fr.setWidth(recalcValue(cr.width()));
				er.setLeft(fr.right() + 1);
			} else {
				fr.setTop(cr.bottom() - recalcValue(cr.height()));
				er.setBottom(fr.top() - 1);
			}
				
			p->setBrushOrigin(cr.topLeft());
			p->fillRect(fr, fb);
			p->fillRect(er, eb);
			
			break;
			
		case Blocked:
			const int margin = 2;
			int max, num, dx, dy;
			if (orient == Horizontal) {
				fr.setHeight(cr.height() - 2 * margin);
				fr.setWidth((int)(0.67 * fr.height()));
				fr.moveTopLeft(QPoint(cr.left() + margin, cr.top() + margin));
				dx = fr.width() + margin;
				dy = 0;
				max = (cr.width() - margin) / (fr.width() + margin) + 1;
				num = recalcValue(max);
			} else {
				fr.setWidth(cr.width() - 2 * margin);
				fr.setHeight((int)(0.67 * fr.width()));
				fr.moveBottomLeft(QPoint(cr.left() + margin, cr.bottom() - margin));
				dx = 0;
				dy = - (fr.height() + margin);
				max = (cr.height() - margin) / (fr.height() + margin) + 1;
				num = recalcValue(max);
			}
			p->setClipRect(cr.x() + margin, cr.y() + margin, 
						   cr.width() - margin, cr.height() - margin);
			for (int i = 0; i < num; i++) {
				p->setBrushOrigin(fr.topLeft());
				p->fillRect(fr, fb);
				fr.moveBy(dx, dy);
			}
			
			if (num != max) {
				if (orient == Horizontal) 
					er.setLeft(fr.right() + 1);
				else
					er.setBottom(fr.bottom() + 1);
				if (!er.isNull()) {
					p->setBrushOrigin(cr.topLeft());
					p->fillRect(er, eb);
				}
			}
			
			break;
	}	
				
	if (text_enabled && bar_style != Blocked)
		drawText(p);
		
}			



