// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#ifndef KROT_SIMULATION_H
#define KROT_SIMULATION_H



#include "calculationParameter.h"
#include "constants.h"
#include "dataset.h"
#include "functor.h"
#include "krotData.h"
#include "transition.h"

#include <algorithm>
#include <cmath>
#include <vector>

#include <qcolor.h>



/**
 * Provides a container for simulations.
 *
 * @short Simulation container.
 * @author Jochen Kpper
 * @version 1998/10/09
 */
class Simulation : virtual public DataSet
{

public:

    /**
     * Enumeration of the different lineshape profiles the simulation can posses.
     */
    enum Lineshape { LINESHAPE_STICK, LINESHAPE_GAUSS, LINESHAPE_LORENTZ, LINESHAPE_VOIGT };

    /**
     * Constructs the object
     */
    Simulation( KRotData *krotdata=0 );

    /**
     * Constructs the object
     */
    Simulation( const vector< Transition >& lines, KRotData* krotdata=0 );

    /**
     * Makes this object a copy of sim.
     *
     * @param sim object to copy.
     */
    Simulation( const Simulation& sim );

    /**
     * Makes this object a copy of sim.
     *
     * @return *this
     * @param sim object to copy.
     */
    virtual Simulation& operator=( const Simulation& sim );

    /**
     * Fold this simulation by the lineshape function defined by the
     * Gaussian and Lorentzian FWHM and the profile stored internally.
     */
    void fold();

    /**
     * @return Gaussian FWHM of this set.
     */
    double gauss() const;

    /**
     * @return Lineshape of this set.
     */
    Lineshape lineshape() const;

    /**
     * @return Lorentzian FWHM of this set.
     */
    double lorentz() const;

    /**
     * @return Reference to the vector of simulated lines this object was build of.
     */
    const vector<Transition>& lines() const;

    /**
     * Set Gaussian FWHM.
     *
     * @param val New gaussian FWHM.
     */
    void setGauss( const double val );

    /**
     * Set Lineshape profile.
     *
     * @param val New lineshape profile.
     */
    void setLineshape( const Lineshape val );

    /**
     * Set Lorentzian FWHM.
     *
     * @param val New lorentzian FWHM.
     */
    void setLorentz( const double val );


protected:

    /**
     * Berechnet den Funktionswert einer auf die Flche eins normierten Gaussfunktion mit der
     * Zentralfrequenz 1 und mit der FWHM Simulation::_gauss an der Stelle x.
     *
     * See P.F. Bernath, Spectra of atoms and molecules, 1995, p. 31
     *
     * @param x Position to calculate Gauss function at [MHz].
     */
    double gauss( double x );

    /**
     * @return Frequency range we need to include in folding procedure ( -ret ... +ret).
     */
    double gaussLimit();

    /**
     * Berechnet das Voigt-Profil nherungsweise nach:
     * J. Humlicek, J. Quant. Spectrosc. Radiat. Transfer, Vol. 21, p. 309, 1978
     *
     * Voigt-Profil: V(v) = 2/pi^(1.5) * y^2/FWHM_L * integral[-inf,+inf](e^(-y^2)/(x+y)^2+
     *
     * @param x no clue.
     * @param y no clue.
     */
    double humlicek_v12( double x, double y );

    /**
     * Import lines.dat file.
     */
    bool loadLinesDat( const QString& filename );

    /**
     * Calculate the value of a Lorentz function (normalized to area one)
     * centered at the frequency zero and characterized by the internally stored lorentzian FWHM.
     *
     * See P.F. Bernath, Spectra of atoms and molecules, 1995, p. 27
     *
     * @param x Position to calculate function at [MHz].
     */
    double lorentz( double x );

    /**
     * @return Frequency range we need to include in folding procedure ( -ret ... +ret).
     */
    double lorentzLimit();

    /**
     * Recalculate the lineshape profile and mark it valid.
     */ 
    void reProfile();
    
    /**
     * Calculate the value of a Voigt function (normalized to area one)
     * centered at the frequency zero and characterized by the internally stored
     * gaussian and lorentzian FWHM.
     *
     * @param x Position to calculate function at [MHz].
     */
    double voigt( double x );

    /**
     * Calculates the frequency range we need to include in the folding of a voigt
     * profile by the parameters stored in _gauss and _lorentz to obtain an area
     * of (1 - _epsilon).
     *
     * @param epsilon Area error we want to achieve.
     * @return Frequency range we need to include in folding procedure ( -ret ... +ret).
     */
    double voigtLimit();


protected:

    double epsilon;

    double wG, wL;

    vector< Transition > lns;

    CalculationParameter parameter;

    /**
     * Lineshape to use for folding.
     */
    Lineshape profile;

    /**
     * What index of the lineshape function array (profiledata) corresponds to
     * frequency zero.
     */
    int profilecenter;
    
    /**
     * lineshape function. This is only current if profilevalid is true.
     */
    double *profiledata;
    
    /**
     * Is the data in profiledata valid ?
     */
    bool profilevalid;
    
    /**
     * Scale intensities up to get better resolution at mapping on integers.
     */
    const double scaling;
};



#include "simulation_inline.h"



#endif



//* Local Variables:
//* c-file-style: "Stroustrup"
//* mode: C++
//* End:
