// -*- mode: C++; c-file-style: "Stroustrup" -*-
//
// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#ifndef KROT_DIALOG_IMPORT_H
#define KROT_DIALOG_IMPORT_H



#include <qdialog.h>
#include <qwidget.h>

#include <ktmainwindow.h>


class Floated;
class KConfig;
class KRotData;
class QCheckBox;
class QRadioButton;
class RawData;


/**
 * Drawing canvas for DialogImportRawData.
 *
 * @short Drawing area widget for DialogImportRawData.
 * @author Jochen Kpper
 * @version 98/10/09
 */
class RawDataDrawArea : public QWidget
{
    Q_OBJECT

public:

    /**
     * Constructor.
     *
     * @param parent Passed to QWidget constructor.
     * @param name   Passed to QWidget constructor.
     */
    RawDataDrawArea( RawData& khidaq, QWidget *parent=0, const char *name=0 );


protected:

    /**
     * Mouse button pressed inside the canvas.
     *
     * The left button is used to add or remove marker,
     * the middle button leads to an assignment of an absolute frequency.
     */
    void mousePressEvent( QMouseEvent * );

    /**
     * Repaints the entire canvas.
     */
    void paintEvent( QPaintEvent * );


protected slots:
    
    /**
     * Calculate start index from new scrollbar value.
     *
     * @param val Scrollbar value.
     */
    void setStart( int val );


signals:

    /**
     * User wants to make an frequency assignment to index.
     *
     * @param index Index the user selected.
     */
    void assignment( const int index );

    /**
     * User wants to toggle a marker on/off.
     *
     * @param index Index the user selected.
     */
    void marker( const int index );


protected:

    /**
     * keys for KConfig entries of this class
     */
    static const char config_ShowEtalon[] = "ShowEtalon";
    static const char config_ShowIodine[] = "ShowIodine";
    static const char config_ShowPower[]  = "ShowPower";
    static const char config_ShowSignal[] = "ShowSignal";
    static const char config_ShowMarker[] = "ShowMarker";

    /**
     * Pointer to application config file.
     */
    KConfig *config;

    /**
     * Reference of the KHiDaq object we are displaying.
     */
    RawData& data;

    /**
     * Index of the leftmost point in the canvas.
     */
    int start;
};




/**
 * This is a working area canvas for DialogImportRawData, consisting of a
 * horizontal scrollbar and a drawing area. 
 *
 * @short Work area canvas for DialogImport.
 * @author Jochen Kpper
 * @version 1998/10/09
 */
class RawDataWorkArea : public QWidget
{
    Q_OBJECT

public:

    enum Trace { ETALON=0, IODINE, MARKER, POWER, SIGNAL };

    /**
     * Constructor.
     *
     * @param data   Raw data to work on.
     * @param parent Passed to QWidget constructor.
     * @param name   Passed to QWidget constructor.
     */
    RawDataWorkArea( RawData& data, QWidget *parent=0, const char *name=0 );

    /**
     * @param  t    Trace to ask about.
     * @return True If the trace shall be drawn, false otherwise.
     */
    bool show( Trace t );


protected:

    /**
     * Repaints the DrawArea child.
     */
    void paintEvent( QPaintEvent * );

    /**
     * Update Scrollbar range/steps/value.
     *
     */       
    void RawDataWorkArea::resizeEvent( QResizeEvent * );


protected slots:

    /**
     * Assignment of absolute position
     *
     * @param pos Index of point to assign an absolute frequency to.
     */
    void setAssignment( const int pos );

    /**
     * Toggle marker.
     *
     * @param pos Position to add/remove a marker at.
     */
    void setMarker( const int pos );


signals:

    /**
     * Assignment of absolute position
     *
     * @param pos Index of point to assign an absolute frequency to.
     */
    void assignment( const int pos );

    /**
     * User wants to toggle a marker on/off.
     *
     * @param index Index the user selected.
     */
    void marker( const int index );


private:

    /**
     * Reference of the KHiDaq object we are displaying.
     */
    RawData& data;

    RawDataDrawArea *drawarea;

    QScrollBar *scrollbar;
};





/**
 * Dialog to display raw data (etalon, iodine, power signal) and calculated
 * marker positions.
 *
 * It should be possible to define new marker and to recalculate them.
 *
 * @short Display raw data.
 * @author Jochen Kpper
 * @version 98/10/09
 */
class DialogImportRawData : public KTMainWindow
{
    Q_OBJECT
  
    enum ToolBarItems { ID_AVERAGING };
    
public:

    /**
     * Constructor.
     *
     * @param data  data to work on.
     * @param name  Passed to @ref KTopLevelWidget constructor.
     */
    DialogImportRawData( RawData& data, const char *name=0 );


protected slots:

    /**
     * Confirm the import and return.
     */
    void accept();

    /**
     * Input for fsr.
     */
    void fsr();

    /**
     * Set linearization model.
     */
    void linModel();

    /**
     * Discard the import and return.
     */
    void reject();

    /**
     * Assignment of absolute position
     */
    void setAssignment( const int pos );

    /**
     * Averaging interval width changed.
     * Store the new value for further drawing.
     */
    void setAveraging( const char * );

    /**
     * Toggle marker.
     */
    void setMarker( const int pos );

    /**
     * Toggle visibility of this trace.
     */
    void toggleEtalon();

    /**
     * Toggle visibility of this trace.
     */
    void toggleIodine();

    /**
     * Toggle visibility of this trace.
     */
    void toggleMarker();

    /**
     * Toggle visibility of this trace.
     */
    void togglePower();

    /**
     * Toggle visibility of this trace.
     */
    void toggleSignal();


protected:
    
    void menu();

    void tool();


signals:

    /**
     * User canceled the import - take the approbiate action to finish up.
     */
    void cancel();

    /**
     * Import the spectrum in its current state.
     *
     * @param khidaq data
     */
    void import( const RawData& khidaq );
    
    /**
     * The user manually added/deleted an etalon marker.
     *
     * @param pos Index of the new marker.
     */
    void marker( const int pos );


protected:

    /**
     * Pointer to application config file.
     */
    KConfig *config;

    /**
     * Menu IDs.
     */
    int id_etalon, id_iodine, id_power, id_signal, id_marker;

    /**
     * Data object to work on.
     */
    RawData& data;

    /**
     * Menu entries for traces visibility.
     */
    QPopupMenu *traces;

    /**
     * WorkArea widget.
     */
    RawDataWorkArea *view;
};



/**
 * Dialog to display raw KHiDaq data (etalon, iodine, power signal) and calculated
 * marker positions.
 *
 * It should be possible to define new marker and to recalculate them.
 *
 * @short Display raw KHiDaq data.
 * @author Jochen Kpper
 * @version 98/10/09
 */
class DialogImportKHiDaq : public DialogImportRawData
{
    Q_OBJECT

public:

    /**
     * Constructor.
     *
     * @param data  data to work on.
     * @param name  Passed to @ref KTopLevelWidget constructor.
     */
    DialogImportKHiDaq( RawData& data, const char *name=0 );
    

private:

    /**
     * Read in KHiDaq data file
     *
     * @param file Stream to read data from.
     */
    void read();
};




class DialogIodineFrequency : public QDialog
{
    Q_OBJECT

public:
    
    /**
     * Keys for KConfig entries of this class
     */
    static const char config_AbsFreq[] = "AbsFreq";
    static const char config_AbsFreqUnit[] = "AbsFreqUnit";
    static const char config_AbsIndex[] = "AbsIndex";
    static const char config_Double[] = "AbsDouble";

    /**
     * Constructor.
     *
     * @param pos    Position to make assignment at; this is emitted later
     * together with the entered frequency to form an complete assignment.
     * @param parent Passed to QDialog constructor.
     * @param name   Passed to QDialog constructor.
     */
    DialogIodineFrequency( const int pos, QWidget *parent=0, const char *name=0 );


protected slots:

    /**
     * Accept user input, emit signal @ref assignment.
     */
    void accept();


signals:

    /**
     * @param pos Position of assignment.
     * @param val Frequency of assignment.
     */
    void assignment( const int pos, const double val );


protected:

    /**
     * Input widget.
     */
    Floated *input;

    /**
     * KConfig for this instance.
     */
    KConfig *config;

    /**
     * CheckBox for frequency doubling.
     */
    QCheckBox *dbl;

    /**
     * RadioButtons for unit-selection.
     */
    QRadioButton *cm, *mhz;
};



class DialogFSR : public QDialog
{
    Q_OBJECT

public:
    
    /**
     * keys for KConfig entries of this class
     */
    static const char config_FSR[] = "FSR";
    static const char config_UnitFSR[] = "UnitFSR";

    /**
     * Constructor.
     *
     * @param pos    Position to make assignment at; this is emitted later
     * together with the entered frequency to form an complete assignment.
     * @param parent Passed to QDialog constructor.
     * @param name   Passed to QDialog constructor.
     */
    DialogFSR( QWidget *parent=0, const char *name=0 );


protected slots:

    /**
     * Accept user input.
     */
    void accept();


protected:

    /**
     * Pointer to application config file.
     */
    KConfig *config;

    /**
     * Input widget.
     */
    QLineEdit *input;

    /**
     * RadioButtons for unit-selection.
     */
    QRadioButton *cm, *mhz;
};



class DialogLinModel : public QDialog
{
    Q_OBJECT

public:
    
    /**
     * Constructor.
     *
     * @param parent Passed to QDialog constructor.
     * @param name   Passed to QDialog constructor.
     */
    DialogLinModel( QWidget *parent=0, const char *name=0 );


protected slots:

    /**
     * Accept user input.
     */
    void accept();


public:

    /**
     * keys for KConfig entries of this class
     */
    static const char config_LinModel[] = "LinModel";


protected:

    /**
     * Pointer to application config file.
     */
    KConfig *config;

    /**
     * RadioButtons for unit-selection.
     */
    QRadioButton *linear, *quadratic, *cubic;
};



#include "importRawData_inline.h"



#endif
