c     This file is part of krot,
c     a program for the simulation, assignment and fit of HRLIF spectra.
c
c     Copyright (C) 1994-1999 Arnim Westphal
c     Copyright (C) 1997-1999 Jochen Kpper
c
c     If you use this program for your scientific work, please cite it according to
c     the file CITATION included with this package.
c
c     krot-arnirot
c     a program to calculate rotational resolved vibrational/vibronic bands


#include "arni.h"


      subroutine rotdi( a, n, np, d, v, e, diaalg, shorti )
c     call diagonalization and eigenvector/eigenvalue sorting routines for COMPLETE hamiltonian matrix in Wang basis

      implicit none

      integer        diaalg, ierr, n, np, shorti
      real*8         a(np,np), d(np), v(np,np), e(np)


      ARNIROT_LAUNCH ( "Launching rotdi." )

      ierr = 0
c     tridiagonal diagonalization technique (two steps)
c     - reduction to tridiagonal form
      call tred2( np, n, a, d, e, v )
c     - complete diagonalization of tridiagonal matrix
      call tql2( np, n, d, e, v, ierr )
      if ( ( ierr .ne. 0 ) .and. ( shorti .eq. 0 ) )
     *     write(*,*) "ERROR: diagonalization error for J =", (n - 1)/2, ", l = ", ierr

      return
      end


c-----------------------------------------------------------------------------
      subroutine rotdip( J, iact, dmham, dmfwng, a, d, v, diaalg, shorti, Jmxout )
c     call diagonalization and eigenvector/eigenvalue sorting routines for FACTORIZED Wang hamiltonian SUBMATRICES

      implicit none

c     maximum dimension of unfactorized wang matrix  : dmham  = 2*Jmax+1
      integer        dmham, dmhm
      parameter      ( dmhm = 2 * ARNIROT_JMAX + 1 )
c     maximum dimension of factorized submatrices E,O: dmfwng = Jmax/2+1
      integer        dmfwng, dmfhm
      parameter      ( dmfhm = ARNIROT_JMAX / 2 + 1 )

c     flag for diagonalization algorithm
      integer        diaalg
c     actual size of J block (2J+1)
      integer        iact
c     diagonalization error flag
      integer        ierr
c     loop variables
      integer        i, ii, ij, ic, ir, l
c     current rotational quantum number
      integer        J
c     J modulo 2
      integer        Jmod2
      integer        Jmxout
c     actual dimensions of submatrices E-, E+, O-, O+
      integer        nem, nep, nom, nop
c     IO format flag
      integer        shorti

c     original Wang matrix
      real*8         a(dmham,dmham)
c     submatrices E-, E+, O-, O+
      real*8         aem(dmfhm,dmfhm), aep(dmfhm,dmfhm)
      real*8         aom(dmfhm,dmfhm), aop(dmfhm,dmfhm)
c     eigenvalues of Wang hamiltonian matrix
      real*8         d(dmham)
c     eigenvalues of submatrices E-, E+, O-, O+
      real*8         dem(dmfhm), dep(dmfhm), dom(dmfhm), dop(dmfhm)
c     temporary vector for subdiagonal matrix elements
      real*8         eem(dmfhm), eep(dmfhm), eom(dmfhm), eop(dmfhm)
c     eigenvectors of Wang hamiltonian matrix
      real*8         v(dmham,dmham)
c     eigenvectors of submatrices E-, E+, O-, O+
      real*8         vem(dmfhm,dmfhm), vep(dmfhm,dmfhm)
      real*8         vom(dmfhm,dmfhm), vop(dmfhm,dmfhm)


      ARNIROT_LAUNCH ( "Launching rotdip." )

      Jmod2 = mod( J, 2 )
      nem   = J/2
      nep   = nem + 1
      nom   = nem + Jmod2
      nop   = nom

c     factorize Wang hamiltonian matrix a into four submatrices E-, O-, E+ and O+
c     (aem, aom, aep and aop)
      do ij = 1, nem, 1
         ic = 2*ij - 1 + Jmod2
         do ii = 1, nem, 1
            ir = 2*ii - 1 + Jmod2
            aem(ii,ij) = a(ir,ic)
         end do
      end do
      do ij = 1, nom, 1
         ic = 2*ij - Jmod2
         do ii = 1, nom, 1
            ir = 2*ii - Jmod2
            aom(ii,ij) = a(ir,ic)
         end do
      end do
      do ij = 1, nep, 1
         ic = 2*ij - 1 + J
         do ii = 1, nep, 1
            ir = 2*ii - 1 + J
            aep(ii,ij) = a(ir,ic)
         end do
      end do
      do ij = 1, nop, 1
         ic = 2*ij + J
         do ii = 1, nop, 1
            ir = 2*ii + J
            aop(ii,ij) = a(ir,ic)
         end do
      end do

#ifdef DEBUG_VERBOSE
      if ( ( shorti .eq. 0 ) .and. ( J .le. Jmxout ) ) then
         write(*,*) '\nfactorized hamiltonian submatrices (J = ',J,'):'
         if ( nep .gt. 0 ) then
            write(*,*) '\nE+'
            do ir = 1, nep, 1
               write(*,'(6(f9.1))') (aep(ir,ic), ic = 1, nep, 1)
            end do
         end if
         if ( nem .gt. 0 ) then
            write(*,*) '\nE-'
            do ir = 1, nem, 1
               write(*,'(6(f9.1))') (aem(ir,ic), ic = 1, nem, 1)
            end do
         end if
         if ( nop .gt. 0 ) then
            write(*,*) '\nO+'
            do ir = 1, nop, 1
               write(*,'(6(f9.1))') (aop(ir,ic), ic = 1, nop, 1)
            end do
         end if
         if ( nom .gt. 0 ) then
            write(*,*) '\nO-'
            do ir = 1, nom, 1
               write(*,'(6(f9.1))') (aom(ir,ic), ic = 1, nom, 1)
            end do
         end if
      end if
#endif

c     prepare parameters D, E, and Z (diagonal, subdiagonal elements and unity
c     matrix, respectively) for a direct entrance into TQL2 (the submatrices
c     are tridiagonal by nature)
      do i = 1, nep, 1
         do l = 1, nep, 1
            vep(i,l) = 0.d0
            vem(i,l) = 0.d0
            vop(i,l) = 0.d0
            vom(i,l) = 0.d0
         end do
         dep(i)   = aep(i,i)
         vep(i,i) = 1.d0
         dem(i)   = aem(i,i)
         vem(i,i) = 1.d0
         dop(i)   = aop(i,i)
         vop(i,i) = 1.d0
         dom(i)   = aom(i,i)
         vom(i,i) = 1.d0
      end do
      do i = 2, nep, 1
         eep(i) = aep(i,i-1)
         eem(i) = aem(i,i-1)
         eop(i) = aop(i,i-1)
         eom(i) = aom(i,i-1)
      end do

      ierr = 0
c     tridiagonal diagonalization technique
      call tql2( dmfwng, nem, dem, eem, vem, ierr )
      call tql2( dmfwng, nom, dom, eom, vom, ierr )
      call tql2( dmfwng, nep, dep, eep, vep, ierr )
      call tql2( dmfwng, nop, dop, eop, vop, ierr )
      if( ( ierr .ne. 0 ) .and. ( shorti .eq. 0 ) )
     *     write(*,*) "ERROR: diagonalization error for J =", J, ", l = ", ierr

c     zero all v elements now (the easy way, but anything else would cause overhead)
      do ir = 1, iact, 1
         do ic = 1, iact, 1
            v(ir,ic) = 0.d0
         end do
      end do
      
c     fill d vec and v matrix in CORRECT ASCENDING order (E+ O- O+ E- E+ ...)
      ic = -3
      do ij = 1, nep, 1
c     // ic = 4*ij - 3
         ic = ic + 4
         d(ic) = dep(ij)
         ir = J - 1
         do ii = 1, nep, 1
c        // ir = 2*ii - 1 + J
            ir = ir + 2
            v(ir,ic) = vep(ii,ij)
         end do
      end do
      ic = -2
      do ij = 1, nom, 1
c     // ic = 4*ij - 2
         ic = ic + 4
         d(ic) = dom(ij)
         ir = -Jmod2
         do ii = 1, nom, 1
c        // ir = 2*ii - Jmod2
            ir = ir + 2
            v(ir,ic) = vom(ii,ij)
         end do
      end do
      ic = -1
      do ij = 1, nop, 1
c     // ic = 4*ij - 1
         ic = ic + 4
         d(ic) = dop(ij)
         ir = J
         do ii = 1, nop, 1
c        // ir = 2*ii + J
            ir = ir + 2
            v(ir,ic) = vop(ii,ij)
         end do
      end do
      ic = 0
      do ij = 1, nem, 1
c     // ic = 4*ij
         ic = ic + 4
         d(ic) = dem(ij)
         ir = Jmod2 - 1
         do ii = 1, nem, 1
c        // ir = 2*ii - 1 + Jmod2
            ir = ir + 2
            v(ir,ic) = vem(ii,ij)
         end do
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine tred2( nm, n, a, d, e, z )
c     convert a nondiagonal matrix to tridiagonal form

c     Househoulder reduction of a real symmetric, N by N matrix A, stored in
c     a NM by NM physical array. On output, Z is the orthogonal matrix Q
c     effecting the transformation. D returns the diagonal elements of the
c     tridiagonal matrix, and E the off-diagonal elements, with E(1)=0.
c     Array A is left unchanged.


      implicit none

      integer        i, j, k, l, n, nm
      real*8         a(nm,nm), d(nm), e(nm), z(nm,nm)
      real*8         f, g, h, hh, scale

      ARNIROT_LAUNCH ( "Launching tred2." )

c     copy diagonal and subdiagonal elements of A into Z in order not to destroy array A
      do j = 1, n
         do i = j, n
            z( i, j ) = a( i , j )
         end do
      end do

      do i = n, 2, -1
         l = i - 1
         h = 0.d0
         scale = 0.d0
         if ( l .gt. 1 ) then
            do k = 1, l, 1
               scale = scale + dabs( z( i, k ) )
            end do
            if ( scale .eq. 0. ) then
c              skip transformation
               e( i ) = z( i, l )
            else
               do k = 1, l, 1
c                 use scaled a's for transformation
                  z( i, k ) = z( i, k ) / scale
c                 form sigma in H
                  h = h + z( i, k ) * z( i, k )
               end do
               f = z( i, l )
               g = -dsign( dsqrt( h ), f )
               e( i ) = scale * g
c              now H is equation (11.2.4)
               h = h - f*g
c              store u in the ith row of Z
               z( i, l ) = f - g
               f = 0.d0
               do j = 1, l, 1
c                 store u/H in the ith column of Z
                  z( j, i ) = z( i, j ) / h
c                 form an element of Z*u in G
                  g = 0.d0
                  do k = 1, j, 1
                     g = g + z( j, k ) * z( i, k )
                  end do
                  if ( l .gt. j ) then
                     do k = j+1, l, 1
                        g = g + z( k, j ) * z( i, k )
                     end do
                  end if
c                 form an element of p in temporarily unused element of E
                  e( j ) = g / h
                  f = f + e( j ) * z( i, j )
               end do
c              form K, equation (11.2.11)
               hh = f / ( h + h )
c              form q and store in E overwriting p
c              note that E(L)=E(I-1) survives
               do j = 1, l, 1
                  f = z( i, j )
                  g = e( j ) - hh*f
                  e( j ) = g
c                 reduce Z, equation (11.2.13)
                  do k = 1, j, 1
                     z( j, k ) = z( j, k ) - f*e( k ) - g*z( i, k )
                  end do
               end do
            end if
         else
            e( i ) = z( i, l )
         end if
         d( i ) = h
      end do

      d( 1 ) = 0.d0
      e( 1 ) = 0.d0
c     begin accumulation of transformation matrices
      do i = 1, n, 1
         l = i - 1
c        this block skipped when I=1
         if ( d( i ) .ne. 0. ) then
            do j = 1, l, 1
               g = 0.d0
c              use u and u/H stored in Z to form P*Q
               do k = 1, l
                  g = g + z( i, k ) * z( k, j )
               end do
               do k = 1, l
                  z( k, j ) = z( k, j ) - g * z( k, i )
               end do
            end do
         end if
         d( i ) = z( i, i )
c        reset row and column of Z to identity matrix for next iteration
         z( i, i ) = 1.d0
         if( l .ge. 1 ) then
            do j = 1, l, 1
               z( i, j ) = 0.d0
               z( j, i ) = 0.d0
            end do
         end if
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine tql2( dim, n, d, e, z, error )
c     Diagonalize a tridiagonal matrix by the QL algorithm.
c     See Numerical Recipes, chapter 11.3, for a description of the algorithm.

c     Input:  dim       Physical dimension of the matrix.
c     Input:  n         Logical dimension of the matrix.
c     InOut:  d         On input, its first n elements are the diagonal elements
c                       of the tridiagonal matrix.
c                       On output, it returns the eigenvalues.
c     InOut:  e         On input the subdiagonal elements of the tridiagonal matrix
c                       with e(1) arbitrary.
c                       CAUTION: Destroyed on output.
c     InOut:  z         On input, the transformation matrix to obtain the tridiagonal
c                       matrix as obtained from tred2.
c                       If you want to diagonalize a naturally tridiagonal matrix
c                       without using tred2, give the unity matrix.
c                       On output, the i-th column of z returns the normalized
c                       eigenvalue corresponding to d( k ).

c     This implementation of the QL algorithm uses implicit shifts to determine the
c     eigenvalues and eigenvectors of a real, symmetric, tridiagonal matrix. ( Thus
c     all eigenvalues will be real. )
c     Usually input will be a matrix previously reduced to the appropriate form
c     by tred2 (v.s.).


      implicit none

      integer           dim, error, n
      double precision  d( dim ), e( dim ), z( dim, dim )

      integer           i, iter, k, l, l1, m
      double precision  b, c, f, g, h, p, r, s, machep
c     machep is a machine dependent parameter specifying the relative precision of floating point arithmetic
      parameter      (machep = 1.0d-14)

      integer           idamax

      ARNIROT_LAUNCH ( "Launching tql2." )

c     error = 0
      if( n .gt. 1 ) then
c        convenient to renumber the elements of E
         do i = 2, n
            e( i - 1 ) = e( i )
         end do
         e( n ) = 0.d0
         b = 0.d0
         f = 0.d0
         do l = 1, n
            iter = 0
            h = machep * ( dabs( d( l ) ) + dabs( e( l ) ) )
            if( b .lt. h )
     *           b = h
c           look for a single small subdiagonal element to split the matrix
            do m = l, n
               if( dabs( e( m ) ) .le. b )
     *              goto 120
c              e(n) is always zero, so there is no exit through the bottom of the loop
            end do
  120       if( m .ne. l ) then
  130          if( iter .eq. 30 ) then
c                 no convergence to an eigenvalue after 30 iterations: set error and return
                  error = error + l
                  goto 1000
               end if
               iter = iter + 1
c              form shift
               l1 = l + 1
               g = d( l )
               p = ( d( l1 ) - g ) / ( 2.d0 * e( l ) )
               r = dsqrt( p*p + 1.d0 )
               d( l ) = e( l ) / ( p + dsign( r, p ) )
               h = g - d( l )
               do i = l1, n
                  d( i ) = d( i ) - h
               end do
               f = f + h
c              QL transformation
               p = d( m )
               c = 1.d0
               s = 0.d0
               do i = m-1, l, -1
                  g = c * e( i )
                  h = c * p
                  if( dabs( p ) .ge. dabs( e( i ) ) ) then
                     c = e( i ) / p
                     r = dsqrt( c*c + 1.d0 )
                     e( i+1 ) = s * p * r
                     s = c / r
                     c = 1.d0 / r
                  else
                     c = p / e( i )
                     r = dsqrt( c*c + 1.d0 )
                     e( i + 1 ) = s * e( i ) * r
                     s = 1.d0 / r
                     c = c * s
                  end if
                  p = c * d( i ) - s * g
                  d( i + 1 ) = h + s * ( c * g + s * d( i ) )
c                 form eigenvector
                  do k = 1, n
                     h = z( k, i+1 )
                     z( k, i + 1 ) = s *z( k, i ) + c * h
                     z( k, i )     = c *z( k, i ) - s * h
                  end do
               end do
               e( l ) = s * p
               d( l ) = c * p
               if( dabs( e( l ) ) .gt. b )
     *              goto 130
            end if
            d( l ) = d( l ) + f
         end do

c        order eigenvalues and corresponding eigenvectors in ascending order (of eigenvalues)
         do i = n, 2, -1
c           find largets eigenvalue with index <= i
            k = idamax( i, d, 1 )
            if( k .ne. i ) then
c              swap eigenvalues
               p = d( k )
               d( k ) = d( i )
               d( i ) = p
c              swap eigenvectors
               call dswap( n, z( 1, i ), 1, z( 1, k ), 1 )
            end if
         end do
         
      end if
      
 1000 return
      end


c-----------------------------------------------------------------------------
      subroutine diagc( a, b, c, ener, enei, eneb, vecr, veci, vecb, J, dmham, eb, diaalg, shorti )
c     set up and diagonalize the full complex hamiltonian matrix

      implicit none

      integer        dmham
      integer        diaalg
      integer        i, ii, k, icol, ioff
      integer        J
      integer        shorti
      real*8         a(dmham,dmham), b(dmham,dmham)
      real*8         c(2*dmham,2*dmham)
      real*8         ener(dmham), enei(dmham)
      real*8         eneb(2*dmham), eb(2*dmham)
      real*8         vecr(dmham,dmham), veci(dmham,dmham)
      real*8         vecb(2*dmham,2*dmham)

      ARNIROT_LAUNCH ( "Launching diagc." )

c     prepare matrix in block diagonal form with real and imaginary blocks
      call matpre( a, b, c, 2*J+1, dmham )

#ifdef DEBUG_VERBOSE
      if ( ( shorti .eq. 0 ) .and. ( J .le. 10 ) ) then
         write(*,*) 'debugging module: matrix C'
         do i = 1, 4*J+2, 1
            write(*,'(42(f12.3,2x))') (c(i,k), k = 1, 4*J+2, 1)
         end do
      end if
#endif

      call rotdi( c, 4*J+2, 2*dmham, eneb, vecb, eb, diaalg, shorti )

c     copy the combined vector matrix into the correct real and imaginary arrays
c     across the top skipping one column
      icol = 0
      ioff = 2*J+1
      do k = 1, 4*J+2, 2
         icol = icol + 1
         do i = 1, 2*J+1, 1
c           load in the real components
            vecr(i,icol) = vecb(i,k)
c           load in the imaginary components
            veci(i,icol) = vecb(i+ioff,k)
         end do
      end do

c     copy the eigenvalues from doubly degenerate eneb to ener
      i = 1
      do ii = 2, 4*J+2, 2
         ener(i) = eneb(ii)
         i = i + 1
      end do

      return
      end

c------------------------------------------------------------------------------
      subroutine matpre( a, b, c, n, nmax )
c     take an N by N real matrix A and an N by N imaginary matrix B, both stored
c     in NMAX by NMAX arrays, and PREpare a 2*N by 2*N MATrix C of the form
c          / A  -B \
c     C = |         |
c          \ B   A /

      implicit none

      integer        i, k, n, nmax
      real*8         a(nmax,nmax), b(nmax,nmax), c(2*nmax,2*nmax)

      ARNIROT_LAUNCH ( "Launching matpre." )

      do k = 1, n
         do i = 1, n
            c(i  ,k  ) =  a(i,k)
            c(i  ,k+n) = -b(i,k)
            c(i+n,k  ) =  b(i,k)
            c(i+n,k+n) =  a(i,k)
         end do
      end do

      return
      end

cc Local Variables:
cc mode: FORTRAN
cc End:
