/*

Formula Interpreter

Version 0.3

A simple, (hopefully) fast formula interpreter, written for the
Korigin project, coauthered by Martin Haefner.

Operations:
  *, /, +, -, ^, ()

Standard Functions:
  cos,sin,tan,acos,asin,atan,cosh,sinh,tanh,acosh,asinh,atanh
  ln, log, exp
  sqrt, sqr,
  abs, ceil, floor

Advanced Functions:
  erf(x)   := 2/sqrt(pi)* integral from 0 to x of exp(-t*t) dt
  j_n(x)   Bessel function first kind, order "bessel_n"
  y_n(x)   Bessel function second kind, order "bessel_n"

Spreadsheet access:
  col(col_name,index)   Name of the column, plus an optional
                        index expression (defaults to "i")


BNF of the formula interpreter:

  <formula> ::= <term> { + <term> | - <term> }
  <term> ::= <factor> { * <factor> | / <factor> }
  <factor> ::= <function> | <factor> ^ <factor> | (<formula>) | <number>
  <function> ::= <name> [ (<formula>) ]
  <number> ::= [ + | - ] <digit> { <digit> }

Note: a variable is a function without a parameter.

  (taken from the lecture "Introduction to Fortran 90", by Dr. R. Lohner,
   Institute for Applied Maths, University of Karlsruhe (TH), Germany)


(C) 1997 Patrick Schemitz

History:
97/08/09 Got started. Imperative form works.
97/08/10 Started objective form (class Formula).
         Introduced silliest function ever (THREE const's!):
         const char* const Variable::getName () const;
         Got it workin' !
97/08/12 Callback function for unknown variables introduced.
97/10/01 Bessel functions included. Who needs 'em?
         Table access implemented. Nice thing.
97/10/02 Cleanup, documentation; global variables;
         "running" variable unified.
97/10/03 state() function
97/10/19 Bug in setName() fixed. I *hate* Heisenbugs!
97/10/28 Both local and global variables. If Martin could only decide...
         Major reorganization. Docu now outdated :-(
97/10/30 Docu updated :-)
98/01/22 Error reporting via exceptions. RNG added.
98/01/28 Formula and Variable now deal with whitespaces.

TODO: - add (statistical) functions over whole columns, i.e. sum(column)
*/

#include "formula.h"


static const char* msgParseErr = "Parse error";
static const char* msgDivisionZeroErr = "Division by Zero";
static const char* msgColumnNameErr = "Column name omitted";
static const char* msgNoLookupErr = "No Lookup function defined";
static const char* msgFunctionErr = "No such function";
static const char* msgVariableErr = "No such variable";
static const char* msgGlobalVarErr = "Can't allocate global variable";
static const char* msgLocalVarErr  = "Can't allocate local variable";


static char* newStr (const char* str)
{
  char* tmp = new char [strlen(str)+1];
  strcpy(tmp,str);
  return tmp;
}

/*
 * Variable class.
 * (C) 1997 Patrick Schemitz
 */

Variable::Variable ()
{
  setName("");
  setValue(0.0);
}

Variable::Variable (const char* n, double v)
{
  setName(n);
  setValue(v);
}

void Variable::setName (const char* raw)
{
  int i, j, n=strlen(raw);
  for (i=j=0; i<n && j<maxNameLen-1; i++)
    if (!isspace(raw[i]))
      name[j++] = raw[i];
  name[j] = 0;
}

/*
 * Default error handler.
 * (C) 1997 Patrick Schemitz
 */

void defaultErrorFunction (const char* msg, const char* str)
{
  cerr << "Error in Formula evaluation:" << endl;
  cerr << msg;
  if (str && *str)
	cerr << ": " << str;
  cerr << endl << "Terminating." << endl;
  exit(42);
}

/*
 * Global variables for Formula
 * (C) 1997 Patrick Schemitz
 */

int Formula::globalVarCount = 0;
Variable Formula::globalVar [Formula::maxGlobalVars];

bool Formula::caseSensitive = true;

FI_ErrorFunc Formula::error = defaultErrorFunction;
FI_LookupFunc Formula::lookup = NULL;


/*
 * Class for Formula Error exceptions.
 *
 * Instances of this class are throw()n when an error
 * occurs while evaluationg the expression.
 *
 * Now this is the canonical example for well-applied exception
 * handling. See chapter 8.3.3 in: Bjarne Stroustrup, The C++ Programming
 * Language, 3rd ed. - Addison-Wesley, 1997.
 *
 * (C) 1998 Patrick Schemitz
 */

struct FormulaError
{
  FormulaError (const char* the_message, const char* the_context);
  FormulaError (const FormulaError& rhs);
  ~FormulaError ();
  char* message;
  char* context;
};

FormulaError::FormulaError (const char* the_message, const char* the_context)
{
  message = new char [strlen(the_message)+1];
  context = new char [strlen(the_context)+1];
  strcpy(message,the_message);
  strcpy(context,the_context);
}

FormulaError::FormulaError (const FormulaError& rhs)
{
  message = new char [strlen(rhs.message)+1];
  context = new char [strlen(rhs.context)+1];
  strcpy(message,rhs.message);
  strcpy(context,rhs.context);
}

FormulaError::~FormulaError ()
{
  delete [] message;
  delete [] context;
}


/*
 * Initialization class for the RNG. Trivial.
 * (C) 1998 Patrick Schemitz
 */

struct RNG_Init {
  RNG_Init () { srand48(0); }
};


/*
 * Formula implementation, public functions.
 * (C) 1997 Patrick Schemitz
 */

void Formula::init ()
{
  static RNG_Init rng_init;
  if (globalVarCount < 2)
	globalVarCount = 2;
  globalVar[0] = Variable("pi",M_PI);
  globalVar[1] = Variable("e",exp(1.0));
}

Formula::Formula (const char* expression)
{
  init();
  the_expression = newStr(expression);
  localVarCount = 0;
}

Formula::Formula (const Formula& f)
{
  int i;
  init();
  if (this == &f) return;
  the_expression = newStr(f.the_expression);
  localVarCount = f.localVarCount;
  for (i=0; i<f.localVarCount; i++)
	localVar[i] = f.localVar[i];
}

Formula& Formula::operator= (const Formula& f)
{
  int i;
  if (this == &f) return (*this);
  delete [] the_expression;
  the_expression = newStr(f.the_expression);
  localVarCount = f.localVarCount;
  for (i=0; i<f.localVarCount; i++)
	localVar[i] = f.localVar[i];
  return (*this);
}

Formula& Formula::operator= (const char* exprezzione)
{
  delete [] the_expression;
  the_expression = newStr(exprezzione);
  localVarCount = 0;
  return (*this);
}

Formula::~Formula ()
{
  delete [] the_expression;
}

void Formula::setExpression (const char* ex)
{
  delete [] the_expression;
  the_expression = newStr(ex);
}

FI_LookupFunc Formula::setLookupFunction (FI_LookupFunc f)
{
  FI_LookupFunc g = lookup;
  lookup = f;
  return g;
}

FI_ErrorFunc Formula::setErrorFunction (FI_ErrorFunc f)
{
  FI_ErrorFunc g = error;
  error = f;
  return g;
}

void Formula::setLocalVariable (Variable X)
{
  int i;
  for (i=0; i<localVarCount; i++)
	{
	  if (stringcmp(X.getName(),localVar[i].getName())==0)
		{
		  localVar[i] = X;
		  return;
		}
	}
  if (localVarCount >= maxLocalVars)
	error(msgLocalVarErr,X.getName());
  else
	localVar[localVarCount++] = X;
}

void Formula::setGlobalVariable (Variable X)
{
  int i;
  for (i=0; i<globalVarCount; i++)
	{
	  if (stringcmp(X.getName(),globalVar[i].getName())==0)
		{
		  globalVar[i] = X;
		  return;
		}
	}
  if (globalVarCount >= maxGlobalVars)
	error(msgGlobalVarErr,X.getName());
  else
	globalVar[globalVarCount++] = X;
  init();
}

int Formula::getLocalVariables (int max, Variable* v)
{
  int i, stop;
  stop = (localVarCount <= max ? localVarCount : max);
  for (i=0; i<stop; i++)
	v[i] = localVar[i];
  return stop;
}

int Formula::getGlobalVariables (int max, Variable* v)
{
  init();
  int i, stop;
  stop = (globalVarCount <= max ? globalVarCount : max);
  for (i=0; i<stop; i++)
	v[i] = globalVar[i];
  return stop;
}

char* Formula::stripped_string (const char* raw)
{
  int i, j, n=strlen(raw);
  char* espresso = new char [n+1];
  for (i=j=0; i<n; i++)
    if (!isspace(raw[i]))
      espresso[j++] = raw[i];
  espresso[j] = 0;
  return espresso;
}

double Formula::operator() ()
{
  double val;
  char* espresso;
  workcopy = espresso = stripped_string(the_expression);
  fi_state = 0;
  try {
    val = eval(workcopy);
    delete [] espresso;
  } catch (FormulaError e) {
    error(e.message,e.context);
    fi_state = 1;
    return NAN;
  }
  return val;
}

/*
 * Internal functions for formula evaluation.
 * (C) 1997 Patrick Schemitz
 */

double Formula::eval (char*& expr)
{
  double val = term(expr);
  while (*expr)
	{
	  if (*expr=='+')
		val += term(++expr);
      else if (*expr=='-')
		val -= term(++expr);
      else if (*expr==')')
		{
		  expr++;
		  return val;
		}
      else
		throw FormulaError(msgParseErr,expr);
	}
  return val;
}

double Formula::term (char*& expr)
{
  double val = factor(expr);
  double more;
  while (expr)
	{
      if (*expr=='*')
		val *= factor(++expr);
      else if (*expr=='/')
		{
		  more = factor(++expr);
		  if (more == 0.0)
			throw FormulaError(msgDivisionZeroErr,expr);
		  else
			val /= more;
		}
      else
		break;
	}
  return val;
}

double Formula::factor (char*& expr)
{
  double val;
  if (*expr=='(')
	val = eval(++expr);
  else if (isdigit(*expr))
	val = number(expr);
  else
	val = function(expr);
  if (*expr=='^')
	val = pow(val,factor(++expr));
  return val;
}

double Formula::number (char*& expr)
{
  char* end = 0;
  double val;
  val = strtod(expr,&end);
  expr = end;
  return val;
}

double Formula::function (char*& expr)
{
  char name [32], *np;
  double param = 0.0;
  np = name;
  while (*expr)
	{
      if (*expr=='(')
		{
		  *np = 0;
		  if (stringcmp(name,"col")==0)
			return table(++expr);
		  else
			{
			  param = eval(++expr);
			  return callf(name,param);
			}
		} else
		  if (isalnum(*expr) || (*expr=='_'))
			*np++ = *expr++;
		  else
			break;
	}
  *np = 0;
  return variable(name);
}

double Formula::table (char*& expr)
{
  char col_name [64], *np;
  int index = (int)variable("i");
  col_name[0] = 0;
  np = col_name;
  while (*expr)
	{
	  if (*expr==')') break;
	  if (*expr==',') 
		{
		  index = (int)eval(++expr);
		  break;
		}
	  else
		{
		  *np++ = *expr++;
		  *np = 0;
		}
	}
  if (*expr==')') expr++;
  if (col_name[0] == 0)
	throw FormulaError(msgColumnNameErr,0);
  if (lookup == 0)
	throw FormulaError(msgNoLookupErr,col_name);
  return lookup(col_name,index);
}

double Formula::callf (char* f, double x)
{
  if (stringcmp(f,"cos")==0)
	return cos(x);
  if (stringcmp(f,"sin")==0)
	return sin(x);
  if (stringcmp(f,"tan")==0)
	return tan(x);
  if (stringcmp(f,"acos")==0)
	return acos(x);
  if (stringcmp(f,"asin")==0)
	return asin(x);
  if (stringcmp(f,"atan")==0)
	return atan(x);
  if (stringcmp(f,"cosh")==0)
	return cosh(x);
  if (stringcmp(f,"sinh")==0)
	return sinh(x);
  if (stringcmp(f,"tanh")==0)
	return tanh(x);
  if (stringcmp(f,"acosh")==0)
	return acosh(x);
  if (stringcmp(f,"asinh")==0)
	return asinh(x);
  if (stringcmp(f,"atanh")==0)
	return atanh(x);
  if (stringcmp(f,"sqrt")==0)
	return sqrt(x);
  if (stringcmp(f,"sqr")==0)
	return x*x;
  if (stringcmp(f,"exp")==0)
	return exp(x);
  if (stringcmp(f,"ln")==0)
	return log(x);
  if (stringcmp(f,"log")==0)
	return log10(x);
  if (stringcmp(f,"abs")==0)
	return fabs(x);
  if (stringcmp(f,"ceil")==0)
	return ceil(x);
  if (stringcmp(f,"floor")==0)
	return floor(x);
  if (stringcmp(f,"erf")==0)
	return erf(x);
  if (stringcmp(f,"j_n")==0)
	return jn((int)variable("bessel_n"),x);
  if (stringcmp(f,"delta")==0)
	return (fabs(x) < epsilon ? 1.0 : 0.0);
  if (stringcmp(f,"rnd")==0)
	return x*drand48();
  
  throw FormulaError(msgFunctionErr,f);
}

double Formula::variable (char* name)
{
  int i;
  for (i=0; i<localVarCount; i++)
	{
	  if (stringcmp(name,localVar[i].getName())==0)
		return localVar[i].getValue();
	}
  for (i=0; i<globalVarCount; i++)
	{
	  if (stringcmp(name,globalVar[i].getName())==0)
		return globalVar[i].getValue();
	}
  throw FormulaError(msgVariableErr,name);
}

int Formula::stringcmp (const char* s1, const char* s2)
{
  if (caseSensitive)
	return strcmp(s1,s2);
  return strcasecmp(s1,s2);
}
