#include <signal.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <math.h>
//#include "matrix.h"
//#include "port.h"
#include "fitter.h"
#include <qapplication.h>

#ifndef MIN
# define MIN(a,b) (((a)<(b))?(a):(b))
#endif

char fitbuf[256];

#ifdef INFINITY
# undef INFINITY
#endif

#define INFINITY    1e30
#define NEARLY_ZERO 1e-30

/* create new variables with this value (was NEARLY_ZERO) */
#define INITIAL_VALUE 1.0

/* Relative change for derivatives */
#define DELTA	    0.001

#define MAX_LAMBDA  1e20
#define MIN_LAMBDA  1e-20
#define LAMBDA_UP_FACTOR 10
#define LAMBDA_DOWN_FACTOR 10

#define PLUSMINUS   "+/-"

#define gp_alloc(a,b) malloc(a)
#define gp_realloc(a,b,c) realloc(a,b)

#include "fitter.moc"

//the constructor
Fitter::Fitter(void)
{
   d_epsilon = 1e-8;	/* convergence limit */
   maxiter = 0;		/* HBB 970304: maxiter patch */

   d_autofit=FALSE;
   fit_x = 0; fit_z = 0; err_data = 0;
   
   num_data = 0;
   num_params = 0;
   
   ctrlc_flag = FALSE;

   startup_lambda = 0;
   lambda_down_factor = LAMBDA_DOWN_FACTOR;
   lambda_up_factor = LAMBDA_UP_FACTOR;
   show_iter=FALSE;
   
}
Fitter::~Fitter(void)
{
   
}

/*****************************************************************
    Marquardt's nonlinear least squares fit
*****************************************************************/
Fitter::marq_res Fitter::marquardt(double a[], double **C, double *chisq, double *lambda)
{
    int i, j;
    static double *da = 0,	/* delta-step of the parameter */
    *temp_a = 0,		/* temptative new params set   */
    *d = 0, *tmp_d = 0, **tmp_C = 0, *residues = 0;
    double tmp_chisq;

    /* Initialization when lambda == -1 */

    if (*lambda == -1) {	/* Get first chi-square check */
	bool analyze_ret;

	temp_a = vec(num_params);
	d = vec(num_data + num_params);
	tmp_d = vec(num_data + num_params);
	da = vec(num_params);
	residues = vec(num_data + num_params);
	tmp_C = matr(num_data + num_params, num_params);

	analyze_ret = analyze(a, C, d, chisq);

	/* Calculate a useful startup value for lambda, as given by Schwarz */
	/* FIXME: this is doesn't turn out to be much better, really... */
	if (startup_lambda != 0)
	    *lambda = startup_lambda;
	else {
	    *lambda = 0;
	    for (i = 0; i < num_data; i++)
		for (j = 0; j < num_params; j++)
		    *lambda += C[i][j] * C[i][j];
	    *lambda = sqrt(*lambda / num_data / num_params);
	}

	/* Fill in the lower square part of C (the diagonal is filled in on
	   each iteration, see below) */
	for (i = 0; i < num_params; i++)
	    for (j = 0; j < i; j++)
		C[num_data + i][j] = 0, C[num_data + j][i] = 0;
	/*printmatrix(C, num_data+num_params, num_params); */
	return analyze_ret ? OK : ERROR;
    }
    /* once converged, free dynamic allocated vars */

    if (*lambda == -2) {
	free(d);
	free(tmp_d);
	free(da);
	free(temp_a);
	free(residues);
	free_matr(tmp_C);
	return OK;
    }
    /* Givens calculates in-place, so make working copies of C and d */

    for (j = 0; j < num_data + num_params; j++)
	memcpy(tmp_C[j], C[j], num_params * sizeof(double));
    memcpy(tmp_d, d, num_data * sizeof(double));

    /* fill in additional parts of tmp_C, tmp_d */

    for (i = 0; i < num_params; i++) {
	/* fill in low diag. of tmp_C ... */
	tmp_C[num_data + i][i] = *lambda;
	/* ... and low part of tmp_d */
	tmp_d[num_data + i] = 0;
    }
    /* printmatrix(tmp_C, num_data+num_params, num_params); */

    /* FIXME: residues[] isn't used at all. Why? Should it be used? */

    if(Givens(tmp_C, tmp_d, da, residues, num_params + num_data, num_params, 1)==-1){
       free(d);
       free(tmp_d);
       free(da);
       free(temp_a);
       free(residues);
       free_matr(tmp_C);
       return ERROR; //added
    }
    /*print_matrix_and_vectors (tmp_C, tmp_d, residues,
       num_params+num_data, num_params); */

    /* check if trial did ameliorate sum of squares */

    for (j = 0; j < num_params; j++)
	temp_a[j] = a[j] + da[j];

    if (!analyze(temp_a, tmp_C, tmp_d, &tmp_chisq)) {
	/* FIXME: will never be reached: always returns TRUE */
       free(d);
       free(tmp_d);
       free(da);
       free(temp_a);
       free(residues);
       free_matr(tmp_C);
       return ERROR;
    }

    if (tmp_chisq < *chisq) {	/* Success, accept new solution */
	if (*lambda > MIN_LAMBDA) {
	    *lambda /= lambda_down_factor;
	}
	*chisq = tmp_chisq;
	for (j = 0; j < num_data; j++) {
	    memcpy(C[j], tmp_C[j], num_params * sizeof(double));
	    d[j] = tmp_d[j];
	}
	for (j = 0; j < num_params; j++)
	    a[j] = temp_a[j];
	return BETTER;
    } else {			/* failure, increase lambda and return */
	*lambda *= lambda_up_factor;
	return WORSE;
    }
}


/* FIXME: in the new code, this function doesn't really do enough to be
 * useful. Maybe it ought to be deleted, i.e. integrated with
 * calculate() ?
 */
/*****************************************************************
    compute chi-square and numeric derivations
*****************************************************************/
bool Fitter::analyze(double a[],double **C,double d[], double *chisq)
{
/*
 *  used by marquardt to evaluate the linearized fitting matrix C
 *  and vector d, fills in only the top part of C and d
 *  I don't use a temporary array zfunc[] any more. Just use
 *  d[] instead.
 */
   int i, j;
   double tmp;
   
    *chisq = 0;
    calculate(d, C, a);

    for (i = 0; i < num_data; i++) {
       tmp=(err_data==0) ? 1: (*err_data)[i];
       /* note: order reversed, as used by Schwarz */
	d[i] = (d[i] - (*fit_z)[i]) / tmp;
	*chisq += d[i] * d[i];
	for (j = 0; j < num_params; j++)
	    C[i][j] /= tmp;
    }
    /* FIXME: why return a value that is always TRUE ? */
    return TRUE;
}


/* To use the more exact, but slower two-side formula, activate the
   following line: */
/*#define TWO_SIDE_DIFFERENTIATION */
/*****************************************************************
    compute function values and partial derivatives of chi-square
*****************************************************************/
void Fitter::calculate(double *zfunc, double **dzda, double a[])
{
    int k, p;
    double tmp_a;
    double *tmp_high, *tmp_pars;
#ifdef TWO_SIDE_DIFFERENTIATION
    double *tmp_low;
#endif

    tmp_high = vec(num_data);	/* numeric derivations */
#ifdef TWO_SIDE_DIFFERENTIATION
    tmp_low = vec(num_data);
#endif
    tmp_pars = vec(num_params);

    /* first function values */

    call_parser(a, zfunc);

    /* then derivatives */

    for (p = 0; p < num_params; p++)
	tmp_pars[p] = a[p];
    for (p = 0; p < num_params; p++) {
	tmp_a = fabs(a[p]) < NEARLY_ZERO ? NEARLY_ZERO : a[p];
	tmp_pars[p] = tmp_a * (1 + DELTA);
	call_parser(tmp_pars, tmp_high);
#ifdef TWO_SIDE_DIFFERENTIATION
	tmp_pars[p] = tmp_a * (1 - DELTA);
	call_parser(tmp_pars, tmp_low);
#endif
	for (k = 0; k < num_data; k++)
#ifdef TWO_SIDE_DIFFERENTIATION
	    dzda[k][p] = (tmp_high[k] - tmp_low[k]) / (2 * tmp_a * DELTA);
#else
	    dzda[k][p] = (tmp_high[k] - zfunc[k]) / (tmp_a * DELTA);
#endif
	tmp_pars[p] = a[p];
    }

#ifdef TWO_SIDE_DIFFERENTIATION
    free(tmp_low);
#endif
    free(tmp_high);
    free(tmp_pars);
}


/*****************************************************************
    frame routine for the marquardt-fit
*****************************************************************/
bool Fitter::regress(bool multi /*QArray<double> *params*/)
{
   double **covar, *dpar, **C, chisq, last_chisq, lambda;
   int iter, i, j;
   marq_res res;

   if(/*params==NULL ||*/ fit_x==NULL || fit_z==NULL)return FALSE;
   
   num_data=MIN(fit_x->size(),fit_z->size());
   if(err_data!=NULL)
     num_data=MIN(num_data,err_data->size());
   
   num_params = fpartbl.size();

   if(num_params<1)return FALSE;
   if(num_data<num_params)return FALSE;

   emit fitting();
   
   //mesg="Fitting started...";
   mesg="";
   addMessage("Parameters to fit: %i, Data: %i\n",num_params,num_data);
   
   double *a = new double[num_params];
   for(i=0;i<num_params;++i)a[i]=fpartbl[i];
   
   chisq = last_chisq = INFINITY;
   C = matr(num_data + num_params, num_params);
   lambda = -1;		/* use sign as flag */
   iter = 0;			/* iteration counter  */

    /* ctrlc now serves as Hotkey */
//    ctrlc_setup();

    /* Initialize internal variables and 1st chi-square check */
    if ((res = marquardt(a, C, &chisq, &lambda)) == ERROR){
       addMessage("FIT: error occured during 1st square check");
       free_matr(C);
       delete a;
       emit finished();
       return FALSE;
    }
    res = BETTER;

    /* MAIN FIT LOOP: do the regression iteration */

    /* HBB 981118: initialize new variable 'user_break' */
    user_stop = FALSE;

    do {
	if (ctrlc_flag) {
	   ctrlc_flag = FALSE;
	   user_stop=TRUE;
	   break;
	}
	if (res == BETTER) {
	    iter++;
	    last_chisq = chisq;
	}
       if(multi)qApp->processEvents();
       res = marquardt(a, C, &chisq, &lambda);
    } while ((res != ERROR)
	     && (lambda < MAX_LAMBDA)
	     && ((maxiter == 0) || (iter <= maxiter))
	     && (res == WORSE
		 || ((chisq > NEARLY_ZERO)
		     ? ((last_chisq - chisq) / chisq)
		     : (last_chisq - chisq)) > d_epsilon
	     )
	);

    /* fit done */

    /* HBB 970304: the maxiter patch: */
   if ((maxiter > 0) && (iter > maxiter)) {
      addMessage("\nMaximum iteration count (%d) reached. Fit stopped.\n", maxiter);
   } else if (user_stop) {
      addMessage("\nThe fit was stopped by the user after %d iterations.\n", iter);
   } else {
      addMessage("\nAfter %d iterations the fit converged.\n", iter);
   }
   
   addMessage("final sum of squares of residuals : %g\n", chisq);
   if (chisq > NEARLY_ZERO) {
      addMessage("rel. change during last iteration : %g\n\n", (chisq - last_chisq) / chisq);
   } else {
      addMessage("abs. change during last iteration : %g\n\n", (chisq - last_chisq));
   }
   
   if (res == ERROR){
      addMessage("FIT: error occured during fit");
      free_matr(C);
      delete a;
      emit finished();
      return FALSE;
   }
   
   /* compute errors in the parameters */

    if (num_data == num_params) {
	int i;

	addMessage("\nExactly as many data points as there are parameters.\n");
	addMessage("In this degenerate case, all errors are zero by definition.\n\n");
	addMessage("Final set of parameters \n");
	addMessage("======================= \n\n");
//	for (i = 0; i < num_params; i++)
//	    Dblf3("%-15.15s = %-15g\n", par_name[i], a[i]);
    } else if (chisq < NEARLY_ZERO) {
	int i;

	addMessage("\nHmmmm.... Sum of squared residuals is zero. Can't compute errors.\n\n");
	addMessage("Final set of parameters \n");
	addMessage("======================= \n\n");
//	for (i = 0; i < num_params; i++)
//	    Dblf3("%-15.15s = %-15g\n", par_name[i], a[i]);
    } else {
	    addMessage("degrees of freedom (ndf) : %d\n",  num_data - num_params);
	    addMessage("rms of residuals (stdfit) = sqrt(WSSR/ndf): %g\n", sqrt(chisq / (num_data - num_params)));
 	    addMessage("variance of residuals (reduced chi^2) = WSSR/ndf: %g\n\n", chisq / (num_data - num_params));

	/* get covariance-, Korrelations- and Kurvature-Matrix */
	/* and errors in the parameters                     */

	/* compute covar[][] directly from C */
        if(Givens(C, 0, 0, 0, num_data, num_params, 0)==-1){
	   free_matr(C);
	   delete a;
	   emit finished();
	   return FALSE;
	}
	/*printmatrix(C, num_params, num_params); */

	/* Use lower square of C for covar */
	covar = C + num_data;
	if(Invert_RtR(C, covar, num_params)==-1){
	   free_matr(C);
	   delete a;
	   emit finished();
	   return FALSE;
	}
	/*printmatrix(covar, num_params, num_params); */

	/* calculate unscaled parameter errors in dpar[]: */
	dpar = vec(num_params);
	for (i = 0; i < num_params; i++) {
	    /* FIXME: can this still happen ? */
	    if (covar[i][i] <= 0.0){	/* HBB: prevent floating point exception later on */
	       addMessage("Calculation error: non-positive diagonal element in covar. matrix");
	       free_matr(C);
	       free(dpar);
	       delete a;
	       emit finished();
	       return FALSE;
	    }
	    dpar[i] = sqrt(covar[i][i]);
	}
       
	/* transform covariances into correlations */
	for (i = 0; i < num_params; i++) {
	    /* only lower triangle needs to be handled */
	    for (j = 0; j <= i; j++)
		covar[i][j] /= dpar[i] * dpar[j];
	}

	/* scale parameter errors based on chisq */
	chisq = sqrt(chisq / (num_data - num_params));
	for (i = 0; i < num_params; i++)
	    dpar[i] *= chisq;

	addMessage("Final set of parameters   Asymptotic Standard Error\n");
	addMessage("=======================   ==========================\n\n");

	for (i = 0; i < num_params; i++) {
	    double temp =
	    (fabs(a[i]) < NEARLY_ZERO) ? 0.0 : fabs(100.0 * dpar[i] / a[i]);
	    addMessage("%-8s = %-15g %-3.3s %-12.4g (%.4g%%)\n",
		  parName(i), a[i], PLUSMINUS, dpar[i], temp);
	}
       
	addMessage("\n\nCorrelation matrix of the fit parameters:\n\n");
	addMessage("         ");
      
	for (j = 0; j < num_params; j++)
	    addMessage("%-6.6s ", parName(j));

	addMessage("\n");
	for (i = 0; i < num_params; i++) {
	    addMessage("%-8s", parName(i));
	    for (j = 0; j <= i; j++) {
		/* Only print lower triangle of symmetric matrix */
		addMessage("%6.3f ", covar[i][j]);
	    }
	    addMessage("\n");
	}

	free(dpar);
    }

    /* call destructor for allocated vars */
   lambda = -2;		/* flag value, meaning 'destruct!' */
   marquardt(a, C, &chisq, &lambda);
   
   free_matr(C);

   delete a;
   emit finished();
   return TRUE;
}


/*****************************************************************
    display actual state of the fit
 *****************************************************************/
/*
void Fitter::show_fit(int i,double  chisq,double  last_chisq,double *a,double lambda,FILE *device)
{
   int k;
   
   if(show_iter){
   
   fprintf(device, "\n\n\
Iteration %d\n\
WSSR       : %-15g  delta(WSSR)/WSSR   : %g\n\
delta(WSSR): %-15g  limit for stopping : %g\n\
lambda	   : %g\n\n%s parameter values\n\n",
	   i, chisq, chisq > NEARLY_ZERO ? (chisq - last_chisq) / chisq : 0.0,
	   chisq - last_chisq, d_epsilon, lambda,
	   (i > 0 ? "resultant" : "initial set of free"));
   for (k = 0; k < num_params; k++)
     fprintf(device, "%-8s = %g\n", parName(k), a[k]);
   }
}
*/


/*****************************************************************
    call parser function
*****************************************************************/
void Fitter::call_parser(double *par, double *data)
{
   int i;
   
   for(i=0;i<num_params;++i){
      fpartbl[i]=par[i];
   }
   
   for(i=0;i<num_data;++i){
      data[i]=function((*fit_x)[i]);
      //if isinf Eex undefined value
   
   }
}


void Fitter::setData(QArray<double> *x, QArray<double> *z, QArray<double> *err)
{
   fit_x=x;
   fit_z=z;
   err_data=err;
   
   
}

/*****************************************************************
    internal prototypes
*****************************************************************/

static inline int fsign(double x);

static inline int fsign(double x)
{
    return (int)( x>0.0 ? 1 : (x < 0.0) ? -1 : 0.0) ;
}

 
/*****************************************************************
    first straightforward vector and matrix allocation functions
*****************************************************************/
 
double *Fitter::vec (int n)
{
   double *dp=NULL;
   
   if( n < 1 )
     return (double *) NULL;
   dp = (double *) gp_alloc ( n * sizeof(double), "vec");
    return dp;
}


double **Fitter::matr (int rows, int cols)
{
    /* allocates a double matrix */

    register int i;
    register double **m;

    if ( rows < 1  ||  cols < 1 )
        return NULL;
    m = (double **) gp_alloc ( rows * sizeof(double *) , "matrix row pointers");
    m[0] = (double *) gp_alloc ( rows * cols * sizeof(double), "matrix elements");
    for ( i = 1; i<rows ; i++ )
    	m[i] = m[i-1] + cols;
    return m;
}


void Fitter::free_matr (double **m)
{
   free (m[0]);
   free (m);
}


double *Fitter::redim_vec (double **v, int n)
{
    if ( n < 1 ) 
      *v = NULL;
    else
      *v = (double *) gp_realloc (*v, n * sizeof(double), "vec");
    return *v;
}


/*****************************************************************

     Solve least squares Problem C*x+d = r, |r| = min!, by Given rotations
     (QR-decomposition). Direct implementation of the algorithm
     presented in H.R.Schwarz: Numerische Mathematik, 'equation'
     number (7.33)

     If 'd == NULL', d is not accesed: the routine just computes the QR
     decomposition of C and exits.

     If 'want_r == 0', r is not rotated back (\hat{r} is returned
     instead).

*****************************************************************/

int Fitter::Givens (double **C, double *d, double *x, double *r, int N, int  n, int want_r)
{
    int i, j, k;
    double w, gamma, sigma, rho, temp;
    double d_epsilon = DBL_EPSILON; /* FIXME (?)*/

/* 
 * First, construct QR decomposition of C, by 'rotating away'
 * all elements of C below the diagonal. The rotations are
 * stored in place as Givens coefficients rho.
 * Vector d is also rotated in this same turn, if it exists 
 */
    for (j = 0; j<n; j++) 
    	for (i = j+1; i<N; i++) 
    	    if (C[i][j]) {
    	    	if (fabs(C[j][j])<d_epsilon*fabs(C[i][j])) { /* find the rotation parameters */
    	    	    w = -C[i][j];
    	    	    gamma = 0;
    	    	    sigma = 1;
    	    	    rho = 1;
		} else {
		    w = fsign(C[j][j])*sqrt(C[j][j]*C[j][j] + C[i][j]*C[i][j]);
		    if (w == 0){
		       addMessage( "ERROR: w = 0 in Givens();  Cjj = %g,  Cij = %g", C[j][j], C[i][j]);
		       return -1;
		    }
		    gamma = C[j][j]/w;
		    sigma = -C[i][j]/w;
		    rho = (fabs(sigma)<gamma) ? sigma : fsign(sigma)/gamma;
		}
		C[j][j] = w;
		C[i][j] = rho;           /* store rho in place, for later use */
		for (k = j+1; k<n; k++) {   /* rotation on index pair (i,j) */
		    temp =    gamma*C[j][k] - sigma*C[i][k];
		    C[i][k] = sigma*C[j][k] + gamma*C[i][k];
		    C[j][k] = temp;
		    
		}
		if (d) {               /* if no d vector given, don't use it */
		    temp = gamma*d[j] - sigma*d[i];  /* rotate d */
		    d[i] = sigma*d[j] + gamma*d[i];
		    d[j] = temp;
	        }
	    }
    if (!d)               /* stop here if no d was specified */
         return 0;

    for (i = n-1; i >= 0; i--) {   /* solve R*x+d = 0, by backsubstitution */
        double s = d[i];
        r[i] = 0;              /* ... and also set r[i] = 0 for i<n */
        for (k = i+1; k<n; k++) 
            s += C[i][k]*x[k];
	if (C[i][i] == 0){
	   addMessage ( "ERROR: Singular matrix in Givens()");
	   return -1;
	}
        x[i] = - s / C[i][i];
	}
    for (i = n; i < N; i++) 
    	r[i] = d[i];	     	/* set the other r[i] to d[i] */
    	
    if (!want_r)        	/* if r isn't needed, stop here */
    	return 0;
    	
    /* rotate back the r vector */
    for (j = n-1; j >= 0; j--)
    	for (i = N-1; i >= 0; i--) {
    	    if ((rho = C[i][j]) == 1) { /* reconstruct gamma, sigma from stored rho */
    	     	gamma = 0;
    	     	sigma = 1;
    	    } else if (fabs(rho)<1) {
    	    	sigma = rho; 
    	    	gamma = sqrt(1-sigma*sigma);
    	    } else {
    	    	gamma = 1/fabs(rho);
    	    	sigma = fsign(rho)*sqrt(1-gamma*gamma);
    	    }
	    temp = gamma*r[j] + sigma*r[i];	/* rotate back indices (i,j) */
	    r[i] = -sigma*r[j] + gamma*r[i];
	    r[j] = temp;
    }
   return 0;
}


/* Given a triangular Matrix R, compute (R^T * R)^(-1), by forward
 * then back substitution
 * 
 * R, I are n x n Matrices, I is for the result. Both must already be
 * allocated.
 * 
 * Will only calculate the lower triangle of I, as it is symmetric 
 */

int Fitter::Invert_RtR ( double **R, double **I, int n)
{
   int i, j, k;
   
   /* fill in the I matrix, and check R for regularity : */

   for (i = 0; i<n; i++) {
      for (j = 0; j<i; j++)  /* upper triangle isn't needed */
	I[i][j] = 0;
      I[i][i] = 1;
      if (! R[i][i]){
	 addMessage ("ERROR: Singular matrix in Invert_RtR");
	 return -1;
      }
   }
   
   /* Forward substitution: Solve R^T * B = I, store B in place of I */
   
   for (k = 0; k<n; k++) 
     for (i = k; i<n; i++) {  /* upper half needn't be computed */
	double s = I[i][k];
	for (j = k; j<i; j++)  /* for j<k, I[j][k] always stays zero! */
	  s -= R[j][i] * I[j][k];
	I[i][k] = s / R[i][i];
     }
   
   /* Backward substitution: Solve R * A = B, store A in place of B */
   
   for (k = 0; k<n; k++)
     for (i = n-1; i >= k; i--) {  /* don't compute upper triangle of A */
	double s = I[i][k];
	for (j = i+1; j<n; j++)
	  s -= R[i][j] * I[j][k];
	I[i][k] = s / R[i][i]; 
     }
   return 0;
}

void Fitter::addMessage( const char *format, ... )
{
   char *buf=new char[512];
   
   va_list ap;
   va_start( ap, format );
   vsprintf( buf, format, ap );
   
   mesg+=buf;
   delete buf;
}

