/**************************************************************************
 * $Id: pcl818.c 1.1 Thu, 03 Dec 1998 12:49:42 +0100 samo $
 * $ReleaseVersion: 1.3.1 $
 *
 * This is the Linux driver for Advantech PCL-818 card
 *
 * This file is part of SampLin data acquisition software
 * Copyright (C) 1997,98 Samuel Kvasnica
 *
 * SampLin is free software; you can redistribute it and/or modify it
 * under the terms of the version 2 of GNU General Public License as
 * published by the Free Software Foundation.
 *
 * SampLin is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * (see the file LICENSE) along with SampLin package; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **************************************************************************/

#include <linux/module.h>
#include <linux/kernel.h>

#include <linux/errno.h>
#include <linux/mm.h>
#include <linux/ioport.h>
#include <linux/interrupt.h>
#include <linux/timer.h>

#include <linux/fcntl.h>

#include <asm/segment.h>
#include <asm/io.h>


#include "pcl818-driver.h"

#include "pcl818.h"
//#define PCL818_DEBUG

unsigned char forceirq=0;

#ifdef PCL818_DEBUG
   #define DPRINTK1(a)          printk(PCL818_ID a)
   #define DPRINTK2(a, b)       printk(PCL818_ID a, b)
   #define DPRINTK3(a, b, c)    printk(PCL818_ID a, b, c)
   #define DPRINTK4(a, b, c, d) printk(PCL818_ID a, b, c, d)
   #define DPRINTK5(a, b, c, d, e) printk(PCL818_ID a, b, c, d, e)
#else
   #define DPRINTK1(a)
   #define DPRINTK2(a, b)
   #define DPRINTK3(a, b, c)
   #define DPRINTK4(a, b, c, d)
   #define DPRINTK5(a, b, c, d, e)
#endif




/*  Global structures for card info.                                         */
static struct pcl818_data pcl818_card[PCL818_MAX_CARDS];
static int pcl818_num_cards = 0;




static int pcl818_read(
   struct inode *inode,
   struct file *file,
   char *buffer,
   int count
) {
   int i;
   int cardnum;
   unsigned int c;

   unsigned int minor;


   cardnum = PCL818_CARD_NUM(MINOR(inode->i_rdev));
   minor = MINOR(inode->i_rdev) - PCL818_MINOR_START(cardnum);

   if(verify_area(VERIFY_READ,buffer,count)==-EFAULT){
      printk(PCL818_ID " Got invalid buffer !\n");
      return(-EFAULT);
   }

   if(minor == PCL818_MINOR_DIGITAL) {
      for(i = 0; i+1 < count; i+=2) {
	 cli();
	 c=PCL818_DIGITAL_GET(cardnum);
	 put_fs_word(c,buffer);
	 DPRINTK3("card %c, digital lines, read 0x%04x\n", 'A' + cardnum, c);
	 sti();
	 buffer+=2;
      }
      return(i);
   }
   
   if(minor <= PCL818_MINOR_ADC15) {
      down(&pcl818_card[cardnum].sem);

      if(count == 2) {
/*  Disable automatic conversion start, select ADC, read the one value.      */
	 DPRINTK3("card %c, adc %d, disabling ATO\n", 'A' + cardnum, minor);
//	 pcl818_card[cardnum].adcmode[minor] &= 0xff - PCL818_ATO;
         PCL818_ADC_SELECT(minor, cardnum);
//	 PCL818_ADC_SET_RANGE(cardnum,5);
	 DPRINTK3("card %c, adc %d, clearing interrupt\n", 'A' + cardnum, minor);
	 PCL818_ADC_RESET_INT_REQ(cardnum);

	 DPRINTK1("entering critical code\n");
	 DPRINTK3("card %c, adc %d, starting single conversion\n", 'A' + cardnum, minor);
         cli();
	 PCL818_ADC_START_CONVERSION(cardnum);

	 sleep_on(&pcl818_card[cardnum].wait_q);
         DPRINTK1("out of critical code\n");
         DPRINTK3("card %c, adc %d woken up\n", 'A' + cardnum, minor);

	 PCL818_ADC_RESET_INT_REQ(cardnum); //convert again - to reduce crosstalk
	 PCL818_ADC_START_CONVERSION(cardnum);	 
	 sleep_on(&pcl818_card[cardnum].wait_q);

	 if(pcl818_card[cardnum].adcmode[minor]>3 && pcl818_card[cardnum].adcmode[minor]<8)
	   put_fs_word(PCL818_ADC_GET(cardnum), buffer);
	 else
	   put_fs_word((short)(PCL818_ADC_GET(cardnum))-(short)(2047), buffer);
      }
/*      else {
	 DPRINTK3("card %c, adc %d, enabling ATO\n", 'A' + cardnum, minor);
	 pcl818_card[cardnum].adcmode[minor] |= PCL818_ATO;
         PCL818_ADC_SELECT(minor, cardnum);

	 DPRINTK3("card %c, adc %d, clearing interrupt\n", 'A' + cardnum, minor);
	 PCL818_ADC_RESET_INT_REQ(cardnum);

	 DPRINTK1("entering critical code\n");
	 DPRINTK3("card %c, adc %d, starting initial conversion\n", 'A' + cardnum, minor);
         cli();
	 PCL818_ADC_START_CONVERSION(cardnum);

	 for(i = 0; i < count - 1; i ++) {
	    sleep_on(&pcl818_card[cardnum].wait_q);
	    DPRINTK3("card %c, adc %d woken up, copying value to user space\n", 'A' + cardnum, minor);

            cli();
	    put_fs_byte(PCL818_ADC_GET(cardnum), buffer);

	    buffer ++;
	 }
	 sleep_on(&pcl818_card[cardnum].wait_q);
	 DPRINTK3("card %c, adc %d woken up\n", 'A' + cardnum, minor);

	 DPRINTK2("card %c, disabling ATO\n", 'A' + cardnum);
	 pcl818_card[cardnum].adcmode[minor] &= 0xff - PCL818_ATO;
	 PCL818_ADC_SELECT(minor, cardnum);

	 DPRINTK3("card %c, adc %d copying last value to user space\n", 'A' + cardnum, minor);
	 put_fs_byte(PCL818_ADC_GET(cardnum), buffer);
      }*/

      up(&pcl818_card[cardnum].sem);

      return(count);
   }

   printk(PCL818_ID " bungled minor number %d\n", minor + (cardnum * PCL818_MINOR_RANGE));
   return(0);
}


static int pcl818_write(
   struct inode *inode,
   struct file *file,
   const char *buffer,
   int count
) {
   int cardnum;
   int i;
   unsigned int val;
#ifdef  PCL818_DEBUG
   unsigned short counter_value;
#endif

   unsigned int minor = MINOR(inode->i_rdev);


   cardnum = PCL818_CARD_NUM(minor);
   minor -= PCL818_MINOR_START(cardnum);

   if(verify_area(VERIFY_WRITE,buffer,count)==-EFAULT){
      printk(PCL818_ID " Got invalid buffer !\n");
      return(-EFAULT);
   }
   
   
   if((minor == PCL818_MINOR_DAC0) || (minor == PCL818_MINOR_DAC1)) {
      minor -= PCL818_MINOR_DAC0;
      for(i = 0; i+1 < count; i+=2) {
//	 val=*buffer + ((*(buffer+1))<<8);
	 pcl818_card[cardnum].dac[minor]=get_fs_word(buffer);
	 PCL818_DAC_SET(get_fs_word(buffer), minor, cardnum);
 	 DPRINTK4("card %c, dac %d, wrote 0x%04x\n", 'A' + cardnum, minor, (unsigned short)get_fs_word(buffer));
	 buffer+=2;
      }

      return(count);
   }

   if(minor == PCL818_MINOR_DIGITAL) {
      for(i = 0; i+1 < count; i+=2) {
	 
	 cli();
	 pcl818_card[cardnum].diglines = get_fs_word(buffer);
	 PCL818_DIGITAL_SET(pcl818_card[cardnum].diglines,cardnum);
   	 DPRINTK3("card %c, digital lines set to 0x%04x\n", 'A' + cardnum, pcl818_card[cardnum].diglines);
	 sti();
	 buffer+=2;
      }
      return(i);
   }

/*  To make the compiler happy:                                              */
   return(0);
}




static int pcl818_ioctl(
   struct inode *inode,
   struct file *file,
   unsigned int command,
   unsigned long argument
) {
   int minor = MINOR(inode->i_rdev);

   int cardnum;


   cardnum = PCL818_CARD_NUM(minor);
   minor -= PCL818_MINOR_START(cardnum);


   switch(command) {

    case PCL818_SET_ADC_RANGE:
      cli();
      PCL818_ADC_SELECT(minor,cardnum);
      PCL818_ADC_SET_RANGE(cardnum, argument&0xf);
      pcl818_card[cardnum].adcmode[minor]=argument&0xf;
      sti();
      break;
      
    case PCL818_SETBIT_DO:
      if(minor == PCL818_MINOR_DIGITAL) {
	 int bitnum = (argument & 0xf0) >> 4;
	 unsigned char val = (argument & 0x01);
	 
	 DPRINTK4("card %c, setting bit %d of digital lines to %d\n", 'A' + cardnum, bitnum, val);
	 
	 cli();
	 
	 if(val == 0x01) {
	    pcl818_card[cardnum].diglines |= (0x01 << bitnum);
	 }
	 else {
	    pcl818_card[cardnum].diglines &= 0xff - (0x01 << bitnum);
	 }
	 
	 PCL818_DIGITAL_SET(pcl818_card[cardnum].diglines, cardnum);
	 
	 sti();
	 
	 break;
      }
      else {
	 printk(PCL818_ID "bungled minor number %d in ioctl SETBIT\n", MINOR(inode->i_rdev));
	 return -ENOTTY;
      }
    case PCL818_SET_DO:
      if(minor == PCL818_MINOR_DIGITAL) {
	 unsigned int val = argument;
	 
	 DPRINTK3("card %c, setting digital lines to %d\n", 'A' + cardnum, val);
	 
	 cli();
	 
	 pcl818_card[cardnum].diglines = val;
	 
	 PCL818_DIGITAL_SET(pcl818_card[cardnum].diglines, cardnum);
	 
	 sti();
	 
	 break;
      }
      else {
	 printk(PCL818_ID "bungled minor number %d in ioctl SETBIT\n", MINOR(inode->i_rdev));
	 return -ENOTTY;
      }
    case PCL818_SET_DAC:
      if((minor == PCL818_MINOR_DAC0) || (minor == PCL818_MINOR_DAC1)) {
	 unsigned int val = argument;

	 minor-=PCL818_MINOR_DAC0;

	 pcl818_card[cardnum].dac[minor] = val;
	 
	 DPRINTK3("card %c, setting DAC0 to %d\n", 'A' + cardnum, val);
	 
	 cli();

	 
	 PCL818_DAC_SET(val,minor,cardnum);
	 
	 sti();
	 
	 break;
      }
      else {
	 printk(PCL818_ID "bungled minor number %d in ioctl SETBIT\n", MINOR(inode->i_rdev));
	 return -ENOTTY;
      }
      
/*
      case PCL818_IO_UNIPOLAR:
         if(minor <= PCL818_MINOR_ADC15) {
	    DPRINTK3("card %c, adc %d ioctl, unipolar input range\n", 'A' + cardnum, minor);

	    pcl818_card[cardnum].adcmode[minor] &= 0xff - PCL818_BU;

	    DPRINTK4("card %c, adc %d mode now 0x%02x\n", 'A' + cardnum, minor, pcl818_card[cardnum].adcmode[minor]);

         }
         else {
            printk(PCL818_ID "bungled minor number %d in ioctl UNIPOLAR\n", MINOR(inode->i_rdev));
            return -ENOTTY;
         }

         break;
*/
/*
      case PCL818_IO_BIPOLAR:
         if(minor <= PCL818_MINOR_ADC15) {
	    DPRINTK3("card %c, adc %d ioctl, bipolar input range\n", 'A' + cardnum, minor);

	    pcl818_card[cardnum].adcmode[minor] |= PCL818_BU;

	    DPRINTK4("card %c, adc %d mode now 0x%02x\n", 'A' + cardnum, minor, pcl818_card[cardnum].adcmode[minor]);
         }
         else {
            printk(PCL818_ID "bungled minor number %d in ioctl BIPOLAR\n", MINOR(inode->i_rdev));
            return -ENOTTY;
         }

         break;
*/
/*
      case PCL818_IO_LOW_RANGE:
         if(minor <= PCL818_MINOR_ADC15) {
	    DPRINTK3("card %c, adc %d ioctl, go to low input range\n", 'A' + cardnum, minor);

	    pcl818_card[cardnum].adcmode[minor] &= 0xff - PCL818_RGE;

	    DPRINTK4("card %c, adc %d mode now 0x%02x\n", 'A' + cardnum, minor, pcl818_card[cardnum].adcmode[minor]);
         }
         else {
            printk(PCL818_ID "bungled minor number %d in ioctl LOW_RANGE\n", MINOR(inode->i_rdev));
            return -ENOTTY;
         }

         break;
*/
/*
      case PCL818_IO_HIGH_RANGE:
         if(minor <= PCL818_MINOR_ADC15) {
	    DPRINTK3("card %c, adc %d ioctl, go to high input range\n", 'A' + cardnum, minor);

	    pcl818_card[cardnum].adcmode[minor] |= PCL818_RGE;

	    DPRINTK4("card %c, adc %d mode now 0x%02x\n", 'A' + cardnum, minor, pcl818_card[cardnum].adcmode[minor]);
         }
         else {
            printk(PCL818_ID "bungled minor number %d in ioctl HIGH_RANGE\n", MINOR(inode->i_rdev));
            return -ENOTTY;
         }

         break;
*/


/*
      case PCL818_IO_CT_MODE:
         if(minor < PCL818_MINOR_CT0) {
            printk(PCL818_ID "bungled minor number %d in ioctl PCL818_IO_CT_MODE\n", MINOR(inode->i_rdev));
            return -ENOTTY;
         }

         DPRINTK4("card %d, c/t %d, switching to mode %ld\n", cardnum, minor - PCL818_MINOR_CT0, argument);

         DPRINTK4("card %d, c/t %d, original mode is 0x%02x\n", cardnum, minor - PCL818_MINOR_CT0, pcl818_card[cardnum].ctmode[minor - PCL818_MINOR_CT0]);
	 pcl818_card[cardnum].ctmode[minor - PCL818_MINOR_CT0] &= (0xff - PCL818_CT_MODE_MASK);
	 DPRINTK4("card %d, c/t %d, cleared mode is 0x%02x\n", cardnum, minor - PCL818_MINOR_CT0, pcl818_card[cardnum].ctmode[minor - PCL818_MINOR_CT0]);

	 switch(argument) {
	    case 0:
	       pcl818_card[cardnum].ctmode[minor - PCL818_MINOR_CT0] |= PCL818_CT_MODE_0;
	       break;

	    case 1:
	       pcl818_card[cardnum].ctmode[minor - PCL818_MINOR_CT0] |= PCL818_CT_MODE_1;
	       break;

	    case 2:
	       pcl818_card[cardnum].ctmode[minor - PCL818_MINOR_CT0] |= PCL818_CT_MODE_2;
	       break;

	    case 3:
	       pcl818_card[cardnum].ctmode[minor - PCL818_MINOR_CT0] |= PCL818_CT_MODE_3;
	       break;

	    case 4:
	       pcl818_card[cardnum].ctmode[minor - PCL818_MINOR_CT0] |= PCL818_CT_MODE_4;
	       break;

	    case 5:
	       pcl818_card[cardnum].ctmode[minor - PCL818_MINOR_CT0] |= PCL818_CT_MODE_5;
	       break;

	    default:
               DPRINTK2("invalid counter/timer mode %ld\n", argument);
               break;
	 }

	 outb(pcl818_card[cardnum].ctmode[minor - PCL818_MINOR_CT0] | PCL818_CT_W_LSB_MSB, (pcl818_card[cardnum].baseaddress + PCL818_CT_CONTROL));
         outb(0xff, pcl818_card[cardnum].baseaddress + PCL818_CT0_WRITE + (minor - PCL818_MINOR_CT0));
         outb(0xff, pcl818_card[cardnum].baseaddress + PCL818_CT0_WRITE + (minor - PCL818_MINOR_CT0));

	 DPRINTK4("card %d, c/t %d, final mode is 0x%02x\n", cardnum, minor - PCL818_MINOR_CT0, pcl818_card[cardnum].ctmode[minor - PCL818_MINOR_CT0]);
         break;
*/
/*
      case  PCL818_IO_INFO:
         if(minor <= PCL818_MINOR_ADC15) {
	    put_fs_byte(pcl818_card[cardnum].adcmode[minor], (unsigned char *)argument);
            break;
         }

         if(minor >= PCL818_MINOR_CT0) {
            DPRINTK4("card %d, c/t %d, mode is 0x%02x\n", cardnum, minor - PCL818_MINOR_CT0, pcl818_card[cardnum].ctmode[minor - PCL818_MINOR_CT0]);
	    put_fs_byte(pcl818_card[cardnum].ctmode[minor - PCL818_MINOR_CT0], (unsigned char *)argument);
            break;
         }

	 printk(PCL818_ID "bungled minor number %d in ioctl INFO\n", MINOR(inode->i_rdev));
	 return -ENOTTY;
*/
      
#ifdef PCL818_DEBUG
/*
 case PCL818_DEBUG_IO_EN_AUTOCON:
         if(minor <= PCL818_MINOR_ADC15) {
	    printk(PCL818_ID "card %c, adc %d, debug ioctl enable autoconversion\n", 'A' + cardnum, minor);

	    pcl818_card[cardnum].adcmode[minor] |= PCL818_ATO;
	    printk(PCL818_ID "card %c, adc %d mode now 0x%02x\n", 'A' + cardnum, minor, pcl818_card[cardnum].adcmode[minor]);
         }
         else {
            printk(PCL818_ID "bungled minor number %d in debug ioctl EN_AUTOCONVERT\n", MINOR(inode->i_rdev));
            return -ENOTTY;
         }

         break;


      case PCL818_DEBUG_IO_DIS_AUTOCON:
         if(minor <= PCL818_MINOR_ADC15) {
	    printk(PCL818_ID "card %c, adc %d, debug ioctl disable autoconversion\n", 'A' + cardnum, minor);

	    pcl818_card[cardnum].adcmode[minor] &= 0xff - PCL818_ATO;
	    printk(PCL818_ID "card %c, adc %d mode now 0x%02x\n", 'A' + cardnum, minor, pcl818_card[cardnum].adcmode[minor]);
         }
         else {
            printk(PCL818_ID "bungled minor number %d in debug ioctl DIS_AUTOCONVERT\n", MINOR(inode->i_rdev));
            return -ENOTTY;
         }

         break;


      case PCL818_DEBUG_IO_RIR:
         if(minor <= PCL818_MINOR_ADC15) {
	    printk(PCL818_ID "card %c, adc %d, debug ioctl RIR\n", 'A' + cardnum, minor);
	    outb(0x00, pcl818_card[cardnum].baseaddress + PCL818_RESET_INT_REQ);
         }
         else {
            printk(PCL818_ID "bungled minor number %d in debug ioctl RIR\n", MINOR(inode->i_rdev));
            return -ENOTTY;
         }

	 break;


      case PCL818_DEBUG_IO_ADC_START:
         if(minor <= PCL818_MINOR_ADC15) {
	    printk(PCL818_ID "card %c, adc %d, debug ioctl ADC_START\n", 'A' + cardnum, minor);
	    outb(0x00, pcl818_card[cardnum].baseaddress + PCL818_ADC_START);
         }
         else {
            printk(PCL818_ID "bungled minor number %d in debug ioctl ADC_START\n", MINOR(inode->i_rdev));
            return -ENOTTY;
         }

	 break;


      case PCL818_DEBUG_IO_ADC_READ:
         if(minor <= PCL818_MINOR_ADC15) {
	    printk(PCL818_ID "card %c, adc %d, debug ioctl ADC_READ\n", 'A' + cardnum, minor);
	    printk(PCL818_ID "card %c, adc %d read 0x%02x\n", 'A' + cardnum, minor, inb(pcl818_card[cardnum].baseaddress + PCL818_ADC_IN));
         }
         else {
            printk(PCL818_ID "bungled minor number %d in debug ioctl ADC_READ\n", MINOR(inode->i_rdev));
            return -ENOTTY;
         }

	 break;


      case PCL818_DEBUG_IO_ADC_SELECT:
         if(minor <= PCL818_MINOR_ADC15) {
	    printk(PCL818_ID "card %c, adc %d, debug ioctl ADC_SELECT\n", 'A' + cardnum, minor);
	    outb(pcl818_card[cardnum].adcmode[minor], pcl818_card[cardnum].baseaddress + PCL818_COMMAND);
         }
         else {
            printk(PCL818_ID "bungled minor number %d in debug ioctl ADC_SELECT\n", MINOR(inode->i_rdev));
            return -ENOTTY;
         }

	 break;
*/
#endif /* PCL818_DEBUG */

      default:
         return -EINVAL;
   }

   return 0;
}




static int pcl818_open(struct inode *inode, struct file *file) {
   unsigned int minor = MINOR(inode->i_rdev);

   int cardnum;


   cardnum = PCL818_CARD_NUM(minor);
   minor -= PCL818_MINOR_START(cardnum);
   if ((minor > PCL818_MINOR_LARGEST) || (cardnum >= pcl818_num_cards)) {
      printk(PCL818_ID "trying to open bad minor number %d\n", MINOR(inode->i_rdev));
      return -ENODEV;
   }


   switch(file->f_flags & O_ACCMODE) {
      case O_RDONLY:
         if((pcl818_card[cardnum].read_in_use & (1L << minor)) != 0) {
#ifdef PCL818_DEBUG
	    printk(PCL818_ID "card %c, ", 'A' + cardnum);
	    if(minor == PCL818_MINOR_DIGITAL) {
	       printk("digital lines ");
	    }
	    else if(minor < PCL818_MINOR_DIGITAL) {
	       printk("analog line %d ", minor);
	    }
	    else {
	       printk("counter/timer %d ", minor - PCL818_MINOR_CT0);
	    }
	    printk("already opened for reading\n");
#endif
	    return -EBUSY;
	 }

	 pcl818_card[cardnum].read_in_use |= (1L << minor);

#ifdef PCL818_DEBUG
	 printk(PCL818_ID "card %c, ", 'A' + cardnum);
	 if(minor == PCL818_MINOR_DIGITAL) {
	    printk("digital lines ");
	 }
	 else if(minor < PCL818_MINOR_DIGITAL) {
	    printk("analog line %d ", minor);
	 }
	 else {
	    printk("counter/timer %d ", minor - PCL818_MINOR_CT0);
	 }
	 printk("opened for reading\n");
#endif

         break;


      case O_WRONLY:
         if ((minor != PCL818_MINOR_DAC0) && (minor != PCL818_MINOR_DAC1) &&  \
            (minor != PCL818_MINOR_DIGITAL) && (minor != PCL818_MINOR_CT0) &&  \
            (minor != PCL818_MINOR_CT1) && (minor != PCL818_MINOR_CT2)) {

	    DPRINTK3("card %c, analog line %d cannot be written to\n", 'A' + cardnum, minor);

            return -EPERM;
         }

	 if((pcl818_card[cardnum].write_in_use & (1L << minor)) != 0) {
#ifdef PCL818_DEBUG
	    printk(PCL818_ID "card %c, ", 'A' + cardnum);
	    if(minor == PCL818_MINOR_DIGITAL) {
	       printk("digital lines ");
	    }
	    else if(minor < PCL818_MINOR_DIGITAL) {
	       printk("analog line %d ", minor);
	    }
	    else {
	       printk("counter/timer %d ", minor - PCL818_MINOR_CT0);
	    }
	    printk("already opened for writing\n");
#endif
	    return -EBUSY;
	 }

	 pcl818_card[cardnum].write_in_use |= (1L << minor);

#ifdef PCL818_DEBUG
	 printk(PCL818_ID "card %c, ", 'A' + cardnum);
	 if(minor == PCL818_MINOR_DIGITAL) {
	    printk("digital lines ");
	 }
	 else if(minor < PCL818_MINOR_DIGITAL) {
	    printk("analog line %d ", minor);
	 }
	 else {
	    printk("counter/timer %d ", minor - PCL818_MINOR_CT0);
	 }
	 printk("opened for writing\n");
#endif

	 break;


      case O_RDWR:
         if ((minor != PCL818_MINOR_DIGITAL) && (minor != PCL818_MINOR_CT0) &&  \
            (minor != PCL818_MINOR_CT1) && (minor != PCL818_MINOR_CT2)) {

	    DPRINTK3("card %c, analog line %d cannot be written to\n", 'A' + cardnum, minor);

            return -EPERM;
         }

	 if (((pcl818_card[cardnum].write_in_use & (1L << minor)) != 0) || \
	    ((pcl818_card[cardnum].read_in_use & (1L << minor)) != 0)) {
#ifdef PCL818_DEBUG
	    printk(PCL818_ID "card %c, ", 'A' + cardnum);
	    if(minor == PCL818_MINOR_DIGITAL) {
	       printk("digital lines ");
	    }
	    else if(minor < PCL818_MINOR_DIGITAL) {
	       printk("analog line %d ", minor);
	    }
	    else {
	       printk("counter/timer %d ", minor - PCL818_MINOR_CT0);
	    }
	    printk("already opened for reading and/or writing\n");
#endif
	    return -EBUSY;
	 }

	 pcl818_card[cardnum].read_in_use |= (1L << minor);
	 pcl818_card[cardnum].write_in_use |= (1L << minor);

#ifdef PCL818_DEBUG
	 printk(PCL818_ID "card %c, ", 'A' + cardnum);
	 if(minor == PCL818_MINOR_DIGITAL) {
	    printk("digital lines ");
	 }
	 else if(minor < PCL818_MINOR_DIGITAL) {
	    printk("analog line %d ", minor);
	 }
	 else {
	    printk("counter/timer %d ", minor - PCL818_MINOR_CT0);
	 }
	 printk("opened for reading and writing\n");
#endif

	 break;
   }
	 
/*  only adc0-7 are available for input when card is configured for
 *  Differential operation
 */

   MOD_INC_USE_COUNT;

   return 0;
}




static void pcl818_close(struct inode *inode, struct file *file) {
   unsigned int minor = MINOR(inode->i_rdev);
   int cardnum;


   cardnum = PCL818_CARD_NUM(minor);
   minor -= PCL818_MINOR_START(cardnum);


   switch(file->f_flags & O_ACCMODE) {
      case O_RDONLY:
	 pcl818_card[cardnum].read_in_use &= ~(1L << minor);
         break;


      case O_WRONLY:
	 pcl818_card[cardnum].write_in_use &= ~(1L << minor);
	 break;


      case O_RDWR:
	 pcl818_card[cardnum].read_in_use &= ~(1L << minor);
	 pcl818_card[cardnum].write_in_use &= ~(1L << minor);
	 break;
   }

#ifdef PCL818_DEBUG
   printk(PCL818_ID "card %c, ", 'A' + cardnum);
   if(minor == PCL818_MINOR_DIGITAL) {
      printk("digital lines ");
   }
   else if(minor < PCL818_MINOR_DIGITAL) {
      printk("analog line %d ", minor);
   }
   else {
      printk("counter/timer %d ", minor - PCL818_MINOR_CT0);
   }
   printk("closed\n");
#endif

   MOD_DEC_USE_COUNT;
}




static struct file_operations pcl818_fops =
{
   NULL,                   /*  pcl818_lseek               */
   pcl818_read,             /*  pcl818_read                */
   pcl818_write,            /*  pcl818_write               */
   NULL,                   /*  pcl818_readaddr            */
   NULL,                   /*  pcl818_select              */
   pcl818_ioctl,            /*  pcl818_ioctl               */
   NULL,                   /*  pcl818_mmap                */
   pcl818_open,             /*  pcl818_open                */
   pcl818_close,            /*  pcl818_close               */
   NULL,                   /*  pcl818_fsync               */
   NULL,                   /*  pcl818_fasync              */
   NULL,                   /*  pcl818_check_media_change  */
   NULL                    /*  pcl818_revalidate          */
};




void pcl818_probe_wakeup(unsigned long wait_q) {
   DPRINTK1("card probe waker-upper here\n");

   wake_up((struct wait_queue **)wait_q);
}




void pcl818_irq_handler(int irq, void *dev_id, struct pt_regs *regs) {
   unsigned char cardnum;


   cardnum = pcl818_num_cards - 1;

   while(pcl818_card[cardnum].irq != irq) {
      if(cardnum == 0) {
         return;
      }

      cardnum --;
   }

   DPRINTK3("card %c, irq %d, irq handler running\n", 'A' + cardnum, irq);

   wake_up(&pcl818_card[cardnum].wait_q);
}




void pcl818_probe_isr(int irq, void *dev_id, struct pt_regs *regs) {
   DPRINTK2("prober ISR running, recieved irq %d\n", irq);

   pcl818_card[pcl818_num_cards].irq = irq;
}




int init_module(void) {
   int i;
   unsigned char irq;
   unsigned int irqs = 0;
   unsigned char use_irq=0;
   
   unsigned short probe_address;

   struct wait_queue *probe_wait_q;
   struct timer_list probe_timer;

   printk(PCL818_ID PCL818_VERSION ", loading...\n");

/*  Register the device with the kernel.                                     */
   DPRINTK2("registering driver, major number %d\n", PCL818_MAJOR);
   if(register_chrdev(PCL818_MAJOR, "pcl818", &pcl818_fops)) {
      printk(PCL818_ID "unable to get major number %d\n", PCL818_MAJOR);
      return -EBUSY;
   }

/*  Grab all free irq's.  Normally one should use probe_irq_on() and         */
/*  probe_irq_off(), but that didnt seem appropriate here.                   */
   DPRINTK1("aquiring irq probe mask\n");
   for(irq = PCL818_PROBE_IRQ_START; irq <= PCL818_PROBE_IRQ_END; irq ++) {
      if(request_irq(irq, pcl818_probe_isr, 0, "pcl818 probe", NULL)) {
         DPRINTK2("irq %d not available for probe\n", irq);
      }
      else {
         DPRINTK2("irq %d aquired for probe\n", irq);
         irqs |= (unsigned int)(1 << irq);
	 use_irq=irq;
      }
   }

   if(forceirq!=0)use_irq=forceirq;
/*  Now test to see if there's an ML16-P card at each available I/O address  */
/*  To find out, start an ADC conversion and listen for the                  */
/*  End-of-Conversion interrupt.                                             */
   for(probe_address = PCL818_PROBE_IO_START; probe_address <= PCL818_PROBE_IO_END; probe_address += 0x010) {
      DPRINTK2("checking 0x10 I/O ports starting at 0x%03x\n", probe_address);
      if(check_region(probe_address, 16)) {
         continue;
      }
      DPRINTK1("I/O ports are available\n");

      DPRINTK1("probing...\n");
      pcl818_card[pcl818_num_cards].irq = 0;

      init_timer(&probe_timer);
      probe_timer.expires = PCL818_PROBE_TIME;
      probe_timer.data = (unsigned long )&probe_wait_q;
      probe_timer.function = pcl818_probe_wakeup;
      probe_wait_q = NULL;

/*  Select an ADC, any ADC.                                                  */
      DPRINTK2("Setting IRQ %d\n",use_irq);
      outb(128+(use_irq<<4), probe_address + PCL818_COMMAND);

/*  Clear the End-of-Conversion interrupt.                                   */
      outb(0x00, probe_address + PCL818_RESET_INT_REQ);

/*  Start timer.                                                             */
      cli();
      add_timer(&probe_timer);

/*  Start conversion.                                                        */
      outb(0x00, probe_address + PCL818_ADC_START);

/*  Will be woken by pcl818_probe_wakeup() in PCL818_PROBE_TIME jiffies.       */
      sleep_on(&probe_wait_q);

/*  PCL818_PROBE_TIME jiffies has passed, if the card is installed at this    */
/*  base address and a free IRQ, then pcl818_probe_isr has set the irq field  */
/*  for this pcl818_card.                                                     */

      del_timer(&probe_timer);

      if(pcl818_card[pcl818_num_cards].irq == 0) {
	 DPRINTK2("no card found at address 0x%03x\n", probe_address);

	 continue;
      }

      DPRINTK4("card %c detected at base address 0x%03x, irq %d\n", 'A' + pcl818_num_cards, probe_address, pcl818_card[pcl818_num_cards].irq);

      pcl818_card[pcl818_num_cards].baseaddress = probe_address;
      pcl818_card[pcl818_num_cards].analog_type = PCL818_SINGLE_ENDED;
      pcl818_card[pcl818_num_cards].sem = MUTEX;
      pcl818_card[pcl818_num_cards].wait_q = NULL;


/*  Re-register the irq to the real IRQ handler.                             */
      DPRINTK2("freeing irq %d from probe ISR\n", pcl818_card[pcl818_num_cards].irq);
      free_irq(pcl818_card[pcl818_num_cards].irq, NULL);
      irqs ^= (unsigned int)(1 << pcl818_card[pcl818_num_cards].irq);

      DPRINTK3("requesting irq %d for real ISR for card %c\n", pcl818_card[pcl818_num_cards].irq, 'A' + pcl818_num_cards);
      if(request_irq(pcl818_card[pcl818_num_cards].irq, pcl818_irq_handler, SA_INTERRUPT, "pcl818", NULL)) {
	 DPRINTK3("unable to get irq %d for card %c\n", pcl818_card[pcl818_num_cards].irq, 'A' + pcl818_num_cards);

/*  we should clean up a little better here... */

	 continue;
      }

/*  Claim the I/O ports.                                                     */
      DPRINTK3("requesting I/O port 0x%03x for card %c\n", pcl818_card[pcl818_num_cards].baseaddress, 'A' + pcl818_num_cards);
      request_region(pcl818_card[pcl818_num_cards].baseaddress, 16, "pcl818");

/*  Initialize the digital output lines.                                     */
      PCL818_DIGITAL_SET(0x00, pcl818_num_cards);
/*  Initialize the ADCs.                                                     */
      for(i = 0x00; i <= 0x0f; i ++) {
	 pcl818_card[pcl818_num_cards].adcmode[i] = 0;
//	 outb(pcl818_card[pcl818_num_cards].adcmode[i], (pcl818_card[pcl818_num_cards].baseaddress + PCL818_COMMAND));

//	 DPRINTK4("card %c, adc %d, mode=0x%02x\n", 'A' + pcl818_num_cards, i, pcl818_card[pcl818_num_cards].adcmode[i]);
      }

/*  Initialize the counter/timers.                                           */
/*      for(i = 0x00; i <= 0x02; i ++) {
	 pcl818_card[pcl818_num_cards].ctmode[i] = (i << 6) |  PCL818_CT_MODE_2 | PCL818_CT_BIN;
	 outb(pcl818_card[pcl818_num_cards].ctmode[i] | PCL818_CT_W_LSB_MSB, (pcl818_card[pcl818_num_cards].baseaddress + PCL818_CT_CONTROL));
         outb(0xff, pcl818_card[pcl818_num_cards].baseaddress + PCL818_CT0_WRITE + i);
         outb(0xff, pcl818_card[pcl818_num_cards].baseaddress + PCL818_CT0_WRITE + i);

	 DPRINTK4("card %c, c/t %d, mode=0x%02x\n", 'A' + pcl818_num_cards, i, pcl818_card[pcl818_num_cards].ctmode[i]);
      }
*/
      
      printk(PCL818_ID "card %c initialized (addr=0x%03x, irq=%d status=%d)\n", 'A' + pcl818_num_cards, pcl818_card[pcl818_num_cards].baseaddress, pcl818_card[pcl818_num_cards].irq, inb(pcl818_card[pcl818_num_cards].baseaddress+8));

      pcl818_num_cards ++;
      if(pcl818_num_cards == PCL818_MAX_CARDS) {
	 probe_address = PCL818_PROBE_IO_END + 0x010;
      }
   }


/*  Free all IRQ's not used by ML16-P cards.                                 */
   for(irq = PCL818_PROBE_IRQ_START; irq <= PCL818_PROBE_IRQ_END; irq ++) {
      if(irqs & (unsigned int)(1 << irq)) {
         DPRINTK2("freeing irq %d\n", irq);
         free_irq(irq, NULL);
      }
   }

   if(pcl818_num_cards == 0) {
      DPRINTK2("unregistering device %d\n", PCL818_MAJOR);
      if(unregister_chrdev(PCL818_MAJOR, "pcl818")) {
	 printk(PCL818_ID "device unregistering failed\n");
      }

      printk(PCL818_ID "no cards found, driver not loaded\n");

      return -EBUSY;
   }

   return 0;
}




void cleanup_module(void) {
   unsigned char cardnum;


   for(cardnum = 0; cardnum < pcl818_num_cards; cardnum ++) {
      DPRINTK3("card %c, releasing io port 0x%03x\n", 'A' + cardnum, pcl818_card[cardnum].baseaddress);
      release_region(pcl818_card[cardnum].baseaddress, 16);

      DPRINTK3("card %c, releasing irq %d\n", 'A' + cardnum, pcl818_card[cardnum].irq);
      free_irq(pcl818_card[cardnum].irq, NULL);

      printk(PCL818_ID "card %c released\n", 'A' + cardnum);
   }

   DPRINTK1("unregistering device\n");
   if(unregister_chrdev(PCL818_MAJOR, "pcl818")) {
      printk(PCL818_ID "device unregistering failed\n");
   } else {
      printk(PCL818_ID "driver unloaded\n");
   }
}


