/**************************************************************************
 * $Id: SamplinMain.cpp 1.2 Thu, 18 Feb 1999 16:48:37 +0100 samo $
 * $ReleaseVersion: 1.3.1 $
 *
 * This file is part of SampLin data acquisition software
 * Copyright (C) 1997,98 Samuel Kvasnica
 *
 * SampLin is free software; you can redistribute it and/or modify it
 * under the terms of the version 2 of GNU General Public License as
 * published by the Free Software Foundation.
 *
 * SampLin is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * (see the file LICENSE) along with SampLin package; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **************************************************************************/

#include "SamplinMain.h"
//#include "main.h"
#include "PrefDirs.h"
#include "PrefComp.h"
#include "DeviceDlg.h"
#include "About.h"
#include <fcntl.h>
#include <unistd.h>
#include <termios.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <qdir.h>
#include <qsplitter.h>
#include <qtabdlg.h>

#define BUTTON_START 10
#define BUTTON_STOP 11
#define BUTTON_STEP 12
#define BUTTON_RESET 13
#define BUTTON_GOTO 14

#include "SamplinMain.moc"

Samplin_main::Samplin_main()
	:KTopLevelWidget()
{
   QPixmap pixmap;
   KIconLoader IconLoader;

   text_changed=FALSE;
   text_changed_warn=FALSE;
   prog_active=FALSE;
   
   resize((int)(QApplication::desktop()->width()*0.7),(int)(QApplication::desktop()->height()*0.7));
   
   setFontPropagation(NoChildren);
   
   compiler = new SamplinScript(this);
   connect(this, SIGNAL(stopSignal()),compiler,SLOT(stop()));
   connect(compiler, SIGNAL(changedState(int)),this,SLOT(stateScript(int)));   
   connect(compiler, SIGNAL(openGraphSignal(uint,GraphWindow *)),this,SLOT(openGraphSlot(uint,GraphWindow *)));      
   connect(compiler, SIGNAL(closeGraphSignal(uint)),this,SLOT(closeGraphSlot(uint)));      

   QSplitter *split = new QSplitter( this );
   split->setOrientation(QSplitter::Vertical);
   editor = new SamplinEdit(KApplication::getKApplication(), split );
   editor->mySetFont(QFont("courier",12,QFont::DemiBold));
   editor->setFontPropagation(NoChildren);
   messagewin = new Edit(split,"Messages");
   messagewin->setReadOnly(TRUE);
   editor->resize(width(),(int)(height()*0.7));
   messagewin->resize(width(),(int)(height()*0.3));
   updateCaption();

  
   QPopupMenu *m_file = new QPopupMenu;
   CHECK_PTR( m_file );
   m_file->insertItem( "&New...", this, SLOT(newScript()) );
   m_file->insertItem( "&Open...", this, SLOT(editOpen()), CTRL+Key_O );
   m_file->insertItem( "&Save", this, SLOT(editSave()), CTRL+Key_S );
   m_file->insertItem( "Save &as...", this, SLOT(editSaveAs()) );   
   m_file->insertSeparator();
   m_file->insertItem( "&Ext. editor", this, SLOT(externEdit()), CTRL+Key_E );   
   m_file->insertSeparator();
   m_file->insertItem( "&Quit",  this, SLOT(closeMe()), CTRL+Key_W );
   
   QPopupMenu *m_edit = new QPopupMenu;
   CHECK_PTR( m_edit );
   m_edit->insertItem( "&Copy", this, SLOT(editCopy()), CTRL+Key_C );
   m_edit->insertItem( "&Paste", this, SLOT(editPaste()), CTRL+Key_V );   
   m_edit->insertItem( "C&ut", this, SLOT(editCut()), CTRL+Key_X );
   m_edit->insertItem( "&Select all", this, SLOT(editSelectAll()));
   m_edit->insertSeparator();
   m_edit->insertItem( "&Find...", this, SLOT(editFind()), CTRL+Key_F );
   m_edit->insertItem( "Find &again", this, SLOT(editFindAgain()), CTRL+Key_L );
   m_edit->insertItem( "&Replace...", this, SLOT(editReplace()), CTRL+Key_A );   
   m_edit->insertSeparator();
   m_edit->insertItem( "&Goto line...", this, SLOT(editGoto()), CTRL+Key_J );   
   
   QPopupMenu *m_basic = new QPopupMenu;
   CHECK_PTR( m_basic );
   m_basic->insertItem( "&Run/Continue", this, SLOT(runScript()), Key_F8, BUTTON_START );
   m_basic->insertItem( "Ste&p", this, SLOT(stepScript()), Key_F7, BUTTON_STEP );   
   m_basic->insertItem( "&Stop", this, SLOT(stopScript()), Key_F9, BUTTON_STOP );
   m_basic->insertItem( "R&eset", this, SLOT(resetScript()), Key_F2, BUTTON_RESET );
   m_basic->insertItem( "&Goto line", this, SLOT(gotoScript()), Key_F4, BUTTON_GOTO);

   QPopupMenu *m_conf = new QPopupMenu;
   CHECK_PTR( m_conf );
   m_conf->insertItem( "&Preferences...", this, SLOT(setupScript()) );      
   m_conf->insertItem( "&Devices...", this, SLOT(editDevices()) );      
   
   QPopupMenu *m_graph = new QPopupMenu;
   CHECK_PTR( m_graph );
   m_graph->insertItem( "&New graph", this, SLOT(openGraph()) );
   m_graph->insertItem( "&Graph list", &m_graphs );
   m_graph->insertSeparator();
   m_graph->insertItem( "&Close all", this, SLOT(closeAllGraphs()) );   

   QPopupMenu *m_help = new QPopupMenu;
   CHECK_PTR( m_help );
   m_help->insertItem( "&Contents", this, SLOT(helpContents()), Key_F1 );
   m_help->insertSeparator();
   m_help->insertItem( "&About", this, SLOT(helpAbout()) );
   
   menubar = new KMenuBar(this);
   menubar->insertItem( "&File", m_file );
   menubar->insertItem( "&Edit", m_edit );
   menubar->insertSeparator();
   menubar->insertItem( "&Script", m_basic );
   menubar->insertItem( "&Graph", m_graph );
   menubar->insertItem( "&Configuration", m_conf );
   menubar->insertSeparator();
   menubar->insertItem( "&Help", m_help );

   setMenu(menubar);
   
   statusbar = new KStatusBar(this);
    
   statusbar->insertItem("Line:000000/000000 Col: 000", ID_LINE_COLUMN);
   statusbar->insertItem("XXX", ID_INS_OVR);
   statusbar->insertItem("**", ID_CHANGED);
   statusbar->insertItem("Status: Idle            ", ID_STATUS);
   
   statusbar->changeItem("Line: 1/1 Col: 1", ID_LINE_COLUMN);
   statusbar->changeItem("INS", ID_INS_OVR);

   statusbar->setAlignment(ID_INS_OVR,AlignCenter);
   statusbar->setAlignment(ID_CHANGED,AlignCenter);
   statusbar->setInsertOrder(KStatusBar::RightToLeft);   
   statusbar->setBorderWidth(2);
   setStatusBar(statusbar);
  
  
   toolbar = new KToolBar(this);
   toolbar->insertButton(IconLoader.loadIcon("samplin-new.xpm"), 0, SIGNAL(clicked()),
			 this,
			 SLOT(newScript()), TRUE, klocale->translate("New script"));
   toolbar->insertButton(IconLoader.loadIcon("samplin-open.xpm"), 0, SIGNAL(clicked()),
			 this,
			 SLOT(editOpen()), TRUE, klocale->translate("Open script"));
   toolbar->insertButton(IconLoader.loadIcon("samplin-save.xpm"), 0, SIGNAL(clicked()),
			 this,
			 SLOT(editSave()), TRUE, klocale->translate("Save script"));   
   toolbar->insertButton(IconLoader.loadIcon("samplin-edit.xpm"), 0, SIGNAL(clicked()),
			 this,
			 SLOT(externEdit()), TRUE, klocale->translate("Ext. editor"));
   toolbar->insertSeparator();
   toolbar->insertSeparator();   

   toolbar->insertButton(IconLoader.loadIcon("samplin-copy.xpm"), 0, SIGNAL(clicked()),
			 this,
			 SLOT(editCopy()), TRUE, klocale->translate("Copy"));
   
   toolbar->insertButton(IconLoader.loadIcon("samplin-cut.xpm"), 0, SIGNAL(clicked()),
			 this,
			 SLOT(editCut()), TRUE, klocale->translate("Cut"));
   toolbar->insertButton(IconLoader.loadIcon("samplin-paste.xpm"), 0, SIGNAL(clicked()),
			 this,
			 SLOT(editPaste()), TRUE, klocale->translate("Paste"));
      
   toolbar->insertSeparator();
   toolbar->insertSeparator();   

   toolbar->insertButton(IconLoader.loadIcon("samplin-run.xpm"), BUTTON_START, SIGNAL(clicked()),
			   this,
			   SLOT(runScript()), TRUE, klocale->translate("Start/continue script"));
   toolbar->insertButton(IconLoader.loadIcon("samplin-goto.xpm"), BUTTON_STEP, SIGNAL(clicked()),
			   this,
			   SLOT(gotoScript()), TRUE, klocale->translate("Goto cursor"));
   toolbar->insertButton(IconLoader.loadIcon("samplin-step.xpm"), BUTTON_STEP, SIGNAL(clicked()),
			   this,
			   SLOT(stepScript()), TRUE, klocale->translate("Step script"));
   toolbar->insertButton(IconLoader.loadIcon("samplin-stop.xpm"), BUTTON_STOP, SIGNAL(clicked()),
			   this,
			   SLOT(stopScript()), FALSE, klocale->translate("Stop script"));
   toolbar->insertButton(IconLoader.loadIcon("samplin-turn.xpm"), BUTTON_RESET, SIGNAL(clicked()),
			    this,
			    SLOT(resetScript()), FALSE, klocale->translate("Reset script"));
   toolbar->insertSeparator();
   toolbar->insertSeparator();

   toolbar->insertButton(IconLoader.loadIcon("samplin-graph.xpm"), 0, SIGNAL(clicked()),
			 this, SLOT(openGraph()),
			 TRUE, klocale->translate("Open graph window"));

   toolbar->insertSeparator();
   toolbar->insertSeparator();
   
   toolbar->insertButton(IconLoader.loadIcon("samplin-setup.xpm"), 0, SIGNAL(clicked()),
			   this,
			   SLOT(setupScript()), TRUE, klocale->translate("Preferences"));

   toolbar->insertButton(IconLoader.loadIcon("samplin-devices.xpm"), 0, SIGNAL(clicked()),
			 this, SLOT(editDevices()),
			 TRUE, klocale->translate("Configure devices"));
  
   toolbar->insertButton(IconLoader.loadIcon("samplin-cross.xpm"), 1, SIGNAL(clicked()),
			 this,
			 SLOT(closeMe()), TRUE, klocale->translate("Quit"));
   toolbar->alignItemRight(1,TRUE);   
   toolbar->setBarPos(KToolBar::Top);
   addToolBar(toolbar);

   setView(split);

   connect(editor,SIGNAL(CursorPositionChanged()),this,SLOT(cursorPositionSlot()));
   connect(editor,SIGNAL(fileChanged()),this,SLOT(cursorPositionSlot()));
   connect(editor,SIGNAL(toggle_overwrite_signal()),this,SLOT(toggleOverwriteSlot()));
   connect(editor, SIGNAL(textChanged()), this, SLOT(textChangedSlot()));   
      
   newScript();
   
   stateScript(SamplinScript::FINISHED);
}


Samplin_main::~Samplin_main()
{
   delete editor;
   delete compiler;
}

void Samplin_main::closeMe()
{
   close();
}

void Samplin_main::helpAbout()
{
   
   SamplinAbout *about;
   
   about = new SamplinAbout(this, "About", FALSE);
   about->show();
   
   delete about;
 }

void Samplin_main::helpContents()
{
   kapp->invokeHTMLHelp("","");
}


void Samplin_main::editDevices()
{
   DeviceDlg *ddlg;
   
   ddlg = new DeviceDlg(this, "Configure devices");
   if(ddlg->setFile(cnf.cnf_dir+cnf.devices_file))
     ddlg->show();
   delete ddlg;
   
}

void Samplin_main::resizeEvent( QResizeEvent *e )
{
   KTopLevelWidget::resizeEvent(e);

}

void Samplin_main::openGraph(void)
{
   
   compiler->openGraph("New graph");

}

void Samplin_main::openGraphSlot(uint id, GraphWindow *gwin)
{
   QString label;
   
   label.sprintf("%u. %s",id,gwin->name());
   m_graphs.insertItem(label,id);
}

bool Samplin_main::closeAllGraphs()
{
   return compiler->closeAllGraphs();
   
}

void Samplin_main::closeGraphSlot(uint id)
{
   m_graphs.removeItem(id);
}

void Samplin_main::setupScript()
{
   QTabDialog *dlg;
   PrefDirs *w1;
   PrefComp *w2;
   
   dlg=new QTabDialog(this,"Preferences",TRUE);
   w1=new PrefDirs(dlg,"Paths");
   w2=new PrefComp(dlg,"Script");
   w1->setData(&cnf);
   w2->setData(&cnf);
   dlg->setCaption("SampLin preferences");
   dlg->addTab(w1,"Paths");
   dlg->addTab(w2,"Script");
   dlg->setCancelButton();
   dlg->setOkButton();
   connect(dlg,SIGNAL(applyButtonPressed()), w1,SLOT(apply()));
   connect(dlg,SIGNAL(applyButtonPressed()), w2,SLOT(apply()));   
   dlg->setFixedSize(400,330);
   dlg->show();
   writeConfig();
}

void Samplin_main::externEdit(void)
{
   externEdit(scriptname);
}

void Samplin_main::externEdit(const char *filename)
{
   int ret;

   int pid,i;
   char *argv[64], *ptr;
   
   pid=fork();
   if(pid==-1)return;
   if(pid==0){
      
      ptr=cnf.edit_command.data();
      argv[0]=strtok(ptr," ");
      i=0;
      do{
	 ++i;
	 argv[i]=strtok(NULL," ");
	 
	 if(!strcmp("%f",argv[i]) && strlen(argv[i])==strlen("%f"))
	   argv[i]=(char *)filename;
      }while(argv[i]!=NULL && i<63);
      argv[i]=NULL;
      
      chdir(cnf.root_dir+cnf.scripts_dir);
      execvp(argv[0],argv);
      exit(0);
      }
   else {
      usleep(250000);
      ret=waitpid(pid,NULL,WNOHANG);
      if(ret==-1)
	QMessageBox::warning(NULL,"Error", "Could not execute command '"+cnf.edit_command+"'");
   }
}



void Samplin_main::stopScript()
{
   compiler->stop();
}

int Samplin_main::compileScript()
{
   int ret;
   
   if(text_changed){
      switch( QMessageBox::information( this, "SampLin Main",
				       "Source file not saved"
				       "Do you want to save & compile ?",
				       "&Yes", "&No",0,
				       0 ) ) {
       case 0:   editSave();
	 break;
       case 1: return(-1);
	 break;
	}
   }

   if(cnf.mesg_win==TRUE)compiler->setMsgwin(messagewin);
   else compiler->setMsgwin(NULL);
   
   compiler->setDevicesFile(cnf.cnf_dir+cnf.devices_file);
   compiler->setInfoLevel(cnf.info_level);
   
   ret=compiler->compile(scriptname);

   if(ret!=0){
      editor->markLine(compiler->currentLine()-1,QColor(230,110,110));
      editor->setCursorPosition(compiler->currentLine()-1,0);
   }
   
   return ret;
}

void Samplin_main::runScript()
{
   int ret=0;
   
   if(compiler->programState()==SamplinScript::FINISHED)
     ret=compileScript();
   
   if(ret==0){
      compiler->go(-1);
   }
}
void Samplin_main::resetScript()
{
   compiler->reset();
}
void Samplin_main::stepScript()
{
   if(compiler->programState()==SamplinScript::FINISHED){
      if(!compileScript()){
	 editor->markLine(compiler->currentLine()-1,QColor(210,200,170));
	 editor->setCursorPosition(compiler->currentLine()-1,0);	 
      }
   }
   else{
      compiler->go(0);
   }
}

void Samplin_main::gotoScript()
{
   int ret=0;
   
   if(compiler->programState()==SamplinScript::FINISHED){
      ret=compileScript();
   }
   if(!ret){
      int x,y;
      editor->cursorPosition(&y,&x);
      compiler->go(y+1);
   }
}

  
void Samplin_main::stateScript(int state)
{
   static int old_state=0;
   
   if(state!=old_state){

      cursorPositionSlot();
      
      switch(state){
	 
       case SamplinScript::COMPILING:
	 statusbar->changeItem("Status: Compiling", ID_STATUS);
	 toolbar->setItemEnabled(BUTTON_START,FALSE);
	 toolbar->setItemEnabled(BUTTON_STOP,FALSE);
	 toolbar->setItemEnabled(BUTTON_STEP,FALSE);
	 toolbar->setItemEnabled(BUTTON_RESET,FALSE);
	 
	 menubar->setItemEnabled(BUTTON_START,FALSE);
	 menubar->setItemEnabled(BUTTON_STOP,FALSE);
	 menubar->setItemEnabled(BUTTON_STEP,FALSE);
	 menubar->setItemEnabled(BUTTON_RESET,FALSE);
	 menubar->setItemEnabled(BUTTON_GOTO,FALSE);
	 prog_active=TRUE;
	 editor->markLine(-1);
	 break;
       case SamplinScript::RUNNING:
	 statusbar->changeItem("Status: Running", ID_STATUS);
	 toolbar->setItemEnabled(BUTTON_START,FALSE);
	 toolbar->setItemEnabled(BUTTON_STOP,TRUE);
	 toolbar->setItemEnabled(BUTTON_STEP,FALSE);
	 toolbar->setItemEnabled(BUTTON_RESET,TRUE);

	 menubar->setItemEnabled(BUTTON_START,FALSE);
	 menubar->setItemEnabled(BUTTON_STOP,TRUE);
	 menubar->setItemEnabled(BUTTON_STEP,FALSE);
	 menubar->setItemEnabled(BUTTON_RESET,TRUE);
	 menubar->setItemEnabled(BUTTON_GOTO,FALSE);

	 prog_active=TRUE;
	 editor->markLine(-1);
	 break;      
       case SamplinScript::FINISHED:
	 statusbar->changeItem("Status: Idle", ID_STATUS);
	 toolbar->setItemEnabled(BUTTON_START,TRUE);
	 toolbar->setItemEnabled(BUTTON_STOP,FALSE);
	 toolbar->setItemEnabled(BUTTON_STEP,TRUE);
	 toolbar->setItemEnabled(BUTTON_RESET,FALSE);
	 
	 menubar->setItemEnabled(BUTTON_START,TRUE);
	 menubar->setItemEnabled(BUTTON_STOP,FALSE);
	 menubar->setItemEnabled(BUTTON_STEP,TRUE);
	 menubar->setItemEnabled(BUTTON_RESET,FALSE);
	 menubar->setItemEnabled(BUTTON_GOTO,TRUE);

	 prog_active=FALSE;
	 editor->markLine(-1);
      break;      
       case SamplinScript::STOPPED:
	 statusbar->changeItem("Status: Stopped", ID_STATUS);
	 toolbar->setItemEnabled(BUTTON_START,TRUE);
	 toolbar->setItemEnabled(BUTTON_STOP,FALSE);
	 toolbar->setItemEnabled(BUTTON_STEP,TRUE);
	 toolbar->setItemEnabled(BUTTON_RESET,TRUE);

	 menubar->setItemEnabled(BUTTON_START,TRUE);
	 menubar->setItemEnabled(BUTTON_STOP,FALSE);
	 menubar->setItemEnabled(BUTTON_STEP,TRUE);
	 menubar->setItemEnabled(BUTTON_RESET,TRUE);
	 menubar->setItemEnabled(BUTTON_GOTO,TRUE);

	 prog_active=TRUE;
//	 if(!text_changed_warn){
	    editor->markLine(compiler->currentLine()-1,QColor(210,200,170));
	    editor->setCursorPosition(compiler->currentLine()-1,0);
//	 }
	 break;      
       case SamplinScript::COMPILED:
	 statusbar->changeItem("Status: Compiled", ID_STATUS);
	 toolbar->setItemEnabled(BUTTON_START,TRUE);
	 toolbar->setItemEnabled(BUTTON_STOP,FALSE);
	 toolbar->setItemEnabled(BUTTON_STEP,TRUE);
	 toolbar->setItemEnabled(BUTTON_RESET,TRUE);
	 
	 menubar->setItemEnabled(BUTTON_START,TRUE);
	 menubar->setItemEnabled(BUTTON_STOP,FALSE);
	 menubar->setItemEnabled(BUTTON_STEP,TRUE);
	 menubar->setItemEnabled(BUTTON_RESET,TRUE);
	 menubar->setItemEnabled(BUTTON_GOTO,TRUE);
	 prog_active=TRUE;
	 break;      
      }
      old_state=state;
   }
   
}

void Samplin_main::updateCaption()
{
   QString title;
   
   title=QString("SampLin Main ")+"["+scriptname+"]";
   setCaption(title);
   setIconText(title);
}

void Samplin_main::closeEvent(QCloseEvent *e)
{
   if(closeAllGraphs()){
      resetScript();
      if(!editor->newFile()){
	 e->accept();
      }
      else e->ignore();
   }
   else e->ignore();
}

void Samplin_main::cursorPositionSlot()
{
   static int old_line=-1,old_col=-1,old_lines=-1;
   QString linenumber;
   int line,col,lines;
   
   editor->getCursorPosition(&line,&col);
   lines=editor->numLines();

   if(line!=old_line || col!=old_col || lines!=old_lines){
      linenumber.sprintf("Line: %d/%d Col: %d",line+1,lines,col+1);
      statusbar->changeItem(linenumber.data(),ID_LINE_COLUMN);
      old_line=line;old_col=col;old_lines=lines;
   }
   
}

void Samplin_main::toggleOverwriteSlot()
{
   
   if(editor->isOverwriteMode()){
      statusbar->changeItem("OVR",ID_INS_OVR);
   }
   else{
      statusbar->changeItem("INS",ID_INS_OVR);
   }
   
}

void Samplin_main::textChangedSlot()
{
   int cstate=compiler->programState();
   
   if(cstate==SamplinScript::FINISHED)
     editor->markLine(-1);   
   text_changed=TRUE;
   statusbar->changeItem("*",ID_CHANGED);

   if((cstate==SamplinScript::COMPILED || cstate==SamplinScript::STOPPED) && !text_changed_warn){
      QMessageBox::warning(NULL,"SampLin Main", "Source file changed\nLine numbers may not match");     
      text_changed_warn=TRUE;
   }

}

void Samplin_main::editCopy()
{
   editor->copyText();
}

void Samplin_main::editPaste()
{
   editor->paste();
}

void Samplin_main::editCut()
{
   editor->cut();
}

void Samplin_main::editSelectAll()
{
   editor->selectAll();
}

void Samplin_main::editInsert()
{
   editor->insertFile();
}

void Samplin_main::editFind()
{
   editor->Search();
}

void Samplin_main::editFindAgain()
{
   editor->repeatSearch();
}

void Samplin_main::editReplace()
{
   editor->Replace();
}

void Samplin_main::editGoto()
{
   editor->doGotoLine();
}

void Samplin_main::editOpen()
{
   if(editor->openFile(SamplinEdit::OPEN_READWRITE)==SamplinEdit::KEDIT_OK){
      scriptname=editor->getName();
      updateCaption();
      text_changed=FALSE;
      statusbar->changeItem("",ID_CHANGED);
   }
}
   
   
void Samplin_main::editSave()
{
   if(editor->doSave()==SamplinEdit::KEDIT_OK){
      scriptname=editor->getName();
      updateCaption();
      text_changed=FALSE;
      text_changed_warn=FALSE;
      statusbar->changeItem("",ID_CHANGED);
   }
}

void Samplin_main::editSaveAs()
{

   if(editor->saveAs()==SamplinEdit::KEDIT_OK){
      scriptname=editor->getName();
      updateCaption();
      text_changed=FALSE;
      text_changed_warn=FALSE;
      statusbar->changeItem("",ID_CHANGED);      
   }
}
   
void Samplin_main::newScript()
{
   if(editor->newFile()==SamplinEdit::KEDIT_OK){
      scriptname=editor->getName();
      updateCaption(); 
   }
}

