/***************************************************************************
              ktexshell.cpp  -  methods of the core application
                             -------------------
    begin                : Son Okt  1 17:17:42 CEST 2000
    copyright            : (C) 2000 by Arnd Fischer
    email                : lxuser@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <limits.h>
#include <string.h>
#include <string>
#include <iostream.h>

#include <qmessagebox.h>
#include <qstring.h>
#include <qfile.h>
#include <qtextstream.h>

#include <kiconloader.h>
#include <kfiledialog.h>
#include <kconfig.h>

#include "ktexshell.h"

KTeXShell::KTeXShell(){

	w = new QWidget (this, "Main Widget");
	setView(w);

	// read the configuration
	readConfig();
	// set up a menubar
  setupMenubar();
  // set up a toolbar
  setupToolbar();
  // set up a statusbar
  setupStatusbar();
	// set up the dislpay/interactive widget
	outwidget = new QMultiLineEdit(this, 0);
	outwidget->setReadOnly(TRUE);	
	outwidget->setText(KTS_COPYRIGHT);
	// determine working dir
	getWDFromPrim();
	// prepare the process list
	processlist = new QList<pid_t>;
	processlist->setAutoDelete(true);
}

KTeXShell::~KTeXShell(){
}

/** slot: which toolbar item is clicked */
void KTeXShell::slotToolbarClicked(int item){

  // Handle all buttons in toolbar.
  switch (item)
    {
		case TOOLBAR_WD:
			slotSetWorkingDir();
			break;
		case TOOLBAR_PRIM:
			slotSetPrimary();
			break;
		case TOOLBAR_EDIT:
			slotEditPrimary();
			break;
		case TOOLBAR_NPFT:
			slotNewPrimFromTemplate();
			break;
		case TOOLBAR_COMPOSE:
			slotCompose();
			break;
		case TOOLBAR_VIEW:
			slotView();
			break;
		case TOOLBAR_PRINT:
			slotPrint();
			break;
		case TOOLBAR_CONF:
			slotConfSettings();
			break;
    case TOOLBAR_QUIT:
      slotQuit();
      break;
    case TOOLBAR_HELP:
      kapp->invokeHTMLHelp( "", "" );
      break;
    default:
      // OOps! This shouldn't happen.
      break;
    }

}

/** slot: quit the app */
void KTeXShell::slotQuit(){

	// first save the config
	saveConfig();

	// check for running processes that have registered
	// in the process list
	if(!processlist->isEmpty())
	{
		QMessageBox::warning (this, i18n("Error"),
			i18n("There are still child processes open.\n"\
			"Please close them first."),
			i18n("OK"), 0, 0, 0, -1);
		return;
	}

	// really quit
  kapp->quit();
}

/** set up the menubar */
void KTeXShell::setupMenubar(){

  // Create an KAccel instance
  accel = new KAccel( this );

  // Each entry in a menubar is of the type QPopupMenu.
  QPopupMenu *f = new QPopupMenu; // file menu
	QPopupMenu *a = new QPopupMenu; // action menu
	QPopupMenu *u = new QPopupMenu; // utils menu
	QPopupMenu *c = new QPopupMenu; // config menu
	QPopupMenu *t = new QPopupMenu; // TeX formats

  // Create a file menu, connect the accelerator key to our slot, and
  // tell KAccel to show the key in the menu.
	int id;
	id = f->insertItem(i18n("Set &Primary"), this, SLOT(slotSetPrimary()));
  id = f->insertItem(i18n("&Edit Primary"), this, SLOT(slotEditPrimary()));
  id = f->insertItem(i18n("&New Primary from Template"), \
			this, SLOT(slotNewPrimFromTemplate()));
	id = f->insertItem(i18n("Edit &File"), this, SLOT(slotEditFile()));
	     f->insertSeparator();
  id = f->insertItem(i18n("&Quit"), this, SLOT(slotQuit()));
  accel->connectItem(KAccel::Quit, this, SLOT(slotQuit()));
  accel->changeMenuAccel(f, id, KAccel::Quit);

  // The file menu p goes into the menubar.
  menuBar()->insertItem(i18n("&File"), f);

  // Create an action menu, connect the accelerator key to our slot, and
  // tell KAccel to show the key in the menu.
	id = a->insertItem(i18n("&Compose"), this, SLOT(slotCompose()));
	id = a->insertItem(i18n("Make &Index"), this, SLOT(slotMakeIndex()));
  id = a->insertItem(i18n("&View"), this, SLOT(slotView()));
  id = a->insertItem(i18n("&Print"), this, SLOT(slotPrint()));
  accel->connectItem(KAccel::Print, this, SLOT(slotPrint()));
  accel->changeMenuAccel(a, id, KAccel::Print);

  // The action menu a goes into the menubar.
  menuBar()->insertItem(i18n("&Action"), a);

  // Create an utils menu, connect the accelerator key to our slot, and
  // tell KAccel to show the key in the menu.
  id = u->insertItem(i18n("View &Log"), this, SLOT(slotViewLog()));
  id = u->insertItem(i18n("Clear &Output"), this, SLOT(slotClearOutput()));
  id = u->insertItem(i18n("Clear &Working Dir"), this, SLOT(slotClearWD()));
	
  // The action menu a goes into the menubar.
  menuBar()->insertItem(i18n("&Utils"), u);

	// create a dynamic submenu for TeX Formats under the config menu
	if(ktsConfPrograms.formats.first())
	{
		QString item = ktsConfPrograms.formats.first();
		id = t->insertItem(item);
		while(item = ktsConfPrograms.formats.next())
			id = t->insertItem(item);
		// shit! Connecting to SLOT(slotMenuFormatChoosen(item) doesn't work
		// so we need an extra slot
		connect(t, SIGNAL(activated(int)), this, SLOT(slotMenuFormatChoosen(int)));
	}

	// the submenu goes into the config menu
	id = c->insertItem(i18n("Set TeX Format"), t);

  // Create an config menu, connect the accelerator key to our slot, and
  // tell KAccel to show the key in the menu.
	id = c->insertItem(i18n("Set &Working dir"), this, SLOT(slotSetWorkingDir()));
	     c->insertSeparator();
  id = c->insertItem(i18n("&Settings"), this, SLOT(slotConfSettings()));

  // The action menu a goes into the menubar.
  menuBar()->insertItem(i18n("&Config"), c);

  // provide the 'About KTeXShell' entry.
  f = kapp->getHelpMenu(false, KTS_COPYRIGHT);
  menuBar()->insertSeparator();
  menuBar()->insertItem(i18n("&Help"), f);
}

/** set up the toolbar */
void KTeXShell::setupToolbar(){

  // Some buttons for the toolbar.
  toolBar()->insertButton(Icon("tick.xpm"), TOOLBAR_PRIM, \
			true, i18n("Set Primary"));
  toolBar()->insertButton(Icon("pencil.xpm"), TOOLBAR_EDIT, \
			true, i18n("Edit Primary"));
  toolBar()->insertButton(Icon("filenew.xpm"), TOOLBAR_NPFT, \
			true, i18n("New Primary from Template"));
  toolBar()->insertButton(Icon("bottom.xpm"), TOOLBAR_COMPOSE, \
			true, i18n("Compose"));
  toolBar()->insertButton(Icon("find.xpm"), TOOLBAR_VIEW, \
			true, i18n("View"));
  toolBar()->insertButton(Icon("fileprint.xpm"), TOOLBAR_PRINT, \
			true, i18n("Print"));
  toolBar()->insertButton(Icon("configure.xpm"), TOOLBAR_CONF, \
			true, i18n("Configuration"));
  toolBar()->insertButton(Icon("exit.xpm"), TOOLBAR_QUIT, \
			true, i18n("Exit"));
  toolBar()->insertButton(Icon("help.xpm"), TOOLBAR_HELP, \
			true, i18n("Help"));

  // and connect the whole toolbar to a slot
  connect(toolBar(), SIGNAL(clicked(int)), SLOT(slotToolbarClicked(int)));
}

/** set up the status bar */
void KTeXShell::setupStatusbar(){

  // standard status entry is "Ready."
  statusBar()->insertItem(i18n("Ready."), 0 );
}

/** set the selected file als primary */
void KTeXShell::slotSetPrimary(){

	//open a file dialog and set the selected file to primary
	primary = KFileDialog::getOpenFileName(workingDir, "*.tex", \
			this, i18n("Set Primary File"));
	if(!testPrimary())
	{
		outwidget->append(i18n("\nNo Primary file set!"));	
		return;
	}
	// set and display primary and working dir
	getWDFromPrim();
}

/** edit the primary file */
void KTeXShell::slotEditPrimary(){

	// test if we have a primary file to edit
	if(!testPrimary())
		return;

	// construct and call editor process
	process = new KShellProcess();
	*process << editor << " " << primary;

	// reset output and connect the exit notify
	outwidget->setText("");
	connect(process, SIGNAL(processExited(KProcess*)), \
			this, SLOT(slotProcessOK(KProcess*)));

	// start editor, register process and set statusbar
	if(!(process->start(KProcess::NotifyOnExit, KProcess::NoCommunication)))
			QMessageBox::warning (this, i18n("Error"),
				i18n("Could not start editor."), i18n("OK"), 0, 0, 0, -1);

	pid_t actpid = process->getPid();
	pid_t *actpidp = new pid_t; *actpidp = actpid;
	processlist->append(actpidp);

	statusBar()->changeItem(i18n("Running Editor..."), 0);
}

/** call TeX to compose the document */
void KTeXShell::slotCompose(){

	// test if we have a primary file to edit
	if(!testPrimary())
		return;

	// set and display primary and working dir
	getWDFromPrim();

	// construct and call compose process
	texprocess = new KShellProcess();
	*texprocess << "cd " << workingDir << " && ";
	*texprocess << texFormat << " " << primary;

	// reset output and connect the process
	outwidget->setText("");
	connect(texprocess, SIGNAL( receivedStdout(KProcess*, char*, int) ), \
			this, SLOT(slotFilterOutput(KProcess*, char*, int ) ) );
	connect(texprocess, SIGNAL( receivedStderr(KProcess*, char*, int) ), \
			this, SLOT(slotFilterOutput(KProcess*, char*, int ) ) );
	connect(this, SIGNAL(signalProcessOutput( const char*) ), \
			outwidget, SLOT(append( const char* ) ) );
	connect(texprocess, SIGNAL(processExited(KProcess*)), \
			this, SLOT(slotProcessOK(KProcess*)));

	// start compose
	if(!(texprocess->start(KProcess::NotifyOnExit, KProcess::All)))
			QMessageBox::warning (this, i18n("Error"),
				i18n("Could not start composer."), i18n("OK"), 0, 0, 0, -1);

	// set up the problem handling dialog
	closeTeXDialog = new KTSCloseTeX();
	connect(closeTeXDialog, SIGNAL(signalActionSelected(char*)),
		this, SLOT(slotGetAction(char*)));

  // change status and check for possible TeX error messages
	statusBar()->changeItem(i18n("Running Compose..."), 0);
	while(texprocess->isRunning())
		slotTestTeXOK();
}

/** call the dvi viewer */
void KTeXShell::slotView(){

	// test if we have a primary file to view
	if(!testPrimary())
		return;

	// construct and call dvi viewer process
	char *dviname = tex2other(primary, "dvi");
	process = new KShellProcess();
	*process << dviViewer << " " << dviname;

	// reset output and connect process
	outwidget->setText("");
	connect(process, SIGNAL( receivedStdout(KProcess*, char*, int) ), \
			this, SLOT(slotFilterOutput(KProcess*, char*, int ) ) );
	connect(process, SIGNAL( receivedStderr(KProcess*, char*, int) ), \
			this, SLOT(slotFilterOutput(KProcess*, char*, int ) ) );
	connect(this, SIGNAL(signalProcessOutput( const char*) ), \
			outwidget, SLOT(append( const char* ) ) );
	connect(process, SIGNAL(processExited(KProcess*)), \
			this, SLOT(slotProcessOK(KProcess*)));

	// start viewer and register process
	if(!(process->start(KProcess::NotifyOnExit, KProcess::AllOutput)))
			QMessageBox::warning (this, i18n("Error"),
				i18n("Could not start dvi viewer."), "OK", 0, 0, 0, -1);

	pid_t actpid = process->getPid();
	pid_t *actpidp = new pid_t; *actpidp = actpid;
	processlist->append(actpidp);

	// inform the user and change status
	outwidget->append(i18n("\nStarted DVI-Viewer for "));
	outwidget->append(dviname);
	statusBar()->changeItem(i18n("Running DVI viewer..."), 0);
}

/** filter the TeX- (and other) Output */
void KTeXShell::slotFilterOutput(KProcess*, char* buffer, int len){

	// terminate buffer string & emit buffer
	buffer[len]='\0';
	emit signalProcessOutput(buffer);
}

/** make other filenames from .tex-name */
char* KTeXShell::tex2other(const char* texname, const char* suffix){

	// convert xyz.tex to xyz.[dvi|log|idx|whatwever]
	int len = strlen(texname);
	char* newname = new char[len];
  strncpy(newname, texname, len-3);
	// now please close your eyes...
	newname[len]='\0'; newname[len-1]=suffix[2]; newname[len-2]=suffix[1];
	newname[len-3]=suffix[0];
	return newname;	
}

/** external process is done */
void KTeXShell::slotProcessOK(KProcess* process){

	// reset status bar
	statusBar()->changeItem(i18n("Ready."), 0);

  // disconnect stdout-slots to avoid multiple outputs
	disconnect(process, SIGNAL(receivedStdout(KProcess*, char*, int)), \
		this, SLOT(slotFilterOutput(KProcess*, char*, int)));
	disconnect(process, SIGNAL(receivedStderr(KProcess*, char*, int)), \
		this, SLOT(slotFilterOutput(KProcess*, char*, int)));
	disconnect(this, SIGNAL(signalProcessOutput(const char*)), \
		outwidget, SLOT(append(const char*)));

	// unregister process from process list
	pid_t closed = process->getPid();
	pid_t *actpidp = new pid_t;
	pid_t *closedp = new pid_t; *closedp = closed;
	for(actpidp=processlist->first(); actpidp != 0; actpidp=processlist->next())
		if(*actpidp == *closedp)
			processlist->remove();

	// set cursor to the output end and remove process
	outwidget->setCursorPosition(INT_MAX, INT_MAX);
  delete process;
}

/** set the working directory */
void KTeXShell::slotSetWorkingDir(){

	//open a file dialog and set the dir to wd
	outwidget->setText("");
	workingDir = KFileDialog::getDirectory(0, this,
		i18n("Set Working Dir"));

  // test if we have a working dir
  if(!testWorkingDir())
	{
		outwidget->append(i18n("\nNo Working dir set!"));		
		return;
	}

	// inform the user about the change
	outwidget->append(i18n("\nWorking dir set to: "));
	outwidget->append(workingDir);
}

/** make the index */
void KTeXShell::slotMakeIndex(){

	// test if we have a primary file to view
	if(!testPrimary())
		return;

	// construct and call dvi viewer process
	char *idxname = tex2other(primary, "idx");
	process = new KShellProcess();
	*process << "makeindex " << idxname;

	// reset output and connect process
	outwidget->setText("");
	connect(process, SIGNAL(receivedStdout(KProcess*, char*, int)), \
			this, SLOT(slotFilterOutput(KProcess*, char*, int)));
	connect(process, SIGNAL(receivedStderr(KProcess*, char*, int)), \
			this, SLOT(slotFilterOutput(KProcess*, char*, int)));
	connect(this, SIGNAL(signalProcessOutput(const char*)), \
			outwidget, SLOT(append(const char*)));
	connect(process, SIGNAL(processExited(KProcess*)), \
			this, SLOT(slotProcessOK(KProcess*)));

	// start viewer
	if(!(process->start(KProcess::NotifyOnExit, KProcess::AllOutput)))
			QMessageBox::warning (this, i18n("Error"),
				i18n("Could not start makeindex."), "OK", 0, 0, 0, -1);

	// inform the user and change status
	outwidget->append(i18n("\nStarted makeindex for "));
	outwidget->append(idxname);
	statusBar()->changeItem(i18n("Running makeindex..."), 0);
}

/** view the logfile */
void KTeXShell::slotViewLog(){

	// do we have a primary file?
	if(!testPrimary())
		return;

	// derive logname from primary and display it in the
	// output window
	char *logname = tex2other(primary, "log");
  QFile f(logname);
		if ( f.open(IO_ReadOnly) ) {
			outwidget->setText("");
			QTextStream t( &f );
			QString s;
			while ( !t.eof() ) {
				s = t.readLine();
				outwidget->append(s);
			}
		}
		else
			QMessageBox::warning (this, i18n("Warning"),
				i18n("Could not open logfile."), i18n("OK"), 0, 0, 0, -1);
	f.close();
}

/** the resize event for the main window */
void KTeXShell::resizeEvent(QResizeEvent *){

	// adjust outwidget to maxwidth and maxheight - KTMW-widgets
  outwidget->setGeometry(0, toolBar()->height() + menuBar()->height(),
		this->width(),
		this->height() - statusBar()->height() \
		- toolBar()->height() - menuBar()->height());

	// call internal resize event
	this->updateRects();
}

/** test if we have a primary file to edit, compose etc. */
bool KTeXShell::testPrimary(){

	bool prim_pres = TRUE;
	if(primary == 0)
	{
		QMessageBox::warning (this, i18n("Error"),
			i18n("No primary file set!"), i18n("OK"), 0, 0, 0, -1);
		prim_pres = FALSE;
	}
	return prim_pres;
}

/** test if we have a working dir */
bool KTeXShell::testWorkingDir(bool WARNING = TRUE){

	bool wd_pres = TRUE;
	if(workingDir == 0)
	{
		if(WARNING == TRUE)
			QMessageBox::warning (this, i18n("Warning"),
				i18n("No working dir set."), i18n("OK"), 0, 0, 0, -1);
		wd_pres = FALSE;
	}
	return wd_pres;
}

/** generate and set a new primay from template */
void KTeXShell::slotNewPrimFromTemplate(){

	// get template name
	QString temp = KFileDialog::getOpenFileName(templateDir, "*.tex", \
			this, i18n("Select template"));
	if(temp.isEmpty())
	{
		outwidget->append(i18n("\nNo template selected!"));	
		return;
	}

	// get new file name
	QString newprim = KFileDialog::getSaveFileName(workingDir, "*.tex", \
			this, i18n("Save new File"));
	if(newprim.isEmpty())
	{
		outwidget->append(i18n("\nNo name selected!"));	
		return;
	}

	// copy template to new file
	process = new KShellProcess();
	*process << "cp " << temp << " " << newprim;
	connect(process, SIGNAL(processExited(KProcess*)), \
			this, SLOT(slotProcessOK(KProcess*)));

	// run process and update status
	if(!(process->start(KProcess::NotifyOnExit, KProcess::NoCommunication)))
			QMessageBox::warning (this, i18n("Error"),
				i18n("Could not make primary from template."),
					i18n( "OK" ), 0, 0, 0, -1);
	statusBar()->changeItem(i18n("Making new primary..."), 0);

	// make new file primary and display prim. and wd
	primary = newprim;
	getWDFromPrim();
}

/** test if TeX returned an error */
void KTeXShell::slotTestTeXOK(){

	// get the last written line
	int actLine = outwidget->numLines()-1;
	outwidget->setCursorPosition(actLine,1);
	kapp->processEvents();

	// do we have a question mark, i.e. does TeX wait for input?
	if(!(strncmp(outwidget->textLine(actLine),"?",1)))
		closeTeXDialog->show();

	// is TeX missing a file?	Then replace it with null.tex
	if(!(strncmp(outwidget->textLine(actLine),"Enter file name:",16)))
	{
		texprocess->writeStdin("null.tex\n", 9);
		QMessageBox::warning (this, i18n("Warning"),
			i18n("TeX missed a file.\nI Replaced it by null.tex."),
			i18n( "OK" ), 0, 0, 0, -1);
	}
	if(!(strncmp(outwidget->textLine(actLine),"Please type",11)))
	{
		texprocess->writeStdin("null.tex\n", 9);
		QMessageBox::warning (this, i18n("Warning"),
			i18n("TeX missed a file.\nI replaced it by null.tex."),
			i18n( "OK" ), 0, 0, 0, -1);
	}
}

/** get the selected TeX error action */
void KTeXShell::slotGetAction(char* action){

	// pass the selected action to TeX
	if(!strcmp(action, i18n("Close TeX")))
	{
		// maybe a bad hack to get rid of multiple
		// dialogs on close TeX
		delete closeTeXDialog;
		texprocess->writeStdin("x\n", 2);
	}	else
	if(!strcmp(action, i18n("Help")))
		texprocess->writeStdin("h\n", 2);  else
	if(!strcmp(action, i18n("Proceed")))
		texprocess->writeStdin("\n", 1);  else
	if(!strcmp(action, i18n("Scroll future messages")))
		texprocess->writeStdin("s\n", 2); else
	if(!strcmp(action, i18n("Run without stopping")))
		texprocess->writeStdin("r\n", 2); else
	if(!strcmp(action, i18n("Run quietly")))
		texprocess->writeStdin("q\n", 2);
}

/** open editor for a project file */
void KTeXShell::slotEditFile(){

	// open file dialog
	QString prfile = KFileDialog::getOpenFileName(workingDir, "*", \
			this, i18n("Open File for Edit"));

	// no file selected?
	if(prfile.isEmpty())
	{
		outwidget->append(i18n("\nNo file selected!"));	
		return;
	}

	// construct editor process
	process = new KShellProcess();

	// build command line
	*process << editor << " ";
	*process << prfile;

	// connect process
	connect(process, SIGNAL(processExited(KProcess*)), \
			this, SLOT(slotProcessOK(KProcess*)));

	// run, register process and update status
	if(!(process->start(KProcess::NotifyOnExit, KProcess::NoCommunication)))
			QMessageBox::warning (this, i18n("Error"),
				i18n("Could not start editor."), i18n("OK"), 0, 0, 0, -1);

	pid_t actpid = process->getPid();
	pid_t *actpidp = new pid_t; *actpidp = actpid;
	processlist->append(actpidp);

	statusBar()->changeItem(i18n("Running Editor..."), 0);
}

/** get the working dir from the primary file */
void KTeXShell::getWDFromPrim(){

	// inform the user about the change
	outwidget->append(i18n("\nPrimary file set to: "));
	outwidget->append(primary);

	// set working dir according to new primary
	int cutpos = primary.findRev("/",-1,FALSE);
	QString dir = primary;
	dir.truncate(cutpos);
  workingDir = dir;

	// inform the user about the change in wd
	outwidget->append(i18n("\nWorking dir set to: "));
	outwidget->append(workingDir);
}

/** the whole configuration */
void KTeXShell::slotConfSettings(){

	// build config dialog
	configDialog = new KTSConfig(this, 0);

	// connect conf. signals
	connect(configDialog, SIGNAL(signalTempDirChanged(QString)),
		this, SLOT(slotSetTempDir(QString)));
	connect(configDialog, SIGNAL(signalFormatChanged(QString)),
		this, SLOT(slotSetFormat(QString)));
	connect(configDialog, SIGNAL(signalEditorChanged(QString)),
		this, SLOT(slotSetEditor(QString)));
	connect(configDialog, SIGNAL(signalViewerChanged(QString)),
		this, SLOT(slotSetViewer(QString)));
	connect(configDialog, SIGNAL(signalConfChanged(ConfPrograms *)),
		this, SLOT(slotSetConf(ConfPrograms *)));

	// show config dialog and save configuration on return
	configDialog->show();
	saveConfig();

	// make new menubar in order to see format changes
	menuBar()->clear();
	setupMenubar();
}

/** save the configuration */
void KTeXShell::saveConfig(){

	// make configuration instance and some variables
	KConfig *config = kapp->getConfig();
	unsigned int i;
	QString name;
	QString number;

	config->setGroup("Environment");
	config->writeEntry("Primary File", primary);
	config->writeEntry("Working Dir", workingDir);
	config->writeEntry("Template Dir", templateDir);

	config->setGroup("Settings");
	config->writeEntry("Format", texFormat);
	config->writeEntry("Editor", editor);
	config->writeEntry("DVI Viewer", dviViewer);

	config->setGroup("Formats");
	config->writeEntry("Number", ktsConfPrograms.formats.count());

	// build and write entries for formats
	for(i = 0; i < ktsConfPrograms.formats.count(); i++)
	{
		name = "Format_";
		name += number.sprintf("%d",i);
		config->writeEntry(name, ktsConfPrograms.formats.at(i));
	}

	// build and write entries for editors
	config->setGroup("Editors");
	config->writeEntry("Number", ktsConfPrograms.editors.count());
	for(i = 0; i < ktsConfPrograms.editors.count(); i++)
	{
		name = "Editor_";
		name += number.sprintf("%d",i);
		config->writeEntry(name, ktsConfPrograms.editors.at(i));
	}

	// build and write entries for viewers
	config->setGroup("Viewers");
	config->writeEntry("Number", ktsConfPrograms.viewers.count());
	for(i = 0; i < ktsConfPrograms.viewers.count(); i++)
	{
		name = "Viewer_";
		name += number.sprintf("%d",i);
		config->writeEntry(name, ktsConfPrograms.viewers.at(i));
	}
}

/** read the configuration */
void KTeXShell::readConfig(){

	// make configuration instance and some variables
	KConfig *config = kapp->getConfig();
	unsigned int i, j;
	QString name;
	QString number;

	config->setGroup("Environment");
	primary    	= config->readEntry("Primary File");
	workingDir 	= config->readEntry("Working Dir");
	templateDir = config->readEntry("Template Dir");

	config->setGroup("Settings");
	texFormat   = config->readEntry("Format");
	editor      = config->readEntry("Editor");
	dviViewer   = config->readEntry("DVI Viewer");

	// build and read entries for formats
	config->setGroup("Formats");
	j = config->readNumEntry("Number");
	for(i = 0; i < j; i++)
	{
		name = "Format_";
		name += number.sprintf("%d",i);
		ktsConfPrograms.formats.append(config->readEntry(name));
	}

	// build and read entries for editors
	config->setGroup("Editors");
	j = config->readNumEntry("Number");
	for(i = 0; i < j; i++)
	{
		name = "Editor_";
		name += number.sprintf("%d",i);
		ktsConfPrograms.editors.append(config->readEntry(name));
	}

	// build and read entries for viewers
	config->setGroup("Viewers");
	j = config->readNumEntry("Number");
	for(i = 0; i < j; i++)
	{
		name = "Viewer_";
		name += number.sprintf("%d",i);
		ktsConfPrograms.viewers.append(config->readEntry(name));
	}
	
	// check if a TeX format is selected
	if(texFormat.isEmpty())
		QMessageBox::warning (this, i18n("Warning"),
			i18n("No TeX Format setting found.\nPlease configure the application."),
			"OK", 0, 0, 0, -1);	
}

/** close app via "X" */
bool KTeXShell::queryClose(){
	saveConfig();

	// check for running processes that have registered
	// in the process list
	if(!processlist->isEmpty())
	{
		QMessageBox::warning (this, i18n("Error"),
			i18n("There are still child processes open.\n"\
			"Please close them first."),
			i18n("OK"), 0, 0, 0, -1);
		return false;
	}

	return true;
}

/** set the template dir */
void KTeXShell::slotSetTempDir(QString td){
	templateDir = td;
}

/** set the editor */
void KTeXShell::slotSetEditor(QString ed){
	editor = ed;
}

/** set the compose format */
void KTeXShell::slotSetFormat(QString f){
	texFormat = f;
}

/** set the dvi viewer */
void KTeXShell::slotSetViewer(QString v){
	dviViewer = v;
}

/** set the available programs from config */
void KTeXShell::slotSetConf(ConfPrograms *confPrograms){

	ktsConfPrograms.editors.clear();
	ktsConfPrograms.editors = confPrograms->editors;	

	ktsConfPrograms.formats.clear();
	ktsConfPrograms.formats = confPrograms->formats;		

	ktsConfPrograms.viewers.clear();
	ktsConfPrograms.viewers = confPrograms->viewers;			
}

/** clear the output window */
void KTeXShell::slotClearOutput(){
	outwidget->setText("");
}

/** clear the working dir from all TeX output */
void KTeXShell::slotClearWD(){

  // test if we have a working dir
  if(!testWorkingDir())
	{
		outwidget->append(i18n("\nNo Working dir set!"));		
		return;
	}

	// construct clear process
	process = new KShellProcess();

	// build command line
	*process << "cd " << workingDir << " && ";
	*process << "rm -f ";
	*process << " *.log *.aux *.toc *.lof *.lot *.bit *.idx *.glo *.bbl *.ilg";
	*process << " *.dvi *.ind";

	// connect process
	connect(process, SIGNAL(processExited(KProcess*)), \
			this, SLOT(slotProcessOK(KProcess*)));

	// run process and update status
	if(!(process->start(KProcess::NotifyOnExit, KProcess::NoCommunication)))
			QMessageBox::warning (this, i18n("Error"),
				i18n("Could not clear working dir."), i18n("OK"), 0, 0, 0, -1);
	statusBar()->changeItem( i18n("Removing TeX output..."), 0);
}

/** print the primary DVI-file */
void KTeXShell::slotPrint(){

  // test if we have a primary file
  if(!testPrimary())
		return;

	// construct clear process
	process = new KShellProcess();

	// build command line
	*process << "cd " << workingDir << " && ";
	*process << "dvips ";
	QString to_print = tex2other(primary, "dvi");
  *process << to_print;

	// connect process
	connect(process, SIGNAL(receivedStdout(KProcess*, char*, int)), \
			this, SLOT(slotFilterOutput(KProcess*, char*, int)));
	connect(process, SIGNAL(receivedStderr(KProcess*, char*, int)), \
			this, SLOT(slotFilterOutput(KProcess*, char*, int)));
	connect(this, SIGNAL(signalProcessOutput(const char*)), \
			outwidget, SLOT(append(const char*)));
	connect(process, SIGNAL(processExited(KProcess*)), \
			this, SLOT(slotProcessOK(KProcess*)));

	// run process and update status
	if(!(process->start(KProcess::NotifyOnExit, KProcess::AllOutput)))
			QMessageBox::warning (this, i18n("Error"),
				i18n("Could not start printing process."), i18n("OK"), 0, 0, 0, -1);
	statusBar()->changeItem( i18n("Printing..."), 0);
}

/** an entry from the dynamic menu is choosen */
void KTeXShell::slotMenuFormatChoosen(int item){

	// set format to item-matching entry
	slotSetFormat(ktsConfPrograms.formats.at(item));
	outwidget->append("\nTeX Format set to:");
	outwidget->append(ktsConfPrograms.formats.at(item));
}
