;;
;; Runtime library for Siag
;; Copyright 1996-1998 Ulric Eriksson
;;

; Load the runtime library from Siod
(require (string-append SIAGHOME "/siod/siod.scm"))

; These are missing
(define (caaaar l) (caar (caar l)))
(define (cdaaar l) (cdar (caar l)))
(define (cadaar l) (cadr (caar l)))
(define (cddaar l) (cddr (caar l)))
(define (caadar l) (caar (cdar l)))
(define (cdadar l) (cdar (cdar l)))
(define (caddar l) (cadr (cdar l)))
(define (cdddar l) (cddr (cdar l)))
(define (caaadr l) (caar (cadr l)))
(define (cdaadr l) (cdar (cadr l)))
(define (cadadr l) (cadr (cadr l)))
(define (cddadr l) (cddr (cadr l)))
(define (caaddr l) (caar (cddr l)))
(define (cdaddr l) (cdar (cddr l)))
(define (cadddr l) (cadr (cddr l)))
(define (cddddr l) (cddr (cddr l)))

; Set up the menu
(require (string-append SIAGHOME "/siag/menu.scm"))

; Set up the key bindings
(require (string-append SIAGHOME "/siag/keytable.scm"))

; X string definitions
(require (string-append SIAGHOME "/xcommon/StringDefs.scm"))

; Form interface
(require (string-append SIAGHOME "/xcommon/form.scm"))

; Data interface
(require (string-append SIAGHOME "/siag/data.scm"))

(define EMPTY 0)
(define LABEL 1)
(define ERROR 2)
(define EXPRESSION 4)
(define STRING 5)
(define EMBED 6)

(define (llpr x)
	(puts x))

;; Handle the "position"
(define (make-position row col)
	(list row col))

(define (position-row p)
	(car p))

(define (position-col p)
	(cadr p))

;; Functions that do approximately the same as position.c
;;
;; The following functions are implemented in C:
;; get-point, set-point, get-mark, set-mark, get-blkl, set-blkl,
;; get-blku, set-blku, get-prot, set-prot
;; line-last-used, col-last-used, max-lines, max-columns

(define (get-point-row)
  (position-row (get-point)))

(define (get-point-col)
  (position-col (get-point)))

(define (set-point-row row)
  (set-point (make-position row (get-point-col))))

(define (set-point-col col)
  (set-point (make-position (get-point-row) col)))

(define (find-beginning-of-buffer)
  (get-prot))

(define (find-end-of-buffer)
	(let ((row (line-last-used)))
		(make-position row (col-last-used row))))

(define (line-forward p)
	(if (< (position-row p) (max-lines))
		(make-position (+ (position-row p) 1) (position-col p))
		p))

(define (line-backward p)
	(if (> (position-row p) (position-row (get-prot)))
		(make-position (- (position-row p) 1) (position-col p))
		p))

(define (cell-forward p)
	(if (< (position-col p) (max-columns))
		(make-position (position-row p) (+ (position-col p) 1))
		p))

(define (cell-backward p)
	(if (> (position-col p) 1)
		(make-position (position-row p) (- (position-col p) 1))
		p))

;; And now: commands written in Scheme
(define (backward-cell)
	(set-point (cell-backward (get-point))))

(define (forward-cell)
	(set-point (cell-forward (get-point))))


(define (set-mark-command)
	(let ()
		(set-mark (get-point))
		(llpr "Mark set")))

(define (exchange-point-and-mark)
	(let ((p (get-point)))
		(set-point (get-mark))
		(set-mark p)))

(define (beginning-of-buffer)
	(set-mark (get-point))
	(set-point (find-beginning-of-buffer)))

(define (end-of-buffer)
	(set-mark (get-point))
	(set-point (find-end-of-buffer)))

(define (top-of-buffer)
	(set-point-row 1))

(define (bottom-of-buffer)
	(set-point-row (line-last-used)))

(define (beginning-of-line)
	(set-point-col 1))

(define (end-of-line)
	(set-point-col (col-last-used (get-point-row))))

(define (next-line)
	(set-point (line-forward (get-point))))

(define (previous-line)
	(set-point (line-backward (get-point))))

(setq BUFFER-ROWS 100000)
(setq BUFFER-COLS 100000)

(define (select-all)
	(let ((p (get-point)))
		(set-mark (make-position 1 1))
		(set-point (make-position BUFFER-ROWS BUFFER-COLS))
		(set-block)
		(set-point p)))

;;
;; A few shorthand functions for my convenience
;;

; These two are *variables* now, not functions
;(define (R) (row))
;(define (C) (col))
(define (@ r c) (get-cell r c))
(define (@$ r c) (get-string r c))
(define +$ string-append)
(define (?$ t s) (string-search t s))
(define (up$ s) (string-upcase s))
(define (down$ s) (string-downcase s))
(define (href x) (@ R (+ C x)))
(define (vref x) (@ (+ R x) C))
(define (lastyear) (href -12))
(define (lastmonth) (href -1))

;;
;; Some more commands
;;

(define (execute-extended-command)
  (execute-interpreter-command 'SIOD))

(define (new-siag)
  (spawn "siag"))

(define (do-help helpfile)
  (spawn (string-append SIAGHELP " file:"
	 SIAGDOCS "/" helpfile)))

(define (help-contents)
  (do-help "siag/docs/siag.html"))

(define (help-copyright)
  (do-help "siag/docs/COPYING"))

(define (help-for-help)
  (do-help "common/docs/siaghelp.html"))

(define (help-search)
  (do-help "common/docs/search.html"))

(define (keyboard-quit)
  (llpr "Quit"))

(define (no-op)
  (llpr "This command does nothing"))

(define (get-cell-coords top point)
  (let ((top-row (position-row top))
	(top-col (position-col top))
	(cell-row (position-row point))
	(cell-col (position-col point))
	(cell-y 0)
	(cell-x 0))
    (while (< cell-row top-row)
      (set! cell-row (+ cell-row 1))
      (set! cell-y (- cell-y (get-cell-height cell-row))))
    (while (> cell-row top-row)
      (set! cell-row (- cell-row 1))
      (set! cell-y (+ cell-y (get-cell-height cell-row))))
    (while (< cell-col top-col)
      (set! cell-col (+ cell-col 1))
      (set! cell-x (- cell-x (get-cell-width cell-col))))
    (while (> cell-col top-col)
      (set! cell-col (- cell-col 1))
      (set! cell-x (+ cell-x (get-cell-width cell-col))))
    (list cell-x cell-y)))

; move point inside visible area
(define (fix-point width height)
  (let ((cur-x (car (get-cell-coords (get-top) (get-point))))
	(cur-y (cadr (get-cell-coords (get-top) (get-point))))
	(row (position-row (get-point)))
	(col (position-col (get-point))))
    (while (< cur-y 0)
      (set! cur-y (+ cur-y (get-cell-height row)))
      (set! row (+ row 1)))
    (while (> (+ cur-y (get-cell-height row)) height)
      (set! row (- row 1))
      (set! cur-y (- cur-y (get-cell-height row))))
    (while (< cur-x 0)
      (set! cur-x (+ cur-x (get-cell-width col)))
      (set! col (+ col 1)))
    (while (> (+ cur-x (get-cell-width col)) width)
      (set! col (- col 1))
      (set! cur-x (- cur-x (get-cell-width col))))
    (set-point (make-position row col))))

(define (scroll-down)
  (let ((width (caddr (get-geometry)))
	(height (cadddr (get-geometry)))
	(top (get-top))
	(new-y 0)
	(new-row 0))
    (set! new-y (- 20 height))
    (set! new-row (position-row top))
    (while (< new-y 20)
      (set! new-y (+ new-y 20))
      (set! new-row (- new-row 1)))
    (if (< new-row 1)
      (set! new-row 1))
    (set-top (make-position new-row (position-col top)))
    (fix-point width height)
    (set-pr-scr)))

(define (scroll-cell-down)
  (let ((width (caddr (get-geometry)))
	(height (cadr (cddr (get-geometry))))
	(top (get-top))
	(new-y 0)
	(new-row 0))
    (set! new-y -20)
    (set! new-row (position-row top))
    (while (< new-y 0)
      (set! new-y (+ new-y 20))
      (set! new-row (- new-row 1)))
    (if (< new-row 1)
      (set! new-row 1))
    (set-top (make-position new-row (position-col top)))
    (fix-point width height)
    (set-pr-scr)))

(define (scroll-up)
  (let ((width (caddr (get-geometry)))
	(height (cadr (cddr (get-geometry))))
	(top (get-top))
	(new-y 0)
	(new-row 0))
    (set! new-y (- height 20))
    (set! new-row (position-row top))
    (while (> new-y 0)
      (set! new-y (- new-y 20))
      (set! new-row (+ new-row 1)))
    (if (> new-row BUFFER-ROWS)
      (set! new-row BUFFER-ROWS))
    (set-top (make-position new-row (position-col top)))
    (fix-point width height)
    (set-pr-scr)))

(define (scroll-cell-up)
  (let ((width (caddr (get-geometry)))
	(height (cadr (cddr (get-geometry))))
	(top (get-top))
	(new-y 0)
	(new-row 0))
    (set! new-y 20)
    (set! new-row (position-row top))
    (while (> new-y 0)
      (set! new-y (- new-y 20))
      (set! new-row (+ new-row 1)))
    (if (> new-row BUFFER-ROWS)
      (set! new-row BUFFER-ROWS))
    (set-top (make-position new-row (position-col top)))
    (fix-point width height)
    (set-pr-scr)))

(define (scroll-left)
  (let ((width (caddr (get-geometry)))
	(height (cadr (cddr (get-geometry))))
	(top (get-top))
	(new-x 0)
	(new-col 0))
    (set! new-x (- 80 width))
    (set! new-col (position-col top))
    (while (< new-x 0)
      (set! new-x (+ new-x 80))
      (set! new-col (- new-col 1)))
    (if (< new-col 1)
      (set! new-col 1))
    (set-top (make-position (position-row top) new-col))
    (fix-point width height)
    (set-pr-scr)))

(define (scroll-cell-left)
  (let ((width (caddr (get-geometry)))
	(height (cadr (cddr (get-geometry))))
	(top (get-top))
	(new-x 0)
	(new-col 0))
    (set! new-x -80)
    (set! new-col (position-col top))
    (while (< new-x 0)
      (set! new-x (+ new-x 80))
      (set! new-col (- new-col 1)))
    (if (< new-col 1)
      (set! new-col 1))
    (set-top (make-position (position-row top) new-col))
    (fix-point width height)
    (set-pr-scr)))

(define (scroll-right)
  (let ((width (caddr (get-geometry)))
	(height (cadr (cddr (get-geometry))))
	(top (get-top))
	(new-x 0)
	(new-col 0))
    (set! new-x (- width 80))
    (set! new-col (position-col top))
    (while (> new-x 0)
      (set! new-x (- new-x 80))
      (set! new-col (+ new-col 1)))
    (if (> new-col BUFFER-COLS)
      (set! new-col BUFFER-COLS))
    (set-top (make-position (position-row top) new-col))
    (fix-point width height)
    (set-pr-scr)))

(define (scroll-cell-right)
  (let ((width (caddr (get-geometry)))
	(height (cadr (cddr (get-geometry))))
	(top (get-top))
	(new-x 0)
	(new-col 0))
    (set! new-x 80)
    (set! new-col (position-col top))
    (while (> new-x 0)
      (set! new-x (- new-x 80))
      (set! new-col (+ new-col 1)))
    (if (> new-col BUFFER-COLS)
      (set! new-col BUFFER-COLS))
    (set-top (make-position (position-row top) new-col))
    (fix-point width height)
    (set-pr-scr)))

(define data-type-list
  (list (cons EMPTY "EMPTY")
	(cons ERROR "ERROR")
	(cons LABEL "LABEL")
	(cons EXPRESSION "EXPRESSION")
	(cons STRING "STRING")
	(cons EMBED "EMBED")))

(define (type->string type)
  (or
    (cdr (assoc type data-type-list))
    "ERROR"))

(define (what-cursor-position)
  (llpr
    (string-append "["
      (number->string (row) 10) ","
      (number->string (col) 10) "] is "
      (type->string (get-type nil (row) (col))) ": "
      (get-text (row) (col)))))

(define (insert-line)
  (downshift-matrix nil (position-row (get-point)))
  (buffer-changed nil)
  (set-pr-scr))

(define (remove-line)
  (upshift-matrix nil (position-row (get-point)))
  (buffer-changed nil)
  (set-pr-scr))

(define (insert-col)
  (rightshift-matrix nil (position-col (get-point)))
  (buffer-changed nil)
  (set-pr-scr))

(define (remove-col)
  (leftshift-matrix nil (position-col (get-point)))
  (buffer-changed nil)
  (set-pr-scr))

(define (delete-block)
  (let ((blku (get-blku))
	(blkl (get-blkl))
	(i 0)
	(j 0))
    (undo-save (position-row blku)
	       (position-col blku)
	       (position-row blkl)
	       (position-col blkl))
    (set! i (position-row blku))
    (while (and (<= i (position-row blkl)) (<= i (line-last-used)))
      (set! j (position-col blku))
      (while (and (<= j (position-col blkl)) (<= j (col-last-used i)))
	(if (not (null? (set-data nil "" 0 EMPTY (make-position i j))))
	  (buffer-changed nil))
	(set! j (+ j 1)))
      (set! i (+ i 1)))
    (calc-matrix nil)
    (set-pr-scr)))

(define *cut-buffer* nil)

(define (copy-block)
  (set! *cut-buffer* (pack-area (position-row (get-blku))
				(position-col (get-blku))
				(position-row (get-blkl))
				(position-col (get-blkl))))
  (buffer-changed nil)
  (calc-matrix nil)
  (set-pr-scr))

(define (cut-block)
  (copy-block)
  (delete-block))

(define (paste-block)
  (unpack-area *cut-buffer* (get-point-row) (get-point-col))
  (buffer-changed nil)
  (calc-matrix nil)
  (set-pr-scr))

(define (recalc-matrix)
  (llpr "calculating...")
  (calc-matrix nil)
  (set-pr-scr))

(define (recenter)
  (set-pr-scr))

(define (delete-cell)
  (if (not (null? (set-data nil "" 0 EMPTY (get-point))))
    (buffer-changed nil))
  (set-pr-scr))

(define (edit-label . args)
  (let* ((row (position-row (get-point)))
	 (col (position-col (get-point)))
	 (b (get-text row col)))
    (if args
      (set! b (edit-cell "Label:" (car args)))
      (set! b (edit-cell "Label:" b)))
    (if (not (null? b))
      (if (not (null? (set-data nil b 0 LABEL (get-point))))
	(begin
	  (calc-matrix nil)
	  (set-pr-scr)
	  (buffer-changed nil))
	(llpr "String pool full")))))

; The following function always inserts as error. This gets resolved
; during the first recalculation.
; This function superseded by interpreter-test (useless name)
;(define (edit-expression)
;  (let ((row (position-row (get-point)))
;	(col (position-col (get-point))))
;    (set! b (ask-for-str "Expression:" (get-text row col)))
;    (if (not (null? b))
;      (if (not (null? (set-data nil b 0 ERROR (get-point))))
;	(begin
;	  (calc-matrix nil)
;	  (set-pr-scr)
;	  (buffer-changed nil))
;	(llpr "String pool full")))))


(define (r_sum . l)
  (if (null? l)
    0
    (if (eq (car l) 'RANGE)
      (+ (area-sum (cadr l)
		   (caddr l)
		   (cadddr l)
		   (cadddr (cdr l)))
	 (apply r_sum (cddddr (cdr l))))
      (+ (car l)
	 (apply r_sum (cdr l))))))

(define (r_max . l)
  (if (null? l)
    -2000000000
    (if (eq (car l) 'RANGE)
      (max (area-max (cadr l)
		     (caddr l)
		     (cadddr l)
		     (cadddr (cdr l)))
	   (apply r_max (cddddr (cdr l))))
      (max (if (number? (car l)) (car l) 0)
	   (apply r_max (cdr l))))))

(define (r_min . l)
  (if (null? l)
    2000000000
    (if (eq (car l) 'RANGE)
      (min (area-min (cadr l)
		     (caddr l)
		     (cadddr l)
		     (cadddr (cdr l)))
	   (apply r_min (cddddr (cdr l))))
      (min (car l)
	   (apply r_min (cdr l))))))


; this is cheating
(define (r_avg . l)
  (apply area-avg (cdr l)))

(define (area-sum r1 c1 r2 c2)
  (let ((sum 0)
	(r r1))
    (while (<= r r2)
      (let ((c c1)
	    (val nil))
	(while (<= c c2)
	  (set! val (get-cell r c))
	  (if (number? val)
	    (set! sum (+ sum val)))
	  (set! c (+ c 1))))
      (set! r (+ r 1)))
    sum))

(define (area-max r1 c1 r2 c2)
  (let ((mmax nil)
	(r r1))
    (while (<= r r2)
      (let ((c c1)
	    (val nil))
	(while (<= c c2)
	  (set! val (get-cell r c))
	  (if (number? val)
	    (if (or (null? mmax) (> val mmax))
	      (set! mmax val)))
	  (set! c (+ c 1))))
      (set! r (+ r 1)))
    mmax))

(define (area-min r1 c1 r2 c2)
  (let ((mmin nil)
	(r r1))
    (while (<= r r2)
      (let ((c c1)
	    (val nil))
	(while (<= c c2)
	  (set! val (get-cell r c))
	  (if (number? val)
	    (if (or (null? mmin) (< val mmin))
	      (set! mmin val)))
	  (set! c (+ c 1))))
      (set! r (+ r 1)))
    mmin))

(define (area-avg r1 c1 r2 c2)
  (let ((sum 0)
	(count 0)
	(r r1))
    (while (<= r r2)
      (let ((c c1)
	    (val nil))
	(while (<= c c2)
	  (set! val (get-cell r c))
	  (if (number? val)
	    (begin
	      (set! sum (+ sum val))
	      (set! count (+ count 1))))
	  (set! c (+ c 1))))
      (set! r (+ r 1)))
    (/ sum count)))

(define (a1-digit d)
  (substring " ABCDEFGHIJKLMNOPQRSTUVWXYZ" d (+ d 1)))

(define (a1-coord c1)
  (let ((b "") (digit 0))
    (while (> c1 0)
      (set! digit (fmod c1 26))
      (if (eq? digit 0)
	(set! digit 26))
      (set! b (string-append (a1-digit digit) b))
      (set! c1 (- c1 digit))
      (set! c1 (trunc (/ c1 26))))
    b))

(define (make-reference r1 c1)
  (if (eq? (a1-refs-get) 0)
    (string-append "R" (number->string r1) "C" (number->string c1))
    (string-append (a1-coord c1) (number->string r1))))

(define (r_fcn fcn)
  (let ((r1 (position-row (get-blku)))
	(c1 (position-col (get-blku)))
	(r2 (position-row (get-blkl)))
	(c2 (position-col (get-blkl)))
	(b nil))
    (set! b (string-append
	      "("
	      fcn
	      " "
	      (make-reference r1 c1)
	      ".."
	      (make-reference r2 c2)
	      ")"))
    (set-data nil b 0 ERROR (get-point))
    (calc-matrix nil)
    (set-pr-scr)
    (buffer-changed nil)))

(define (block-fcn fcn)
  (let ((r1 (position-row (get-blku)))
	(c1 (position-col (get-blku)))
	(r2 (position-row (get-blkl)))
	(c2 (position-col (get-blkl)))
	(b nil))
    (set! b (string-append
	      "("
	      fcn
	      " "
	      (number->string r1)
	      " "
	      (number->string c1)
	      " "
	      (number->string r2)
	      " "
	      (number->string c2)
	      ")"))
    (set-data nil b 0 ERROR (get-point))
    (calc-matrix nil)
    (set-pr-scr)
    (buffer-changed nil)))

(define (block-sum)
  (r_fcn "r_sum"))

(define (block-min)
  (r_fcn "r_min"))

(define (block-max)
  (r_fcn "r_max"))

(define (block-avg)
  (r_fcn "r_avg"))


; Load the plotting functions
; Don't load until actually needed
(define (plot . args)
  (require (string-append SIAGHOME "/siag/plot.scm"))
  (apply plot args))

(define (splot . args)
  (require (string-append SIAGHOME "/siag/splot.scm"))
  (apply splot args))

(define viewer-command "kghostview")
(define lpr-command "lpr")

(define (preview)
  (let ((fn (string-append "/tmp/siagprint" (number->string (getpid)) ".ps")))
    (if (savematrix fn nil (psformat))
      (spawn (string-append viewer-command " " fn))
      (llpr "Can't make postscript"))))

(define (print)
  (let ((fn (string-append "/tmp/siagprint" (number->string (getpid)) ".ps")))
    (if (savematrix fn nil (psformat))
      (spawn (string-append lpr-command " " fn))
      (llpr "Can't make postscript"))))

(define (print-format)
  (let* ((props (form-record "Previewer" "Print Command"))
	 (viewcmd (cdr (assoc "Previewer" props)))
	 (printcmd (cdr (assoc "Print Command" props))))
    (if (not (equal? viewcmd ""))
      (set! viewer-command viewcmd))
    (if (not (equal? printcmd ""))
      (set! lpr-command printcmd))))

; To consider: make sure that top is outside the protected area
; Also: change (set-point) et al to do this
(define (protect-cells)
  (set-prot (get-point))
  (set-pr-scr))

(define (remove-protection)
  (set-prot (make-position 1 1))
  (set-pr-scr))

(define BLACK 0)
(define RED 1)
(define GREEN 2)
(define BLUE 3)
(define YELLOW 4)
(define MAGENTA 5)
(define CYAN 6)
(define WHITE 7)

(define *colors*
  (list (cons "black" BLACK)
	(cons "red" RED)
	(cons "green" GREEN)
	(cons "blue" BLUE)
	(cons "yellow" YELLOW)
	(cons "magenta" MAGENTA)
	(cons "cyan" CYAN)
	(cons "white" WHITE)))

(define (string->color color)
  (or (cdr (assoc color *colors*)) BLACK))

(define (ask-for-color)
  (form-begin)
  (form-label "Color:")
  (form-menu "color")
  (form-properties XtNwidth 100)
  (let ((c *colors*))
    (while c
      (form-menuentry (caar c))
      (set! c (cdr c))))
  (form-newline)
  (form-okbutton "Ok")
  (form-property XtNwidth 80)
  (form-cancelbutton "Cancel")
  (form-property XtNwidth 80)
  (cdr (assoc "color" (form-end))))

(define (cell-color)
  (set-color nil
	     (get-point)
	     (bit-and 7 (string->color (ask-for-color)))))

(define (block-color)
  (let ((blku (get-blku nil))
	(blkl (get-blkl nil))
	(color (bit-and 7 (string->color (ask-for-color))))
	(row 0)
	(col 0))
    (set! row (position-row blku))
    (while (<= row (position-row blkl))
      (set! col (position-col blku))
      (while (<= col (position-col blkl))
	(set-color nil (make-position row col) color)
	(set! col (+ col 1)))
      (set! row (+ row 1)))))

; A hack to allow simple hyperlinking.
; Calls execute on the text in the next cell.
(define (hyperlink)
  (execute (get-text (row) (+ (col) 1))))

(define (search-forward)
  (require (string-append SIAGHOME "/siag/find.scm"))
  (search-forward))

(define (search-backward)
  (require (string-append SIAGHOME "/siag/find.scm"))
  (search-backward))

; (change-format fromr tor fromc toc format mask)
; format = new format 
; mask = which bits to change
(define (change-format fromr tor fromc toc format mask)
  (set! tor (min tor (line-last-used)))
  (while (<= fromr tor)
    (let ((c fromc)
	  (c2 (min toc (col-last-used fromr))))
      (while (<= c c2)
	(let ((oldfmt (get-format nil (make-position fromr c))))
	  (set-format nil (make-position fromr c)
			  (bit-or (bit-and format mask)
				  (bit-and oldfmt (bit-not mask)))))
	(set! c (+ c 1))))
    (set! fromr (+ fromr 1)))
  (set-pr-scr))

(define (in-block point blku blkl)
  (and (>= (position-row point) (position-row blku))
       (<= (position-row point) (position-row blkl))
       (>= (position-col point) (position-col blku))
       (<= (position-col point) (position-col blkl))))

(define FONT_MASK 96)
(define COURIER 0)
(define HELVETICA 32)
(define NEW_CENTURY 64)
(define TIMES 96)

(define SIZE_MASK 7)
(define SIZE_8 0)
(define SIZE_10 1)
(define SIZE_12 2)
(define SIZE_14 3)
(define SIZE_18 4)
(define SIZE_24 5)
(define SIZE_20 6)
(define SIZE_30 7)

(define FMT_SHIFT 16)
(define FMT_MASK (ash 15 FMT_SHIFT))
(define FMT_DEFAULT (ash 0 FMT_SHIFT))
(define FMT_INVISIBLE (ash 1 FMT_SHIFT))
(define FMT_INTEGER (ash 2 FMT_SHIFT))
(define FMT_SCIENTIFIC (ash 3 FMT_SHIFT))
(define FMT_FIXED (ash 4 FMT_SHIFT))
(define FMT_DATE (ash 5 FMT_SHIFT))
(define FMT_TIME (ash 6 FMT_SHIFT))
(define FMT_TIMEDIFF (ash 7 FMT_SHIFT))
(define FMT_PERCENT (ash 8 FMT_SHIFT))
(define FMT_HEX (ash 9 FMT_SHIFT))
(define FMT_CURRENCY (ash 10 FMT_SHIFT))
(define FMT_USER1 (ash 11 FMT_SHIFT))
(define FMT_USER2 (ash 12 FMT_SHIFT))
(define FMT_USER3 (ash 13 FMT_SHIFT))
(define FMT_USER4 (ash 14 FMT_SHIFT))
(define FMT_USER5 (ash 15 FMT_SHIFT))

(define COLOR_SHIFT 20)
(define COLOR_MASK (ash 7 COLOR_SHIFT))
(define COLOR_BLACK (ash 0 COLOR_SHIFT))
(define COLOR_RED (ash 1 COLOR_SHIFT))
(define COLOR_GREEN (ash 2 COLOR_SHIFT))
(define COLOR_BLUE (ash 3 COLOR_SHIFT))
(define COLOR_YELLOW (ash 4 COLOR_SHIFT))
(define COLOR_MAGENTA (ash 5 COLOR_SHIFT))
(define COLOR_CYAN (ash 6 COLOR_SHIFT))
(define COLOR_WHITE (ash 7 COLOR_SHIFT))

(define HADJ_MASK (bit-or 4096 8192))
(define HADJ_LEFT 0)
(define HADJ_CENTER 8192)
(define HADJ_RIGHT (bit-or 4096 8192))

(define ITALIC 8)
(define BOLD 16)

; Not needed, really
;(define (name->font name)
;  (cond ((equal? name "Times") TIMES)
;	((equal? name "Courier") COURIER)
;	((equal? name "Helvetica") HELVETICA)
;	((equal? name "New Century Schoolbook") NEW_CENTURY)
;	(t HELVETICA)))

(define (change-font font mask)
  (if (in-block (get-point) (get-blku) (get-blkl))
    (change-format (position-row (get-blku))
		   (position-row (get-blkl))
		   (position-col (get-blku))
		   (position-col (get-blkl))
		   font
		   mask)
    (change-format (position-row (get-point))
		   (position-row (get-point))
		   (position-col (get-point))
		   (position-col (get-point))
		   font
		   mask)))

(define (toggle-fontflag flag)
  (let ((oldfmt (get-format nil (get-point))))
    (if (= (bit-and oldfmt flag) flag)
      (change-font 0 flag)
      (change-font flag flag))))

(define (exec-siod)
  (execute-interpreter-command 'SIOD))

(define (exec-c)
  (execute-interpreter-command 'C))

(define (exec-guile)
  (execute-interpreter-command 'Guile))

(define (exec-tcl)
  (execute-interpreter-command 'Tcl))

(define (edit-siod)
  (interpreter-test 'SIOD))

(define (edit-guile)
  (interpreter-test 'Guile))

(define (edit-tcl)
  (interpreter-test 'Tcl))

(define (edit-c)
  (interpreter-test 'C))

; This is the interpreter that is bound to the '=' key.
;(define *standard-interpreter* 'SIOD)
(define *standard-interpreter* 'C)
(define *tooltip-mode* 2)

(define (edit-expression . args)
  (if args (interpreter-test *standard-interpreter* (car args))
    (interpreter-test *standard-interpreter* nil)))

(define (change-interpreter)
  (set! *standard-interpreter*
	(ask-for-str "New expression interpreter:" *standard-interpreter*)))

(define (add-property)
  (let ((property (form-record "Key" "Value")))
    (put-property nil
		  (cdr (assoc "Key" property))
		  (cdr (assoc "Value" property)))))

; Load 1-2-3 compatibility functions
(require (string-append SIAGHOME "/siag/123.scm"))

; These are temporarily moved to .siagrc
;(plugin-register "Oclock" "oclock" "/usr/local/siag/plugins/dummy")
;(plugin-register "Xeyes" "xeyes" "/usr/local/siag/plugins/xeyes")
;(plugin-register "Xload" "xload" "/usr/local/siag/plugins/xload")

(require (string-append SIAGHOME "/plugins/plugin.scm"))

(define (enter-date)
  (let* ((s (ask-for-str "Date:" ""))
	 (t (strptime s "%x"))
	 (b (number->string (mktime t) 10)))
    (set-data nil b 0 ERROR (get-point))
    (set-format nil (get-point) (+ FMT_DATE HELVETICA SIZE_10))
    (calc-matrix nil)
    (set-pr-scr)
    (buffer-changed nil)))

(define (enter-time)
  (let* ((s (ask-for-str "Time:" ""))
	 (t (strptime s "%X"))
	 (ti (append (list (assq 'sec t) (assq 'min t) (assq 'hour t))
		     (cdddr (localtime 0))))
	 (b (number->string (mktime ti) 10)))
    (set-data nil b 0 ERROR (get-point))
    (set-format nil (get-point) (+ FMT_TIME HELVETICA SIZE_10))
    (calc-matrix nil)
    (set-pr-scr)
    (buffer-changed nil)))

(a1-refs-set 1)
(require (string-append SIAGHOME "/siag/sort.scm"))

