// -*- C++ -*-
/* This file is part of
 * ======================================================
 * 
 *           LyX, the High Level Word Processor
 * 	 
 *	    Copyright (C) 1995 Matthias Ettrich
 *
 *======================================================*/
#ifndef _LYXFONT_H
#define _LYXFONT_H

#include "LString.h"
#include <X11/Xlib.h>
#include <stdio.h>

class LyXLex;

///
class LyXFont {
public:
	/** The value INHERIT_* means that the font attribute is
	  inherited from the layout. In the case of layout fonts, the
	  attribute is inherited from the default font.
	  The value IGNORE_* is used with LyXFont::update() when the
	  attribute should not be changed.
	  */
	enum FONT_FAMILY {
		///
		ROMAN_FAMILY, // fontstruct rely on this to be 0
		///
		SANS_FAMILY,
		///
		TYPEWRITER_FAMILY,
		///
		SYMBOL_FAMILY,
		///
		INHERIT_FAMILY,
		///
		IGNORE_FAMILY
	};

	///
	enum FONT_SERIES {
		///
		MEDIUM_SERIES, // fontstruct rely on this to be 0
		///
		BOLD_SERIES,
		///
		INHERIT_SERIES,
		///
		IGNORE_SERIES
	};

	///
	enum FONT_SHAPE {
		///
		UP_SHAPE, // fontstruct rely on this to be 0
		///
		ITALIC_SHAPE,
		///
		SLANTED_SHAPE,
		///
		SMALLCAPS_SHAPE,
		///
		INHERIT_SHAPE,
		///
		IGNORE_SHAPE
	};

	///
	enum FONT_SIZE {
		///
		SIZE_TINY, // fontstruct rely on this to be 0
		///
		SIZE_SCRIPT,
		///
		SIZE_FOOTNOTE,
		///
		SIZE_SMALL,
		///
		SIZE_NORMAL,
		///
		SIZE_LARGE,
		///
		SIZE_LARGER,
		///
		SIZE_LARGEST,
		///
		SIZE_HUGE,
		///
		SIZE_HUGER,
		///
		INCREASE_SIZE,
		///
		DECREASE_SIZE,
		///
		INHERIT_SIZE,
		///
		IGNORE_SIZE
	};
 
	/// Used for emph, underbar, noun and latex toggles
	enum FONT_MISC_STATE {
		///
		OFF,
		///
		ON,
// 		///
// 		TOGGLE,
// 		///
		INHERIT,
		///
		IGNORE
	};
 
	///
	enum FONT_COLOR {
		///
		NONE,
		///
		BLACK,
		///
		WHITE,
		///
		RED,
		///
		GREEN,
		///
		BLUE,
		///
		CYAN,
		///
		MAGENTA,
		///
		YELLOW,
		///
		MATH,
		///
		INSET,
		///
		INHERIT_COLOR,
		///
		IGNORE_COLOR
	};

	/// Trick to overload constructor and make it megafast
	enum FONT_INIT1 {
		///
		ALL_INHERIT
	};
	///
	enum FONT_INIT2 {
		///
		ALL_IGNORE
	};
	///
	enum FONT_INIT3 {
		///
		ALL_SANE
	};

	///
	LyXFont();

	/// LyXFont x(LyXFont ...) and LyXFont x = LyXFont ...
	LyXFont(LyXFont const & x);
 
	/// Shortcut initialization
	LyXFont(LyXFont::FONT_INIT1);
	/// Shortcut initialization
	LyXFont(LyXFont::FONT_INIT2);
	/// Shortcut initialization
	LyXFont(LyXFont::FONT_INIT3);

	/// LyXFont x,y; x=y;
	LyXFont& operator=(LyXFont const & x);
 
	/// Decreases font size by one
	LyXFont & decSize();
 
	/// Increases font size by one
	LyXFont & incSize();
 
	///
	FONT_FAMILY family() const;
 
	///
	FONT_SERIES series() const;
 
	///
	FONT_SHAPE shape() const;
 
	///
	FONT_SIZE size() const;
 
	///
	FONT_MISC_STATE emph() const;
 
	///
	FONT_MISC_STATE underbar() const;
 
	///
	FONT_MISC_STATE noun() const;

	///
	FONT_MISC_STATE latex() const;
 
	///
	FONT_COLOR color() const;
 
	///
	LyXFont & setFamily(LyXFont::FONT_FAMILY f);
	///
	LyXFont & setSeries(LyXFont::FONT_SERIES s);
	///
	LyXFont & setShape(LyXFont::FONT_SHAPE s);
	///
	LyXFont & setSize(LyXFont::FONT_SIZE s);
	///
	LyXFont & setEmph(LyXFont::FONT_MISC_STATE e);
	///
	LyXFont & setUnderbar(LyXFont::FONT_MISC_STATE u);
	///
	LyXFont & setNoun(LyXFont::FONT_MISC_STATE n);
	///
	LyXFont & setLatex(LyXFont::FONT_MISC_STATE l);
	///
	LyXFont & setColor(LyXFont::FONT_COLOR c);
 
	/// Set family after LyX text format
	LyXFont & setLyXFamily(LString const &);
 
	/// Set series after LyX text format
	LyXFont & setLyXSeries(LString const &);
 
	/// Set shape after LyX text format
	LyXFont & setLyXShape(LString const &);
 
	/// Set size after LyX text format
	LyXFont & setLyXSize(LString const &);
 
	/// Returns misc flag after LyX text format
	LyXFont::FONT_MISC_STATE setLyXMisc(LString const &);

	/// Sets color after LyX text format
	LyXFont & setLyXColor(LString const &);
 
	/// Sets size after GUI name
	LyXFont & setGUISize(LString const &);
 
	/// Returns size of font in LaTeX text notation
	LString latexSize() const;
 
	/** Updates font settings according to request. If an
	  attribute is IGNORE, the attribute is left as it is. */
	void update(LyXFont const & newfont);

	/** Toggle font settings. If an attribute is IGNORE, the 
	    attribute is left as it is. */
	void toggle();

	/** returns true if the argument equals the font
	  in all defined attributes*/
	bool containsPart(LyXFont const & otherfont);

 
	/** Reduce font to fall back to template where possible.
	  Equal fields are reduced to INHERIT */
	void reduce(LyXFont const & tmplt);
 
	/// Realize font from a template (INHERIT are realized)
	LyXFont & realize(LyXFont const & tmplt);

	/// Is a given font fully resolved?
	bool resolved() const;
 
	/// Read a font specification from LyXLex. Used for layout files.
	LyXFont & lyxRead(LyXLex&);
 
	/// Writes the changes from this font to orgfont in .lyx format in file
	void lyxWriteChanges(LyXFont const & orgfont, FILE *) const;
 
	/** Writes the head of the LaTeX needed to change to this font.
	  Writes to file, the head of the LaTeX needed to change to this font.
	    Returns number of chars written. Base is the font state active now.
	*/
	int latexWriteStartChanges(FILE *, LyXFont const & base) const;

	/** Writes to LString, the head of the LaTeX needed to change
	  to this font. Returns number of chars written. Base is the
	  font state active now.
	*/
	int latexWriteStartChanges(LString &, LyXFont const & base) const;

	/** Writes the tail of the LaTeX needd to change to this font.
	  Returns number of chars written. Base is the font state we want
	  to achieve.
	*/
	int latexWriteEndChanges(FILE *, LyXFont const & base) const;

	/** Writes tha tail of the LaTeX needed to chagne to this font.
	  Returns number of chars written. Base is the font state we want
	    to achieve.
	*/
	int latexWriteEndChanges(LString &, LyXFont const & base) const;
 
	/// Build GUI description of font state
	LString stateText() const;

	///
	int maxAscent() const; 

	///
	int maxDescent() const;

	///
	int ascent(char c) const;

	///
	int descent(char c) const;

	///
	int width(char c) const;

	///
	int textWidth(char const *s, int n) const;

	///
	int stringWidth(LString const & s) const;

	///
	int signedStringWidth(LString const & s) const;

	/// Draws text and returns width of text
	int drawText(char const*, int n, Pixmap, int baseline, int x) const;

	///
	int drawString(LString const &, Pixmap pm, int baseline, int x) const;

	///
	GC getGC() const;

	///
	friend inline
	bool operator==(LyXFont const & font1, LyXFont const & font2) {
		return font1.bits == font2.bits;
	}

	///
	friend inline
	bool operator!=(LyXFont const & font1, LyXFont const & font2) {
		return font1.bits != font2.bits;
	}
private:
	/// This have to be at least 32 bits, but 64 or more does not hurt
	typedef unsigned int ui32;

	/** Representation: bit table
	  Layout of bit table:
	       	   11 1111 111 122 222 222 2233
	  Bit 012 34 567 8901 2345 678 901 234 567 8901
	      FFF SS SSS SSSS CCCC EEE UUU NNN LLL
	      aaa ee hhh iiii oooo mmm nnn ooo aaa
	      mmm rr aaa zzzz llll ppp ddd uuu ttt

	  Some might think this is a dirty representation, but it gives
	  us at least 25% speed-up, so why not?
	*/
	ui32 bits;

	///
	enum FONT_POSITION {
		///
		Fam_Pos =  0,
		///
		Ser_Pos =  3,
		///
		Sha_Pos =  5,
		///
		Siz_Pos =  8,
		///
		Col_Pos = 12,
		///
		Emp_Pos = 16,
		///
		Und_Pos = 19,
		///
		Nou_Pos = 22,
		///
		Lat_Pos = 25
	};

	///
	enum FONT_MASK {
		///
		Fam_Mask = 0x07,
		///
		Ser_Mask = 0x03,
		///
		Sha_Mask = 0x07,
		///
		Siz_Mask = 0x0f,
		///
		Col_Mask = 0x0f,
		///
		Misc_Mask = 0x07
	};
 
	/// Sane font
	enum {	sane = ui32(ROMAN_FAMILY) << Fam_Pos
		| ui32(MEDIUM_SERIES) << Ser_Pos
		| ui32(UP_SHAPE) << Sha_Pos
		| ui32(SIZE_NORMAL) << Siz_Pos
		| ui32(NONE) << Col_Pos
		| ui32(OFF) << Emp_Pos
		| ui32(OFF) << Und_Pos
		| ui32(OFF) << Nou_Pos
		| ui32(OFF) << Lat_Pos};
 
	/// All inherit font
	enum{ inherit = ui32(INHERIT_FAMILY) << Fam_Pos
		| ui32(INHERIT_SERIES) << Ser_Pos
		| ui32(INHERIT_SHAPE) << Sha_Pos
		| ui32(INHERIT_SIZE) << Siz_Pos
		| ui32(INHERIT_COLOR) << Col_Pos
		| ui32(INHERIT) << Emp_Pos
		| ui32(INHERIT) << Und_Pos
		| ui32(INHERIT) << Nou_Pos
		| ui32(INHERIT) << Lat_Pos};
 
	/// All ignore font
	enum{ ignore = ui32(IGNORE_FAMILY) << Fam_Pos
	       | ui32(IGNORE_SERIES) << Ser_Pos
	       | ui32(IGNORE_SHAPE) << Sha_Pos
	       | ui32(IGNORE_SIZE) << Siz_Pos
	       | ui32(IGNORE_COLOR) << Col_Pos
	       | ui32(IGNORE) << Emp_Pos
	       | ui32(IGNORE) << Und_Pos
	       | ui32(IGNORE) << Nou_Pos
	       | ui32(IGNORE) << Lat_Pos};
 
	/// Updates a misc setting according to request
	LyXFont::FONT_MISC_STATE setMisc(LyXFont::FONT_MISC_STATE newfont,
					 LyXFont::FONT_MISC_STATE org);

	/// Converts logical attributes to concrete shape attribute
	LyXFont::FONT_SHAPE realShape() const;

	///
	XFontStruct* getXFontstruct() const;

};


inline LyXFont::LyXFont()
{
	bits = sane;
}


inline LyXFont::LyXFont(LyXFont const & x)
{
	bits = x.bits;
}


inline LyXFont::LyXFont(LyXFont::FONT_INIT1)
{
	bits = inherit;
}


inline LyXFont::LyXFont(LyXFont::FONT_INIT2)
{
	bits = ignore;
}


inline LyXFont::LyXFont(LyXFont::FONT_INIT3)
{
	bits = sane;
}


inline LyXFont & LyXFont::operator=(LyXFont const & x) 
{
	bits = x.bits;
	return *this;
}


 // You don't have to understand the stuff below :-)
 // It works, and it's bloody fast. (Asger)
inline LyXFont::FONT_FAMILY LyXFont::family() const 
{
	return LyXFont::FONT_FAMILY((bits >> Fam_Pos) & Fam_Mask);
}


inline LyXFont::FONT_SERIES LyXFont::series() const
{
	return LyXFont::FONT_SERIES((bits >> Ser_Pos) & Ser_Mask);
}


inline LyXFont::FONT_SHAPE LyXFont::shape() const
{
	return LyXFont::FONT_SHAPE((bits >> Sha_Pos) & Sha_Mask);
}


inline LyXFont::FONT_SIZE LyXFont::size() const
{
	return LyXFont::FONT_SIZE((bits >> Siz_Pos) & Siz_Mask);
}


inline LyXFont::FONT_MISC_STATE LyXFont::emph() const
{
	return LyXFont::FONT_MISC_STATE((bits >> Emp_Pos) & Misc_Mask);
}


inline LyXFont::FONT_MISC_STATE LyXFont::underbar() const
{
	return LyXFont::FONT_MISC_STATE((bits >> Und_Pos) & Misc_Mask);
}


inline LyXFont::FONT_MISC_STATE LyXFont::noun() const
{
	return LyXFont::FONT_MISC_STATE((bits >> Nou_Pos) & Misc_Mask);
}


inline LyXFont::FONT_MISC_STATE LyXFont::latex() const 
{
	return LyXFont::FONT_MISC_STATE((bits >> Lat_Pos) & Misc_Mask);
}


inline LyXFont::FONT_COLOR LyXFont::color() const 
{
	return LyXFont::FONT_COLOR((bits >> Col_Pos) & Col_Mask);
}


inline LyXFont & LyXFont::setFamily(LyXFont::FONT_FAMILY f)
{
	bits &= ~(Fam_Mask << Fam_Pos);
	bits |= ui32(f) << Fam_Pos;
	return *this;
}


inline LyXFont & LyXFont::setSeries(LyXFont::FONT_SERIES s)
{
	bits &= ~(Ser_Mask << Ser_Pos);
	bits |= ui32(s) << Ser_Pos;
	return *this;
}


inline LyXFont & LyXFont::setShape(LyXFont::FONT_SHAPE s)
{
	bits &= ~(Sha_Mask << Sha_Pos);
	bits |= ui32(s) << Sha_Pos;
	return *this;
}


inline LyXFont & LyXFont::setSize(LyXFont::FONT_SIZE s)
{
	bits &= ~(Siz_Mask << Siz_Pos);
	bits |= ui32(s) << Siz_Pos;
	return *this;
}


inline LyXFont & LyXFont::setEmph(LyXFont::FONT_MISC_STATE e)
{
	bits &= ~(Misc_Mask << Emp_Pos);
	bits |= ui32(e) << Emp_Pos;
	return *this;
}


inline LyXFont & LyXFont::setUnderbar(LyXFont::FONT_MISC_STATE u)
{
	bits &= ~(Misc_Mask << Und_Pos);
	bits |= ui32(u) << Und_Pos;
	return *this;
}


inline LyXFont & LyXFont::setNoun(LyXFont::FONT_MISC_STATE n)
{
	bits &= ~(Misc_Mask << Nou_Pos);
	bits |= ui32(n) << Nou_Pos;
	return *this;
}

inline LyXFont & LyXFont::setLatex(LyXFont::FONT_MISC_STATE l)
{
	bits &= ~(Misc_Mask << Lat_Pos);
	bits |= ui32(l) << Lat_Pos;
	return *this;
}


inline LyXFont & LyXFont::setColor(LyXFont::FONT_COLOR c)
{
	bits &= ~(Col_Mask << Col_Pos);
	bits |= ui32(c) << Col_Pos;
	return *this;
}
#endif
