// -*- C++ -*-
/* This file is part of
 * ======================================================
 * 
 *           LyX, the High Level Word Processor
 * 	 
 *	    Copyright (C) 1997 Asger Alstrup
 *           and the LyX Team.
 *
 *======================================================*/

#include <config.h>

#include <kapp.h>
#include <klocale.h>

#include <math.h>	// fabs()
#include <stdlib.h>	// atoi()

#include "FontLoader.h"
#include "FontInfo.h"
#include "error.h"
#include "lyxrc.h"	// lyxrc.font_*
extern LyXRC * lyxrc;
#include "minibuffer.h"
#include "BufferView.h"

extern BufferView *current_view;


// Initialize font loader
FontLoader::FontLoader()
{
	reset();
}

// Destroy font loader
FontLoader::~FontLoader()
{
	unload();
}

// Update fonts after zoom, dpi, font names, or norm change
// For now, we just ditch all fonts we have. Later, we should
// reuse the ones that are already loaded.
// fixed. Matthias
void FontLoader::update(unsigned int _zoom)
{
  zoom = _zoom;
  XFontStruct* fontstruct2[4][2][4][10];
  for (int i1=0; i1<4; i1++)
    for (int i2=0; i2<2; i2++)
      for (int i3=0; i3<4; i3++){
	if (fontinfo[i1][i2][i3]) {
	  delete fontinfo[i1][i2][i3];
	  fontinfo[i1][i2][i3] = 0;
	}
	for (int i4=0; i4<10; i4++) 
	  fontstruct2[i1][i2][i3][i4] = fontstruct[i1][i2][i3][i4];
      }
  for (int i1=0; i1<4; i1++)
    for (int i2=0; i2<2; i2++)
      for (int i3=0; i3<4; i3++) 
	for (int i4=0; i4<10; i4++) {
	  if (fontstruct[i1][i2][i3][i4]) {
	    doLoad((LyXFont::FONT_FAMILY)i1, (LyXFont::FONT_SERIES)i2, 
		   (LyXFont::FONT_SHAPE)i3, (LyXFont::FONT_SIZE)i4);
	    XFreeFont(qt_display, fontstruct2[i1][i2][i3][i4]);
	  }
	}
}

unsigned int FontLoader::getZoom(){
  if (zoom == 0){
    KConfigGroupSaver saver( kapp->getConfig(), "ScreenOptions" );
    zoom =  kapp->getConfig()->readNumEntry( "FontZoom", 150 );
  }
  return zoom; 
}

// Reset font loader
void FontLoader::reset()
{
	// Clear font infos, font structs and font metrics
	for (int i1=0; i1<4; i1++)
		for (int i2=0; i2<2; i2++)
			for (int i3=0; i3<4; i3++) {
				fontinfo[i1][i2][i3] = 0;
				for (int i4=0; i4<10; i4++) {
					fontstruct[i1][i2][i3][i4] = 0;
				}
			}
	zoom = 0;
}

// Unload all fonts
void FontLoader::unload() 
{
	// Unload all fonts
	for (int i1=0; i1<4; i1++)
		for (int i2=0; i2<2; i2++)
			for (int i3=0; i3<4; i3++) {
				if (fontinfo[i1][i2][i3]) {
					delete fontinfo[i1][i2][i3];
					fontinfo[i1][i2][i3] = 0;
				}
				for (int i4=0; i4<10; i4++) {
					if (fontstruct[i1][i2][i3][i4]) {
						XFreeFont(qt_display, fontstruct[i1][i2][i3][i4]);
						fontstruct[i1][i2][i3][i4] = 0;
					}
				}
			}
}

// Get font info
/* Takes care of finding which font that can match the given request. Tries
different alternatives. */
void FontLoader::getFontinfo(LyXFont::FONT_FAMILY family, 
			     LyXFont::FONT_SERIES series, 
			     LyXFont::FONT_SHAPE shape)
{
  // Do we have the font info already?
  if (fontinfo[family][series][shape] != 0)
	return;
  
  // Special code for the symbol family
  if (family == LyXFont::SYMBOL_FAMILY){
	fontinfo[family][series][shape] = new FontInfo("-*-symbol-*");
	return;
  }
  
  KConfig* config = kapp->getConfig();
  QString oldgroup = config->group();
  config->setGroup( "ScreenOptions" );

	// Normal font. Let's search for an existing name that matches.
	LString ffamily;
	LString fseries;
	LString fshape;
	LString norm = config->readEntry( "FontNorm", DEFCHSET ).data();
	LString fontname;

	FontInfo * fi = new FontInfo();
	fontinfo[family][series][shape] = fi;

	for (int cfam=0; cfam < 2; cfam++) {
		// Determine family name
		switch (family) {
		case LyXFont::ROMAN_FAMILY:
			switch (cfam) {
			case 0: ffamily = config->readEntry( "RomanFontName", "-*-utopia" ); break;
			case 1: ffamily = "-*-times";
			default: cfam = 100;
			}
			break;
		case LyXFont::SANS_FAMILY:
			switch (cfam) {
			case 0: ffamily = config->readEntry( "SansFontName","-*-helvetica" ); break;
			case 1: ffamily = "-*-helvetica";
			default: cfam = 100;
			}
			break;
		case LyXFont::TYPEWRITER_FAMILY:
			switch (cfam) {
			case 0: ffamily = config->readEntry( "TypewriterFontName","-*-courier" ); break;
			case 1: ffamily = "-*-courier";
			default: cfam = 100;
			}
			break;
		default: ;
		}

		for (int cser=0; cser < 4; cser++) {
			// Determine series name
			switch (series) {
			case LyXFont::MEDIUM_SERIES:
				switch (cser) {
				case 0: fseries = "-medium"; break;
				case 1: fseries = "-book"; break;
				case 2: fseries = "-light";
				default: cser = 100;
				}
				break;
			case LyXFont::BOLD_SERIES:
				switch (cser) {
				case 0: fseries = "-bold"; break;
				case 1: fseries = "-black"; break;
				case 2: fseries = "-demi"; break;
				case 3: fseries = "-demibold";
				default: cser = 100;
				}
				break;
			default: ;
			}

			for (int csha=0; csha < 2; csha++) {
				// Determine shape name
				switch (shape) {
				case LyXFont::UP_SHAPE:
				case LyXFont::SMALLCAPS_SHAPE:
					switch (csha) {
					case 0: fshape = "-r";
					default: csha = 100;
					}
					break;
				case LyXFont::ITALIC_SHAPE:
					switch (csha) {
					case 0: fshape = "-i"; break;
					case 1: fshape = "-o";
					default: csha = 100;
					}
					break;
				case LyXFont::SLANTED_SHAPE:
					switch (csha) {
					case 0: fshape = "-o"; break;
					case 1: fshape = "-i";
					default: csha = 100;
					}
					break;
				default: ;
				}
				//
				fontname = ffamily + fseries + fshape +
					   "-normal-*-*-*-*-*-*-*-" + norm;
				fi->setPattern(fontname);
				if (fi->exist()) {
					return;
				}
			}
		}
	}

	config->setGroup( oldgroup );
}

/// Do load font
XFontStruct* FontLoader::doLoad(LyXFont::FONT_FAMILY family, 
				LyXFont::FONT_SERIES series, 
				LyXFont::FONT_SHAPE shape, 
				LyXFont::FONT_SIZE size)
{
  KConfigGroupSaver saver( kapp->getConfig(), "ScreenOptions" );
  
	getFontinfo(family,series,shape);
	int fsize = (int) ( (lyxrc->font_sizes[size] * lyxrc->dpi * 
			     (getZoom()/100.0) ) / 72.27 + 0.5 );

	LString font = fontinfo[family][series][shape]->getFontname(fsize);

	if (font.empty()) {
		lyxerr.print("No font matches request. Using 'fixed'.");
		font = "fixed";
	}

	current_view->getMiniBuffer()->Store();
	current_view->getMiniBuffer()->Set(i18n("Loading font into X-Server..."));

	XFontStruct* fs = XLoadQueryFont(qt_display, font.c_str());

	if (fs == 0) {
		if (font=="fixed") {
			lyxerr.print("We're doomed. Can't get 'fixed' font.");
		} else {
			lyxerr.print("Could not get font. Using 'fixed'.");
			fs = XLoadQueryFont(qt_display, "fixed");
		}
	} else {
		if (lyxerr.debugging(Error::FONT)) {
			// Tell user the font matching
			LyXFont f;
			f.setFamily(family);
			f.setSeries(series);
			f.setShape(shape);
			f.setSize(size);
			// The rest of the attributes are not interesting
			f.setEmph(LyXFont::INHERIT);
			f.setUnderbar(LyXFont::INHERIT);
			f.setNoun(LyXFont::INHERIT);
			f.setLatex(LyXFont::INHERIT);
			f.setColor(LyXFont::INHERIT_COLOR);
			lyxerr.print(LString("Font '") + f.stateText() 
				      + "' matched by\n" + font + '\n');
		}
	}
	current_view->getMiniBuffer()->Reset();

	fontstruct[family][series][shape][size] = fs;
	return fs;
}
