/*
 Copyright (C) 1998, Mark W J Redding <mark@grawlfang.demon.co.uk>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <getopt.h>

#include "pppsvr.h"

static void Count(int);
static void Status(int);
static void LinkUp(char);
static void LinkDown();
static void KeepAlive(int);
static void Terminate(int);
static void Death(int);
static void Proxy(char);

static void usage(char *);

static void ProcessProxyCommand(int);

static int references;	/* count of "attaches" */
static int keepalive;	/* count of "keep alive" requests */
static int ppppid;	/* pid of pppd daemon */
static int pippid;	/* pid of proxy command daemon */
static int running;	/* status of pppd */
static int restart;	/* flag to indicate a restart is required */

int main(int argc, char *argv[])
{
int soc,msgsock,rval,run,sts,loop,hostPort=0;
struct sockaddr_in server;
char cmd[2],rts[5];
int c,opt_ind;
int proxypipes[2];
static struct option long_opts[] = {{"port",0,0,'p'},{"usage",0,0,'h'}};

	do
	{
		c=getopt_long(argc,argv,"p:h",long_opts,&opt_ind);
		switch(c)
		{
		case 'p' : /* server port address */
			hostPort=atoi(optarg);
			break;
		case 'h' : /* display help */
			usage(argv[0]);
			break;
		}
	}
	while(c != -1);

	if(pipe(proxypipes) < 0)
	{
		perror("opening pipe");
		exit(1);
	}

	if(!(pippid=fork()))
	{
		close(proxypipes[1]);
		ProcessProxyCommand(proxypipes[0]);
		exit(0);
	}
	else
		close(proxypipes[0]);

	if(!hostPort) hostPort=PPPSVR_PORT;

	if((soc=socket(AF_INET,SOCK_STREAM,0)) < 0)
	{
		perror("opening socket");
		exit(1);
	}

	memset(&server,0,sizeof(server));
	server.sin_family=AF_INET;
	server.sin_addr.s_addr=INADDR_ANY;
	server.sin_port=htons(hostPort);

	for(loop=0;loop<PPPSVR_RETRY;loop++)
	{
		if(!bind(soc,(struct sockaddr *)&server,sizeof(server))) break;
		sleep(PPPSVR_SLEEP);
	}
	if(loop==PPPSVR_RETRY)
	{
		perror("binding socket");
		exit(2);
	}

	listen(soc,PPPSVR_QUEUE);

	if(fork()) exit(0);

	for(run=1;run;)
	{
		if(restart && !running) LinkUp('R');
		if((msgsock=accept(soc,0,0)) == -1)
		{
			if(errno != ENOENT && errno != EINTR)
			{
				perror("accept");
				exit(2);
			}
			close(msgsock);
			continue;
		}
		memset(cmd,0,sizeof(cmd));
		rval=read(msgsock,cmd,sizeof(cmd)-1);
		if(rval < 0)
		{
			perror("reading stream");
			continue;
		}

		sts=1;
		switch(cmd[0])
		{
		case 'R' :
		case 'U' : /* open connection */
			LinkUp(cmd[0]);
			break;
		case 'D' : /* close connection */
			LinkDown();
			break;
		case 'C' : /* count of connections */
			Count(msgsock);
			break;
		case 'T' : /* terminate */
			Terminate(proxypipes[1]);
			run=0;	
			break;
		case 'S' : /* report status of link */
			Status(msgsock);
			break;
		case 'K' : /* Keep alive */
		case 'Z' : /* cancel keep alive */
			KeepAlive(cmd[0]=='K');
			break;
		case 'M' : /* Issue command to send mail */
		case 'G' :
		case 'O' :
		case 'L' :
		case 'F' : /* Issue commands to proxy server */
			write(proxypipes[1],cmd,1);
			break;
		default  : /* unknown message */
			sts=0;
		}

		write(msgsock,(sts) ? ((run) ? "OKAY" : "TERM") : "BAD ",4);
		close(msgsock);
	}

	close(soc);
	close(proxypipes[1]);

	waitpid(pippid,&sts,0);

	exit(0);
}


static void Count(int soc)
{
char count[5];

	memset(count,0,sizeof(count));
	sprintf(count,"%4d",references);
	write(soc,count,4);
}


static void Status(int soc)
{
	write(soc,(running) ? "UP  " : "DOWN",4);
}


static void LinkUp(char mode)
{
	restart=0;
	if(mode == 'R' && !references) return;

	if(!running)
	{
		signal(SIGCHLD,Death);
		ppppid=fork();
		if(ppppid == -1)
		{
			perror("starting pppd");
			exit(2);
		}
		if(!ppppid) execl(PPPSVR_PROCESS,PPPSVR_PROCESS,PPPSVR_PARAMS,0);
		running=1;
	}
	if(mode != 'R') references++;
}


static void LinkDown()
{
int sts;

	if(references)
	{
		references--;
		if(!references)
		{
			signal(SIGCHLD,SIG_DFL);
			kill(ppppid,SIGINT);
			waitpid(ppppid,&sts,0);
			running=0;
		}
		if(keepalive > references) keepalive=references;
	}
}


static void Terminate(int pip)
{
int sts=0;

	if(running)
	{
		signal(SIGCHLD,SIG_DFL);
		kill(ppppid,SIGINT);
		waitpid(ppppid,&sts,0);
	}
	running=0;
	write(pip,"T",1);
}


static void Death(int snum)
{
int sts;

	waitpid(ppppid,&sts,0);
	running=0;
	if(keepalive) restart=1;
}


static void usage(char *prog)
{
	printf("%s [-p port]\n",prog);
}


static void KeepAlive(int state)
{
	keepalive += (state) ? 1 : -1;
	if(keepalive<0) keepalive=0;
}


static void Proxy(char cmd)
{
char syscmd[128];
int pid,sts;

	memset(syscmd,0,sizeof(syscmd));

	if(cmd!='M')
	{

		strncpy(syscmd,PPPSVR_PROXY,sizeof(syscmd)-1);

		switch(cmd)
		{
		case 'O' : /* on-line */
			strcat(syscmd," -online");
			break;
		case 'L' : /* off-line */
			strcat(syscmd," -offline");
			break;
		case 'F' : /* fetch */
			strcat(syscmd," -fetch");
			break;
		case 'G' : /* purge */
			strcat(syscmd," -purge");
			break;
		default  : /* bad parameter */
			return;
		}
	}
	else
		strncpy(syscmd,PPPSVR_MAIL,sizeof(syscmd)-1);

	if(pid=fork())
	{
		waitpid(pid,&sts,0);
		return;
	}
	system(syscmd);
	exit(0);
}


static void ProcessProxyCommand(int pip)
{
char cmd[2];
int rval;

	for(;;)
	{
		memset(cmd,0,sizeof(cmd));
		rval=read(pip,cmd,sizeof(cmd)-1);
		if(cmd[0]=='T') break;
		Proxy(cmd[0]);
	}
}

