// -*- c++ -*-
// **************************************************************
// $Source: /home/proj/mmm/cvsroot/mmm/modules/MAdsr.cc,v $
// $Revision: 1.1 $
// $Date: 1999/05/13 08:02:16 $
// $State: Exp $
// **************************************************************

#define MODULE_NAME "adsr";

#include "ModuleMacros.h"

BEGIN_MODULE_DEFINITION(Adsr);
    Slot *nslot_attack;
    Slot *nslot_decay;
    Slot *nslot_sustain;
    Slot *nslot_release;
    Slot *nslot_cutoff;
END_MODULE_DEFINITION(Adsr);

class SSAdsrOutput : public Signal
{
public:
    SSAdsrOutput(Module *m) : Signal(SOUND_TYPE, "output", "ADSR curve", m) {};
    PreparedSignal *prepareSignal(Metainfo *, const Parameterset *);
};


class PSSAdsrOutput : public PreparedSoundSignal
{
    PreparedNumberSignal *pnsig_attack;
    PreparedNumberSignal *pnsig_decay;
    PreparedNumberSignal *pnsig_sustain;
    PreparedNumberSignal *pnsig_release;
    PreparedNumberSignal *pnsig_cutoff;
    Number attack, decay, sustain, release, cutoff;
    Number sampling_interval;
    Number duration;
    Number attack_time;
    long   cut_to;
public:
    PSSAdsrOutput(PreparedNumberSignal *pnsig_attack,
		  PreparedNumberSignal *pnsig_decay,
		  PreparedNumberSignal *pnsig_sustain,
		  PreparedNumberSignal *pnsig_release,
		  PreparedNumberSignal *pnsig_cutoff,
		  Number sampling_interval, Number duration);
    ~PSSAdsrOutput() { delete pnsig_attack; delete pnsig_decay; delete pnsig_sustain; 
                       delete pnsig_release; delete pnsig_cutoff; };
    long getCutFrom() const { return (long)(-attack_time / sampling_interval); };
    /**
     * Returns the index of the last sample that must be calculated.
     * Returns a negative value, if there is no upper limit.
     */
    long getCutTo() const;
protected:
    void computeSamples(Number *, long start_time, long nsamples);
private:
    void computeConstants();
    Number levelAt(long time) const { return levelAt(time * sampling_interval); };
    Number levelAt(Number realtime) const;
};


// ---------------------------------------------------------------------------
//                             Implementation
// ---------------------------------------------------------------------------

#include <math.h>
#include "NullSound.h"

MAdsr::MAdsr(string)
{
    addConnector(nslot_attack  = new Slot(NUMBER_TYPE, "attack",   "Attack value 0 <= a < 1",    this, 10)); 
    addConnector(nslot_decay   = new Slot(NUMBER_TYPE, "decay",    "Decay factor 0 <= d <= 1",   this, 11));
    addConnector(nslot_sustain = new Slot(NUMBER_TYPE, "sustain",  "Sustain level 0 <= s <= 1",  this, 12));
    addConnector(nslot_release = new Slot(NUMBER_TYPE, "release",  "Release factor 0 <= r <= 1", this, 13));
    addConnector(nslot_cutoff  = new Slot(NUMBER_TYPE, "cutoff",   "Cutoff level 0 <= c <= 1",   this, 14));
    addConnector(new SSAdsrOutput(this)); 
}


PreparedSignal *SSAdsrOutput::prepareSignal(Metainfo *mi, const Parameterset *parset)
{
    Metainfo asked_for = *mi;
    mi->clear();

    Seconds sampling_interval = getSamplingInterval(getModule(), mi, parset);
    
    // I'm interested in the parameter duration. If it is not provided, I will use
    // a default duration of 1 sec.

    Number duration = parset->containsDurationFactor() ? parset->getDurationFactor() : 1.0;
    if (asked_for.containsDuration()) mi->setDuration(duration);

    PSSAdsrOutput *pss =  new PSSAdsrOutput(
	getPreparedNumberSignal(this, ((MAdsr *)getModule())->nslot_attack,  0.0),
	getPreparedNumberSignal(this, ((MAdsr *)getModule())->nslot_decay,   0.5),
	getPreparedNumberSignal(this, ((MAdsr *)getModule())->nslot_sustain, 0.5),
	getPreparedNumberSignal(this, ((MAdsr *)getModule())->nslot_release, 0.5),
	getPreparedNumberSignal(this, ((MAdsr *)getModule())->nslot_cutoff,  0.01),
	sampling_interval,
	duration);

    // To compute CutFrom and CutTo, PSSAdsrOutput will have to ask
    // the signals. It is doing this in its constructor.

    if (asked_for.containsCutFrom()) mi->setCutFrom(pss->getCutFrom());
    if (asked_for.containsCutTo())   {
	long cut_to = pss->getCutTo();
	if (cut_to >= 0) mi->setCutTo(pss->getCutTo());
	else mi->clearCutTo();
    }

    return pss;
}


PSSAdsrOutput::PSSAdsrOutput(PreparedNumberSignal *pnsig_attack,
			     PreparedNumberSignal *pnsig_decay,
			     PreparedNumberSignal *pnsig_sustain,
			     PreparedNumberSignal *pnsig_release,
			     PreparedNumberSignal *pnsig_cutoff,
			     Number sampling_interval, Number duration) :
    pnsig_attack(pnsig_attack),
    pnsig_decay(pnsig_decay),
    pnsig_sustain(pnsig_sustain),
    pnsig_release(pnsig_release),
    pnsig_cutoff(pnsig_cutoff),
    sampling_interval(sampling_interval), 
    duration(duration) 
{
    computeConstants();
}


void PSSAdsrOutput::computeConstants()
{
    // Get current values from NumberSlots.
    // TODO: handle range-errors instead of suppressing them.
    attack  = max(0.0, min(1.0, pnsig_attack->getNumber()));
    decay   = max(0.0, min(1.0, pnsig_decay->getNumber()));
    sustain = max(0.0, min(1.0, pnsig_sustain->getNumber()));
    release = max(0.0, min(1.0, pnsig_release->getNumber()));
    cutoff  = max(0.0, min(1.0, pnsig_cutoff->getNumber()));

    if (attack == 1) attack_time = MINNUMBER;
    else attack_time = 1.0 / (1.0 - attack) - 1;
}


#include <stdio.h>
long PSSAdsrOutput::getCutTo() const
{
    if (cutoff <= 0) return -1;
    else {
	Number level_at_duration = levelAt(duration);
	if (level_at_duration <= cutoff) return (long) (duration / sampling_interval);
	Number cut_to = (0.5 + ((log(cutoff / level_at_duration) / log(release) )
			       + duration) / sampling_interval);
	if (cut_to <= (Number)MAXLONG) return (long)cut_to;
	else return -1;
    }
}

Number PSSAdsrOutput::levelAt(Number realtime) const
{
    if (realtime <= -attack_time) return 0.0;
    else if (realtime <= 0)       return 1.0 + (realtime / attack_time);
    else if (realtime <= duration) return sustain + (1.0 - sustain) * pow(decay, realtime);
    else return levelAt(duration) * pow(release, realtime - duration);
}


void PSSAdsrOutput::computeSamples(Number *output,
				   long start_time, long number_of_samples)
{
    computeConstants(); // actualize input values.

    if (start_time * sampling_interval < -attack_time) { // pre-attack phase	
	long samples_to_write =
	    min(number_of_samples, long(0.5 - attack_time / sampling_interval) - start_time);
	start_time        += samples_to_write;
	number_of_samples -= samples_to_write;
	while (samples_to_write--) *output++ = 0.0;
    }
    
    if (number_of_samples && start_time < 0) { // attack-phase
	Number level = levelAt(start_time - 1);
	const Number increment = sampling_interval/ attack_time;
	long samples_to_write = min(number_of_samples, - start_time);
	start_time        += samples_to_write;
	number_of_samples -= samples_to_write;
	while (samples_to_write--) *output++ = (level += increment);
    }

    if (number_of_samples && start_time * sampling_interval < duration) // decay-phase
    {
	Number slevel = levelAt(start_time) - sustain;
	const Number factor = pow(decay, sampling_interval);
	long samples_to_write = 
	    min(number_of_samples, long(duration / sampling_interval) - start_time);
	start_time        += samples_to_write;
	number_of_samples -= samples_to_write;
	while (samples_to_write--) {
	    slevel *= factor;
	    *output++ = sustain + slevel;
	}
    }

    if (number_of_samples) // Rest is release-phase
    {
	Number level = levelAt(start_time - 1);
	const Number factor = pow(release, sampling_interval);
	while (number_of_samples--) *output++ = (level *= factor);
    }
}
