	/*

	Copyright (C) 1998 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include "autorouter.h"
#include <assert.h>
#include <stdlib.h>
#include <stdio.h>

AutoRouter::AutoRouter(int width, int height)
{
	int i;
	const PRSIZE = 16;

//	field = new Field[width][height];	  anybody knows how to do this correct?

	field = (Field **)malloc(sizeof(Field *)*width);

	for(i=0;i<width;i++)
		field[i] = (Field *)malloc(sizeof(Field)*height);

	for(i=0;i<255;i++) directionmask[i] = 0xffff;

	directionmask['d'] = ~(left | right);
	directionmask['u'] = ~(left | right);
	directionmask['l'] = ~(up | down);
	directionmask['r'] = ~(up | down);

	ownerindex['d'] = OWNER_UD;
	ownerindex['u'] = OWNER_UD;
	ownerindex['l'] = OWNER_LR;
	ownerindex['r'] = OWNER_LR;

	prstart = (long *)malloc((PRSIZE + 1) * sizeof(long));
	prp = prstart;

	int remaining[4];	
	for(i=0;i<PRSIZE;i++)
	{
		if((i & 3) == 0)
		{
			int j;
			for (j=0;j<4;j++) remaining[j] = j+1;
			printf("\n");
		}

		int rnd;
		do
		{
			rnd = random()&3;
		} while(!remaining[rnd]);

		*prp++ = remaining[rnd];
		printf("%d ",remaining[rnd]);
		remaining[rnd]=0;
	}
	*prp = 0;
	printf("\n");
	
	this->width = width;
	this->height = height;

	newowner = 1;
	clear();
}

void AutoRouter::clear()
{
	int x,y;

	for (x=0;x<width;x++)
		for (y=0;y<height;y++)
		{
			field[x][y].data = none;
			field[x][y].penalty = 0;
			field[x][y].owner[0] = -1;
			field[x][y].owner[1] = -1;
		}

	//newowner = 1;
}

long AutoRouter::get(int x,int y)
{
	assert(x >= 0 && x < width);
	assert(y >= 0 && y < height);

	return(field[x][y].data);
}

void AutoRouter::getowners(int x, int y, long& ud_owner, long& lr_owner)
{
	assert(x >= 0 && x < width);
	assert(y >= 0 && y < height);

	ud_owner = field[x][y].owner[OWNER_UD];
	lr_owner = field[x][y].owner[OWNER_LR];
}

void AutoRouter::set(int x,int y, long lt)
{
	assert(x >= 0 && x < width);
	assert(y >= 0 && y < height);

	if(lt & solid)
	{
		if((y-1) >= 0)
			field[x][y-1].penalty += 5;

		if((y-2) >= 0)
			field[x][y-2].penalty += 2;

		if((y+1) < height)
			field[x][y+1].penalty += 5;

		if((y+2) < height)
			field[x][y+2].penalty += 2;
	}
	field[x][y].data = lt;
	field[x][y].owner[0] = 0;
	field[x][y].owner[1] = 0;		// don't pass
}


int gms, gmsd;

long AutoRouter::connect(int x1, int y1, int x2, int y2, long owner)
{
	int x,y;
	long dat_source, dat_dest;

	if(owner == 0) owner = newowner++;
	currentowner = owner;

	// clear data(source) & data(dest) first and restore later, since they
	// might be solid

	printf("-field[x1][y1].owner[0..1] = %ld,%ld\n",field[x1][y1].owner[0],
													field[x1][y1].owner[1]);
	printf("-field[x2][y2].owner[0..1] = %ld,%ld\n",field[x2][y2].owner[0],
													field[x2][y2].owner[1]);

	dat_source = field[x1][y1].data; field[x1][y1].data = none;
	dat_dest   = field[x2][y2].data; field[x2][y2].data = none;

	for (x=0;x<width;x++)
		for (y=0;y<height;y++)
			field[x][y].minscore = 20000;

	printf("autorouter: trying to connect %d,%d with %d,%d (owner %d)\n",
								x1,y1,x2,y2,owner);

	QString history("");

	gmsd = 0; gms=20000;

	prp = prstart;
	q_try_connect(x1,y1,x2,y2,0,0,history);

	while(trylist.begin() != trylist.end())
	{
		list<tryinfo>::iterator ti,minti;
		int mintry=20000;

		minti = trylist.begin();
		ti = trylist.begin();
		while(ti != trylist.end())
		{
			if((*ti).score < mintry)
			{
				mintry = (*ti).score;
				minti = ti;
			}
			ti++;
		}

		try_connect((*minti).x1, (*minti).y1,
					(*minti).x2, (*minti).y2, (*minti).score, (*minti).depth, (*minti).history);
		trylist.erase(minti);
	}
	//printf("gmsd= %d\n",gmsd);

	field[x1][y1].data = dat_source;
	field[x2][y2].data = dat_dest;

	if(gms != 20000)
	{
		//printf("minhistory for this connection is %s\n",minhistory.data());
		//printf("minscore for that was %d\n",gms);

		char *walk = minhistory.data();

		int x = x1;
		int y = y1;

		while(*walk)
		{
			field[x][y].owner[ownerindex[*walk]] = currentowner;
			switch(*walk)
			{
				case 'd':	field[x][y++].data |= down;
							field[x][y].data |= up;
						break;
				case 'u':	field[x][y--].data |= up;
							field[x][y].data |= down;
						break;
				case 'l':	field[x--][y].data |= left;
							field[x][y].data |= right;
						break;
				case 'r':	field[x++][y].data |= right;
							field[x][y].data |= left;
						break;
			}
			field[x][y].owner[ownerindex[*walk]] = currentowner;
			walk++;
		}
	}
	else
	{
		printf("!! sorry, this connection is impossible !!\n");
	}
	return(owner);
}

void AutoRouter::q_try_connect(int x1, int y1, int x2, int y2, int score, int depth, QString history)
{
	tryinfo newtry;

	newtry.x1 = x1;
	newtry.x2 = x2;
	newtry.y1 = y1;
	newtry.y2 = y2;
	newtry.score = score;
	newtry.depth = depth;
	newtry.history = history;

	trylist.push_back(newtry);
}

void AutoRouter::try_connect(int x1, int y1, int x2, int y2, int score, int depth, QString history)
{
	char *walk = history.data();

// check if we can go here:

	if(x1 < 0 || x1 >= width) return;
	if(y1 < 0 || y1 >= width) return;

	if(depth > 0)
	{
		if(field[x1][y1].data != 0) score += 100;
		// going over a field where already connections are is bad...

		if(directionmask[walk[depth-1]] & field[x1][y1].data)
		{
			// someone already uses that field... we can continue
			// only if the connection has the same sourceport

			long fieldowner = field[x1][y1].owner[ownerindex[walk[depth-1]]];

			if(fieldowner != -1)		// used?
			{
				if(fieldowner != currentowner) return;
				score -= 100;
				// ... or a good idea, if the connections are from the
				// same owner
			}
		}
	}

	//score++;
	score += abs(x1-x2) + abs(y1-y2);
	score += field[x1][y1].penalty;

	if(depth > 2)
		if(walk[depth-2] != walk[depth-1]) score += 100;
	
	if(field[x1][y1].minscore <= score) return;
	if(score > gms) { gmsd++; return; }

	field[x1][y1].minscore = score;

// check if we are where we wanted to be:

	if(x1 == x2 && y1 == y2) {
		// success

		if(score < gms)
		{
			// best solution until now
			minhistory = history;
			gms = score;
		}
		return;
	}

// search next place to go; take some pseudo random direction order
// this method improves search speed ;)

	depth++;

	int i;
	for(i=0;i<4;i++)
	{
		switch(*prp++)
		{
			case 0: i--;
					prp = prstart;
				break;
			case 1: q_try_connect(x1-1,y1,x2,y2,score,depth,history+"l");
				break;
			case 2: q_try_connect(x1+1,y1,x2,y2,score,depth,history+"r");
				break;
			case 3: q_try_connect(x1,y1-1,x2,y2,score,depth,history+"u");
				break;
			case 4: q_try_connect(x1,y1+1,x2,y2,score,depth,history+"d");
				break;
		}
	}
}
