/***************************************************************************
                          ksoundwidget.cpp  -  description
                             -------------------
    begin                : Sun Dec 26 1999
    copyright            : (C) 1999 by Tobias Wollgam
    email                : Tobias.Wollgam@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include "ksoundwidget.h"

#include <stdlib.h>
#include <math.h>

#include <qcursor.h>
#include <qpaintdevice.h>

KSoundWidget::KSoundWidget(QWidget *parent) :
	KDNDWidget(parent)
{
	m_sample = new CSample();
	m_deletesample = true;
	
	setup();
}

KSoundWidget::KSoundWidget(QWidget *parent,int c,uint mode,int size = 1) :
	KDNDWidget(parent)
{
	m_sample = new CSample(c,mode,size);
	m_deletesample = true;
	
	setup();
}

KSoundWidget::KSoundWidget(QWidget *parent,CSample *s) :
	KDNDWidget(parent)
{
	m_sample = s;
	m_deletesample = false;

	setup();
}

KSoundWidget::KSoundWidget(QWidget *parent,CSample &s) :
	KDNDWidget(parent)
{
	m_sample = new CSample(s);
	m_deletesample = true;

	setup();
}

KSoundWidget::~KSoundWidget()
{
	painter->end();
	
	delete dnddz;
	if(m_sample && m_deletesample)
		delete m_sample;
}

void	KSoundWidget::droped(KDNDDropZone *dz)
{
	const char	*d;
	char		*dd;
	int		s,t,x,y,pos;
	QPoint		g,l;

	d = dz->getData();
	s = dz->getDataSize();
	t = dz->getDataType();
	x = dz->getMouseX();
	y = dz->getMouseY();

	g.setX(x);
	g.setY(y);

	l = mapFromGlobal(g);

	x = l.x();
	y = l.y();

	//printf("droped: size=%i type=%i pos=%i\n",s,t,x);

	switch(t)
	{
		case DndRawData:
			printf("Got RawData\n");
		break;
		case DndURL:
			dd = strdup(d);
			pos = MAX(0,MIN((int)m_sample->getLength(),screen2sample(x)));
			printf("\nGot URL=%s  pos=%i\n\n",dd,pos);
			emit gotDndURL(dd,pos);
			free(dd);
		break;
	}
}

#include "zoom.xbm"
#include "zoom_mask.xbm"

void	KSoundWidget::setup()
{
	setMouseTracking(true);

	isdrag = false;
	startdrag = false;

	dnddz = new KDNDDropZone(this,DndRawData);

	connect(dnddz,SIGNAL(dropAction(KDNDDropZone*)),this,SLOT(droped(KDNDDropZone*)));


	setMinimumSize(48,48);

	m_disp_start = 0;
	m_disp_zoom = 0;
	m_disp_mode = 0;

	m_sel_start = -1;
	m_sel_end = -1;
	m_cursor_mode = KSoundWidget::Select;
	m_zsel_start = -1;
	m_zsel_end = -1;

	m_pointer = 0;

	scroll = 0;


	QBitmap	br(zoom_width,zoom_height,(const uchar*)zoom_bits,true);
	QBitmap	mr(zoom_mask_width,zoom_mask_height,(const uchar*)zoom_mask_bits,true);
	
	zoom_cursor = QCursor(br,mr,3,7);

	m_disp_zoom = maxZoom();
	//printf("Zoom: %i %i\n",len,disp_zoom);

	painter = new QPainter(this);

	setMode(KSoundWidget::Select);
	
	setDefaultColors();
}

void	KSoundWidget::getColors(QColor *color)
{
	color[0] = bgcolor;
	color[1] = fgcolor;
	color[2] = nocolor;
	color[3] = midlinecolor;
	color[4] = devidecolor;
	color[5] = sbgcolor;
	color[6] = sfgcolor;
	color[7] = smidlinecolor;
	color[8] = pointercolor;
	color[9] = spointercolor;
	color[10] = zoomcolor;
}

void	KSoundWidget::setColors(QColor *color)
{
	bgcolor = color[0];
	fgcolor = color[1];
	nocolor = color[2];
	midlinecolor = color[3];
	devidecolor = color[4];
	sbgcolor = color[5];
	sfgcolor = color[6];
	smidlinecolor = color[7];
	pointercolor = color[8];
	spointercolor = color[9];
	zoomcolor = color[10];

	layout();
}

void	KSoundWidget::setDefaultColors()
{
	nocolor = QColor(128,128,160);
	devidecolor = QColor(128,128,128);
	
	bgcolor = QColor(255,255,255);
	fgcolor = QColor(0,0,0);
	midlinecolor = QColor(255,0,0);
	
	sbgcolor = QColor(0,0,0);
	sfgcolor = QColor(255,255,255);
	smidlinecolor = QColor(0,255,0);
	
	pointercolor = QColor(0,0,128);
	spointercolor = QColor(255,255,0);
	zoomcolor = QColor(128,0,128);

	layout();
}

int	KSoundWidget::sample2screen(int samplepos)
{
	if(samplepos < 0 || samplepos > (int)m_sample->getLength())
		return -1;
		
	return (samplepos - m_disp_start) / (1 << m_disp_zoom);
}

int	KSoundWidget::screen2sample(int screenpos)
{
	return m_disp_start + screenpos * (1 << m_disp_zoom);
}

void	KSoundWidget::paintEvent(QPaintEvent *pe)
{
	if(pe == 0) return;
	//draw(pe->rect().left() - 1,pe->rect().right() + 1);
	draw();
}

void	KSoundWidget::resizeEvent(QResizeEvent *re)
{
	if(re == 0) return;
	//printf("resize\n");
	if(m_disp_zoom > maxZoom())
	{
		//printf("Zoom: %i\n",maxZoom());
		setDisplayZoom(maxZoom());
	}
	emit changedDisplayZoom(getDisplayZoom());
}

void	KSoundWidget::mousePressEvent(QMouseEvent *me)
{
	int		sd,ed,ss,se;
	
	sel_on = false;
	
	printf("Pressed %i %i\n",me->state(),me->button());
	if(me->button() == RightButton)
	{
		printf("drag\n");
		//isdrag = true;
		//startdrag = true;
		return;
	}
	if(me->button() != LeftButton)
		return;

  ss = sample2screen(m_sel_start);
  se = sample2screen(m_sel_end);
  sd = labs(me->x() - ss);
  ed = labs(me->x() - se);

  if(sd < 3 && ed < 3)
  {
  	if(sd > ed)
  	{
//	  	printf("Move from End of Selection\n");
			mouse_left = ss;
			mouse_right = me->x();
			first_mouse_pos = ss;
			last_mouse_pos = me->x();
  	}
  	else
  	{
//	  	printf("Move from Start of Selection\n");
			mouse_left = me->x();
 			mouse_right = se;
			first_mouse_pos = se;
			last_mouse_pos = se;
  	}
  }
  else if(ed < 3)
  {
//  	printf("Move from End of Selection\n");
		mouse_left = ss;
		mouse_right = me->x();
		first_mouse_pos = ss;
		last_mouse_pos = ss;
  }
  else if(sd < 3)
  {
//  	printf("Move from Start of Selection\n");
 		mouse_left = me->x();
 		mouse_right = se;
		first_mouse_pos = se;
		last_mouse_pos = se;
	}
  else
  {
		mouse_right =
			mouse_left =
			first_mouse_pos =
			last_mouse_pos = me->x();
		if(m_cursor_mode == KSoundWidget::Select)
			draw(sample2screen(m_sel_start) - 1,sample2screen(m_sel_end) + 1,0);
  }

	sel_on = true;

	if(m_cursor_mode == KSoundWidget::Select)
	{
//		draw(sample2screen(sel_start) - 1,sample2screen(sel_end) + 1,0);
		if(screen2sample(first_mouse_pos) > (int)m_sample->getLength())
			first_mouse_pos = last_mouse_pos = mouse_left = mouse_right = sample2screen(m_sample->getLength());
		if(screen2sample(first_mouse_pos) < 0)
			first_mouse_pos = last_mouse_pos = mouse_left = mouse_right = sample2screen(0);
	}
	else if(m_cursor_mode == KSoundWidget::Zoom)
	{
	}
}

void	KSoundWidget::mouseReleaseEvent(QMouseEvent *me)
{
	if((me->state() & RightButton) == RightButton)
	{
		startdrag = false;
		isdrag = true;
		return;
	}

	sel_on = false;
	scroll = 0;
	//printf("scroll end\n");
	killTimers();
	if(m_cursor_mode == KSoundWidget::Select)
	{
		//printf("%i %i\n",sel_start,sel_end);
		if(first_mouse_pos == me->x())
		{
			m_sel_start = m_sel_end = -1;
			m_pointer = screen2sample(first_mouse_pos);
		}
		else
		{
			if(me->x() < 0)
				m_sel_start = 0;
			if(m_sel_start < 0)
				m_sel_start = 0;
			if(me->x() > sample2screen(m_sample->getLength()))
				m_sel_end = m_sample->getLength();
			if(m_sel_end > (int)m_sample->getLength())
				m_sel_end = m_sample->getLength();
		}
		//printf("%i %i\n",sel_start,sel_end);
		draw();
	}
	else if(m_cursor_mode == KSoundWidget::Zoom)
	{
	}
//	draw();
}

#include "dnd.xpm"

void	KSoundWidget::drop(QMouseEvent *_mouse)
{
	QPixmap		dndpm(dnd_xpm);
	char			*s;
	int				size,typesize,t;
	QPoint		p = mapToGlobal( _mouse->pos() );
	int				dx = - dndpm.width() / 2;
	int				dy = - dndpm.height() / 2;

	printf("drag\n");

	typesize = sizeof(float);

	size = sizeof(int) * (1 + 1);
	size += m_sample->getNChannel() * m_sample->getLength() * typesize;

	if((s = (char*)malloc(size)) == 0) return;

	*((int*)s + 0) = (int)m_sample->getNChannel();
	*((int*)s + 1) = (int)m_sample->getLength();

	for(t = 0;t < m_sample->getNChannel();t++)
	{
		m_sample->get(t,0,m_sample->getLength(),(float*)((int*)s + 2) + (t * m_sample->getLength()));
	}
	startDrag(new KDNDIcon(dndpm,p.x() + dx,p.y() + dy),s,8,DndRawData,dx,dy);
}

void	KSoundWidget::dragEndEvent(void)
{
	isdrag = false;
	startdrag = false;
}

void	KSoundWidget::dndMouseMoveEvent(QMouseEvent *me)
{
	int		new_mouse_pos;

//	printf("drag %i startdrag %i \n",drag,startdrag);
	if(drag)
		return;
		
	if(startdrag)
	{
		drop(me);

		return;
	}

	if(!sel_on)
	{
		int				sd,ed;
		QCursor		c,&cr = c;

  	if(m_sel_start < 0 || m_sel_end < 0)
  		return;
  		
  	sd = labs(me->x() - sample2screen(m_sel_start));
		ed = labs(me->x() - sample2screen(m_sel_end));

  	if(sd < 3 && ed < 3)
  	{
  		if(sd > ed)
  		{
				c.setShape(SizeHorCursor);
				setCursor(cr);
  		}
  		else
  		{
				c.setShape(SizeHorCursor);
				setCursor(cr);
  		}
		}
		else if(ed < 3)
		{
			c.setShape(SizeHorCursor);
			setCursor(cr);
		}
		else if(sd < 3)
		{
			c.setShape(SizeHorCursor);
			setCursor(cr);
  	}
  	else
  	{
			c.setShape(IbeamCursor);
			setCursor(cr);
		}
		
		return;
	}
	
	new_mouse_pos = me->x();

	if(new_mouse_pos < 0)
	{
		if(screen2sample(new_mouse_pos) > 0)
		{
			if(!scroll) startTimer(150);
			scroll = new_mouse_pos;
			return;
		}
		if(scroll) killTimers();
		scroll = 0;
	}
	else if(new_mouse_pos >= width())
	{
		if(screen2sample(new_mouse_pos) < (int)m_sample->getLength())
		{
			if(!scroll) startTimer(150);
			scroll = new_mouse_pos - width();
			return;
		}
		if(scroll) killTimers();
		scroll = 0;
	}
	else
	{
		if(scroll) killTimers();
		scroll = 0;
	}

	if(new_mouse_pos < sample2screen(0))
		new_mouse_pos = sample2screen(0);
	if(new_mouse_pos > sample2screen(m_sample->getLength()))
		new_mouse_pos = sample2screen(m_sample->getLength());

	if(new_mouse_pos > first_mouse_pos)
	{
		mouse_left = first_mouse_pos;
		mouse_right = new_mouse_pos;
	}
	else
	{
		mouse_left = new_mouse_pos;
		mouse_right = first_mouse_pos;
	}

	if(m_cursor_mode == KSoundWidget::Select)
	{
		// if(m_sel_start >= screen2sample(0) || m_sel_start == -1)
			m_sel_start = screen2sample(mouse_left);
		// if(m_sel_end <= screen2sample(width() - 1) || m_sel_end == -1)
			m_sel_end = screen2sample(mouse_right);
		if(last_mouse_pos > new_mouse_pos)
			draw(new_mouse_pos - 2,last_mouse_pos + 2);
		else
			draw(last_mouse_pos - 2,new_mouse_pos + 2);
	}
	else if(m_cursor_mode == KSoundWidget::Zoom)
	{
	}

	last_mouse_pos = new_mouse_pos;
}

void	KSoundWidget::timerEvent(QTimerEvent *te)
{
	int		new_mouse_pos = 0;

	if(te == 0) return;
	//printf("scroll %i\n",scroll);

	scroll *= 2;

	if(m_cursor_mode == KSoundWidget::Select)
	{
		if(scroll < 0)
		{
			new_mouse_pos = 0;
		}
		else if(scroll > 0)
		{
			new_mouse_pos = width();
			if(screen2sample(new_mouse_pos) > (int)m_sample->getLength())
				new_mouse_pos = sample2screen(m_sample->getLength());
		}
		if(first_mouse_pos == sample2screen(m_sel_start))
		{
			if(!setDispStart(m_disp_start + scroll * (1 << m_disp_zoom)))
			{
				scroll = 0;
				killTimers();
				return;
			}
			first_mouse_pos = sample2screen(m_sel_start);
		}
		else if(first_mouse_pos == sample2screen(m_sel_end))
		{
			if(!setDispStart(m_disp_start + scroll * (1 << m_disp_zoom)))
			{
				scroll = 0;
				killTimers();
				return;
			}
			first_mouse_pos = sample2screen(m_sel_end);
		}

		if(new_mouse_pos > first_mouse_pos)
		{
			mouse_left = first_mouse_pos;
			mouse_right = new_mouse_pos;
		}
		else
		{
			mouse_left = new_mouse_pos;
			mouse_right = first_mouse_pos;
		}

		m_sel_start = screen2sample(mouse_left);
		if(m_sel_start < 0)
			m_sel_start = 0;
		m_sel_end = screen2sample(mouse_right);
		if(m_sel_end > (int)m_sample->getLength())
			m_sel_end = (int)m_sample->getLength();

		last_mouse_pos = new_mouse_pos;

		draw();
	}
	else if(m_cursor_mode == KSoundWidget::Zoom)
	{
	}
	emit changedDisplayStart(getDisplayStart());
	draw();
}

void	KSoundWidget::layout()
{
	draw();
}

void	KSoundWidget::draw(int l,int r,bool selected)
{
	int			w,h,y,c,t;
	int			a,z,zs;
	int			i,ch;
	double	min,max,d = 0,lmin,lmax;

	w = width();
	h = height();

	if((c = m_sample->getNChannel()) == 0) return;

	ch = (h - c) / c;

	zs = (1 << (m_disp_zoom / 2));
	zs *= MAX(1,zs / 8);

	for(t = 0;t < c;t++)
	{
		y = (ch / 2) + ch * t + t;

		lmin = lmax = 0;

		for(i = MAX(0,l);i < MIN(r,width());i++)
		{
			min = 2;
			max = -2;
			for(z = 0;z <= (1 << m_disp_zoom);z+=zs)
			{
				a = m_disp_start + i * (1L << m_disp_zoom) + z;
				if(a >= (int)m_sample->getLength())
					break;
				m_sample->get(t,a,1,&d);
				d /= -2;
				min = MIN(min,d);
				max = MAX(max,d);
			}
			if(min == 2 || max == -2)
			{
				painter->setPen(nocolor);
				painter->drawLine(i,t * ch + t,i,(t + 1) * ch + t);
				painter->setPen(midlinecolor);
				painter->drawPoint(i,y);
			}
			else
			{
				d = min;
				if(min > lmax) min = lmax;
				lmin = d;
				d = max;
				if(max < lmin) max = lmin;
				lmax = d;
				if(selected)
					painter->setPen(sbgcolor);
				else
					painter->setPen(bgcolor);
				painter->drawLine(i,t * ch + t,i,(int)(min * (double)(ch - 5)) + y - 1 + 1);
				painter->drawLine(i,(int)(max * (double)(ch - 5)) + y + 1 + t,i,(t + 1) * ch + t);
				if(selected)
					painter->setPen(sfgcolor);
				else
					painter->setPen(fgcolor);
				painter->drawLine(i,
						(int)(min * (double)(ch - 5)) + y + t,
						  i,
						(int)(max * (double)(ch - 5)) + y + t);
				if(fabs(min) > 0.01 && fabs(max) > 0.01)
				{
					if(selected)
						painter->setPen(smidlinecolor);
					else
						painter->setPen(midlinecolor);
					painter->drawPoint(i,y);
				}
			}
			if(i == sample2screen(m_pointer))
			{
				painter->setPen(pointercolor);
				painter->drawLine(i,t * ch + t,i,(t + 1) * ch + t);
			}
		}
		painter->setPen(zoomcolor);
		a = sample2screen(m_zsel_start);
		if(a > 0 && a < w)
			painter->drawLine(a,0,a,h);
		a = sample2screen(m_zsel_end);
		if(a > 0 && a < w)
			painter->drawLine(a,0,a,h);

		if(t + 1 < c)
		{
			painter->setPen(devidecolor);
			painter->drawLine(l,(t + 1) * ch + t,r,(t + 1) * ch + t);
		}
	}
}

void	KSoundWidget::draw()
{
	draw(0,width());
}

void	KSoundWidget::draw(int l,int r)
{
	int		sl,sr;

	if(l < 0) l = 0;
	if(l > width()) return;
	if(r > width()) r = width();
	if(r < 0) return;

	sl = sample2screen(m_sel_start);
	sr = sample2screen(m_sel_end);

	//printf("draw(%i,%i): %i %i %i %i\n",l,r,l,sl,sr,r);
	if(m_sel_start < 0 || m_sel_end < 0)
	{
		//printf("LINE %i\n",__LINE__);
		draw(l,r,0);
		return;
	}
	if(sr < 0 || sl > width())
	{
		//printf("LINE %i\n",__LINE__);
		draw(l,r,0);
		return;
	}
	if(sl - 1> l && sl - 1<= r)
	{
		//printf("LINE %i\n",__LINE__);
		draw(l,sl - 1,0);
	}
	if(sl <= l && sr >= r)
	{
		//printf("LINE %i\n",__LINE__);
		draw(l,r,1);
	}
	else if(sl <= l && sr <= r && sr >= l)
	{
		//printf("LINE %i\n",__LINE__);
		draw(l,sr,1);
	}
 	else if(sl >= l && sr >= r && sl <= r)
	{
		//printf("LINE %i\n",__LINE__);
		draw(sl,r,1);
	}
	else if(sl >= l && sr <= r)
	{
		//printf("LINE %i\n",__LINE__);
		draw(sl,sr,1);
	}
	if(sr + 1 < r && sr + 1>= l)
	{
		//printf("LINE %i\n",__LINE__);
		draw(sr + 1,r,0);
	}
	//printf("\n");
}

void	KSoundWidget::draw(int x)
{
	int		w,h,y,c,t,sl,sr,sel;
	int		ch;
	unsigned int	z;
	double		min,max,d = 0;

	w = width();
	h = height();

	c = m_sample->getNChannel();

	sl = screen2sample(x);
	sr = screen2sample(x + 1);

	sel = (sl < m_sel_start && sr > m_sel_end);

	if(sel)
		painter->fillRect(x,0,x,h,sbgcolor);
	else
		painter->fillRect(x,0,x,h,bgcolor);

	for(t = 1;t < c;t++)
	{
		painter->setPen(devidecolor);
		painter->drawPoint(x,h / c * t);
	}

	ch = h / c;

	for(t = 0;t < c;t++)
	{
		if(sel)
			painter->setPen(smidlinecolor);
		else
			painter->setPen(midlinecolor);

		y = (h / c / 2) + h / c * t;
		painter->drawPoint(x,y);

		min = 2;
		max = -2;
		for(z = sl;(int)z <= sr;z++)
		{
			if(z >= m_sample->getLength())
				break;
			
			m_sample->get(t,z,1,&d);
			
			d /= -2;
			min = MIN(min,d);
			max = MAX(max,d);
		}
		if(sel)
			painter->setPen(sfgcolor);
		else
			painter->setPen(fgcolor);
		if(!(min == 2 || max == -2))
			painter->drawLine(x,
				  (int)(min * (double)(ch - 5)) + y,
				  x,
				  (int)(max * (double)(ch - 5)) + y);
		if(x == sample2screen(m_pointer))
		{
			painter->setPen(pointercolor);
			painter->drawLine(x,t * ch + t,x,(t + 1) * ch + t);
		}
	}
}

void	KSoundWidget::setDisplayValues(int ds,int dz,bool repaint)
{
	int		redraw = false;

	if(dz < 0)
		dz = 0;
	if(dz > maxZoom())
		dz = maxZoom();

	if(m_disp_zoom != dz)
	{
		m_disp_zoom = dz;
		emit changedDisplayZoom(getDisplayZoom());
		redraw = true;
	}

	if(ds + (width() - KSW_NAREA) * (1 << m_disp_zoom) > (int)m_sample->getLength())
		ds = m_sample->getLength() - (width() - KSW_NAREA) * (1 << m_disp_zoom);
	if(ds < 0) ds = 0;

	if(labs(m_disp_start - ds) > 0)
	{
		m_disp_start = ds;
		emit changedDisplayStart(getDisplayStart());
		redraw = true;
	}
	if(repaint && redraw)
		draw();
}

bool	KSoundWidget::setDispStart(int ds)
{
	if(ds + (width() - KSW_NAREA) * (1 << m_disp_zoom) > (int)m_sample->getLength())
		ds = m_sample->getLength() - (width() - KSW_NAREA) * (1 << m_disp_zoom);
	if(ds < 0)
		ds = 0;
	if(m_disp_start != ds)
	{
		m_disp_start = ds;
		return true;
	}
	return false;
}

void	KSoundWidget::setDisplayStart(int ds,bool repaint)
{
	int		oldpos = getDisplayStart();
	
	if(setDispStart(ds))
	{
		if(repaint)
		{
			int		dx,sx;
			
			dx = sample2screen(oldpos);
			sx = sample2screen(ds);
			if(sx >= width() || sx < -width() || dx >= width() || dx < -width())
				draw();
			else
			{
				if(dx < 0)
				{
					sx -= dx;
					dx = 0;
				}
				if(sx < dx)
				{
					bitBlt(this,dx,0,this,sx,0,width() - (dx - sx),height(),CopyROP,true);
					draw(0,(dx - sx));
				}
				else
				{
					bitBlt(this,dx,0,this,sx,0,width() - (sx - dx),height(),CopyROP,true);
					draw(width() - (sx - dx) - 1,width());
				}
			}
		}
		emit changedDisplayStart(getDisplayStart());
	}
}

void	KSoundWidget::setDisplayEnd(int de,bool repaint)
{
	setDisplayStart(screen2sample(sample2screen(de) - width()));
/*
	if(setDispStart(screen2sample(sample2screen(de) - width())))
	{
		if(repaint)
			draw();
		emit changedDisplayStart(getDisplayStart());
	}
*/
}

void	KSoundWidget::setDisplayCenter(int dc,bool repaint)
{
	setDisplayStart(screen2sample(sample2screen(dc) - width() / 2));
/*
	if(setDispStart(screen2sample(sample2screen(dc) - width() / 2)))
	{
		if(repaint)
			draw();
		emit changedDisplayStart(getDisplayStart());
	}
*/
}

void	KSoundWidget::setDisplayZoom(int dz,bool repaint)
{
	int		ds;

	if(dz < 0)
		dz = 0;
	if(dz > maxZoom())
		dz = maxZoom();

	//printf("Zoom: %i\n",dz);

	if(m_disp_zoom != dz)
	{
		m_disp_zoom = dz;
		ds = m_disp_start;
		setDisplayStart(m_disp_start,repaint);
//		if(ds == m_disp_start && repaint)
		if(repaint)
			draw();
		emit changedDisplayZoom(getDisplayZoom());
	}
}

int	KSoundWidget::getDisplayStart()
{
	return m_disp_start;
}

int	KSoundWidget::getDisplayEnd()
{
	return screen2sample(sample2screen(m_disp_start) + width());
}

int	KSoundWidget::getDisplayCenter()
{
	return screen2sample(sample2screen(m_disp_start) + width() / 2);
}

int	KSoundWidget::getDisplayZoom()
{
	return m_disp_zoom;
}

void	KSoundWidget::resize(int size)
{
	m_sample->resize(size);
}

void	KSoundWidget::resize(int w,int h)
{
	QWidget::resize(w,h);
}

void	KSoundWidget::setMode(KSoundWidget::Mode m)
{
	QCursor		c,&cr = c,&zcr = zoom_cursor;

	switch(m)
	{
		case KSoundWidget::Zoom:
			m_cursor_mode = KSoundWidget::Zoom;
		break;
		case KSoundWidget::Select:
		default:
				m_cursor_mode = KSoundWidget::Select;
		break;
	}
	if(m_busy)
	{
		c.setShape(WaitCursor);
		setCursor(cr);
	}
	else
	{
		switch(m)
		{
			case KSoundWidget::Zoom:
				setCursor(zcr);
			break;
			case KSoundWidget::Select:
				c.setShape(IbeamCursor);
				setCursor(cr);
			break;
		}
	}
}

KSoundWidget::Mode		KSoundWidget::getMode()
{
	return m_cursor_mode;
}

CSample	*KSoundWidget::getSample()
{
	return	m_sample;
}
	
/** returns the pointer position in samples */
int		KSoundWidget::getPointer()
{
	return m_pointer;
}

int	KSoundWidget::getSelectionStart()
{
	return m_sel_start;
}

int	KSoundWidget::getSelectionEnd()
{
	return m_sel_end;
}

void	KSoundWidget::setSelectionStart(int ss,bool repaint)
{
	if(ss == m_sel_start)
		return;
	if(ss < 0)
	{
		m_sel_start = -1;
		m_sel_end = -1;
	}
	else
	{
		m_sel_start = MIN(m_sample->getLength(),(uint)ss);
		if(m_sel_start > m_sel_end)
		{
			int	t;

			t = m_sel_start;
			m_sel_start = m_sel_end;
			m_sel_end = t;
		}
	}
	if(repaint)
		draw();
}

void	KSoundWidget::setSelectionEnd(int se,bool repaint)
{
	if(se == m_sel_end) return;
	if(se < 0)
	{
		m_sel_start = -1;
		m_sel_end = -1;
	}
	else
	{
		m_sel_start = MIN(m_sample->getLength(),(uint)se);
		if(m_sel_start > m_sel_end)
		{
			int	t;

			t = m_sel_start;
			m_sel_start = m_sel_end;
			m_sel_end = t;
		}
	}
	if(repaint)
		draw();
}

int		KSoundWidget::calcZoom(int l)
{
	int		t;

	l /= width();
	
	for(t = 0;l > 0;t++)
		l >>= 1;

	return t;
}

int		KSoundWidget::calcZoom(int a,int b)
{
	return calcZoom(b - a);
}

int		KSoundWidget::maxZoom()
{
	return calcZoom(m_sample->getLength());
}

/**  */
bool KSoundWidget::isBusy()
{
	return m_busy;
}

/**  */
void KSoundWidget::setBusy(bool busy)
{
  m_busy = busy;
  setMode(getMode());
}

/** sets the pointer position. If it lays outside the
visible area, the visible area is moved. It returns true
if the position is inside the sample, else it returns false
and sets the pointer to the border of the sample. */
bool KSoundWidget::setPointer(int pos,bool repaint)
{
	bool	inside = true;
	int		old;
	
	old = m_pointer;
	m_pointer = pos;
	if(m_pointer < 0)
	{
		m_pointer = 0;
		inside = false;
	}
	if(m_pointer >= (int)m_sample->getLength())
	{
		m_pointer = m_sample->getLength() - 1;
		inside = false;
	}
		
	if(m_pointer <= getDisplayStart())
		setDisplayStart(m_pointer,repaint);
	else if(m_pointer >= getDisplayEnd())
		setDisplayEnd(m_pointer,repaint);
	else
	{
		if(repaint)
		{
			draw(sample2screen(old) - 2,sample2screen(old) + 2);
			draw(sample2screen(m_pointer) - 2,sample2screen(m_pointer) + 2);
		}
	}
		
	return inside;
}
