/*
  buffer capabilites for a stream
  Copyright (C) 1998  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */

#include <producer/yaf/yafStream.h>

// Counter is used for debugging reasons (to distinguish
// between the threads. And to create an unique (hopefully) fifo name

static int counter=0;




/*Threads and usleep does not work, select is very portable*/
static void doSleep(int msec) {
  struct timeval time;
  time.tv_sec=0;
  time.tv_usec=msec;

  select(0,NULL,NULL,NULL,&time);
}


static void *readloopThread(void *arg){
  ((YafStream*)arg)->readloop();
  return NULL;
}   



YafStream::YafStream(YafGenerator* generator,int bufferSize) {
  int rNum;

  allRead=0;
  this->generator=generator;
  pthread_mutex_init(&mut,NULL);
  pthread_cond_init(&cond,NULL);

  randomName=new Buffer (60);
  streamCounter=counter;
  randomName->append("/tmp/.kmpg.");

  rNum=time(NULL);
  randomName->append(rNum);
  randomName->append(counter);
  randomName->append(getuid());
  randomName->append(getpid());
  counter++;
  

  fifoCommand=new Buffer(50);
  fifoCommand->clear();
  fifoCommand->append("outputfile ");
  fifoCommand->append(randomName->getData());
  fifoCommand->append("\n");
  
  // 8kb
  //  size=1024*16;
  //size=1024*1024*1;
  size=bufferSize;
  isShutdown=true;


  ringBufferNT=new RingBufferNT(size,16384);
  bufferUnderrun=__BUFFER_UNDERRUN_NOTIFIED;
  buffUnderRunSize=size/50;
  buffOverRunSize=size/8;
  lSockCreate=false;
  eof=false;
  start();
  while(lSockCreate==false) {
    doSleep(200);
  }
}


YafStream::~YafStream() {
  stop();
  delete ringBufferNT;
  delete fifoCommand;
 
  delete randomName;
}




void YafStream::start() {
  if (isShutdown == true) {
    isShutdown=false;
    pthread_create(&tr,NULL,readloopThread,this);
    return;
  }
}



void YafStream::stop() {
  void* ret;

  if (isShutdown == false) {
    lockReaderThread();
    isShutdown=true;
    lWriteToRing=false;
    unlockReaderThread();
    ringBufferNT->exitWaitForSpace();
    pthread_join(tr,&ret);
    return;
  }
}


void YafStream::setReaderBlock(int allow) {
  ringBufferNT->setWaitForData(allow);
}



int YafStream::getReaderBlock() {
  return ringBufferNT->getWaitForData();
}

/**
   These Functions are entered by the "reader" thread [START]

   Note: The Streambuffer is _not_ filled if we have a buffer
   we simply set the StartPtr to the memory of the buffer.
   If we have  not buffer we set the ptr to the read stream.

*/



int YafStream::fillBuffer(AudioBuffer* buffer) {

  int pSize=buffer->getPreferredFillGrade();
  MemChunk* gmem;
  float percent;

  // here we do quality of service, when mixing streams
  // under heavy load, we do nnever deliver less than
  // preffered size, except when we are eof.
  buffer->setMemChunk(NULL);

  if (eof==false) {
    if (ringBufferNT->waitForData(pSize) == false) {
      // if we are kicked
      return false;
    }
  }

  gmem=ringBufferNT->requestMemChunk(pSize);
  if (gmem->getLen() != pSize) {
    if (eof == false) {
      return false;
    }
  }

  buffer->setMemChunk(gmem);


  buffer->setReadBytes(ringBufferNT->getReadBytes()+gmem->getLen());
  percent=(float)ringBufferNT->getFillgrade()/(float)ringBufferNT->getSize();
  buffer->setMainBufferFillgrade(percent);




  return true;
}
  




/**
   The readloop is the thread which reads the fifo and fills the
   buffer
*/

void YafStream::readloop() {
  int didRead;
  int lShow=false;
  MemChunk* gmem=new MemChunk(0);
  int preferredSize=4096;
  bindSocket=new BindSocket(randomName->getData());
  bindSocket->bind();
  lSockCreate=true;
  while (isShutdown==false) {
    if (bindSocket->accept() > 0) {
      break;
    }
    doSleep(200);
  }
  pthread_mutex_lock(&mut);
  while(isShutdown==false) {


    pthread_mutex_unlock(&mut);
    ringBufferNT->getWriteArea(gmem,preferredSize);
   
    if (gmem->getLen() < preferredSize){
      ringBufferNT->waitForSpace(preferredSize);
      ringBufferNT->getWriteArea(gmem,preferredSize);
    }
    if (gmem->getLen() == 0) {
      pthread_mutex_lock(&mut);
      continue;
    }
    if (gmem->getLen() < 0) {
      cout << "gmem->getLen() < 0 this means trouble"<<endl;
    }
    
    // because we block on the fifo and have at least
    // one byte to buffer, we are sure, that didRead<=0 means
    // some nasty bug
    didRead=bindSocket->read(gmem->getPtr(),gmem->getLen());
    if (didRead <= 0) {
      if (lShow==false) {
	cout << "bindSocket error. sleeping for recovery"<<endl;
	ringBufferNT->setWaitForData(false);
	lShow=true;
      }
      doSleep(200);
      pthread_mutex_lock(&mut);
      continue;
    }
    allRead+=didRead;
    pthread_mutex_lock(&mut);
    if (lWriteToRing) {
      ringBufferNT->forwardWritePtr(didRead);
     } else {
       //cout << "Skip read bytes :"<<didRead<<endl;
     }
  }
  pthread_mutex_unlock(&mut);
  delete gmem;
  delete bindSocket;
}


/**
   These Functions are entered by the "reader" thread [END]
*/




  
/**

   
*/
char* YafStream::getFifoCommand() {
  return fifoCommand->getData();
}


void YafStream::lockReaderThread() {
  pthread_mutex_lock(&mut);
}


void YafStream::unlockReaderThread() {
  pthread_cond_signal(&cond);
  pthread_mutex_unlock(&mut);
}




void YafStream::close() {
  setWriteRing(false);
}


void YafStream::open() {
  setWriteRing(true);
}
 

void YafStream::setEOF(int leof) {
  this->eof=leof;
}


int YafStream::getEOF() {
  return eof;
}


void YafStream::setWriteRing(int lWrite) {
  lockReaderThread();
  lWriteToRing=lWrite;
  unlockReaderThread();
}
  

void YafStream::clear() {
  pthread_mutex_lock(&mut);
  ringBufferNT->emptyBuffer();
  bufferUnderrun=__BUFFER_UNDERRUN_NOTIFIED;
  pthread_mutex_unlock(&mut);
}



