/*
  starter for plugin decoder
  Copyright (C) 1999  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */


#include <decoderStarter.h>



DecoderStarter::DecoderStarter(){
  currentSong=new Song("");
  count=0;
  currentTime=0;
  installEventFilter(this);
  currentDecoder=NULL;
  lPlaying=false;
  makePreferences();
}


DecoderStarter::~DecoderStarter() {
  
  delete preferences;
  if (currentDecoder != NULL) {
    delete currentDecoder;
  }
}

char* DecoderStarter::getNodeName() {
  return "decoderStarter";
}


void DecoderStarter::setRefCount(int count) {
  this->count=count;
}


int DecoderStarter::getRefCount() {
  return count;
}


void DecoderStarter::open(Song* song,DecoderID* decoderID) {

  song->copyTo(currentSong);

  if (currentDecoder != NULL) {
    currentDecoder->close();
  }
  launchDecoder(song,decoderID);
 
  if (currentDecoder != NULL) {
    DecoderID* decoderID=getCurrentDecoderID();
    if (decoderID->getID() != _MPEG_VIDEO_DECODER) {
      setLastSong(song);
    }
  }

  play();
}


void DecoderStarter::jump(int second) {

  if (currentDecoder != NULL) {
    currentDecoder->jump(second);
  }
}

void DecoderStarter::play() {
  if (currentDecoder != NULL) {
    currentDecoder->play();
  }
  const char* currentSongName=currentSong->getSongLocation();
  if (strlen(currentSongName) == 0) {
    // empty, emit play next
    emit(playNextEvent());
  }

  lPlaying=true;
}

void DecoderStarter::pause() {
  if (currentDecoder != NULL) {
    currentDecoder->pause();
  }
  lPlaying=false;
}

void DecoderStarter::stop() {
  pause();
  if (currentDecoder != NULL) {
    currentDecoder->close();
  }

  if (GarbageCollector::canMakeCollection()) {
    GarbageCollector::makeGarbageCollection();
  }
}



bool DecoderStarter::eventFilter( QObject* object, QEvent *e ) {

#if (QT_VERSION < 200) 
  if ( e->type() == Event_User)     // yaf global event
#else
  if ( e->type() == QEvent::User)     // yaf global event
#endif
    {
    YafGlobalEvent* yafEvent = (YafGlobalEvent*)e;
    
    if (yafEvent->getMsg() == _GS_SIGNAL_GENERATOR_CRASHED) {
      if (currentDecoder != NULL) {
	if (currentDecoder->isRunning()==false) {
	  decoderCrash();
	}
      }
      return TRUE;                        // eat event
    }
    if (yafEvent->getMsg() == _GS_SIGNAL_PLAYING_READY) {

      lPlaying=false;
      emit(playNextEvent());
      return TRUE;                        // eat event
    }
  }
  return FALSE;  
}


void DecoderStarter::processEvent(char eventId) {
  switch(eventId) {
  default:
    cout << "unknown msg: "<<eventId<<" in DecoderStarter::processEvent"<<endl;
  }
}


void DecoderStarter::decoderCrash() {
  QString msg2;
  const char* url=currentSong->getSongLocation();

  msg2 = i18n("Your decoder takes a trip to nirvana!");
  msg2 += "\n\n";
  msg2 += i18n("song:");
  msg2 += url;
  msg2 += "\n\n";
  msg2 += i18n("crashprotection(tm)+autorecovery(tm) v2.0");

  if (crashConfig->getDoWarn()) {
    KMsgBox::message(NULL,i18n("nirvana warning"),msg2.data());
  }
  delete currentDecoder;
  currentDecoder=NULL;

  // if it was current we play next
  if (crashConfig->getPlayNext()) {
    emit(playNextEvent());
  }
  if (crashConfig->getDoInform()) {
    msg2 = i18n("Decoder restarted!");
    msg2 += "\n\n";
    KMsgBox::message(NULL,i18n("SigSev Handler"),msg2.data());
  }  
  if (GarbageCollector::canMakeCollection()) {
    GarbageCollector::makeGarbageCollection();
  }
}


DecoderID* DecoderStarter::getConfigFor(DecoderID* target) {
  int id=target->getID();

  // check how the current user settings are
  if (id == _MP3_DECODER) {
    return splayDecoderConfig->getDecoderID();
  }
  if (id == _MPEG_VIDEO_DECODER) {
    return mpgDecoderConfig->getDecoderID();
  }
  // no config
  return NULL;
}


void DecoderStarter::launchDecoder(Song* target,DecoderID* launchID) {
  // check how the current user settings are
  // and do not start a new decoder if thes are still the same

  DecoderID* configID=getConfigFor(launchID);

  DecoderID* currentDecoderID=getCurrentDecoderID();

  if (currentDecoderID != NULL) {

    // no config for decoder present
    if (configID == NULL) {
      if (launchID->argumentEqualTo(currentDecoderID)) {
	if (currentDecoder != NULL) {
	  const char* file=target->getSongLocation();
	  currentDecoder->open(file);
	}
	return;
      }
    }
    if (configID != NULL) {

      if (configID->argumentEqualTo(currentDecoderID)) {
	if (currentDecoder != NULL) {
	  const char* file=target->getSongLocation();
	  currentDecoder->open(file);
	}
	return;
      }
      launchID=configID;
    }
  } else {
    if (configID != NULL) {
      launchID=configID;
    }
  }
 
  restartDecoder(target,0,launchID);

}


void DecoderStarter::configure() {
  preferences->show();
}



void DecoderStarter::makePreferences() {

  preferences=new Preferences();


  crashConfig=new CrashConfig("ChrashConfig",preferences);
  preferences->addTab(crashConfig,crashConfig->getGroupName());
  connect(crashConfig,SIGNAL(applyEvent()), this, SLOT(apply()));
  connect(crashConfig,SIGNAL(killDecoder()), this, SLOT(decoderCrash()));
  

  splayDecoderConfig=new SplayDecoderConfig("SPlayDecoder",preferences);
  preferences->addTab(splayDecoderConfig,splayDecoderConfig->getGroupName());
  connect(splayDecoderConfig,SIGNAL(restart()),this,SLOT(restartSong()));

  mpgDecoderConfig=new MPGDecoderConfig("MPGDecoder",preferences);
  preferences->addTab(mpgDecoderConfig,mpgDecoderConfig->getGroupName());
  connect(mpgDecoderConfig,SIGNAL(restart()),this,SLOT(restartSong()));

 /*
  TplayDecoderConfig* tplayDecoderConfig;
  tplayDecoderConfig=new TplayDecoderConfig("TPlayDecoder",preferences);
  preferences->addTab(tplayDecoderConfig,tplayDecoderConfig->getGroupName());
  */
  
}




void DecoderStarter::apply() {
}


  
  
void DecoderStarter::writeIn(NodeDevice* source,DeviceConfig* config) {

  AudioStream* audioStream=config->getAudioStream();
  TimeInfo* timeInfoStream=audioStream->getTimeInfo();

  // grap current time out of stream:
  currentTime=timeInfoStream->getTotalSec();

  // and on with deliver
  // I am the postman :-)
  deliverToListener(config);
}




DecoderID* DecoderStarter::getCurrentDecoderID() {
  DataGenerator* dataGenerator;

  if (currentDecoder == NULL) {
    return NULL;
  }
  dataGenerator=currentDecoder->getDataGenerator();
  if (dataGenerator == NULL) {
    return NULL;
  } 
  DecoderID* back=dataGenerator->getDecoderId();

  return back;
}



void DecoderStarter::restartDecoder(Song* startSong,
				    int startTime,DecoderID* id) {
  Song* tmp;
  int cnt=0;

  if (currentDecoder != NULL) {
    delete currentDecoder;
    currentDecoder=NULL;
  }


  tmp=new Song("");

 
  startSong->copyTo(tmp);
  
  currentDecoder=Amplifier::createPlayer(id);
  
  if (currentDecoder == NULL) {
    cout << "Aieehh! no decoder registered"<<endl;
    exit(0);
  }
  // we must register to get global _or_ lokal events !!!!!!
  connect(currentDecoder->getEventQueue(),SIGNAL(processEvent(char)),
	  this,SLOT(processEvent(char)));
  currentDecoder->addListener(this);
  const char* file=tmp->getSongLocation();


  currentDecoder->open(file);

  
  if (startTime  != 0) {
    currentDecoder->jump(startTime);
    if (lPlaying) {
      currentDecoder->play();
    }
  }
  delete tmp;
  
}
   

void DecoderStarter::restartSong() {
  if (currentDecoder != NULL) {
    currentDecoder->close();
  }
  
  DecoderID* tmp=getCurrentDecoderID();
  if (tmp == NULL) {
    cout << "no currentDecoder: restart aborted"<<endl;
    return;
  }
  // we must make a lokal copy , because restartDecoder
  // deletes currentdecoder and guess what , the decoderID as well

  DecoderID* target=new DecoderID(tmp->getID(),0,NULL);
  DecoderID* config=getConfigFor(target);

  // if we have a config, use it otherwise
  // use the current version
  tmp->copyTo(target);



  if (config != NULL) {
    restartDecoder(currentSong,currentTime,config);
  } else {
    restartDecoder(currentSong,currentTime,target);
  }
    
  delete target;

}

void DecoderStarter::setLastSong(Song* song) {
  const char* url=song->getSongLocation();

  StartUpConfig::setLastSong((char*)url);
}



