/*
 *	      ModClass - multithreaded C++ wrapper for mikmod
 *	(C)1998, 1999 Roope Anttinen - roope.anttinen@ntc.nokia.com
 *	
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 */

#ifndef _MODCLASS_H
#define _MODCLASS_H

#ifndef _MIT_POSIX_THREADS
# define _MIT_POSIX_THREADS
#endif
#ifndef _REENTRANT
# define _REENTRANT
#endif

#include <unistd.h>
#include <pthread.h>
#include <time.h>
#include <mikmod.h>

#include <qstring.h>

#include "stopwatch.h"
#include "PlayList.h"

struct playdata {	// Updated frequently.
	unsigned int PatPosition;
	unsigned int PatLenght;
	unsigned int PatNumber;
	unsigned int TotPatterns;
	unsigned int Tempo;
};

struct playinfo {	// Updated sometimes.
	QString Status;
	QString Composer;
	QString Comment;
	QString ModType;
	bool changed;
};

struct configuration {
	int md_mixfreq;	// 11025/22050/44100/48000(GUS Pnp)
	int md_mode;		// Mono/Stereo, 8/16b etc...
	int md_device;		// Driver.
	int volume;		// Sound volume.
	long sleep_time;	// Decoder thread sleep time.
	bool fadeout;		// Fadeout at the end of module.
#if defined (_POSIX_THREAD_PRIORITY_SCHEDULING)
	bool use_rt;		//
#else
# warning POSIX THREAD PRIORITY scheduling not supported (no realtime now).
#endif //_POSIX_THREAD_PRIORITY_SCHEDULING
	bool operator==(const configuration &);
	bool operator!=(const configuration &p) {return !operator==(p);}
};

class ModPlayer
{
private:
	PlayList *playlist;	// Pointer to global playlist object.
	MODULE *module;		// Pointer to current song.
	configuration conf;	// Configuration info.
	int relative_spd;	// Relative playing speed.
	
	int player_id;		// Thread id of player.
	int loader_id;		// Thread id of loader.
	pthread_t THplayer;	// Player thread.
	pthread_t THloader;	// Loader thread.

	pthread_mutex_t DL;	// Datalock.
	pthread_mutex_t IL;	// PlayInfolock.
	pthread_mutex_t WL;	// Waitlock for ModReady.
	pthread_cond_t ModReady;// Module loaded - start playing it.
	
	struct timespec sleep_time;

	bool Reset;		// Tell player thread about reconfiguration.
	bool Quit;		// Tell player thread to die.
	bool Pause;		// Pause player thread.
	bool Stop;		// Stop player thread.
	bool Active;		// Is player active (playing).
	bool Loading;		// Not active, but loading...
	bool Alive;		// Is player alive (player thread is created).
	bool Feed;		// Player needs feeding (please load a new module!).
	bool threaded_driver;	// Is modified driver in use.

	unsigned int Memory;	// Remeber position after reset.

	int maxchan;		// Mikmod needs this for loading.

	playdata Data;		// pattern position, bpm etc...
	playinfo Info;		// Song name, module type etc...
	
// Not sure if these are really needed. Have to try on SMP box without 'em.
	playdata d_tmp;		// temporary for Data.
	playinfo i_tmp;		// temporary for Info.
	
	StopWatch * timer;	// Elapsed playing time of current song.
	
	static void *PLstarter(void *); // Used to start player().
	static void *LDstarter(void *);	// Used to start loader().
	void *player();			// Does the actual playing.
	void *loader();			// Loads modules (so that UI doesn't freeze).
	bool loadModule();		// Load playlists current module.
	const QString modName();	// Module name if available/file name.
	inline void message(const QString); 	// Placed in playinfo.Status.
	bool needRestart(const configuration&);
					// Is configuration changed so that
					// player needs reinitialization.
	

public:
	ModPlayer(PlayList *, const configuration &); // Initialize player.
	virtual ~ModPlayer();		// Destruct player.

	virtual void play();		// Play current playlist item.
	virtual void togglePause();	// (Un)pause player.
	virtual void stop();		// Stop player.
	virtual void step(const int);	// Step forward/backward (+/- value).
	virtual void setVolume(const int); // Set player volume (0 - 128).

	virtual bool playerActive();	// Is player active (playing).
	virtual bool feedPlayer();	// Player needs new module.
	virtual void stopFeeding();	// No modules->don't feed.
	virtual const playdata &getData(); // Data changes at every "beat".
	virtual const playinfo &getInfo(); // Info changes occasionally.
	virtual bool infoChanged() {return Info.changed;} // is Info changed.
	virtual time_t playtime() {return timer->elapsed();} // Elapsed time.
	virtual configuration &getConfig(); // Return current configuration.
	virtual void setConfig(const configuration &); // Set configuration.
	virtual char *getDrivers();	// List available drivers.
	virtual const QString getInstruments();	// List all instrument names.
	virtual const QString getSamples();	// List all sample names.
};

#endif // _MODCLASS_H
