#include "mpegsound.h"
#include "splay.h"
#include <string.h>
#include <sys/stat.h>
#include <iostream.h>
#include "rawtoemusic.h"
#include <pthread.h>
#include <time.h>

static Soundinputstream *loader = NULL;
static Rawtoemusic	 *player = NULL;
static Mpegtoraw	 *server = NULL;

static char *song_path;

static int lDecoderLoop;
static int lCreatorLoop;
static int linDecoderLoop;
static int lDecode;
static int eof;

static int downSampleFlag=0;
static int monoFlag=0;
static int autoupdateFlag=true;

typedef struct {
  float totalSize;
  int   counter;
  int   currentLength;
} AvgFrameSize;

static pthread_t tr;
static pthread_mutex_t decoderMut; 
static pthread_mutex_t changeMut; 
static pthread_cond_t decoderCond;


static struct stat mp3stat;

/**
   The idea of this implementation:
   The decoding engine is a single thread. This thread is controlled
   by "a controller".
   The controller is a single thread and all calls to pause,play,open etc..
   are made by the controller.
   The controller calls contruct/destruct as well.
*/

void autoUpdate();

static int getSongLength() {
  int back=0;
  if (server != NULL){
    float totalframes=server->gettotalframe();
    float pcm=server->getpcmperframe();
    float wavfilesize=(totalframes*pcm);
    float frequence=(float)server->getfrequency();
    if (frequence != 0) {
      back=(int)(wavfilesize/frequence);
    }
  }
  return back;
}


static void decoderLock() {
  pthread_mutex_lock(&changeMut);
  pthread_mutex_lock(&decoderMut);
}

static void decoderUnlock() {
  pthread_cond_signal(&decoderCond);
  pthread_mutex_unlock(&changeMut);
  pthread_mutex_unlock(&decoderMut);
}


void splay_seek(int second) {
  decoderLock();
  int length=getSongLength();
  int totalframes;
  float jumpFrame=0.0;
  // race condition: 
  if (server != NULL) {
    totalframes=server->gettotalframe();
    if (totalframes > 0) {
      jumpFrame = ((float)second/(float)length)*(float)totalframes;
    }
    server->clearbuffer();
    server->setframe((int)jumpFrame);
  }
  decoderUnlock();
}


static void decoder_loop() {
  int lInit=false;
  player = new Rawtoemusic();
  player->initialize(NULL);
  server = new Mpegtoraw(loader, player);
  server->initialize(song_path);
 
  server->setdownfrequency(downSampleFlag);
  server->setforcetomono(monoFlag);
#ifdef PTHREADEDMPEG
  server->makethreadedplayer(2);
#endif

  int bps=server->getbitrate();
  if (server->isXingVBR()) {
    // someday I must find out how to convert from big to little endian
    bps=0;
  }
  while(lDecoderLoop && lCreatorLoop) {
    if (pthread_mutex_trylock(&changeMut) == EBUSY) {
      pthread_cond_wait(&decoderCond,&decoderMut);
      continue;
    }
    pthread_mutex_unlock(&changeMut);
    if (lDecode) {
      if (lInit==false) {
	// this is the trick to initilize the output sound
	if (server->run(-1) == false) {
	  continue;
	}
	lInit=true;
      }
      server->run(5);
      if (server->geterrorcode() == SOUND_ERROR_FINISH){
	lDecoderLoop=false;
	eof=true;
	flush_audio();
      }
    } else {
      pthread_cond_wait(&decoderCond,&decoderMut);
   }
  }
  server->clearbuffer();
#ifdef PTHREADEDMPEG
  server->freethreadedplayer();
#endif
  delete server;
  delete player;
  server = NULL;
  player = NULL;
  pthread_mutex_unlock(&decoderMut);
}



int splay_open(char *path) {
  decoderLock();
  int err;
  loader = Soundinputstream::hopen(path, &err);
  if (!loader) {
    decoderUnlock();
    return 0;
  }

  song_path = path;
  lDecoderLoop=true;
  lDecode=false;
  eof=false;
  stat(path, &mp3stat);
  decoderUnlock();
  while(linDecoderLoop==true) {
    usleep(500);
  }
  return 1; 
}

void splay_close() {
  decoderLock();
  lDecode=false;
  lDecoderLoop=false;
  decoderUnlock();
  while(linDecoderLoop==false) {
    usleep(500);
  }
  if (loader != NULL) {
    delete loader;
    loader =NULL;
  }
}
  

void splay_pause(void){
  decoderLock();
  lDecode=false;
  decoderUnlock();
  
}

void splay_play(){
  decoderLock();
  lDecode=true;
  decoderUnlock();
}


void splay_getModuleInfo(char *id, char *version, char *copyright) {
   strcpy(id, "splay");
   strcpy(version, "0.8.2");
   strcpy(copyright, "1998 Woo-jae Jung");
}



void splay_getMusicInfo(playlist_item *song) {
  song->length=getSongLength();
  song->tag.genre = -1;
  if (song->tag.genre == -1)
    strcpy(song->name, strrchr(song->path,'/') + 1);
  else
    strcpy(song->name, song->tag.song);
}


void splay_config(char *left, char *right) {
  if (strcmp(left,"-m")==0) {
    monoFlag=true;
  }
  if (strcmp(left,"-2")==0) {
    downSampleFlag=true;
  }
  if (strcmp(left,"runtime")==0) {
    if (strcmp(right,"on")==0) {
      autoupdateFlag=true;
    } else {
      autoupdateFlag=false;
    }
  }
}



int splay_getStreamState() {
  if (eof) {
    return _STREAM_STATE_EOF;
  } 
  return _STREAM_STATE_NOT_EOF;
}





void* splay_thread(void* arg) {
  pthread_mutex_lock(&decoderMut);
  while(lCreatorLoop) {
    linDecoderLoop=true;
    if (pthread_mutex_trylock(&changeMut) == EBUSY) {
      pthread_cond_wait(&decoderCond,&decoderMut);
      continue;
    }
    pthread_mutex_unlock(&changeMut);
    if (lDecoderLoop) {
      linDecoderLoop=false;
      decoder_loop();
    } else {
      pthread_cond_wait(&decoderCond,&decoderMut);
    }
  }
  pthread_mutex_unlock(&decoderMut);
  return NULL;
}

   

void splay_construct() {
  pthread_cond_init(&decoderCond,NULL);

  pthread_mutex_init(&changeMut,NULL);
  pthread_mutex_init(&decoderMut,NULL);
  loader =NULL;
  server =NULL;
  player =NULL;
  lCreatorLoop=true;
  lDecoderLoop=false;
  linDecoderLoop=false;
  pthread_create(&tr,NULL,splay_thread,NULL);
  while(linDecoderLoop==false) {
    usleep(500);
  }
}


void splay_destruct() {
  void* ret;
  decoderLock();
  lDecoderLoop=false;
  lCreatorLoop=false;
  lDecode=false;
  pthread_join(tr,&ret);
  decoderUnlock();
}






