/*
  Copyright (C) 1999 Rainer Maximini

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package
*/

#include "printer.h"
#include <stdlib.h>

/*
 *********************************************************************
 **                                                                 **
 **    Printer Settings                                             **
 **                                                                 **
 *********************************************************************
 */
PrinterSettings::PrinterSettings(QWidget *parent=0, const char* name=0)
  : QDialog(parent, name, false) {
    
  setMinimumSize(300,280);

  static const char *weights[] = {
    "Light (25)", "Normal (50)", "DemiBold (63)",
    "Bold (75)", "Black (87)", 0
  };   
  static const char *families[] = {
    "Charter", "Clean", "Courier", "Fixed", "Gothic", "Helvetica",
    "Lucida", "Lucidabright", "Lucidatypewriter", "Mincho",
    "New century schoolbook", "Symbol", "Terminal", "Times", "Utopia",
    0
  };     
  const QFont iniFont("lucida", 12, QFont::Normal, false);
  config = (KApplication::getKApplication())->getConfig();
  config->setGroup(KCONFIG_PRINTER);
  font = config->readFontEntry(KCONFIG_PRINTER_FONT,&iniFont );
  tab1 = config->readNumEntry(KCONFIG_PRINTER_TAB1, 200);
  tab2 = config->readNumEntry(KCONFIG_PRINTER_TAB2, 600);
  

  setCaption(i18n("Printer Settings"));

  const char **tmp;

  fontBox = new QGroupBox(i18n("Font Settings"), this, "fontbox");

  familyLabel = new QLabel(i18n("Family:"), fontBox, "falabel");
  family = new QComboBox(FALSE, fontBox, "family");
  tmp = families;
  while(*tmp){
    family->insertItem(*tmp);
    tmp++;
  }
  weightLabel = new QLabel(i18n("Weight:"), fontBox, "folabel");
  weight = new QComboBox(FALSE, fontBox, "weight");
  tmp = weights;
  while(*tmp){
    weight->insertItem(*tmp);
    tmp++;
  }
  sizeLabel = new QLabel(i18n("Point size:"), fontBox, "silabel");
  size   = new QLineEdit(fontBox, "size");
  italic = new QCheckBox(i18n("Italic"), fontBox, "italic" );

  tabBox = new QGroupBox( i18n("Tabulator Settings"), this, "tabs");
  tabLabel = new QLabel(i18n("Tabulator for the title and the time."), tabBox, "tablabel");
  tabOne   = new QLineEdit(tabBox, "size");
  tabTwo   = new QLineEdit(tabBox, "size");

  acceptButton = new QPushButton(i18n("OK"), this, "accept");
  applyButton = new QPushButton(i18n("Accept"), this, "apply");
  cancelButton = new QPushButton(i18n("Cancel"), this, "cancel");

  connect(acceptButton, SIGNAL(clicked()), this, SLOT(accept()));
  connect(applyButton, SIGNAL(clicked()), this, SLOT(apply()));
  connect(cancelButton, SIGNAL(clicked()), this, SLOT(close()));

  refresh();
  emit newParameter(font, tab1, tab2);
}


void PrinterSettings::refresh(){
  const char **tmp;
  int i;
  static const char *families[] = {
    "Charter", "Clean", "Courier", "Fixed", "Gothic", "Helvetica",
    "Lucida", "Lucidabright", "Lucidatypewriter", "Mincho",
    "New century schoolbook", "Symbol", "Terminal", "Times", "Utopia",
    0
  };     

  tmp = families;
  i=0;
  while(*tmp){
    if(strcmp(*tmp,font.family()) == 0){
      family->setCurrentItem(i);
      break;
    }
    tmp++;
    i++;
  }

  switch(font.weight()){
  case 25: i = 0; break;
  case 50: i = 1; break;
  case 63: i = 2; break;
  case 75: i = 3; break;
  case 87: i = 4; break;
  default: i=0;
  }
  weight->setCurrentItem(i);

  QString qstmp;
  qstmp.sprintf("%d",font.pointSize());
  size  ->setText(qstmp);
  italic->setChecked(font.italic());

  qstmp.sprintf("%d", tab1);
  tabOne->setText(qstmp);
  qstmp.sprintf("%d", tab2);
  tabTwo->setText(qstmp);
}


QFont PrinterSettings::getFont() const { return font; }
int PrinterSettings::getTab1() const { return tab1; }
int PrinterSettings::getTab2() const { return tab2; }
void PrinterSettings::close(){  hide(); }
void PrinterSettings::open(){  show(); }
void PrinterSettings::accept(){
  apply();
  close();
}

void PrinterSettings::cancel(){
  const QFont iniFont("lucida", 12, QFont::Normal, false);
  config->setGroup(KCONFIG_PRINTER);
  font = config->readFontEntry(KCONFIG_PRINTER_FONT, &iniFont);
  emit fontChanged();
}  

void PrinterSettings::apply(){
  int i = 0;
  font.setFamily(family->currentText());
  font.setPointSize(atoi(size->text()));
  font.setItalic(italic->isChecked());
  switch(weight->currentItem()){
  case 0: i = 25; break;
  case 1: i = 50; break;
  case 2: i = 63; break;
  case 3: i = 75; break;
  case 4: i = 87; break;
  default: i = 50; break;
  }
  font.setWeight(i);

  tab1 = atoi(tabOne->text());
  tab2 = atoi(tabTwo->text());

  config->setGroup(KCONFIG_PRINTER);
  config->writeEntry(KCONFIG_PRINTER_FONT, font );
  config->writeEntry(KCONFIG_PRINTER_TAB1, tab1 );
  config->writeEntry(KCONFIG_PRINTER_TAB2, tab2 );
  config->sync();
  emit newParameter(font,tab1,tab2);
}

void PrinterSettings::resizeEvent( QResizeEvent * ){
  int m;
  fontBox->setGeometry( 5, 5, width()-10,140);
  familyLabel ->setGeometry( 10, 20, 90, 25);
  family      ->setGeometry(100, 20, fontBox->width()-110, 25);
  weightLabel ->setGeometry( 10, 50, 90, 25);
  weight      ->setGeometry(100, 50, fontBox->width()-110, 25);
  sizeLabel   ->setGeometry( 10, 80, 90, 25);
  size        ->setGeometry(100, 80, fontBox->width()-110, 25);
  italic      ->setGeometry(100,110, fontBox->width()-110, 25);


  tabBox->setGeometry( 5,155, width()-10, 80);
  m = (tabBox->width()-30)/2;
  tabLabel ->setGeometry( 10, 20, tabBox->width()-20, 20);
  tabOne   ->setGeometry( 10, 45,  m, 25);
  tabTwo   ->setGeometry(m+20, 45, m, 25);

  acceptButton ->setGeometry(10,height()-35,80,30);
  applyButton  ->setGeometry(((width()/2)-40),height()-35,80,30);
  cancelButton ->setGeometry(width()-90,height()-35,80,30);
}

/*
 *********************************************************************
 **                                                                 **
 **    Printer Preview Window                                       **
 **                                                                 **
 *********************************************************************
 */

PrinterPreviewWindow::PrinterPreviewWindow(QWidget *parent=0, const char* name=0)
  : QWidget(parent, name) {
  setBackgroundColor(white);
  currentPage = 1;
}

void PrinterPreviewWindow::setPrinter(Printer *tmp){ printer = tmp; }
void PrinterPreviewWindow::setPage(int nr){ 
  currentPage = nr; 
  repaint();
}

void PrinterPreviewWindow::paintEvent(QPaintEvent *) {
  int x = 0;
  int y = 0;
  printer->getPageDimension(x, y);
  QWMatrix m;

  float dx = (float)height() / (float)y;

  m.scale(dx,dx);

  if(printer == 0) return;
  QPainter p;
  p.begin(this);
  p.setWorldMatrix(m);
  p.setBackgroundColor( white );
  //  p.scale(sx,sy);
  printer->draw(&p, currentPage);
  p.end();
}

void PrinterPreviewWindow::resizeEvent( QResizeEvent * ){
  resize(width(), height());
}

/*
 *********************************************************************
 **                                                                 **
 **    Printer Preview                                              **
 **                                                                 **
 *********************************************************************
 */
PrinterPreview::PrinterPreview(QWidget *parent=0, const char* name=0, Printer *tmp=0)
  : QDialog(0, 0, false) {
    
  setMinimumSize(345,500);
  resize(345,500);
  setCaption(i18n("Printing Preview"));

  currentPage = 1;
  totalPages  = 1;

  painter = new PrinterPreviewWindow(this, "painter" );
  printer = tmp;
  painter->setPrinter(tmp);

  printButton   = new QPushButton(i18n("Print"),    this, "print");
  settingButton = new QPushButton(i18n("Settings"), this, "settings");
  closeButton   = new QPushButton(i18n("Close"),    this, "close");
  pagesLabel    = new QLabel(this, "pagesLabel");
  scrollBar     = new QScrollBar(QScrollBar::Horizontal, this, "scrollBar");
  scrollBar->setSteps(1,5);
  scrollBar->setRange(1,1);

  connect(scrollBar, SIGNAL(valueChanged(int)), painter, SLOT(setPage(int)));
  connect(scrollBar, SIGNAL(valueChanged(int)), this, SLOT(setCurrentPage(int)));
  
  connect(printButton,   SIGNAL(clicked()), this, SIGNAL(print()));
  connect(settingButton, SIGNAL(clicked()), this, SIGNAL(openSettings()));
  connect(closeButton,   SIGNAL(clicked()), this, SLOT(close()));
  painter->repaint();
}


void PrinterPreview::setCurrentPage(int page){
  currentPage = page;
  updatePagesLabel();
}
void PrinterPreview::setPages(int pages){
  totalPages = pages;
  scrollBar->setRange(1,pages);
  updatePagesLabel();
}
void PrinterPreview::updatePagesLabel(){
  QString tmp;
  tmp.sprintf("P:%d/%d",currentPage,totalPages);
  pagesLabel->setText(tmp);
}

void PrinterPreview::close(){  hide(); }
void PrinterPreview::open(){  painter->setPage(1); show(); }
void PrinterPreview::refresh(){ painter->repaint(); }
void PrinterPreview::resizeEvent( QResizeEvent * ){
  int x = 0;
  int y = 0;
  scrollBar     ->setGeometry(  5,  7,140, 20);
  pagesLabel    ->setGeometry(150,  5, 40, 25);
  printButton   ->setGeometry(  5,         height()-30, 80, 25);
  settingButton ->setGeometry( 90,         height()-30, 80, 25);
  closeButton   ->setGeometry( width()-85, height()-30, 80,25);

  printer->getPageDimension(x,y);

  int w = width() - 20;
  int h = height()- 70;
  float dx = (float)w/(float)x;
  float dy = (float)h/(float)y;
  if (dx < dy) 
    h = (int) (dx * (float )y);
  else 
    w = (int) (dy * (float )x);

  x = 0;
  y = 0;
  if (w < width()-20){
    x = (width()-20-w)/2;
  }
  if (h < height()-70){
    y = (height()-70-h)/2;
  }

  painter->setGeometry( 10+x, 35+y, w, h);
}



/*
 *********************************************************************
 **                                                                 **
 **    Printer                                                      **
 **                                                                 **
 *********************************************************************
 */
Printer::Printer(QWidget *tmp, const char* name, KStatusBar *sb)
  : QObject(tmp, name) {
  parent = tmp;
  statusBar = sb;
  config = (KApplication::getKApplication())->getConfig();
  printer = new QPrinter();
  restorePrinterSettings();
  settings = new PrinterSettings();
  connect(settings, SIGNAL(newParameter(QFont,int , int )), 
	  this, SLOT(parameterChanged(QFont, int , int )));
  preview = new PrinterPreview(0,0,this);
  connect(preview, SIGNAL(openSettings()), this, SLOT(openSettings()));
  connect(preview, SIGNAL(print()), this, SLOT(print()));
  connect(this,    SIGNAL(numberOfPages(int)), preview, SLOT(setPages(int)));

  font = settings->getFont();
  tab1 = settings->getTab1();
  tab2 = settings->getTab2();
  preview->refresh();
}

void Printer::openSettings(QSongList songList){
  songs = songList;
  openSettings();
}
void Printer::openSettings(){  settings->open(); }

void Printer::openPreview(QSongList songList){
  songs = songList;
  openPreview();
}
void Printer::openPreview(){  preview->open(); }

void Printer::print(QSongList songList){
  songs = songList;
  print();
}
void Printer::print(){
  if ( printer->setup(parent) ) {               // printer dialog
    if(statusBar != 0)
      statusBar->message( "Printing..." );
    savePrinterSettings();
    QPainter p;
    p.begin( printer );                     // paint on printer
    draw(&p,0);
    p.end();                                // send job to device
    statusBar->message( "Printing completed", 2000 );
  } else {
    statusBar->message( "Printing aborted", 2000 );
  }
}

QPrinter* Printer::getPrinter(){
  return printer;
}

void Printer::savePrinterSettings(){
  config->setGroup(KCONFIG_PRINTER);
  config->writeEntry(KCONFIG_PRINTER_ORIENTATION, (int) printer->orientation() );
  config->writeEntry(KCONFIG_PRINTER_PAGE, (int) printer->pageSize() );
}

void Printer::restorePrinterSettings(){
  config->setGroup(KCONFIG_PRINTER);
  printer->setOrientation((QPrinter::Orientation)config->readNumEntry(KCONFIG_PRINTER_ORIENTATION,  
					       QPrinter::Portrait ));
  printer->setPageSize((QPrinter::PageSize)config->readNumEntry(KCONFIG_PRINTER_PAGE, 
					    QPrinter::A4));
}


void Printer::getPageDimension(int &x, int &y){
  switch(printer->pageSize()){
  case QPrinter::A4        : x= 826; y=1170; break; 
  case QPrinter::B5        : x= 717; y=1015; break; 
  case QPrinter::Executive : x= 750; y=1000; break; 
  case QPrinter::Legal     : x= 850; y=1400; break; 
  case QPrinter::Letter    : x= 850; y=1100; break; 
  default: x= 826; y=1170;
  } 
  x *=  72;
  x /= 100;
  y *=  72;
  y /= 100;
  if(printer->orientation() == QPrinter::Landscape){
    int tmp = x;
    x = y;
    y = tmp;
  }
    
}

void Printer::parameterChanged(QFont newFont, int a1, int a2){
  font = newFont;
  tab1 = a1;
  tab2 = a2;
  preview->refresh();
}

void Printer::draw(QPainter *painter, int page=0){
  const int MARGIN = 10;
  int pageNo = 1;
  QString songDescription;
  Song *song = 0;
  QPainter p;
  bool printIt = false;
  int tabStops[] = { tab1, tab2};

  painter->setFont( font );
  painter->setTabArray(tabStops);
  int yPos        = 0;                    // y position for each line
  QFontMetrics fm = painter->fontMetrics();
  QPaintDeviceMetrics metrics( printer ); // need width/height
  // of printer surface
  if(page==0) printIt = true;
  if(page==pageNo) printIt=true;
  for(song=songs.first(); song!=NULL; song=songs.next()){
    if ( MARGIN + yPos > metrics.height() - MARGIN ) {
      pageNo++;
      printer->newPage();             // no more room on this page
      yPos = 0;                       // back to top of page
      if(pageNo==page) printIt=true;
      else{
	if(page==0) printIt=true;
	else printIt=false;
      }
    }
    if (printIt){
      songDescription.sprintf("%s\t%s\t%.2d:%0.2d", 
			      song->getArtist().data(), 
			      song->getTitle().data(), 
			      song->getSeconds()/60,
			      song->getSeconds()%60);
      painter->drawText( MARGIN, MARGIN + yPos,
			 metrics.width(), fm.lineSpacing(),
			 ExpandTabs | DontClip,
			 songDescription );
    }
    yPos = yPos + fm.lineSpacing();
  }
  emit numberOfPages(pageNo);
}







